"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Forecast = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [forecast](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonforecast.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Forecast extends shared_1.PolicyStatement {
    /**
     * Grants permission to create an auto predictor
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_CreateAutoPredictor.html
     */
    toCreateAutoPredictor() {
        return this.to('CreateAutoPredictor');
    }
    /**
     * Grants permission to create a dataset
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_CreateDataset.html
     */
    toCreateDataset() {
        return this.to('CreateDataset');
    }
    /**
     * Grants permission to create a dataset group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_CreateDatasetGroup.html
     */
    toCreateDatasetGroup() {
        return this.to('CreateDatasetGroup');
    }
    /**
     * Grants permission to create a dataset import job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_CreateDatasetImportJob.html
     */
    toCreateDatasetImportJob() {
        return this.to('CreateDatasetImportJob');
    }
    /**
     * Grants permission to create an explainability
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_CreateExplainability.html
     */
    toCreateExplainability() {
        return this.to('CreateExplainability');
    }
    /**
     * Grants permission to create an explainability export using an explainability resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_CreateExplainabilityExport.html
     */
    toCreateExplainabilityExport() {
        return this.to('CreateExplainabilityExport');
    }
    /**
     * Grants permission to create a forecast
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_CreateForecast.html
     */
    toCreateForecast() {
        return this.to('CreateForecast');
    }
    /**
     * Grants permission to create an endpoint using a Predictor resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/what-is-forecast.html
     */
    toCreateForecastEndpoint() {
        return this.to('CreateForecastEndpoint');
    }
    /**
     * Grants permission to create a forecast export job using a forecast resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_CreateForecastExportJob.html
     */
    toCreateForecastExportJob() {
        return this.to('CreateForecastExportJob');
    }
    /**
     * Grants permission to create an monitor using a Predictor resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_CreateMonitor.html
     */
    toCreateMonitor() {
        return this.to('CreateMonitor');
    }
    /**
     * Grants permission to create a predictor
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_CreatePredictor.html
     */
    toCreatePredictor() {
        return this.to('CreatePredictor');
    }
    /**
     * Grants permission to create a predictor backtest export job using a predictor
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_CreatePredictorBacktestExportJob.html
     */
    toCreatePredictorBacktestExportJob() {
        return this.to('CreatePredictorBacktestExportJob');
    }
    /**
     * Grants permission to create a what-if analysis
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_CreateWhatIfAnalysis.html
     */
    toCreateWhatIfAnalysis() {
        return this.to('CreateWhatIfAnalysis');
    }
    /**
     * Grants permission to create a what-if forecast
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_CreateWhatIfForecast.html
     */
    toCreateWhatIfForecast() {
        return this.to('CreateWhatIfForecast');
    }
    /**
     * Grants permission to create a what-if forecast export using what-if forecast resources
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_CreateWhatIfForecastExport.html
     */
    toCreateWhatIfForecastExport() {
        return this.to('CreateWhatIfForecastExport');
    }
    /**
     * Grants permission to delete a dataset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_DeleteDataset.html
     */
    toDeleteDataset() {
        return this.to('DeleteDataset');
    }
    /**
     * Grants permission to delete a dataset group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_DeleteDatasetGroup.html
     */
    toDeleteDatasetGroup() {
        return this.to('DeleteDatasetGroup');
    }
    /**
     * Grants permission to delete a dataset import job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_DeleteDatasetImportJob.html
     */
    toDeleteDatasetImportJob() {
        return this.to('DeleteDatasetImportJob');
    }
    /**
     * Grants permission to delete an explainability
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_DeleteExplainability.html
     */
    toDeleteExplainability() {
        return this.to('DeleteExplainability');
    }
    /**
     * Grants permission to delete an explainability export
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_DeleteExplainabilityExport.html
     */
    toDeleteExplainabilityExport() {
        return this.to('DeleteExplainabilityExport');
    }
    /**
     * Grants permission to delete a forecast
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_DeleteForecast.html
     */
    toDeleteForecast() {
        return this.to('DeleteForecast');
    }
    /**
     * Grants permission to delete an endpoint resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/what-is-forecast.html
     */
    toDeleteForecastEndpoint() {
        return this.to('DeleteForecastEndpoint');
    }
    /**
     * Grants permission to delete a forecast export job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_DeleteForecastExportJob.html
     */
    toDeleteForecastExportJob() {
        return this.to('DeleteForecastExportJob');
    }
    /**
     * Grants permission to delete a monitor resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_DeleteMonitor.html
     */
    toDeleteMonitor() {
        return this.to('DeleteMonitor');
    }
    /**
     * Grants permission to delete a predictor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_DeletePredictor.html
     */
    toDeletePredictor() {
        return this.to('DeletePredictor');
    }
    /**
     * Grants permission to delete a predictor backtest export job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_DeletePredictorBacktestExportJob.html
     */
    toDeletePredictorBacktestExportJob() {
        return this.to('DeletePredictorBacktestExportJob');
    }
    /**
     * Grants permission to delete a resource and its child resources
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_DeleteResourceTree.html
     */
    toDeleteResourceTree() {
        return this.to('DeleteResourceTree');
    }
    /**
     * Grants permission to delete a what-if analysis
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_DeleteWhatIfAnalysis.html
     */
    toDeleteWhatIfAnalysis() {
        return this.to('DeleteWhatIfAnalysis');
    }
    /**
     * Grants permission to delete a what-if forecast
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_DeleteWhatIfForecast.html
     */
    toDeleteWhatIfForecast() {
        return this.to('DeleteWhatIfForecast');
    }
    /**
     * Grants permission to delete a what-if forecast export
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_DeleteWhatIfForecastExport.html
     */
    toDeleteWhatIfForecastExport() {
        return this.to('DeleteWhatIfForecastExport');
    }
    /**
     * Grants permission to describe an auto predictor
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_DescribeAutoPredictor.html
     */
    toDescribeAutoPredictor() {
        return this.to('DescribeAutoPredictor');
    }
    /**
     * Grants permission to describe a dataset
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_DescribeDataset.html
     */
    toDescribeDataset() {
        return this.to('DescribeDataset');
    }
    /**
     * Grants permission to describe a dataset group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_DescribeDatasetGroup.html
     */
    toDescribeDatasetGroup() {
        return this.to('DescribeDatasetGroup');
    }
    /**
     * Grants permission to describe a dataset import job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_DescribeDatasetImportJob.html
     */
    toDescribeDatasetImportJob() {
        return this.to('DescribeDatasetImportJob');
    }
    /**
     * Grants permission to describe an explainability
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_DescribeExplainability.html
     */
    toDescribeExplainability() {
        return this.to('DescribeExplainability');
    }
    /**
     * Grants permission to describe an explainability export
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_DescribeExplainabilityExport.html
     */
    toDescribeExplainabilityExport() {
        return this.to('DescribeExplainabilityExport');
    }
    /**
     * Grants permission to describe a forecast
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_DescribeForecast.html
     */
    toDescribeForecast() {
        return this.to('DescribeForecast');
    }
    /**
     * Grants permission to describe an endpoint resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/what-is-forecast.html
     */
    toDescribeForecastEndpoint() {
        return this.to('DescribeForecastEndpoint');
    }
    /**
     * Grants permission to describe a forecast export job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_DescribeForecastExportJob.html
     */
    toDescribeForecastExportJob() {
        return this.to('DescribeForecastExportJob');
    }
    /**
     * Grants permission to describe an monitor resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_DescribeMonitor.html
     */
    toDescribeMonitor() {
        return this.to('DescribeMonitor');
    }
    /**
     * Grants permission to describe a predictor
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_DescribePredictor.html
     */
    toDescribePredictor() {
        return this.to('DescribePredictor');
    }
    /**
     * Grants permission to describe a predictor backtest export job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_DescribePredictorBacktestExportJob.html
     */
    toDescribePredictorBacktestExportJob() {
        return this.to('DescribePredictorBacktestExportJob');
    }
    /**
     * Grants permission to describe a what-if analysis
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_DescribeWhatIfAnalysis.html
     */
    toDescribeWhatIfAnalysis() {
        return this.to('DescribeWhatIfAnalysis');
    }
    /**
     * Grants permission to describe a what-if forecast
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_DescribeWhatIfForecast.html
     */
    toDescribeWhatIfForecast() {
        return this.to('DescribeWhatIfForecast');
    }
    /**
     * Grants permission to describe a what-if forecast export
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_DescribeWhatIfForecastExport.html
     */
    toDescribeWhatIfForecastExport() {
        return this.to('DescribeWhatIfForecastExport');
    }
    /**
     * Grants permission to get the Accuracy Metrics for a predictor
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_GetAccuracyMetrics.html
     */
    toGetAccuracyMetrics() {
        return this.to('GetAccuracyMetrics');
    }
    /**
     * Grants permission to get the forecast context of a timeseries for an endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/what-is-forecast.html
     */
    toGetRecentForecastContext() {
        return this.to('GetRecentForecastContext');
    }
    /**
     * Grants permission to invoke the endpoint to get forecast for a timeseries
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/what-is-forecast.html
     */
    toInvokeForecastEndpoint() {
        return this.to('InvokeForecastEndpoint');
    }
    /**
     * Grants permission to list all the dataset groups
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_ListDatasetGroups.html
     */
    toListDatasetGroups() {
        return this.to('ListDatasetGroups');
    }
    /**
     * Grants permission to list all the dataset import jobs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_ListDatasetImportJobs.html
     */
    toListDatasetImportJobs() {
        return this.to('ListDatasetImportJobs');
    }
    /**
     * Grants permission to list all the datasets
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_ListDatasets.html
     */
    toListDatasets() {
        return this.to('ListDatasets');
    }
    /**
     * Grants permission to list all the explainabilities
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_ListExplainabilities.html
     */
    toListExplainabilities() {
        return this.to('ListExplainabilities');
    }
    /**
     * Grants permission to list all the explainability exports
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_ListExplainabilityExports.html
     */
    toListExplainabilityExports() {
        return this.to('ListExplainabilityExports');
    }
    /**
     * Grants permission to list all the forecast export jobs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_ListForecastExportJobs.html
     */
    toListForecastExportJobs() {
        return this.to('ListForecastExportJobs');
    }
    /**
     * Grants permission to list all the forecasts
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_ListForecasts.html
     */
    toListForecasts() {
        return this.to('ListForecasts');
    }
    /**
     * Grants permission to list all the monitor evaluation result for a monitor
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_ListMonitorEvaluations.html
     */
    toListMonitorEvaluations() {
        return this.to('ListMonitorEvaluations');
    }
    /**
     * Grants permission to list all the monitor resources
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_ListMonitors.html
     */
    toListMonitors() {
        return this.to('ListMonitors');
    }
    /**
     * Grants permission to list all the predictor backtest export jobs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_ListPredictorBacktestExportJobs.html
     */
    toListPredictorBacktestExportJobs() {
        return this.to('ListPredictorBacktestExportJobs');
    }
    /**
     * Grants permission to list all the predictors
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_ListPredictors.html
     */
    toListPredictors() {
        return this.to('ListPredictors');
    }
    /**
     * Grants permission to list the tags for an Amazon Forecast resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list all the what-if analyses
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_ListWhatIfAnalyses.html
     */
    toListWhatIfAnalyses() {
        return this.to('ListWhatIfAnalyses');
    }
    /**
     * Grants permission to list all the what-if forecast exports
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_ListWhatIfForecastExports.html
     */
    toListWhatIfForecastExports() {
        return this.to('ListWhatIfForecastExports');
    }
    /**
     * Grants permission to list all the what-if forecasts
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_ListWhatIfForecasts.html
     */
    toListWhatIfForecasts() {
        return this.to('ListWhatIfForecasts');
    }
    /**
     * Grants permission to retrieve a forecast for a single item
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_forecastquery_QueryForecast.html
     */
    toQueryForecast() {
        return this.to('QueryForecast');
    }
    /**
     * Grants permission to retrieve a what-if forecast for a single item
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_forecastquery_QueryWhatIfForecast.html
     */
    toQueryWhatIfForecast() {
        return this.to('QueryWhatIfForecast');
    }
    /**
     * Grants permission to resume Amazon Forecast resource jobs
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_ResumeResource.html
     */
    toResumeResource() {
        return this.to('ResumeResource');
    }
    /**
     * Grants permission to stop Amazon Forecast resource jobs
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_StopResource.html
     */
    toStopResource() {
        return this.to('StopResource');
    }
    /**
     * Grants permission to associate the specified tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to delete the specified tags for a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a dataset group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_UpdateDatasetGroup.html
     */
    toUpdateDatasetGroup() {
        return this.to('UpdateDatasetGroup');
    }
    /**
     * Adds a resource of type dataset to the statement
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_CreateDataset.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDataset(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:forecast:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:dataset/${resourceId}`);
    }
    /**
     * Adds a resource of type datasetGroup to the statement
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_CreateDatasetGroup.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDatasetGroup(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:forecast:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:dataset-group/${resourceId}`);
    }
    /**
     * Adds a resource of type datasetImportJob to the statement
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_CreateDatasetImportJob.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDatasetImportJob(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:forecast:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:dataset-import-job/${resourceId}`);
    }
    /**
     * Adds a resource of type algorithm to the statement
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/aws-forecast-choosing-recipes.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAlgorithm(resourceId, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:forecast:::algorithm/${resourceId}`);
    }
    /**
     * Adds a resource of type predictor to the statement
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_CreatePredictor.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPredictor(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:forecast:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:predictor/${resourceId}`);
    }
    /**
     * Adds a resource of type predictorBacktestExportJob to the statement
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_CreatePredictorBacktestExportJob.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPredictorBacktestExportJob(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:forecast:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:predictor-backtest-export-job/${resourceId}`);
    }
    /**
     * Adds a resource of type forecast to the statement
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_CreateForecast.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onForecast(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:forecast:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:forecast/${resourceId}`);
    }
    /**
     * Adds a resource of type forecastExport to the statement
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_CreateForecastExportJob.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onForecastExport(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:forecast:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:forecast-export-job/${resourceId}`);
    }
    /**
     * Adds a resource of type explainability to the statement
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_CreateExplainability.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onExplainability(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:forecast:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:explainability/${resourceId}`);
    }
    /**
     * Adds a resource of type explainabilityExport to the statement
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_CreateExplainabilityExport.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onExplainabilityExport(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:forecast:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:explainability-export/${resourceId}`);
    }
    /**
     * Adds a resource of type monitor to the statement
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_CreateMonitor.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMonitor(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:forecast:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:monitor/${resourceId}`);
    }
    /**
     * Adds a resource of type whatIfAnalysis to the statement
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_CreateWhatIfAnalysis.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWhatIfAnalysis(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:forecast:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:what-if-analysis/${resourceId}`);
    }
    /**
     * Adds a resource of type whatIfForecast to the statement
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_CreateWhatIfForecast.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWhatIfForecast(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:forecast:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:what-if-forecast/${resourceId}`);
    }
    /**
     * Adds a resource of type whatIfForecastExport to the statement
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/API_CreateWhatIfForecastExport.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWhatIfForecastExport(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:forecast:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:what-if-forecast-export/${resourceId}`);
    }
    /**
     * Adds a resource of type endpoint to the statement
     *
     * https://docs.aws.amazon.com/forecast/latest/dg/what-is-forecast.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEndpoint(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:forecast:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:forecast-endpoint/${resourceId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateAutoPredictor()
     * - .toCreateDataset()
     * - .toCreateDatasetGroup()
     * - .toCreateDatasetImportJob()
     * - .toCreateExplainability()
     * - .toCreateExplainabilityExport()
     * - .toCreateForecast()
     * - .toCreateForecastEndpoint()
     * - .toCreateForecastExportJob()
     * - .toCreateMonitor()
     * - .toCreatePredictor()
     * - .toCreatePredictorBacktestExportJob()
     * - .toCreateWhatIfAnalysis()
     * - .toCreateWhatIfForecast()
     * - .toCreateWhatIfForecastExport()
     * - .toResumeResource()
     * - .toStopResource()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - dataset
     * - datasetGroup
     * - datasetImportJob
     * - predictor
     * - predictorBacktestExportJob
     * - forecast
     * - forecastExport
     * - explainability
     * - explainabilityExport
     * - monitor
     * - whatIfAnalysis
     * - whatIfForecast
     * - whatIfForecastExport
     * - endpoint
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateAutoPredictor()
     * - .toCreateDataset()
     * - .toCreateDatasetGroup()
     * - .toCreateDatasetImportJob()
     * - .toCreateExplainability()
     * - .toCreateExplainabilityExport()
     * - .toCreateForecast()
     * - .toCreateForecastEndpoint()
     * - .toCreateForecastExportJob()
     * - .toCreateMonitor()
     * - .toCreatePredictor()
     * - .toCreatePredictorBacktestExportJob()
     * - .toCreateWhatIfAnalysis()
     * - .toCreateWhatIfForecast()
     * - .toCreateWhatIfForecastExport()
     * - .toResumeResource()
     * - .toStopResource()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [forecast](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonforecast.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'forecast';
        this.accessLevelList = {
            Write: [
                'CreateAutoPredictor',
                'CreateDataset',
                'CreateDatasetGroup',
                'CreateDatasetImportJob',
                'CreateExplainability',
                'CreateExplainabilityExport',
                'CreateForecast',
                'CreateForecastEndpoint',
                'CreateForecastExportJob',
                'CreateMonitor',
                'CreatePredictor',
                'CreatePredictorBacktestExportJob',
                'CreateWhatIfAnalysis',
                'CreateWhatIfForecast',
                'CreateWhatIfForecastExport',
                'DeleteDataset',
                'DeleteDatasetGroup',
                'DeleteDatasetImportJob',
                'DeleteExplainability',
                'DeleteExplainabilityExport',
                'DeleteForecast',
                'DeleteForecastEndpoint',
                'DeleteForecastExportJob',
                'DeleteMonitor',
                'DeletePredictor',
                'DeletePredictorBacktestExportJob',
                'DeleteResourceTree',
                'DeleteWhatIfAnalysis',
                'DeleteWhatIfForecast',
                'DeleteWhatIfForecastExport',
                'ResumeResource',
                'StopResource',
                'UpdateDatasetGroup'
            ],
            Read: [
                'DescribeAutoPredictor',
                'DescribeDataset',
                'DescribeDatasetGroup',
                'DescribeDatasetImportJob',
                'DescribeExplainability',
                'DescribeExplainabilityExport',
                'DescribeForecast',
                'DescribeForecastEndpoint',
                'DescribeForecastExportJob',
                'DescribeMonitor',
                'DescribePredictor',
                'DescribePredictorBacktestExportJob',
                'DescribeWhatIfAnalysis',
                'DescribeWhatIfForecast',
                'DescribeWhatIfForecastExport',
                'GetAccuracyMetrics',
                'GetRecentForecastContext',
                'InvokeForecastEndpoint',
                'ListDatasetGroups',
                'ListDatasetImportJobs',
                'ListDatasets',
                'ListExplainabilities',
                'ListExplainabilityExports',
                'ListForecastExportJobs',
                'ListForecasts',
                'ListMonitorEvaluations',
                'ListMonitors',
                'ListPredictorBacktestExportJobs',
                'ListPredictors',
                'ListTagsForResource',
                'ListWhatIfAnalyses',
                'ListWhatIfForecastExports',
                'ListWhatIfForecasts',
                'QueryForecast',
                'QueryWhatIfForecast'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Forecast = Forecast;
//# sourceMappingURL=data:application/json;base64,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