"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Fsx = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [fsx](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonfsx.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Fsx extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate a File Gateway instance with an Amazon FSx for Windows File Server file system
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/filegateway/latest/filefsxw/what-is-file-fsxw.html
     */
    toAssociateFileGateway() {
        return this.to('AssociateFileGateway');
    }
    /**
     * Grants permission to associate DNS aliases with an Amazon FSx for Windows File Server file system
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_AssociateFileSystemAliases.html
     */
    toAssociateFileSystemAliases() {
        return this.to('AssociateFileSystemAliases');
    }
    /**
     * Grants permission to allow deletion of an FSx for ONTAP SnapLock Enterprise volume that contains WORM (write once, read many) files with active retention periods
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/snaplock-enterprise.html#bypass-enterprise
     */
    toBypassSnaplockEnterpriseRetention() {
        return this.to('BypassSnaplockEnterpriseRetention');
    }
    /**
     * Grants permission to cancel a data repository task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_CancelDataRepositoryTask.html
     */
    toCancelDataRepositoryTask() {
        return this.to('CancelDataRepositoryTask');
    }
    /**
     * Grants permission to copy a backup
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - fsx:TagResource
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_CopyBackup.html
     */
    toCopyBackup() {
        return this.to('CopyBackup');
    }
    /**
     * Grants permission to update an existing volume by using a snapshot from another Amazon FSx for OpenZFS file system
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_CopySnapshotAndUpdateVolume.html
     */
    toCopySnapshotAndUpdateVolume() {
        return this.to('CopySnapshotAndUpdateVolume');
    }
    /**
     * Grants permission to create a new backup of an Amazon FSx file system or an Amazon FSx volume
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - fsx:TagResource
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_CreateBackup.html
     */
    toCreateBackup() {
        return this.to('CreateBackup');
    }
    /**
     * Grants permission to create a new data respository association for an Amazon FSx for Lustre file system
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - fsx:TagResource
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_CreateDataRepositoryAssociation.html
     */
    toCreateDataRepositoryAssociation() {
        return this.to('CreateDataRepositoryAssociation');
    }
    /**
     * Grants permission to create a new data respository task for an Amazon FSx for Lustre file system
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - fsx:TagResource
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_CreateDataRepositoryTask.html
     */
    toCreateDataRepositoryTask() {
        return this.to('CreateDataRepositoryTask');
    }
    /**
     * Grants permission to create a new, empty, Amazon file cache
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - ec2:GetSecurityGroupsForVpc
     * - fsx:CreateDataRepositoryAssociation
     * - fsx:TagResource
     * - logs:CreateLogGroup
     * - logs:CreateLogStream
     * - logs:PutLogEvents
     * - s3:ListBucket
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_CreateFileCache.html
     */
    toCreateFileCache() {
        return this.to('CreateFileCache');
    }
    /**
     * Grants permission to create a new, empty, Amazon FSx file system
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:GetSecurityGroupsForVpc
     * - fsx:TagResource
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_CreateFileSystem.html
     */
    toCreateFileSystem() {
        return this.to('CreateFileSystem');
    }
    /**
     * Grants permission to create a new Amazon FSx file system from an existing backup
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:GetSecurityGroupsForVpc
     * - fsx:TagResource
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_CreateFileSystemFromBackup.html
     */
    toCreateFileSystemFromBackup() {
        return this.to('CreateFileSystemFromBackup');
    }
    /**
     * Grants permission to create a new snapshot on a volume
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - fsx:TagResource
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_CreateSnapshot.html
     */
    toCreateSnapshot() {
        return this.to('CreateSnapshot');
    }
    /**
     * Grants permission to create a new storage virtual machine in an Amazon FSx for Ontap file system
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - fsx:TagResource
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_CreateStorageVirtualMachine.html
     */
    toCreateStorageVirtualMachine() {
        return this.to('CreateStorageVirtualMachine');
    }
    /**
     * Grants permission to create a new volume
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifStorageVirtualMachineId()
     * - .ifParentVolumeId()
     *
     * Dependent actions:
     * - fsx:TagResource
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_CreateVolume.html
     */
    toCreateVolume() {
        return this.to('CreateVolume');
    }
    /**
     * Grants permission to create a new volume from backup
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifStorageVirtualMachineId()
     *
     * Dependent actions:
     * - fsx:TagResource
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_CreateVolumeFromBackup.html
     */
    toCreateVolumeFromBackup() {
        return this.to('CreateVolumeFromBackup');
    }
    /**
     * Grants permission to delete a backup, deleting its contents. After deletion, the backup no longer exists, and its data is no longer available
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DeleteBackup.html
     */
    toDeleteBackup() {
        return this.to('DeleteBackup');
    }
    /**
     * Grants permission to delete a data repository association
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DeleteDataRepositoryAssociation.html
     */
    toDeleteDataRepositoryAssociation() {
        return this.to('DeleteDataRepositoryAssociation');
    }
    /**
     * Grants permission to delete a file cache, deleting its contents
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - fsx:DeleteDataRepositoryAssociation
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DeleteFileCache.html
     */
    toDeleteFileCache() {
        return this.to('DeleteFileCache');
    }
    /**
     * Grants permission to delete a file system, deleting its contents and any existing automatic backups of the file system
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - fsx:CreateBackup
     * - fsx:TagResource
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DeleteFileSystem.html
     */
    toDeleteFileSystem() {
        return this.to('DeleteFileSystem');
    }
    /**
     * Required to manage cross-account sharing of FSx volumes through AWS Resource Access Manager (RAM). PutResourcePolicy and GetResourcePolicy are also required
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/on-demand-replication.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to delete a snapshot on a volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DeleteSnapshot.html
     */
    toDeleteSnapshot() {
        return this.to('DeleteSnapshot');
    }
    /**
     * Grants permission to delete a storage virtual machine, deleting its contents
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DeleteStorageVirtualMachine.html
     */
    toDeleteStorageVirtualMachine() {
        return this.to('DeleteStorageVirtualMachine');
    }
    /**
     * Grants permission to delete a volume, deleting its contents and any existing automatic backups of the volume
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifStorageVirtualMachineId()
     * - .ifParentVolumeId()
     *
     * Dependent actions:
     * - fsx:TagResource
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DeleteVolume.html
     */
    toDeleteVolume() {
        return this.to('DeleteVolume');
    }
    /**
     * Grants permission to describe the File Gateway instances associated with an Amazon FSx for Windows File Server file system
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/filegateway/latest/filefsxw/what-is-file-fsxw.html
     */
    toDescribeAssociatedFileGateways() {
        return this.to('DescribeAssociatedFileGateways');
    }
    /**
     * Grants permission to return the descriptions of all backups owned by your AWS account in the AWS Region of the endpoint that you're calling
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeBackups.html
     */
    toDescribeBackups() {
        return this.to('DescribeBackups');
    }
    /**
     * Grants permission to return the descriptions of all data repository associations owned by your AWS account in the AWS Region of the endpoint that you're calling
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeDataRepositoryAssociations.html
     */
    toDescribeDataRepositoryAssociations() {
        return this.to('DescribeDataRepositoryAssociations');
    }
    /**
     * Grants permission to return the descriptions of all data repository tasks owned by your AWS account in the AWS Region of the endpoint that you're calling
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeDataRepositoryTasks.html
     */
    toDescribeDataRepositoryTasks() {
        return this.to('DescribeDataRepositoryTasks');
    }
    /**
     * Grants permission to return the descriptions of all file caches owned by your AWS account in the AWS Region of the endpoint that you're calling
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeFileCaches.html
     */
    toDescribeFileCaches() {
        return this.to('DescribeFileCaches');
    }
    /**
     * Grants permission to return the description of all DNS aliases owned by your Amazon FSx for Windows File Server file system
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeFileSystemAliases.html
     */
    toDescribeFileSystemAliases() {
        return this.to('DescribeFileSystemAliases');
    }
    /**
     * Grants permission to return the descriptions of all file systems owned by your AWS account in the AWS Region of the endpoint that you're calling
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeFileSystems.html
     */
    toDescribeFileSystems() {
        return this.to('DescribeFileSystems');
    }
    /**
     * Grants permission to return the descriptions of whether FSx route table updates from participant accounts are allowed in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeSharedVpcConfiguration.html
     */
    toDescribeSharedVpcConfiguration() {
        return this.to('DescribeSharedVpcConfiguration');
    }
    /**
     * Grants permission to return the descriptions of all snapshots owned by your AWS account in the AWS Region of the endpoint you're calling
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeSnapshots.html
     */
    toDescribeSnapshots() {
        return this.to('DescribeSnapshots');
    }
    /**
     * Grants permission to return the descriptions of all storage virtual machines owned by your AWS account in the AWS Region of the endpoint that you're calling
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeStorageVirtualMachines.html
     */
    toDescribeStorageVirtualMachines() {
        return this.to('DescribeStorageVirtualMachines');
    }
    /**
     * Grants permission to return the descriptions of all volumes owned by your AWS account in the AWS Region of the endpoint that you're calling
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeVolumes.html
     */
    toDescribeVolumes() {
        return this.to('DescribeVolumes');
    }
    /**
     * Grants permission to disassociate a File Gateway instance from an Amazon FSx for Windows File Server file system
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/filegateway/latest/filefsxw/what-is-file-fsxw.html
     */
    toDisassociateFileGateway() {
        return this.to('DisassociateFileGateway');
    }
    /**
     * Grants permission to disassociate file system aliases with an Amazon FSx for Windows File Server file system
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DisassociateFileSystemAliases.html
     */
    toDisassociateFileSystemAliases() {
        return this.to('DisassociateFileSystemAliases');
    }
    /**
     * Required to manage cross-account sharing of FSx volumes through AWS Resource Access Manager (RAM). PutResourcePolicy and DeleteResourcePolicy are also required
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/on-demand-replication.html
     */
    toGetResourcePolicy() {
        return this.to('GetResourcePolicy');
    }
    /**
     * Grants permission to list tags for an Amazon FSx resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to manage backup principal associations through AWS Backup
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_CopyBackup.html
     */
    toManageBackupPrincipalAssociations() {
        return this.to('ManageBackupPrincipalAssociations');
    }
    /**
     * Required to manage cross-account sharing of FSx volumes through AWS Resource Access Manager (RAM). DeleteResourcePolicy and GetResourcePolicy are also required
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/on-demand-replication.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to release file system NFS V3 locks
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_ReleaseFileSystemNfsV3Locks.html
     */
    toReleaseFileSystemNfsV3Locks() {
        return this.to('ReleaseFileSystemNfsV3Locks');
    }
    /**
     * Grants permission to restore volume state from a snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_RestoreVolumeFromSnapshot.html
     */
    toRestoreVolumeFromSnapshot() {
        return this.to('RestoreVolumeFromSnapshot');
    }
    /**
     * Grants permission to start misconfigured state recovery
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_StartMisconfiguredStateRecovery.html
     */
    toStartMisconfiguredStateRecovery() {
        return this.to('StartMisconfiguredStateRecovery');
    }
    /**
     * Grants permission to tag an Amazon FSx resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove a tag from an Amazon FSx resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update data repository association configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_UpdateDataRepositoryAssociation.html
     */
    toUpdateDataRepositoryAssociation() {
        return this.to('UpdateDataRepositoryAssociation');
    }
    /**
     * Grants permission to update file cache configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_UpdateFileCache.html
     */
    toUpdateFileCache() {
        return this.to('UpdateFileCache');
    }
    /**
     * Grants permission to update file system configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_UpdateFileSystem.html
     */
    toUpdateFileSystem() {
        return this.to('UpdateFileSystem');
    }
    /**
     * Grants permission to enable or disable FSx route table updates from participant accounts in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_UpdateSharedVpcConfiguration.html
     */
    toUpdateSharedVpcConfiguration() {
        return this.to('UpdateSharedVpcConfiguration');
    }
    /**
     * Grants permission to update snapshot configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_UpdateSnapshot.html
     */
    toUpdateSnapshot() {
        return this.to('UpdateSnapshot');
    }
    /**
     * Grants permission to update storage virtual machine configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_UpdateStorageVirtualMachine.html
     */
    toUpdateStorageVirtualMachine() {
        return this.to('UpdateStorageVirtualMachine');
    }
    /**
     * Grants permission to update volume configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifStorageVirtualMachineId()
     * - .ifParentVolumeId()
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_UpdateVolume.html
     */
    toUpdateVolume() {
        return this.to('UpdateVolume');
    }
    /**
     * Adds a resource of type file-system to the statement
     *
     * https://docs.aws.amazon.com/fsx/latest/WindowsGuide/access-control-overview.html#access-control-resources
     *
     * @param fileSystemId - Identifier for the fileSystemId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFileSystem(fileSystemId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:fsx:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:file-system/${fileSystemId}`);
    }
    /**
     * Adds a resource of type file-cache to the statement
     *
     * https://docs.aws.amazon.com/fsx/latest/FileCacheGuide/security-iam.html
     *
     * @param fileCacheId - Identifier for the fileCacheId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFileCache(fileCacheId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:fsx:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:file-cache/${fileCacheId}`);
    }
    /**
     * Adds a resource of type backup to the statement
     *
     * https://docs.aws.amazon.com/fsx/latest/WindowsGuide/access-control-overview.html#access-control-resources
     *
     * @param backupId - Identifier for the backupId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBackup(backupId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:fsx:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:backup/${backupId}`);
    }
    /**
     * Adds a resource of type storage-virtual-machine to the statement
     *
     * https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/security-iam.html
     *
     * @param fileSystemId - Identifier for the fileSystemId.
     * @param storageVirtualMachineId - Identifier for the storageVirtualMachineId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStorageVirtualMachine(fileSystemId, storageVirtualMachineId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:fsx:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:storage-virtual-machine/${fileSystemId}/${storageVirtualMachineId}`);
    }
    /**
     * Adds a resource of type task to the statement
     *
     * https://docs.aws.amazon.com/fsx/latest/LustreGuide/access-control-overview.html#access-control-resources
     *
     * @param taskId - Identifier for the taskId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTask(taskId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:fsx:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:task/${taskId}`);
    }
    /**
     * Adds a resource of type association to the statement
     *
     * https://docs.aws.amazon.com/fsx/latest/LustreGuide/access-control-overview.html#access-control-resources
     *
     * @param fileSystemIdOrFileCacheId - Identifier for the fileSystemIdOrFileCacheId.
     * @param dataRepositoryAssociationId - Identifier for the dataRepositoryAssociationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAssociation(fileSystemIdOrFileCacheId, dataRepositoryAssociationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:fsx:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:association/${fileSystemIdOrFileCacheId}/${dataRepositoryAssociationId}`);
    }
    /**
     * Adds a resource of type volume to the statement
     *
     * https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/security-iam.html
     *
     * @param fileSystemId - Identifier for the fileSystemId.
     * @param volumeId - Identifier for the volumeId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVolume(fileSystemId, volumeId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:fsx:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:volume/${fileSystemId}/${volumeId}`);
    }
    /**
     * Adds a resource of type snapshot to the statement
     *
     * https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/access-control-overview.html#access-control-resources
     *
     * @param volumeId - Identifier for the volumeId.
     * @param snapshotId - Identifier for the snapshotId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSnapshot(volumeId, snapshotId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:fsx:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:snapshot/${volumeId}/${snapshotId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCopyBackup()
     * - .toCreateBackup()
     * - .toCreateDataRepositoryAssociation()
     * - .toCreateDataRepositoryTask()
     * - .toCreateFileCache()
     * - .toCreateFileSystem()
     * - .toCreateFileSystemFromBackup()
     * - .toCreateSnapshot()
     * - .toCreateStorageVirtualMachine()
     * - .toCreateVolume()
     * - .toCreateVolumeFromBackup()
     * - .toDeleteFileCache()
     * - .toDeleteFileSystem()
     * - .toDeleteVolume()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - file-system
     * - file-cache
     * - backup
     * - storage-virtual-machine
     * - task
     * - association
     * - volume
     * - snapshot
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCopyBackup()
     * - .toCreateBackup()
     * - .toCreateDataRepositoryAssociation()
     * - .toCreateDataRepositoryTask()
     * - .toCreateFileCache()
     * - .toCreateFileSystem()
     * - .toCreateFileSystemFromBackup()
     * - .toCreateSnapshot()
     * - .toCreateStorageVirtualMachine()
     * - .toCreateVolume()
     * - .toCreateVolumeFromBackup()
     * - .toDeleteFileCache()
     * - .toDeleteFileSystem()
     * - .toDeleteVolume()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by whether the backup is a destination backup for a CopyBackup operation
     *
     * https://docs.aws.amazon.com/fsx/latest/WindowsGuide/using-backups.html#copy-backups
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifIsBackupCopyDestination(value) {
        return this.if(`IsBackupCopyDestination`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by whether the backup is a source backup for a CopyBackup operation
     *
     * https://docs.aws.amazon.com/fsx/latest/WindowsGuide/using-backups.html#copy-backups
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifIsBackupCopySource(value) {
        return this.if(`IsBackupCopySource`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by NFS data repositories which support authentication
     *
     * https://docs.aws.amazon.com/fsx/latest/FileCacheGuide/encryption-in-transit.html
     *
     * Applies to actions:
     * - .toCreateFileCache()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifNfsDataRepositoryAuthenticationEnabled(value) {
        return this.if(`NfsDataRepositoryAuthenticationEnabled`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by NFS data repositories which support encryption-in-transit
     *
     * https://docs.aws.amazon.com/fsx/latest/FileCacheGuide/encryption-in-transit.html
     *
     * Applies to actions:
     * - .toCreateFileCache()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifNfsDataRepositoryEncryptionInTransitEnabled(value) {
        return this.if(`NfsDataRepositoryEncryptionInTransitEnabled`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the containing parent volume for mutating volume operations
     *
     * https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/creating-volumes.html
     *
     * Applies to actions:
     * - .toCreateVolume()
     * - .toDeleteVolume()
     * - .toUpdateVolume()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifParentVolumeId(value, operator) {
        return this.if(`ParentVolumeId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the containing storage virtual machine for a volume for mutating volume operations
     *
     * https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/creating-volumes.html
     *
     * Applies to actions:
     * - .toCreateVolume()
     * - .toCreateVolumeFromBackup()
     * - .toDeleteVolume()
     * - .toUpdateVolume()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifStorageVirtualMachineId(value, operator) {
        return this.if(`StorageVirtualMachineId`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [fsx](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonfsx.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'fsx';
        this.accessLevelList = {
            Write: [
                'AssociateFileGateway',
                'AssociateFileSystemAliases',
                'CancelDataRepositoryTask',
                'CopyBackup',
                'CopySnapshotAndUpdateVolume',
                'CreateBackup',
                'CreateDataRepositoryAssociation',
                'CreateDataRepositoryTask',
                'CreateFileCache',
                'CreateFileSystem',
                'CreateFileSystemFromBackup',
                'CreateSnapshot',
                'CreateStorageVirtualMachine',
                'CreateVolume',
                'CreateVolumeFromBackup',
                'DeleteBackup',
                'DeleteDataRepositoryAssociation',
                'DeleteFileCache',
                'DeleteFileSystem',
                'DeleteSnapshot',
                'DeleteStorageVirtualMachine',
                'DeleteVolume',
                'DisassociateFileGateway',
                'DisassociateFileSystemAliases',
                'ReleaseFileSystemNfsV3Locks',
                'RestoreVolumeFromSnapshot',
                'StartMisconfiguredStateRecovery',
                'UpdateDataRepositoryAssociation',
                'UpdateFileCache',
                'UpdateFileSystem',
                'UpdateSharedVpcConfiguration',
                'UpdateSnapshot',
                'UpdateStorageVirtualMachine',
                'UpdateVolume'
            ],
            'Permissions management': [
                'BypassSnaplockEnterpriseRetention',
                'DeleteResourcePolicy',
                'GetResourcePolicy',
                'ManageBackupPrincipalAssociations',
                'PutResourcePolicy'
            ],
            Read: [
                'DescribeAssociatedFileGateways',
                'DescribeBackups',
                'DescribeDataRepositoryAssociations',
                'DescribeDataRepositoryTasks',
                'DescribeFileCaches',
                'DescribeFileSystemAliases',
                'DescribeFileSystems',
                'DescribeSharedVpcConfiguration',
                'DescribeSnapshots',
                'DescribeStorageVirtualMachines',
                'DescribeVolumes',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Fsx = Fsx;
//# sourceMappingURL=data:application/json;base64,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