"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Honeycode = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [honeycode](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonhoneycode.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Honeycode extends shared_1.PolicyStatement {
    /**
     * Grants permission to approve a team association request for your AWS Account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/team-association.html#approve-team-association
     */
    toApproveTeamAssociation() {
        return this.to('ApproveTeamAssociation');
    }
    /**
     * Grants permission to create new rows in a table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/API_BatchCreateTableRows.html
     */
    toBatchCreateTableRows() {
        return this.to('BatchCreateTableRows');
    }
    /**
     * Grants permission to delete rows from a table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/API_BatchDeleteTableRows.html
     */
    toBatchDeleteTableRows() {
        return this.to('BatchDeleteTableRows');
    }
    /**
     * Grants permission to update rows in a table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/API_BatchUpdateTableRows.html
     */
    toBatchUpdateTableRows() {
        return this.to('BatchUpdateTableRows');
    }
    /**
     * Grants permission to upsert rows in a table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/API_BatchUpsertTableRows.html
     */
    toBatchUpsertTableRows() {
        return this.to('BatchUpsertTableRows');
    }
    /**
     * Grants permission to create a new Amazon Honeycode team for your AWS Account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/team.html#create-team
     */
    toCreateTeam() {
        return this.to('CreateTeam');
    }
    /**
     * Grants permission to create a new tenant within Amazon Honeycode for your AWS Account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/tenant.html#create-tenant
     */
    toCreateTenant() {
        return this.to('CreateTenant');
    }
    /**
     * Grants permission to delete Amazon Honeycode domains for your AWS Account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/domain.html#delete-domains
     */
    toDeleteDomains() {
        return this.to('DeleteDomains');
    }
    /**
     * Grants permission to remove groups from an Amazon Honeycode team for your AWS Account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/group.html#deregister-groups
     */
    toDeregisterGroups() {
        return this.to('DeregisterGroups');
    }
    /**
     * Grants permission to get details about a table data import job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/API_DescribeTableDataImportJob.html
     */
    toDescribeTableDataImportJob() {
        return this.to('DescribeTableDataImportJob');
    }
    /**
     * Grants permission to get details about Amazon Honeycode teams for your AWS Account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/team.html#describe-team
     */
    toDescribeTeam() {
        return this.to('DescribeTeam');
    }
    /**
     * Grants permission to load the data from a screen
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/API_GetScreenData.html
     */
    toGetScreenData() {
        return this.to('GetScreenData');
    }
    /**
     * Grants permission to invoke a screen automation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/API_InvokeScreenAutomation.html
     */
    toInvokeScreenAutomation() {
        return this.to('InvokeScreenAutomation');
    }
    /**
     * Grants permission to list all Amazon Honeycode domains and their verification status for your AWS Account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/domain.html#list-domains
     */
    toListDomains() {
        return this.to('ListDomains');
    }
    /**
     * Grants permission to list all groups in an Amazon Honeycode team for your AWS Account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/group.html#list-groups
     */
    toListGroups() {
        return this.to('ListGroups');
    }
    /**
     * Grants permission to list the columns in a table
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/API_ListTableColumns.html
     */
    toListTableColumns() {
        return this.to('ListTableColumns');
    }
    /**
     * Grants permission to list the rows in a table
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/API_ListTableRows.html
     */
    toListTableRows() {
        return this.to('ListTableRows');
    }
    /**
     * Grants permission to list the tables in a workbook
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/API_ListTables.html
     */
    toListTables() {
        return this.to('ListTables');
    }
    /**
     * Grants permission to list all tags for a resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list all pending and approved team associations with your AWS Account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/team-association.html#list-team-associations
     */
    toListTeamAssociations() {
        return this.to('ListTeamAssociations');
    }
    /**
     * Grants permission to list all tenants of Amazon Honeycode for your AWS Account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/tenant.html#list-tenants
     */
    toListTenants() {
        return this.to('ListTenants');
    }
    /**
     * Grants permission to query the rows of a table using a filter
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/API_QueryTableRows.html
     */
    toQueryTableRows() {
        return this.to('QueryTableRows');
    }
    /**
     * Grants permission to request verification of the Amazon Honeycode domains for your AWS Account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/domain.html#register-domain-for-verification
     */
    toRegisterDomainForVerification() {
        return this.to('RegisterDomainForVerification');
    }
    /**
     * Grants permission to add groups to an Amazon Honeycode team for your AWS Account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/group.html#register-groups
     */
    toRegisterGroups() {
        return this.to('RegisterGroups');
    }
    /**
     * Grants permission to reject a team association request for your AWS Account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/team-association.html#reject-team-association
     */
    toRejectTeamAssociation() {
        return this.to('RejectTeamAssociation');
    }
    /**
     * Grants permission to restart verification of the Amazon Honeycode domains for your AWS Account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/domain.html#restart-domain-verification
     */
    toRestartDomainVerification() {
        return this.to('RestartDomainVerification');
    }
    /**
     * Grants permission to start a table data import job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/API_StartTableDataImportJob.html
     */
    toStartTableDataImportJob() {
        return this.to('StartTableDataImportJob');
    }
    /**
     * Grants permission to tag a resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an Amazon Honeycode team for your AWS Account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/team.html#update-team
     */
    toUpdateTeam() {
        return this.to('UpdateTeam');
    }
    /**
     * Adds a resource of type workbook to the statement
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/resource-workbook.html
     *
     * @param workbookId - Identifier for the workbookId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onWorkbook(workbookId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:honeycode:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workbook:workbook/${workbookId}`);
    }
    /**
     * Adds a resource of type table to the statement
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/resource-table.html
     *
     * @param workbookId - Identifier for the workbookId.
     * @param tableId - Identifier for the tableId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onTable(workbookId, tableId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:honeycode:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:table:workbook/${workbookId}/table/${tableId}`);
    }
    /**
     * Adds a resource of type screen to the statement
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/resource-screen.html
     *
     * @param workbookId - Identifier for the workbookId.
     * @param appId - Identifier for the appId.
     * @param screenId - Identifier for the screenId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onScreen(workbookId, appId, screenId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:honeycode:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:screen:workbook/${workbookId}/app/${appId}/screen/${screenId}`);
    }
    /**
     * Adds a resource of type screen-automation to the statement
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/resource-screen-automation.html
     *
     * @param workbookId - Identifier for the workbookId.
     * @param appId - Identifier for the appId.
     * @param screenId - Identifier for the screenId.
     * @param automationId - Identifier for the automationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onScreenAutomation(workbookId, appId, screenId, automationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:honeycode:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:screen-automation:workbook/${workbookId}/app/${appId}/screen/${screenId}/automation/${automationId}`);
    }
    /**
     * Statement provider for service [honeycode](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonhoneycode.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'honeycode';
        this.accessLevelList = {
            Write: [
                'ApproveTeamAssociation',
                'BatchCreateTableRows',
                'BatchDeleteTableRows',
                'BatchUpdateTableRows',
                'BatchUpsertTableRows',
                'CreateTeam',
                'CreateTenant',
                'DeleteDomains',
                'DeregisterGroups',
                'InvokeScreenAutomation',
                'RegisterDomainForVerification',
                'RegisterGroups',
                'RejectTeamAssociation',
                'RestartDomainVerification',
                'StartTableDataImportJob',
                'UpdateTeam'
            ],
            Read: [
                'DescribeTableDataImportJob',
                'DescribeTeam',
                'GetScreenData',
                'QueryTableRows'
            ],
            List: [
                'ListDomains',
                'ListGroups',
                'ListTableColumns',
                'ListTableRows',
                'ListTables',
                'ListTeamAssociations',
                'ListTenants'
            ],
            Tagging: [
                'ListTagsForResource',
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Honeycode = Honeycode;
//# sourceMappingURL=data:application/json;base64,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