"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Inspector2 = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [inspector2](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoninspector2.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Inspector2 extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate an account with an Amazon Inspector administrator account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_AssociateMember.html
     */
    toAssociateMember() {
        return this.to('AssociateMember');
    }
    /**
     * Grants permission to retrieve information about Amazon Inspector accounts for an account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_BatchGetAccountStatus.html
     */
    toBatchGetAccountStatus() {
        return this.to('BatchGetAccountStatus');
    }
    /**
     * Grants permission to retrieve code snippet information about one or more code vulnerability findings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_BatchGetCodeSnippet.html
     */
    toBatchGetCodeSnippet() {
        return this.to('BatchGetCodeSnippet');
    }
    /**
     * Grants permission to let a customer get enhanced vulnerability intelligence details for findings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_BatchGetFindingDetails.html
     */
    toBatchGetFindingDetails() {
        return this.to('BatchGetFindingDetails');
    }
    /**
     * Grants permission to retrieve free trial period eligibility about Amazon Inspector accounts for an account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_BatchGetFreeTrialInfo.html
     */
    toBatchGetFreeTrialInfo() {
        return this.to('BatchGetFreeTrialInfo');
    }
    /**
     * Grants permission to delegated administrator to retrieve ec2 deep inspection status of member accounts
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_BatchGetMemberEc2DeepInspectionStatus.html
     */
    toBatchGetMemberEc2DeepInspectionStatus() {
        return this.to('BatchGetMemberEc2DeepInspectionStatus');
    }
    /**
     * Grants permission to update ec2 deep inspection status by delegated administrator for its associated member accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_BatchUpdateMemberEc2DeepInspectionStatus.html
     */
    toBatchUpdateMemberEc2DeepInspectionStatus() {
        return this.to('BatchUpdateMemberEc2DeepInspectionStatus');
    }
    /**
     * Grants permission to cancel the generation of a findings report
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_CancelFindingsReport.html
     */
    toCancelFindingsReport() {
        return this.to('CancelFindingsReport');
    }
    /**
     * Grants permission to cancel the generation of an SBOM report
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_CancelSbomExport.html
     */
    toCancelSbomExport() {
        return this.to('CancelSbomExport');
    }
    /**
     * Grants permission to create and define the settings for a CIS scan configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_CreateCisScanConfiguration.html
     */
    toCreateCisScanConfiguration() {
        return this.to('CreateCisScanConfiguration');
    }
    /**
     * Grants permission to create and define the settings for a findings filter
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_CreateFilter.html
     */
    toCreateFilter() {
        return this.to('CreateFilter');
    }
    /**
     * Grants permission to request the generation of a findings report
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_CreateFindingsReport.html
     */
    toCreateFindingsReport() {
        return this.to('CreateFindingsReport');
    }
    /**
     * Grants permission to request the generation of an SBOM report
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_CreateSbomExport.html
     */
    toCreateSbomExport() {
        return this.to('CreateSbomExport');
    }
    /**
     * Grants permission to delete a CIS scan configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_DeleteCisScanConfiguration.html
     */
    toDeleteCisScanConfiguration() {
        return this.to('DeleteCisScanConfiguration');
    }
    /**
     * Grants permission to delete a findings filter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_DeleteFilter.html
     */
    toDeleteFilter() {
        return this.to('DeleteFilter');
    }
    /**
     * Grants permission to retrieve information about the Amazon Inspector configuration settings for an AWS organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_DescribeOrganizationConfiguration.html
     */
    toDescribeOrganizationConfiguration() {
        return this.to('DescribeOrganizationConfiguration');
    }
    /**
     * Grants permission to disable an Amazon Inspector account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_Disable.html
     */
    toDisable() {
        return this.to('Disable');
    }
    /**
     * Grants permission to disable an account as the delegated Amazon Inspector administrator account for an AWS organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_DisableDelegatedAdminAccount.html
     */
    toDisableDelegatedAdminAccount() {
        return this.to('DisableDelegatedAdminAccount');
    }
    /**
     * Grants permission to an Amazon Inspector administrator account to disassociate from an Inspector member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_DisassociateMember.html
     */
    toDisassociateMember() {
        return this.to('DisassociateMember');
    }
    /**
     * Grants permission to enable and specify the configuration settings for a new Amazon Inspector account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_Enable.html
     */
    toEnable() {
        return this.to('Enable');
    }
    /**
     * Grants permission to enable an account as the delegated Amazon Inspector administrator account for an AWS organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_EnableDelegatedAdminAccount.html
     */
    toEnableDelegatedAdminAccount() {
        return this.to('EnableDelegatedAdminAccount');
    }
    /**
     * Grants permission to retrieve a report containing information about completed CIS scans
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_GetCisScanReport.html
     */
    toGetCisScanReport() {
        return this.to('GetCisScanReport');
    }
    /**
     * Grants permission to retrieve information about all details pertaining to one CIS scan and one targeted resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_GetCisScanResultDetails.html
     */
    toGetCisScanResultDetails() {
        return this.to('GetCisScanResultDetails');
    }
    /**
     * Grants permission to retrieve information about the Amazon Inspector configuration settings for an AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_GetConfiguration.html
     */
    toGetConfiguration() {
        return this.to('GetConfiguration');
    }
    /**
     * Grants permission to retrieve information about the Amazon Inspector administrator account for an account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_GetDelegatedAdminAccount.html
     */
    toGetDelegatedAdminAccount() {
        return this.to('GetDelegatedAdminAccount');
    }
    /**
     * Grants permission to retrieve ec2 deep inspection configuration for standalone accounts, delegated administrator and member account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_GetEc2DeepInspectionConfiguration.html
     */
    toGetEc2DeepInspectionConfiguration() {
        return this.to('GetEc2DeepInspectionConfiguration');
    }
    /**
     * Grants permission to retrieve information about the KMS key used to encrypt code snippets with
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_GetEncryptionKey.html
     */
    toGetEncryptionKey() {
        return this.to('GetEncryptionKey');
    }
    /**
     * Grants permission to retrieve status for a requested findings report
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_GetFindingsReportStatus.html
     */
    toGetFindingsReportStatus() {
        return this.to('GetFindingsReportStatus');
    }
    /**
     * Grants permission to retrieve information about an account that's associated with an Amazon Inspector administrator account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_GetMember.html
     */
    toGetMember() {
        return this.to('GetMember');
    }
    /**
     * Grants permission to retrieve a requested SBOM report
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_GetSbomExport.html
     */
    toGetSbomExport() {
        return this.to('GetSbomExport');
    }
    /**
     * Grants permission to retrieve feature configuration permissions associated with an Amazon Inspector account within an organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ListAccountPermissions.html
     */
    toListAccountPermissions() {
        return this.to('ListAccountPermissions');
    }
    /**
     * Grants permission to retrieve information about all CIS scan configurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ListCisScanConfigurations.html
     */
    toListCisScanConfigurations() {
        return this.to('ListCisScanConfigurations');
    }
    /**
     * Grants permission to retrieve information about all checks pertaining to one CIS scan
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ListCisScanResultsAggregatedByChecks.html
     */
    toListCisScanResultsAggregatedByChecks() {
        return this.to('ListCisScanResultsAggregatedByChecks');
    }
    /**
     * Grants permission to retrieve information about all resources pertaining to one CIS scan
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ListCisScanResultsAggregatedByTargetResource.html
     */
    toListCisScanResultsAggregatedByTargetResource() {
        return this.to('ListCisScanResultsAggregatedByTargetResource');
    }
    /**
     * Grants permission to retrieve information about completed CIS scans
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ListCisScans.html
     */
    toListCisScans() {
        return this.to('ListCisScans');
    }
    /**
     * Grants permission to retrieve the types of statistics Amazon Inspector can generate for resources Inspector monitors
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ListCoverage.html
     */
    toListCoverage() {
        return this.to('ListCoverage');
    }
    /**
     * Grants permission to retrieve statistical data and other information about the resources Amazon Inspector monitors
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ListCoverageStatistics.html
     */
    toListCoverageStatistics() {
        return this.to('ListCoverageStatistics');
    }
    /**
     * Grants permission to retrieve information about the delegated Amazon Inspector administrator account for an AWS organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ListDelegatedAdminAccounts.html
     */
    toListDelegatedAdminAccounts() {
        return this.to('ListDelegatedAdminAccounts');
    }
    /**
     * Grants permission to retrieve information about all findings filters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ListFilters.html
     */
    toListFilters() {
        return this.to('ListFilters');
    }
    /**
     * Grants permission to retrieve statistical data and other information about Amazon Inspector findings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ListFindingAggregations.html
     */
    toListFindingAggregations() {
        return this.to('ListFindingAggregations');
    }
    /**
     * Grants permission to retrieve a subset of information about one or more findings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ListFindings.html
     */
    toListFindings() {
        return this.to('ListFindings');
    }
    /**
     * Grants permission to retrieve information about the Amazon Inspector member accounts that are associated with an Inspector administrator account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ListMembers.html
     */
    toListMembers() {
        return this.to('ListMembers');
    }
    /**
     * Grants permission to retrieve the tags for an Amazon Inspector resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to retrieve aggregated usage data for an account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ListUsageTotals.html
     */
    toListUsageTotals() {
        return this.to('ListUsageTotals');
    }
    /**
     * Grants permission to let a customer reset to use an Amazon-owned KMS key to encrypt code snippets with
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ResetEncryptionKey.html
     */
    toResetEncryptionKey() {
        return this.to('ResetEncryptionKey');
    }
    /**
     * Grants permission to list Amazon Inspector coverage details for a specific vulnerability
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_SearchVulnerabilities.html
     */
    toSearchVulnerabilities() {
        return this.to('SearchVulnerabilities');
    }
    /**
     * Grants permission to send CIS health for a CIS scan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_SendCisSessionHealth.html
     */
    toSendCisSessionHealth() {
        return this.to('SendCisSessionHealth');
    }
    /**
     * Grants permission to send CIS telemetry for a CIS scan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_SendCisSessionTelemetry.html
     */
    toSendCisSessionTelemetry() {
        return this.to('SendCisSessionTelemetry');
    }
    /**
     * Grants permission to start a CIS scan session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_StartCisSession.html
     */
    toStartCisSession() {
        return this.to('StartCisSession');
    }
    /**
     * Grants permission to stop a CIS scan session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_StopCisSession.html
     */
    toStopCisSession() {
        return this.to('StopCisSession');
    }
    /**
     * Grants permission to add or update the tags for an Amazon Inspector resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from an Amazon Inspector resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the settings for a CIS scan configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_UpdateCisScanConfiguration.html
     */
    toUpdateCisScanConfiguration() {
        return this.to('UpdateCisScanConfiguration');
    }
    /**
     * Grants permission to update information about the Amazon Inspector configuration settings for an AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_UpdateConfiguration.html
     */
    toUpdateConfiguration() {
        return this.to('UpdateConfiguration');
    }
    /**
     * Grants permission to update ec2 deep inspection configuration by delegated administrator, member and standalone account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_UpdateEc2DeepInspectionConfiguration.html
     */
    toUpdateEc2DeepInspectionConfiguration() {
        return this.to('UpdateEc2DeepInspectionConfiguration');
    }
    /**
     * Grants permission to let a customer use a KMS key to encrypt code snippets with
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_UpdateEncryptionKey.html
     */
    toUpdateEncryptionKey() {
        return this.to('UpdateEncryptionKey');
    }
    /**
     * Grants permission to update the settings for a findings filter
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_UpdateFilter.html
     */
    toUpdateFilter() {
        return this.to('UpdateFilter');
    }
    /**
     * Grants permission to update ec2 deep inspection configuration by delegated administrator for its associated member accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_UpdateOrgEc2DeepInspectionConfiguration.html
     */
    toUpdateOrgEc2DeepInspectionConfiguration() {
        return this.to('UpdateOrgEc2DeepInspectionConfiguration');
    }
    /**
     * Grants permission to update Amazon Inspector configuration settings for an AWS organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_UpdateOrganizationConfiguration.html
     */
    toUpdateOrganizationConfiguration() {
        return this.to('UpdateOrganizationConfiguration');
    }
    /**
     * Adds a resource of type Filter to the statement
     *
     * https://docs.aws.amazon.com/inspector/latest/user/what-is-inspector.html
     *
     * @param ownerId - Identifier for the ownerId.
     * @param filterId - Identifier for the filterId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFilter(ownerId, filterId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:inspector2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:owner/${ownerId}/filter/${filterId}`);
    }
    /**
     * Adds a resource of type Finding to the statement
     *
     * https://docs.aws.amazon.com/inspector/latest/user/what-is-inspector.html
     *
     * @param findingId - Identifier for the findingId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onFinding(findingId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:inspector2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:finding/${findingId}`);
    }
    /**
     * Adds a resource of type CIS Scan Configuration to the statement
     *
     * https://docs.aws.amazon.com/inspector/latest/user/what-is-inspector.html
     *
     * @param ownerId - Identifier for the ownerId.
     * @param cISScanConfigurationId - Identifier for the cISScanConfigurationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCISScanConfiguration(ownerId, cISScanConfigurationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:inspector2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:owner/${ownerId}/cis-configuration/${cISScanConfigurationId}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateCisScanConfiguration()
     * - .toCreateFilter()
     * - .toTagResource()
     * - .toUpdateFilter()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toCreateCisScanConfiguration()
     * - .toDeleteCisScanConfiguration()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateCisScanConfiguration()
     *
     * Applies to resource types:
     * - Filter
     * - CIS Scan Configuration
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateCisScanConfiguration()
     * - .toCreateFilter()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateFilter()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [inspector2](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoninspector2.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'inspector2';
        this.accessLevelList = {
            Write: [
                'AssociateMember',
                'BatchUpdateMemberEc2DeepInspectionStatus',
                'CancelFindingsReport',
                'CancelSbomExport',
                'CreateCisScanConfiguration',
                'CreateFilter',
                'CreateFindingsReport',
                'CreateSbomExport',
                'DeleteCisScanConfiguration',
                'DeleteFilter',
                'Disable',
                'DisableDelegatedAdminAccount',
                'DisassociateMember',
                'Enable',
                'EnableDelegatedAdminAccount',
                'ResetEncryptionKey',
                'SendCisSessionHealth',
                'SendCisSessionTelemetry',
                'StartCisSession',
                'StopCisSession',
                'UpdateCisScanConfiguration',
                'UpdateConfiguration',
                'UpdateEc2DeepInspectionConfiguration',
                'UpdateEncryptionKey',
                'UpdateFilter',
                'UpdateOrgEc2DeepInspectionConfiguration',
                'UpdateOrganizationConfiguration'
            ],
            Read: [
                'BatchGetAccountStatus',
                'BatchGetCodeSnippet',
                'BatchGetFindingDetails',
                'BatchGetFreeTrialInfo',
                'BatchGetMemberEc2DeepInspectionStatus',
                'DescribeOrganizationConfiguration',
                'GetCisScanReport',
                'GetConfiguration',
                'GetDelegatedAdminAccount',
                'GetEc2DeepInspectionConfiguration',
                'GetEncryptionKey',
                'GetFindingsReportStatus',
                'GetMember',
                'GetSbomExport',
                'ListTagsForResource',
                'SearchVulnerabilities'
            ],
            List: [
                'GetCisScanResultDetails',
                'ListAccountPermissions',
                'ListCisScanConfigurations',
                'ListCisScanResultsAggregatedByChecks',
                'ListCisScanResultsAggregatedByTargetResource',
                'ListCisScans',
                'ListCoverage',
                'ListCoverageStatistics',
                'ListDelegatedAdminAccounts',
                'ListFilters',
                'ListFindingAggregations',
                'ListFindings',
                'ListMembers',
                'ListUsageTotals'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Inspector2 = Inspector2;
//# sourceMappingURL=data:application/json;base64,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