"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Iotevents = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [iotevents](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotevents.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Iotevents extends shared_1.PolicyStatement {
    /**
     * Grants permission to send one or more acknowledge action requests to AWS IoT Events
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_BatchAcknowledgeAlarm.html
     */
    toBatchAcknowledgeAlarm() {
        return this.to('BatchAcknowledgeAlarm');
    }
    /**
     * Grants permission to delete a detector instance within the AWS IoT Events system
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_BatchDeleteDetector.html
     */
    toBatchDeleteDetector() {
        return this.to('BatchDeleteDetector');
    }
    /**
     * Grants permission to disable one or more alarm instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_BatchDisableAlarm.html
     */
    toBatchDisableAlarm() {
        return this.to('BatchDisableAlarm');
    }
    /**
     * Grants permission to enable one or more alarm instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_BatchEnableAlarm.html
     */
    toBatchEnableAlarm() {
        return this.to('BatchEnableAlarm');
    }
    /**
     * Grants permission to send a set of messages to the AWS IoT Events system
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_BatchPutMessage.html
     */
    toBatchPutMessage() {
        return this.to('BatchPutMessage');
    }
    /**
     * Grants permission to reset one or more alarm instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_BatchResetAlarm.html
     */
    toBatchResetAlarm() {
        return this.to('BatchResetAlarm');
    }
    /**
     * Grants permission to change one or more alarm instances to the snooze mode
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_BatchSnoozeAlarm.html
     */
    toBatchSnoozeAlarm() {
        return this.to('BatchSnoozeAlarm');
    }
    /**
     * Grants permission to update a detector instance within the AWS IoT Events system
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_BatchUpdateDetector.html
     */
    toBatchUpdateDetector() {
        return this.to('BatchUpdateDetector');
    }
    /**
     * Grants permission to create an alarm model to monitor an AWS IoT Events input attribute or an AWS IoT SiteWise asset property
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_CreateAlarmModel.html
     */
    toCreateAlarmModel() {
        return this.to('CreateAlarmModel');
    }
    /**
     * Grants permission to create a detector model to monitor an AWS IoT Events input attribute
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_CreateDetectorModel.html
     */
    toCreateDetectorModel() {
        return this.to('CreateDetectorModel');
    }
    /**
     * Grants permission to create an Input in IotEvents
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_CreateInput.html
     */
    toCreateInput() {
        return this.to('CreateInput');
    }
    /**
     * Grants permission to delete an alarm model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_DeleteAlarmModel.html
     */
    toDeleteAlarmModel() {
        return this.to('DeleteAlarmModel');
    }
    /**
     * Grants permission to delete a detector model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_DeleteDetectorModel.html
     */
    toDeleteDetectorModel() {
        return this.to('DeleteDetectorModel');
    }
    /**
     * Grants permission to delete an input
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_DeleteInput.html
     */
    toDeleteInput() {
        return this.to('DeleteInput');
    }
    /**
     * Grants permission to retrieve information about an alarm instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_DescribeAlarm.html
     */
    toDescribeAlarm() {
        return this.to('DescribeAlarm');
    }
    /**
     * Grants permission to retrieve information about an alarm model
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_DescribeAlarmModel.html
     */
    toDescribeAlarmModel() {
        return this.to('DescribeAlarmModel');
    }
    /**
     * Grants permission to retriev information about a detector instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_DescribeDetector.html
     */
    toDescribeDetector() {
        return this.to('DescribeDetector');
    }
    /**
     * Grants permission to retrieve information about a detector model
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_DescribeDetectorModel.html
     */
    toDescribeDetectorModel() {
        return this.to('DescribeDetectorModel');
    }
    /**
     * Grants permission to retrieve the detector model analysis information
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_DescribeDetectorModelAnalysis.html
     */
    toDescribeDetectorModelAnalysis() {
        return this.to('DescribeDetectorModelAnalysis');
    }
    /**
     * Grants permission to retrieve an information about Input
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_DescribeInput.html
     */
    toDescribeInput() {
        return this.to('DescribeInput');
    }
    /**
     * Grants permission to retrieve the current settings of the AWS IoT Events logging options
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_DescribeLoggingOptions.html
     */
    toDescribeLoggingOptions() {
        return this.to('DescribeLoggingOptions');
    }
    /**
     * Grants permission to retrieve the detector model analysis results
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_GetDetectorModelAnalysisResults.html
     */
    toGetDetectorModelAnalysisResults() {
        return this.to('GetDetectorModelAnalysisResults');
    }
    /**
     * Grants permission to list all the versions of an alarm model
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_ListAlarmModelVersions.html
     */
    toListAlarmModelVersions() {
        return this.to('ListAlarmModelVersions');
    }
    /**
     * Grants permission to list the alarm models that you created
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_ListAlarmModels.html
     */
    toListAlarmModels() {
        return this.to('ListAlarmModels');
    }
    /**
     * Grants permission to retrieve information about all alarm instances per alarmModel
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_ListAlarms.html
     */
    toListAlarms() {
        return this.to('ListAlarms');
    }
    /**
     * Grants permission to list all the versions of a detector model
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_ListDetectorModelVersions.html
     */
    toListDetectorModelVersions() {
        return this.to('ListDetectorModelVersions');
    }
    /**
     * Grants permission to list the detector models that you created
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_ListDetectorModels.html
     */
    toListDetectorModels() {
        return this.to('ListDetectorModels');
    }
    /**
     * Grants permission to retrieve information about all detector instances per detectormodel
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_ListDetectors.html
     */
    toListDetectors() {
        return this.to('ListDetectors');
    }
    /**
     * Grants permission to list one or more input routings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_ListInputRoutings.html
     */
    toListInputRoutings() {
        return this.to('ListInputRoutings');
    }
    /**
     * Grants permission to lists the inputs you have created
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_ListInputs.html
     */
    toListInputs() {
        return this.to('ListInputs');
    }
    /**
     * Grants permission to list the tags (metadata) which you have assigned to the resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to set or update the AWS IoT Events logging options
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_PutLoggingOptions.html
     */
    toPutLoggingOptions() {
        return this.to('PutLoggingOptions');
    }
    /**
     * Grants permission to start the detector model analysis
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_StartDetectorModelAnalysis.html
     */
    toStartDetectorModelAnalysis() {
        return this.to('StartDetectorModelAnalysis');
    }
    /**
     * Grants permission to adds to or modifies the tags of the given resource.Tags are metadata which can be used to manage a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove the given tags (metadata) from the resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an alarm model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_UpdateAlarmModel.html
     */
    toUpdateAlarmModel() {
        return this.to('UpdateAlarmModel');
    }
    /**
     * Grants permission to update a detector model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_UpdateDetectorModel.html
     */
    toUpdateDetectorModel() {
        return this.to('UpdateDetectorModel');
    }
    /**
     * Grants permission to update an input
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_UpdateInput.html
     */
    toUpdateInput() {
        return this.to('UpdateInput');
    }
    /**
     * Grants permission to update input routing
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_UpdateInputRouting.html
     */
    toUpdateInputRouting() {
        return this.to('UpdateInputRouting');
    }
    /**
     * Adds a resource of type detectorModel to the statement
     *
     * https://docs.aws.amazon.com/iotevents/latest/developerguide/iotevents-getting-started.html
     *
     * @param detectorModelName - Identifier for the detectorModelName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDetectorModel(detectorModelName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotevents:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:detectorModel/${detectorModelName}`);
    }
    /**
     * Adds a resource of type alarmModel to the statement
     *
     * https://docs.aws.amazon.com/iotevents/latest/developerguide/iotevents-getting-started.html
     *
     * @param alarmModelName - Identifier for the alarmModelName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAlarmModel(alarmModelName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotevents:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:alarmModel/${alarmModelName}`);
    }
    /**
     * Adds a resource of type input to the statement
     *
     * https://docs.aws.amazon.com/iotevents/latest/developerguide/iotevents-getting-started.html
     *
     * @param inputName - Identifier for the inputName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onInput(inputName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotevents:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:input/${inputName}`);
    }
    /**
     * Filters access by the tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateAlarmModel()
     * - .toCreateDetectorModel()
     * - .toCreateInput()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - detectorModel
     * - alarmModel
     * - input
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions by the tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateAlarmModel()
     * - .toCreateDetectorModel()
     * - .toCreateInput()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the instanceId (key-value) of the message
     *
     * https://docs.aws.amazon.com/iotevents/latest/developerguide/security_iam_id-based-policy-examples.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifKeyValue(value, operator) {
        return this.if(`keyValue`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [iotevents](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotevents.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'iotevents';
        this.accessLevelList = {
            Write: [
                'BatchAcknowledgeAlarm',
                'BatchDeleteDetector',
                'BatchDisableAlarm',
                'BatchEnableAlarm',
                'BatchPutMessage',
                'BatchResetAlarm',
                'BatchSnoozeAlarm',
                'BatchUpdateDetector',
                'CreateAlarmModel',
                'CreateDetectorModel',
                'CreateInput',
                'DeleteAlarmModel',
                'DeleteDetectorModel',
                'DeleteInput',
                'PutLoggingOptions',
                'StartDetectorModelAnalysis',
                'UpdateAlarmModel',
                'UpdateDetectorModel',
                'UpdateInput',
                'UpdateInputRouting'
            ],
            Read: [
                'DescribeAlarm',
                'DescribeAlarmModel',
                'DescribeDetector',
                'DescribeDetectorModel',
                'DescribeDetectorModelAnalysis',
                'DescribeInput',
                'DescribeLoggingOptions',
                'GetDetectorModelAnalysisResults',
                'ListTagsForResource'
            ],
            List: [
                'ListAlarmModelVersions',
                'ListAlarmModels',
                'ListAlarms',
                'ListDetectorModelVersions',
                'ListDetectorModels',
                'ListDetectors',
                'ListInputRoutings',
                'ListInputs'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Iotevents = Iotevents;
//# sourceMappingURL=data:application/json;base64,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