"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Iottwinmaker = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [iottwinmaker](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiottwinmaker.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Iottwinmaker extends shared_1.PolicyStatement {
    /**
     * Grants permission to set values for multiple time series properties
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iottwinmaker:GetComponentType
     * - iottwinmaker:GetEntity
     * - iottwinmaker:GetWorkspace
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_BatchPutPropertyValues.html
     */
    toBatchPutPropertyValues() {
        return this.to('BatchPutPropertyValues');
    }
    /**
     * Grants permission to cancel a metadata transfer job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_CancelMetadataTransferJob.html
     */
    toCancelMetadataTransferJob() {
        return this.to('CancelMetadataTransferJob');
    }
    /**
     * Grants permission to create a componentType
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_CreateComponentType.html
     */
    toCreateComponentType() {
        return this.to('CreateComponentType');
    }
    /**
     * Grants permission to create an entity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_CreateEntity.html
     */
    toCreateEntity() {
        return this.to('CreateEntity');
    }
    /**
     * Grants permission to create a metadata transfer job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_CreateMetadataTransferJob.html
     */
    toCreateMetadataTransferJob() {
        return this.to('CreateMetadataTransferJob');
    }
    /**
     * Grants permission to create a scene
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_CreateScene.html
     */
    toCreateScene() {
        return this.to('CreateScene');
    }
    /**
     * Grants permission to create a sync job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_CreateSyncJob.html
     */
    toCreateSyncJob() {
        return this.to('CreateSyncJob');
    }
    /**
     * Grants permission to create a workspace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_CreateWorkspace.html
     */
    toCreateWorkspace() {
        return this.to('CreateWorkspace');
    }
    /**
     * Grants permission to delete a componentType
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_DeleteComponentType.html
     */
    toDeleteComponentType() {
        return this.to('DeleteComponentType');
    }
    /**
     * Grants permission to delete an entity
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_DeleteEntity.html
     */
    toDeleteEntity() {
        return this.to('DeleteEntity');
    }
    /**
     * Grants permission to delete a scene
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_DeleteScene.html
     */
    toDeleteScene() {
        return this.to('DeleteScene');
    }
    /**
     * Grants permission to delete a sync job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_DeleteSyncJob.html
     */
    toDeleteSyncJob() {
        return this.to('DeleteSyncJob');
    }
    /**
     * Grants permission to delete a workspace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_DeleteWorkspace.html
     */
    toDeleteWorkspace() {
        return this.to('DeleteWorkspace');
    }
    /**
     * Grants permission to execute query
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_ExecuteQuery.html
     */
    toExecuteQuery() {
        return this.to('ExecuteQuery');
    }
    /**
     * Grants permission to get a componentType
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_GetComponentType.html
     */
    toGetComponentType() {
        return this.to('GetComponentType');
    }
    /**
     * Grants permission to get an entity
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_GetEntity.html
     */
    toGetEntity() {
        return this.to('GetEntity');
    }
    /**
     * Grants permission to get a metadata transfer job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_GetMetadataTransferJob.html
     */
    toGetMetadataTransferJob() {
        return this.to('GetMetadataTransferJob');
    }
    /**
     * Grants permission to get pricing plan
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_GetPricingPlan.html
     */
    toGetPricingPlan() {
        return this.to('GetPricingPlan');
    }
    /**
     * Grants permission to retrieve the property values
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - iottwinmaker:GetComponentType
     * - iottwinmaker:GetEntity
     * - iottwinmaker:GetWorkspace
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_GetPropertyValue.html
     */
    toGetPropertyValue() {
        return this.to('GetPropertyValue');
    }
    /**
     * Grants permission to retrieve the time series value history
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - iottwinmaker:GetComponentType
     * - iottwinmaker:GetEntity
     * - iottwinmaker:GetWorkspace
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_GetPropertyValueHistory.html
     */
    toGetPropertyValueHistory() {
        return this.to('GetPropertyValueHistory');
    }
    /**
     * Grants permission to get a scene
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_GetScene.html
     */
    toGetScene() {
        return this.to('GetScene');
    }
    /**
     * Grants permission to get a sync job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_GetSyncJob.html
     */
    toGetSyncJob() {
        return this.to('GetSyncJob');
    }
    /**
     * Grants permission to get a workspace
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_GetWorkspace.html
     */
    toGetWorkspace() {
        return this.to('GetWorkspace');
    }
    /**
     * Grants permission to list all componentTypes in a workspace
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_ListComponentTypes.html
     */
    toListComponentTypes() {
        return this.to('ListComponentTypes');
    }
    /**
     * Grants permission to list components attached to an entity
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_ListComponents.html
     */
    toListComponents() {
        return this.to('ListComponents');
    }
    /**
     * Grants permission to list all entities in a workspace
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_ListEntities.html
     */
    toListEntities() {
        return this.to('ListEntities');
    }
    /**
     * Grants permission to list all metadata transfer jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_ListMetadataTransferJobs.html
     */
    toListMetadataTransferJobs() {
        return this.to('ListMetadataTransferJobs');
    }
    /**
     * Grants permission to list properties of an entity component
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_ListProperties.html
     */
    toListProperties() {
        return this.to('ListProperties');
    }
    /**
     * Grants permission to list all scenes in a workspace
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_ListScenes.html
     */
    toListScenes() {
        return this.to('ListScenes');
    }
    /**
     * Grants permission to list all sync jobs in a workspace
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_ListSyncJobs.html
     */
    toListSyncJobs() {
        return this.to('ListSyncJobs');
    }
    /**
     * Grants permission to list all sync resources for a sync job
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_ListSyncResources.html
     */
    toListSyncResources() {
        return this.to('ListSyncResources');
    }
    /**
     * Grants permission to list all tags for a resource
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list all workspaces
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_ListWorkspaces.html
     */
    toListWorkspaces() {
        return this.to('ListWorkspaces');
    }
    /**
     * Grants permission to tag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a componentType
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_UpdateComponentType.html
     */
    toUpdateComponentType() {
        return this.to('UpdateComponentType');
    }
    /**
     * Grants permission to update an entity
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_UpdateEntity.html
     */
    toUpdateEntity() {
        return this.to('UpdateEntity');
    }
    /**
     * Grants permission to update pricing plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_UpdatePricingPlan.html
     */
    toUpdatePricingPlan() {
        return this.to('UpdatePricingPlan');
    }
    /**
     * Grants permission to update a scene
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_UpdateScene.html
     */
    toUpdateScene() {
        return this.to('UpdateScene');
    }
    /**
     * Grants permission to update a workspace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_UpdateWorkspace.html
     */
    toUpdateWorkspace() {
        return this.to('UpdateWorkspace');
    }
    /**
     * Adds a resource of type workspace to the statement
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_CreateWorkspace.html
     *
     * @param workspaceId - Identifier for the workspaceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorkspace(workspaceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iottwinmaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workspace/${workspaceId}`);
    }
    /**
     * Adds a resource of type entity to the statement
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_CreateEntity.html
     *
     * @param workspaceId - Identifier for the workspaceId.
     * @param entityId - Identifier for the entityId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEntity(workspaceId, entityId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iottwinmaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workspace/${workspaceId}/entity/${entityId}`);
    }
    /**
     * Adds a resource of type componentType to the statement
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_CreateComponentType.html
     *
     * @param workspaceId - Identifier for the workspaceId.
     * @param componentTypeId - Identifier for the componentTypeId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onComponentType(workspaceId, componentTypeId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iottwinmaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workspace/${workspaceId}/component-type/${componentTypeId}`);
    }
    /**
     * Adds a resource of type scene to the statement
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_CreateScene.html
     *
     * @param workspaceId - Identifier for the workspaceId.
     * @param sceneId - Identifier for the sceneId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onScene(workspaceId, sceneId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iottwinmaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workspace/${workspaceId}/scene/${sceneId}`);
    }
    /**
     * Adds a resource of type syncJob to the statement
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_CreateSyncJob.html
     *
     * @param workspaceId - Identifier for the workspaceId.
     * @param syncJobId - Identifier for the syncJobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSyncJob(workspaceId, syncJobId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iottwinmaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workspace/${workspaceId}/sync-job/${syncJobId}`);
    }
    /**
     * Adds a resource of type metadataTransferJob to the statement
     *
     * https://docs.aws.amazon.com/iot-twinmaker/latest/apireference/API_CreateMetadataTransferJob.html
     *
     * @param metadataTransferJobId - Identifier for the metadataTransferJobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onMetadataTransferJob(metadataTransferJobId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iottwinmaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:metadata-transfer-job/${metadataTransferJobId}`);
    }
    /**
     * Filters access by the tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateComponentType()
     * - .toCreateEntity()
     * - .toCreateScene()
     * - .toCreateSyncJob()
     * - .toCreateWorkspace()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toListTagsForResource()
     *
     * Applies to resource types:
     * - workspace
     * - entity
     * - componentType
     * - scene
     * - syncJob
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateComponentType()
     * - .toCreateEntity()
     * - .toCreateScene()
     * - .toCreateSyncJob()
     * - .toCreateWorkspace()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by destination type of metadata transfer job
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiottwinmaker.html#awsiottwinmaker-policy-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDestinationType(value, operator) {
        return this.if(`destinationType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by workspace linked to services
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiottwinmaker.html#awsiottwinmaker-policy-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLinkedServices(value, operator) {
        return this.if(`linkedServices`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by source type of metadata transfer job
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiottwinmaker.html#awsiottwinmaker-policy-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSourceType(value, operator) {
        return this.if(`sourceType`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [iottwinmaker](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiottwinmaker.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'iottwinmaker';
        this.accessLevelList = {
            Write: [
                'BatchPutPropertyValues',
                'CancelMetadataTransferJob',
                'CreateComponentType',
                'CreateEntity',
                'CreateMetadataTransferJob',
                'CreateScene',
                'CreateSyncJob',
                'CreateWorkspace',
                'DeleteComponentType',
                'DeleteEntity',
                'DeleteScene',
                'DeleteSyncJob',
                'DeleteWorkspace',
                'UpdateComponentType',
                'UpdateEntity',
                'UpdatePricingPlan',
                'UpdateScene',
                'UpdateWorkspace'
            ],
            Read: [
                'ExecuteQuery',
                'GetComponentType',
                'GetEntity',
                'GetMetadataTransferJob',
                'GetPricingPlan',
                'GetPropertyValue',
                'GetPropertyValueHistory',
                'GetScene',
                'GetSyncJob',
                'GetWorkspace'
            ],
            List: [
                'ListComponentTypes',
                'ListComponents',
                'ListEntities',
                'ListMetadataTransferJobs',
                'ListProperties',
                'ListScenes',
                'ListSyncJobs',
                'ListSyncResources',
                'ListTagsForResource',
                'ListWorkspaces'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Iottwinmaker = Iottwinmaker;
//# sourceMappingURL=data:application/json;base64,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