"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Kinesisvideo = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [kinesisvideo](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonkinesisvideostreams.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Kinesisvideo extends shared_1.PolicyStatement {
    /**
     * Grants permission to connect as a master to the signaling channel specified by the endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_ConnectAsMaster.html
     */
    toConnectAsMaster() {
        return this.to('ConnectAsMaster');
    }
    /**
     * Grants permission to connect as a viewer to the signaling channel specified by the endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_ConnectAsViewer.html
     */
    toConnectAsViewer() {
        return this.to('ConnectAsViewer');
    }
    /**
     * Grants permission to create a signaling channel
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_CreateSignalingChannel.html
     */
    toCreateSignalingChannel() {
        return this.to('CreateSignalingChannel');
    }
    /**
     * Grants permission to create a Kinesis video stream
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_CreateStream.html
     */
    toCreateStream() {
        return this.to('CreateStream');
    }
    /**
     * Grants permission to delete the edge configuration of your Kinesis Video Stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DeleteEdgeConfiguration.html
     */
    toDeleteEdgeConfiguration() {
        return this.to('DeleteEdgeConfiguration');
    }
    /**
     * Grants permission to delete an existing signaling channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DeleteSignalingChannel.html
     */
    toDeleteSignalingChannel() {
        return this.to('DeleteSignalingChannel');
    }
    /**
     * Grants permission to delete an existing Kinesis video stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DeleteStream.html
     */
    toDeleteStream() {
        return this.to('DeleteStream');
    }
    /**
     * Grants permission to describe the edge configuration of your Kinesis Video Stream
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DescribeEdgeConfiguration.html
     */
    toDescribeEdgeConfiguration() {
        return this.to('DescribeEdgeConfiguration');
    }
    /**
     * Grants permission to describe the image generation configuration of your Kinesis video stream
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DescribeImageGenerationConfiguration.html
     */
    toDescribeImageGenerationConfiguration() {
        return this.to('DescribeImageGenerationConfiguration');
    }
    /**
     * Grants permission to describe the resource mapped to the Kinesis video stream
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DescribeMappedResourceConfiguration.html
     */
    toDescribeMappedResourceConfiguration() {
        return this.to('DescribeMappedResourceConfiguration');
    }
    /**
     * Grants permission to describe the media storage configuration of a signaling channel
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DescribeMediaStorageConfiguration.html
     */
    toDescribeMediaStorageConfiguration() {
        return this.to('DescribeMediaStorageConfiguration');
    }
    /**
     * Grants permission to describe the notification configuration of your Kinesis video stream
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DescribeNotificationConfiguration.html
     */
    toDescribeNotificationConfiguration() {
        return this.to('DescribeNotificationConfiguration');
    }
    /**
     * Grants permission to describe the specified signaling channel
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DescribeSignalingChannel.html
     */
    toDescribeSignalingChannel() {
        return this.to('DescribeSignalingChannel');
    }
    /**
     * Grants permission to describe the specified Kinesis video stream
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DescribeStream.html
     */
    toDescribeStream() {
        return this.to('DescribeStream');
    }
    /**
     * Grants permission to get a media clip from a video stream
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_reader_GetClip.html
     */
    toGetClip() {
        return this.to('GetClip');
    }
    /**
     * Grants permission to create a URL for MPEG-DASH video streaming
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_reader_GetDASHStreamingSessionURL.html
     */
    toGetDASHStreamingSessionURL() {
        return this.to('GetDASHStreamingSessionURL');
    }
    /**
     * Grants permission to get an endpoint for a specified stream for either reading or writing media data to Kinesis Video Streams
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_GetDataEndpoint.html
     */
    toGetDataEndpoint() {
        return this.to('GetDataEndpoint');
    }
    /**
     * Grants permission to create a URL for HLS video streaming
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_reader_GetHLSStreamingSessionURL.html
     */
    toGetHLSStreamingSessionURL() {
        return this.to('GetHLSStreamingSessionURL');
    }
    /**
     * Grants permission to get the ICE server configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_signaling_GetIceServerConfig.html
     */
    toGetIceServerConfig() {
        return this.to('GetIceServerConfig');
    }
    /**
     * Grants permission to get generated images from your Kinesis video stream
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_reader_GetImages.html
     */
    toGetImages() {
        return this.to('GetImages');
    }
    /**
     * Grants permission to return media content of a Kinesis video stream
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_dataplane_GetMedia.html
     */
    toGetMedia() {
        return this.to('GetMedia');
    }
    /**
     * Grants permission to read and return media data only from persisted storage
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_reader_GetMediaForFragmentList.html
     */
    toGetMediaForFragmentList() {
        return this.to('GetMediaForFragmentList');
    }
    /**
     * Grants permission to get endpoints for a specified combination of protocol and role for a signaling channel
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_GetSignalingChannelEndpoint.html
     */
    toGetSignalingChannelEndpoint() {
        return this.to('GetSignalingChannelEndpoint');
    }
    /**
     * Grants permission to join a storage session for a channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_webrtc_JoinStorageSession.html
     */
    toJoinStorageSession() {
        return this.to('JoinStorageSession');
    }
    /**
     * Grants permission to join a storage session for a channel as viewer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_webrtc_JoinStorageSessionAsViewer.html
     */
    toJoinStorageSessionAsViewer() {
        return this.to('JoinStorageSessionAsViewer');
    }
    /**
     * Grants permission to list an edge agent configurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_ListEdgeAgentConfigurations.html
     */
    toListEdgeAgentConfigurations() {
        return this.to('ListEdgeAgentConfigurations');
    }
    /**
     * Grants permission to list the fragments from archival storage based on the pagination token or selector type with range specified
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_reader_ListFragments.html
     */
    toListFragments() {
        return this.to('ListFragments');
    }
    /**
     * Grants permission to list your signaling channels
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_ListSignalingChannels.html
     */
    toListSignalingChannels() {
        return this.to('ListSignalingChannels');
    }
    /**
     * Grants permission to list your Kinesis video streams
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_ListStreams.html
     */
    toListStreams() {
        return this.to('ListStreams');
    }
    /**
     * Grants permission to fetch the tags associated with your resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to fetch the tags associated with Kinesis video stream
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_ListTagsForStream.html
     */
    toListTagsForStream() {
        return this.to('ListTagsForStream');
    }
    /**
     * Grants permission to send media data to a Kinesis video stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_dataplane_PutMedia.html
     */
    toPutMedia() {
        return this.to('PutMedia');
    }
    /**
     * Grants permission to send the Alexa SDP offer to the master
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_signaling_SendAlexaOfferToMaster.html
     */
    toSendAlexaOfferToMaster() {
        return this.to('SendAlexaOfferToMaster');
    }
    /**
     * Grants permission to start edge configuration update of your Kinesis Video Stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_StartEdgeConfigurationUpdate.html
     */
    toStartEdgeConfigurationUpdate() {
        return this.to('StartEdgeConfigurationUpdate');
    }
    /**
     * Grants permission to attach set of tags to your resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to attach set of tags to your Kinesis video streams
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_TagStream.html
     */
    toTagStream() {
        return this.to('TagStream');
    }
    /**
     * Grants permission to remove one or more tags from your resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to remove one or more tags from your Kinesis video streams
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_UntagStream.html
     */
    toUntagStream() {
        return this.to('UntagStream');
    }
    /**
     * Grants permission to update the data retention period of your Kinesis video stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_UpdateDataRetention.html
     */
    toUpdateDataRetention() {
        return this.to('UpdateDataRetention');
    }
    /**
     * Grants permission to update the image generation configuration of your Kinesis video stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_UpdateImageGenerationConfiguration.html
     */
    toUpdateImageGenerationConfiguration() {
        return this.to('UpdateImageGenerationConfiguration');
    }
    /**
     * Grants permission to create or update an mapping between a signaling channel and stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_UpdateMediaStorageConfiguration.html
     */
    toUpdateMediaStorageConfiguration() {
        return this.to('UpdateMediaStorageConfiguration');
    }
    /**
     * Grants permission to update the notification configuration of your Kinesis video stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_UpdateNotificationConfiguration.html
     */
    toUpdateNotificationConfiguration() {
        return this.to('UpdateNotificationConfiguration');
    }
    /**
     * Grants permission to update an existing signaling channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_UpdateSignalingChannel.html
     */
    toUpdateSignalingChannel() {
        return this.to('UpdateSignalingChannel');
    }
    /**
     * Grants permission to update an existing Kinesis video stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_UpdateStream.html
     */
    toUpdateStream() {
        return this.to('UpdateStream');
    }
    /**
     * Adds a resource of type stream to the statement
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/how-it-works.html
     *
     * @param streamName - Identifier for the streamName.
     * @param creationTime - Identifier for the creationTime.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStream(streamName, creationTime, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:kinesisvideo:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:stream/${streamName}/${creationTime}`);
    }
    /**
     * Adds a resource of type channel to the statement
     *
     * https://docs.aws.amazon.com/kinesisvideostreams-webrtc-dg/latest/devguide/kvswebrtc-how-it-works.html
     *
     * @param channelName - Identifier for the channelName.
     * @param creationTime - Identifier for the creationTime.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onChannel(channelName, creationTime, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:kinesisvideo:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:channel/${channelName}/${creationTime}`);
    }
    /**
     * Filters requests based on the allowed set of values for each of the tags
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateSignalingChannel()
     * - .toCreateStream()
     * - .toTagResource()
     * - .toTagStream()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on tag-value assoicated with the stream
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - stream
     * - channel
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters requests based on the presence of mandatory tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateSignalingChannel()
     * - .toCreateStream()
     * - .toTagResource()
     * - .toTagStream()
     * - .toUntagResource()
     * - .toUntagStream()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [kinesisvideo](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonkinesisvideostreams.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'kinesisvideo';
        this.accessLevelList = {
            Write: [
                'ConnectAsMaster',
                'ConnectAsViewer',
                'CreateSignalingChannel',
                'CreateStream',
                'DeleteEdgeConfiguration',
                'DeleteSignalingChannel',
                'DeleteStream',
                'JoinStorageSession',
                'JoinStorageSessionAsViewer',
                'PutMedia',
                'SendAlexaOfferToMaster',
                'StartEdgeConfigurationUpdate',
                'UpdateDataRetention',
                'UpdateImageGenerationConfiguration',
                'UpdateMediaStorageConfiguration',
                'UpdateNotificationConfiguration',
                'UpdateSignalingChannel',
                'UpdateStream'
            ],
            Read: [
                'DescribeEdgeConfiguration',
                'DescribeImageGenerationConfiguration',
                'DescribeMediaStorageConfiguration',
                'DescribeNotificationConfiguration',
                'GetClip',
                'GetDASHStreamingSessionURL',
                'GetDataEndpoint',
                'GetHLSStreamingSessionURL',
                'GetIceServerConfig',
                'GetImages',
                'GetMedia',
                'GetMediaForFragmentList',
                'GetSignalingChannelEndpoint',
                'ListTagsForResource',
                'ListTagsForStream'
            ],
            List: [
                'DescribeMappedResourceConfiguration',
                'DescribeSignalingChannel',
                'DescribeStream',
                'ListEdgeAgentConfigurations',
                'ListFragments',
                'ListSignalingChannels',
                'ListStreams'
            ],
            Tagging: [
                'TagResource',
                'TagStream',
                'UntagResource',
                'UntagStream'
            ]
        };
    }
}
exports.Kinesisvideo = Kinesisvideo;
//# sourceMappingURL=data:application/json;base64,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