"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Lambda = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [lambda](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awslambda.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Lambda extends shared_1.PolicyStatement {
    /**
     * Grants permission to add permissions to the resource-based policy of a version of an AWS Lambda layer
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_AddLayerVersionPermission.html
     */
    toAddLayerVersionPermission() {
        return this.to('AddLayerVersionPermission');
    }
    /**
     * Grants permission to give an AWS service or another account permission to use an AWS Lambda function
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPrincipal()
     * - .ifFunctionUrlAuthType()
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_AddPermission.html
     */
    toAddPermission() {
        return this.to('AddPermission');
    }
    /**
     * Grants permission to create an alias for a Lambda function version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_CreateAlias.html
     */
    toCreateAlias() {
        return this.to('CreateAlias');
    }
    /**
     * Grants permission to create an AWS Lambda code signing config
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_CreateCodeSigningConfig.html
     */
    toCreateCodeSigningConfig() {
        return this.to('CreateCodeSigningConfig');
    }
    /**
     * Grants permission to create a mapping between an event source and an AWS Lambda function
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFunctionArn()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_CreateEventSourceMapping.html
     */
    toCreateEventSourceMapping() {
        return this.to('CreateEventSourceMapping');
    }
    /**
     * Grants permission to create an AWS Lambda function
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifLayer()
     * - .ifVpcIds()
     * - .ifSubnetIds()
     * - .ifSecurityGroupIds()
     * - .ifCodeSigningConfigArn()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_CreateFunction.html
     */
    toCreateFunction() {
        return this.to('CreateFunction');
    }
    /**
     * Grants permission to create a function url configuration for a Lambda function
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFunctionUrlAuthType()
     * - .ifFunctionArn()
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_CreateFunctionUrlConfig.html
     */
    toCreateFunctionUrlConfig() {
        return this.to('CreateFunctionUrlConfig');
    }
    /**
     * Grants permission to delete an AWS Lambda function alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_DeleteAlias.html
     */
    toDeleteAlias() {
        return this.to('DeleteAlias');
    }
    /**
     * Grants permission to delete an AWS Lambda code signing config
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_DeleteCodeSigningConfig.html
     */
    toDeleteCodeSigningConfig() {
        return this.to('DeleteCodeSigningConfig');
    }
    /**
     * Grants permission to delete an AWS Lambda event source mapping
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFunctionArn()
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_DeleteEventSourceMapping.html
     */
    toDeleteEventSourceMapping() {
        return this.to('DeleteEventSourceMapping');
    }
    /**
     * Grants permission to delete an AWS Lambda function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_DeleteFunction.html
     */
    toDeleteFunction() {
        return this.to('DeleteFunction');
    }
    /**
     * Grants permission to detach a code signing config from an AWS Lambda function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_DeleteFunctionCodeSigningConfig.html
     */
    toDeleteFunctionCodeSigningConfig() {
        return this.to('DeleteFunctionCodeSigningConfig');
    }
    /**
     * Grants permission to remove a concurrent execution limit from an AWS Lambda function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_DeleteFunctionConcurrency.html
     */
    toDeleteFunctionConcurrency() {
        return this.to('DeleteFunctionConcurrency');
    }
    /**
     * Grants permission to delete the configuration for asynchronous invocation for an AWS Lambda function, version, or alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_DeleteFunctionEventInvokeConfig.html
     */
    toDeleteFunctionEventInvokeConfig() {
        return this.to('DeleteFunctionEventInvokeConfig');
    }
    /**
     * Grants permission to delete function url configuration for a Lambda function
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFunctionUrlAuthType()
     * - .ifFunctionArn()
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_DeleteFunctionUrlConfig.html
     */
    toDeleteFunctionUrlConfig() {
        return this.to('DeleteFunctionUrlConfig');
    }
    /**
     * Grants permission to delete a version of an AWS Lambda layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_DeleteLayerVersion.html
     */
    toDeleteLayerVersion() {
        return this.to('DeleteLayerVersion');
    }
    /**
     * Grants permission to delete the provisioned concurrency configuration for an AWS Lambda function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_DeleteProvisionedConcurrencyConfig.html
     */
    toDeleteProvisionedConcurrencyConfig() {
        return this.to('DeleteProvisionedConcurrencyConfig');
    }
    /**
     * Grants permission to disable replication for a Lambda@Edge function
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/lambda-edge-permissions.html
     */
    toDisableReplication() {
        return this.to('DisableReplication');
    }
    /**
     * Grants permission to enable replication for a Lambda@Edge function
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/lambda-edge-permissions.html
     */
    toEnableReplication() {
        return this.to('EnableReplication');
    }
    /**
     * Grants permission to view details about an account's limits and usage in an AWS Region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetAccountSettings.html
     */
    toGetAccountSettings() {
        return this.to('GetAccountSettings');
    }
    /**
     * Grants permission to view details about an AWS Lambda function alias
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetAlias.html
     */
    toGetAlias() {
        return this.to('GetAlias');
    }
    /**
     * Grants permission to view details about an AWS Lambda code signing config
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetCodeSigningConfig.html
     */
    toGetCodeSigningConfig() {
        return this.to('GetCodeSigningConfig');
    }
    /**
     * Grants permission to view details about an AWS Lambda event source mapping
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifFunctionArn()
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetEventSourceMapping.html
     */
    toGetEventSourceMapping() {
        return this.to('GetEventSourceMapping');
    }
    /**
     * Grants permission to view details about an AWS Lambda function
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetFunction.html
     */
    toGetFunction() {
        return this.to('GetFunction');
    }
    /**
     * Grants permission to view the code signing config arn attached to an AWS Lambda function
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetFunctionCodeSigningConfig.html
     */
    toGetFunctionCodeSigningConfig() {
        return this.to('GetFunctionCodeSigningConfig');
    }
    /**
     * Grants permission to view details about the reserved concurrency configuration for a function
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetFunctionConcurrency.html
     */
    toGetFunctionConcurrency() {
        return this.to('GetFunctionConcurrency');
    }
    /**
     * Grants permission to view details about the version-specific settings of an AWS Lambda function or version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetFunctionConfiguration.html
     */
    toGetFunctionConfiguration() {
        return this.to('GetFunctionConfiguration');
    }
    /**
     * Grants permission to view the configuration for asynchronous invocation for a function, version, or alias
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetFunctionEventInvokeConfig.html
     */
    toGetFunctionEventInvokeConfig() {
        return this.to('GetFunctionEventInvokeConfig');
    }
    /**
     * Grants permission to view the recursion configuration of an AWS Lambda function
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetFunctionRecursionConfig.html
     */
    toGetFunctionRecursionConfig() {
        return this.to('GetFunctionRecursionConfig');
    }
    /**
     * Grants permission to read function url configuration for a Lambda function
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifFunctionUrlAuthType()
     * - .ifFunctionArn()
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetFunctionUrlConfig.html
     */
    toGetFunctionUrlConfig() {
        return this.to('GetFunctionUrlConfig');
    }
    /**
     * Grants permission to view details about a version of an AWS Lambda layer. Note this action also supports GetLayerVersionByArn API
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetLayerVersion.html
     */
    toGetLayerVersion() {
        return this.to('GetLayerVersion');
    }
    /**
     * Grants permission to view the resource-based policy for a version of an AWS Lambda layer
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetLayerVersionPolicy.html
     */
    toGetLayerVersionPolicy() {
        return this.to('GetLayerVersionPolicy');
    }
    /**
     * Grants permission to view the resource-based policy for an AWS Lambda function, version, or alias
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetPolicy.html
     */
    toGetPolicy() {
        return this.to('GetPolicy');
    }
    /**
     * Grants permission to view the provisioned concurrency configuration for an AWS Lambda function's alias or version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetProvisionedConcurrencyConfig.html
     */
    toGetProvisionedConcurrencyConfig() {
        return this.to('GetProvisionedConcurrencyConfig');
    }
    /**
     * Grants permission to view the runtime management configuration of an AWS Lambda function
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_GetRuntimeManagementConfig.html
     */
    toGetRuntimeManagementConfig() {
        return this.to('GetRuntimeManagementConfig');
    }
    /**
     * Grants permission to invoke a function asynchronously (Deprecated)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_InvokeAsync.html
     */
    toInvokeAsync() {
        return this.to('InvokeAsync');
    }
    /**
     * Grants permission to invoke an AWS Lambda function
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifEventSourceToken()
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_Invoke.html
     */
    toInvokeFunction() {
        return this.to('InvokeFunction');
    }
    /**
     * Grants permission to invoke an AWS Lambda function through url
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFunctionUrlAuthType()
     * - .ifFunctionArn()
     * - .ifEventSourceToken()
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_InvokeFunctionUrl.html
     */
    toInvokeFunctionUrl() {
        return this.to('InvokeFunctionUrl');
    }
    /**
     * Grants permission to retrieve a list of aliases for an AWS Lambda function
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_ListAliases.html
     */
    toListAliases() {
        return this.to('ListAliases');
    }
    /**
     * Grants permission to retrieve a list of AWS Lambda code signing configs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_ListCodeSigningConfigs.html
     */
    toListCodeSigningConfigs() {
        return this.to('ListCodeSigningConfigs');
    }
    /**
     * Grants permission to retrieve a list of AWS Lambda event source mappings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_ListEventSourceMappings.html
     */
    toListEventSourceMappings() {
        return this.to('ListEventSourceMappings');
    }
    /**
     * Grants permission to retrieve a list of configurations for asynchronous invocation for a function
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_ListFunctionEventInvokeConfigs.html
     */
    toListFunctionEventInvokeConfigs() {
        return this.to('ListFunctionEventInvokeConfigs');
    }
    /**
     * Grants permission to read function url configurations for a function
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifFunctionUrlAuthType()
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_ListFunctionUrlConfigs.html
     */
    toListFunctionUrlConfigs() {
        return this.to('ListFunctionUrlConfigs');
    }
    /**
     * Grants permission to retrieve a list of AWS Lambda functions, with the version-specific configuration of each function
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_ListFunctions.html
     */
    toListFunctions() {
        return this.to('ListFunctions');
    }
    /**
     * Grants permission to retrieve a list of AWS Lambda functions by the code signing config assigned
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_ListFunctionsByCodeSigningConfig.html
     */
    toListFunctionsByCodeSigningConfig() {
        return this.to('ListFunctionsByCodeSigningConfig');
    }
    /**
     * Grants permission to retrieve a list of versions of an AWS Lambda layer
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_ListLayerVersions.html
     */
    toListLayerVersions() {
        return this.to('ListLayerVersions');
    }
    /**
     * Grants permission to retrieve a list of AWS Lambda layers, with details about the latest version of each layer
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_ListLayers.html
     */
    toListLayers() {
        return this.to('ListLayers');
    }
    /**
     * Grants permission to retrieve a list of provisioned concurrency configurations for an AWS Lambda function
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_ListProvisionedConcurrencyConfigs.html
     */
    toListProvisionedConcurrencyConfigs() {
        return this.to('ListProvisionedConcurrencyConfigs');
    }
    /**
     * Grants permission to retrieve a list of tags for an AWS Lambda function, event source mapping or code signing configuration resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_ListTags.html
     */
    toListTags() {
        return this.to('ListTags');
    }
    /**
     * Grants permission to retrieve a list of versions for an AWS Lambda function
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_ListVersionsByFunction.html
     */
    toListVersionsByFunction() {
        return this.to('ListVersionsByFunction');
    }
    /**
     * Grants permission to create an AWS Lambda layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_PublishLayerVersion.html
     */
    toPublishLayerVersion() {
        return this.to('PublishLayerVersion');
    }
    /**
     * Grants permission to create an AWS Lambda function version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_PublishVersion.html
     */
    toPublishVersion() {
        return this.to('PublishVersion');
    }
    /**
     * Grants permission to attach a code signing config to an AWS Lambda function
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCodeSigningConfigArn()
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_PutFunctionCodeSigningConfig.html
     */
    toPutFunctionCodeSigningConfig() {
        return this.to('PutFunctionCodeSigningConfig');
    }
    /**
     * Grants permission to configure reserved concurrency for an AWS Lambda function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_PutFunctionConcurrency.html
     */
    toPutFunctionConcurrency() {
        return this.to('PutFunctionConcurrency');
    }
    /**
     * Grants permission to configures options for asynchronous invocation on an AWS Lambda function, version, or alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_PutFunctionEventInvokeConfig.html
     */
    toPutFunctionEventInvokeConfig() {
        return this.to('PutFunctionEventInvokeConfig');
    }
    /**
     * Grants permission to update the recursion configuration of an AWS Lambda function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_PutFunctionRecursionConfig.html
     */
    toPutFunctionRecursionConfig() {
        return this.to('PutFunctionRecursionConfig');
    }
    /**
     * Grants permission to configure provisioned concurrency for an AWS Lambda function's alias or version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_PutProvisionedConcurrencyConfig.html
     */
    toPutProvisionedConcurrencyConfig() {
        return this.to('PutProvisionedConcurrencyConfig');
    }
    /**
     * Grants permission to update the runtime management configuration of an AWS Lambda function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_PutRuntimeManagementConfig.html
     */
    toPutRuntimeManagementConfig() {
        return this.to('PutRuntimeManagementConfig');
    }
    /**
     * Grants permission to remove a statement from the permissions policy for a version of an AWS Lambda layer
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_RemoveLayerVersionPermission.html
     */
    toRemoveLayerVersionPermission() {
        return this.to('RemoveLayerVersionPermission');
    }
    /**
     * Grants permission to revoke function-use permission from an AWS service or another account
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPrincipal()
     * - .ifFunctionUrlAuthType()
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_RemovePermission.html
     */
    toRemovePermission() {
        return this.to('RemovePermission');
    }
    /**
     * Grants permission to add tags to an AWS Lambda function, event source mapping or code signing configuration resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_TagResources.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from an AWS Lambda function, event source mapping or code signing configuration resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the configuration of an AWS Lambda function's alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateAlias.html
     */
    toUpdateAlias() {
        return this.to('UpdateAlias');
    }
    /**
     * Grants permission to update an AWS Lambda code signing config
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateCodeSigningConfig.html
     */
    toUpdateCodeSigningConfig() {
        return this.to('UpdateCodeSigningConfig');
    }
    /**
     * Grants permission to update the configuration of an AWS Lambda event source mapping
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFunctionArn()
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateEventSourceMapping.html
     */
    toUpdateEventSourceMapping() {
        return this.to('UpdateEventSourceMapping');
    }
    /**
     * Grants permission to update the code of an AWS Lambda function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateFunctionCode.html
     */
    toUpdateFunctionCode() {
        return this.to('UpdateFunctionCode');
    }
    /**
     * Grants permission to update the code signing config of an AWS Lambda function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateFunctionCodeSigningConfig.html
     */
    toUpdateFunctionCodeSigningConfig() {
        return this.to('UpdateFunctionCodeSigningConfig');
    }
    /**
     * Grants permission to modify the version-specific settings of an AWS Lambda function
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifLayer()
     * - .ifVpcIds()
     * - .ifSubnetIds()
     * - .ifSecurityGroupIds()
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateFunctionConfiguration.html
     */
    toUpdateFunctionConfiguration() {
        return this.to('UpdateFunctionConfiguration');
    }
    /**
     * Grants permission to modify the configuration for asynchronous invocation for an AWS Lambda function, version, or alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateFunctionEventInvokeConfig.html
     */
    toUpdateFunctionEventInvokeConfig() {
        return this.to('UpdateFunctionEventInvokeConfig');
    }
    /**
     * Grants permission to update a function url configuration for a Lambda function
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFunctionUrlAuthType()
     * - .ifFunctionArn()
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateFunctionUrlConfig.html
     */
    toUpdateFunctionUrlConfig() {
        return this.to('UpdateFunctionUrlConfig');
    }
    /**
     * Adds a resource of type code signing config to the statement
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html
     *
     * @param codeSigningConfigId - Identifier for the codeSigningConfigId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCodeSigningConfig(codeSigningConfigId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:lambda:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:code-signing-config:${codeSigningConfigId}`);
    }
    /**
     * Adds a resource of type eventSourceMapping to the statement
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html
     *
     * @param uUID - Identifier for the uUID.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEventSourceMapping(uUID, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:lambda:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:event-source-mapping:${uUID}`);
    }
    /**
     * Adds a resource of type function to the statement
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html
     *
     * @param functionName - Identifier for the functionName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFunction(functionName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:lambda:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:function:${functionName}`);
    }
    /**
     * Adds a resource of type function alias to the statement
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html
     *
     * @param functionName - Identifier for the functionName.
     * @param alias - Identifier for the alias.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFunctionAlias(functionName, alias, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:lambda:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:function:${functionName}:${alias}`);
    }
    /**
     * Adds a resource of type function version to the statement
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html
     *
     * @param functionName - Identifier for the functionName.
     * @param version - Identifier for the version.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFunctionVersion(functionName, version, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:lambda:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:function:${functionName}:${version}`);
    }
    /**
     * Adds a resource of type layer to the statement
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html
     *
     * @param layerName - Identifier for the layerName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onLayer(layerName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:lambda:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:layer:${layerName}`);
    }
    /**
     * Adds a resource of type layerVersion to the statement
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html
     *
     * @param layerName - Identifier for the layerName.
     * @param layerVersion - Identifier for the layerVersion.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onLayerVersion(layerName, layerVersion, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:lambda:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:layer:${layerName}:${layerVersion}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateCodeSigningConfig()
     * - .toCreateEventSourceMapping()
     * - .toCreateFunction()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - code signing config
     * - eventSourceMapping
     * - function
     * - function alias
     * - function version
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateCodeSigningConfig()
     * - .toCreateEventSourceMapping()
     * - .toCreateFunction()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ARN of an AWS Lambda code signing config
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html
     *
     * Applies to actions:
     * - .toCreateFunction()
     * - .toPutFunctionCodeSigningConfig()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifCodeSigningConfigArn(value, operator) {
        return this.if(`CodeSigningConfigArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the ID from a non-AWS event source configured for the AWS Lambda function
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html
     *
     * Applies to actions:
     * - .toInvokeFunction()
     * - .toInvokeFunctionUrl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEventSourceToken(value, operator) {
        return this.if(`EventSourceToken`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ARN of an AWS Lambda function
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html
     *
     * Applies to actions:
     * - .toCreateEventSourceMapping()
     * - .toCreateFunctionUrlConfig()
     * - .toDeleteEventSourceMapping()
     * - .toDeleteFunctionUrlConfig()
     * - .toGetEventSourceMapping()
     * - .toGetFunctionUrlConfig()
     * - .toInvokeFunctionUrl()
     * - .toUpdateEventSourceMapping()
     * - .toUpdateFunctionUrlConfig()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifFunctionArn(value, operator) {
        return this.if(`FunctionArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by authorization type specified in request. Available during CreateFunctionUrlConfig, UpdateFunctionUrlConfig, DeleteFunctionUrlConfig, GetFunctionUrlConfig, ListFunctionUrlConfig, AddPermission and RemovePermission operations
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html
     *
     * Applies to actions:
     * - .toAddPermission()
     * - .toCreateFunctionUrlConfig()
     * - .toDeleteFunctionUrlConfig()
     * - .toGetFunctionUrlConfig()
     * - .toInvokeFunctionUrl()
     * - .toListFunctionUrlConfigs()
     * - .toRemovePermission()
     * - .toUpdateFunctionUrlConfig()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFunctionUrlAuthType(value, operator) {
        return this.if(`FunctionUrlAuthType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ARN of a version of an AWS Lambda layer
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html
     *
     * Applies to actions:
     * - .toCreateFunction()
     * - .toUpdateFunctionConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLayer(value, operator) {
        return this.if(`Layer`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by restricting the AWS service or account that can invoke a function
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html
     *
     * Applies to actions:
     * - .toAddPermission()
     * - .toRemovePermission()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPrincipal(value, operator) {
        return this.if(`Principal`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ID of security groups configured for the AWS Lambda function
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html
     *
     * Applies to actions:
     * - .toCreateFunction()
     * - .toUpdateFunctionConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecurityGroupIds(value, operator) {
        return this.if(`SecurityGroupIds`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ARN of the AWS Lambda function from which the request originated
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifSourceFunctionArn(value, operator) {
        return this.if(`SourceFunctionArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the ID of subnets configured for the AWS Lambda function
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html
     *
     * Applies to actions:
     * - .toCreateFunction()
     * - .toUpdateFunctionConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSubnetIds(value, operator) {
        return this.if(`SubnetIds`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ID of the VPC configured for the AWS Lambda function
     *
     * https://docs.aws.amazon.com/lambda/latest/dg/lambda-api-permissions-ref.html
     *
     * Applies to actions:
     * - .toCreateFunction()
     * - .toUpdateFunctionConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpcIds(value, operator) {
        return this.if(`VpcIds`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [lambda](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awslambda.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'lambda';
        this.accessLevelList = {
            'Permissions management': [
                'AddLayerVersionPermission',
                'AddPermission',
                'DisableReplication',
                'EnableReplication',
                'RemoveLayerVersionPermission',
                'RemovePermission'
            ],
            Write: [
                'CreateAlias',
                'CreateCodeSigningConfig',
                'CreateEventSourceMapping',
                'CreateFunction',
                'CreateFunctionUrlConfig',
                'DeleteAlias',
                'DeleteCodeSigningConfig',
                'DeleteEventSourceMapping',
                'DeleteFunction',
                'DeleteFunctionCodeSigningConfig',
                'DeleteFunctionConcurrency',
                'DeleteFunctionEventInvokeConfig',
                'DeleteFunctionUrlConfig',
                'DeleteLayerVersion',
                'DeleteProvisionedConcurrencyConfig',
                'InvokeAsync',
                'InvokeFunction',
                'InvokeFunctionUrl',
                'PublishLayerVersion',
                'PublishVersion',
                'PutFunctionCodeSigningConfig',
                'PutFunctionConcurrency',
                'PutFunctionEventInvokeConfig',
                'PutFunctionRecursionConfig',
                'PutProvisionedConcurrencyConfig',
                'PutRuntimeManagementConfig',
                'UpdateAlias',
                'UpdateCodeSigningConfig',
                'UpdateEventSourceMapping',
                'UpdateFunctionCode',
                'UpdateFunctionCodeSigningConfig',
                'UpdateFunctionConfiguration',
                'UpdateFunctionEventInvokeConfig',
                'UpdateFunctionUrlConfig'
            ],
            Read: [
                'GetAccountSettings',
                'GetAlias',
                'GetCodeSigningConfig',
                'GetEventSourceMapping',
                'GetFunction',
                'GetFunctionCodeSigningConfig',
                'GetFunctionConcurrency',
                'GetFunctionConfiguration',
                'GetFunctionEventInvokeConfig',
                'GetFunctionRecursionConfig',
                'GetFunctionUrlConfig',
                'GetLayerVersion',
                'GetLayerVersionPolicy',
                'GetPolicy',
                'GetProvisionedConcurrencyConfig',
                'GetRuntimeManagementConfig',
                'ListTags'
            ],
            List: [
                'ListAliases',
                'ListCodeSigningConfigs',
                'ListEventSourceMappings',
                'ListFunctionEventInvokeConfigs',
                'ListFunctionUrlConfigs',
                'ListFunctions',
                'ListFunctionsByCodeSigningConfig',
                'ListLayerVersions',
                'ListLayers',
                'ListProvisionedConcurrencyConfigs',
                'ListVersionsByFunction'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Lambda = Lambda;
//# sourceMappingURL=data:application/json;base64,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