"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Launchwizard = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [launchwizard](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awslaunchwizard.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Launchwizard extends shared_1.PolicyStatement {
    /**
     * Grants permission to create an additional node
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/launchwizard/
     */
    toCreateAdditionalNode() {
        return this.to('CreateAdditionalNode');
    }
    /**
     * Grants permission to create a deployment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/launchwizard/latest/APIReference/API_CreateDeployment.html
     */
    toCreateDeployment() {
        return this.to('CreateDeployment');
    }
    /**
     * Grants permission to create an application settings set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/launchwizard/
     */
    toCreateSettingsSet() {
        return this.to('CreateSettingsSet');
    }
    /**
     * Grants permission to delete an additional node
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/launchwizard/
     */
    toDeleteAdditionalNode() {
        return this.to('DeleteAdditionalNode');
    }
    /**
     * Grants permission to delete an application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/launchwizard/
     */
    toDeleteApp() {
        return this.to('DeleteApp');
    }
    /**
     * Grants permission to delete a deployment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/launchwizard/latest/APIReference/API_DeleteDeployment.html
     */
    toDeleteDeployment() {
        return this.to('DeleteDeployment');
    }
    /**
     * Grants permission to delete a settings set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/launchwizard/
     */
    toDeleteSettingsSet() {
        return this.to('DeleteSettingsSet');
    }
    /**
     * Grants permission to describe an additional node
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/launchwizard/
     */
    toDescribeAdditionalNode() {
        return this.to('DescribeAdditionalNode');
    }
    /**
     * Grants permission to describe provisioning applications
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/launchwizard/
     */
    toDescribeProvisionedApp() {
        return this.to('DescribeProvisionedApp');
    }
    /**
     * Grants permission to describe provisioning events
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/launchwizard/
     */
    toDescribeProvisioningEvents() {
        return this.to('DescribeProvisioningEvents');
    }
    /**
     * Grants permission to describe an application settings set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/launchwizard/
     */
    toDescribeSettingsSet() {
        return this.to('DescribeSettingsSet');
    }
    /**
     * Grants permission to get a deployment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/launchwizard/latest/APIReference/API_GetDeployment.html
     */
    toGetDeployment() {
        return this.to('GetDeployment');
    }
    /**
     * Grants permission to get infrastructure suggestion
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/launchwizard/
     */
    toGetInfrastructureSuggestion() {
        return this.to('GetInfrastructureSuggestion');
    }
    /**
     * Grants permission to get customer's ip address
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/launchwizard/
     */
    toGetIpAddress() {
        return this.to('GetIpAddress');
    }
    /**
     * Grants permission to get resource cost estimate
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/launchwizard/
     */
    toGetResourceCostEstimate() {
        return this.to('GetResourceCostEstimate');
    }
    /**
     * Grants permission to get recommendation for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/launchwizard/
     */
    toGetResourceRecommendation() {
        return this.to('GetResourceRecommendation');
    }
    /**
     * Grants permission to get a settings set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/launchwizard/
     */
    toGetSettingsSet() {
        return this.to('GetSettingsSet');
    }
    /**
     * Grants permission to get a workload
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/launchwizard/latest/APIReference/API_GetWorkload.html
     */
    toGetWorkload() {
        return this.to('GetWorkload');
    }
    /**
     * Grants permission to get a workload's asset
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/launchwizard/
     */
    toGetWorkloadAsset() {
        return this.to('GetWorkloadAsset');
    }
    /**
     * Grants permission to get workload assets
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/launchwizard/
     */
    toGetWorkloadAssets() {
        return this.to('GetWorkloadAssets');
    }
    /**
     * Grants permission to get a deployment pattern
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/launchwizard/latest/APIReference/API_GetWorkloadDeploymentPattern.html
     */
    toGetWorkloadDeploymentPattern() {
        return this.to('GetWorkloadDeploymentPattern');
    }
    /**
     * Grants permission to list additional nodes
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/launchwizard/
     */
    toListAdditionalNodes() {
        return this.to('ListAdditionalNodes');
    }
    /**
     * Grants permission to list the allowed resources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/launchwizard/
     */
    toListAllowedResources() {
        return this.to('ListAllowedResources');
    }
    /**
     * Grants permission to list the events that occured during a deployment
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/launchwizard/latest/APIReference/API_ListDeploymentEvents.html
     */
    toListDeploymentEvents() {
        return this.to('ListDeploymentEvents');
    }
    /**
     * Grants permission to list deployments
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/launchwizard/latest/APIReference/API_ListDeployments.html
     */
    toListDeployments() {
        return this.to('ListDeployments');
    }
    /**
     * Grants permission to list provisioning applications
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/launchwizard/
     */
    toListProvisionedApps() {
        return this.to('ListProvisionedApps');
    }
    /**
     * Grants permission to list the cost estimates of resources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/launchwizard/
     */
    toListResourceCostEstimates() {
        return this.to('ListResourceCostEstimates');
    }
    /**
     * Grants permission to list settings sets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/launchwizard/
     */
    toListSettingsSets() {
        return this.to('ListSettingsSets');
    }
    /**
     * Grants permission to list tags for a LaunchWizard resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/launchwizard/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list deployment options of a given workload
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/launchwizard/
     */
    toListWorkloadDeploymentOptions() {
        return this.to('ListWorkloadDeploymentOptions');
    }
    /**
     * Grants permission to list the deployment patterns of a workload
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/launchwizard/latest/APIReference/API_ListWorkloadDeploymentPatterns.html
     */
    toListWorkloadDeploymentPatterns() {
        return this.to('ListWorkloadDeploymentPatterns');
    }
    /**
     * Grants permission to list workloads
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/launchwizard/latest/APIReference/API_ListWorkloads.html
     */
    toListWorkloads() {
        return this.to('ListWorkloads');
    }
    /**
     * Grants permission to create a settings set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/launchwizard/
     */
    toPutSettingsSet() {
        return this.to('PutSettingsSet');
    }
    /**
     * Grants permission to start a provisioning
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/launchwizard/
     */
    toStartProvisioning() {
        return this.to('StartProvisioning');
    }
    /**
     * Grants permission to tag a LaunchWizard resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/launchwizard/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a LaunchWizard resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/launchwizard/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an application settings set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/launchwizard/
     */
    toUpdateSettingsSet() {
        return this.to('UpdateSettingsSet');
    }
    /**
     * Adds a resource of type deployment to the statement
     *
     * @param deploymentId - Identifier for the deploymentId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDeployment(deploymentId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:launchwizard:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:deployment/${deploymentId}`);
    }
    /**
     * Filters access based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateDeployment()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toDeleteDeployment()
     * - .toGetDeployment()
     * - .toListTagsForResource()
     * - .toTagResource()
     *
     * Applies to resource types:
     * - deployment
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateDeployment()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [launchwizard](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awslaunchwizard.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'launchwizard';
        this.accessLevelList = {
            Write: [
                'CreateAdditionalNode',
                'CreateDeployment',
                'CreateSettingsSet',
                'DeleteAdditionalNode',
                'DeleteApp',
                'DeleteDeployment',
                'DeleteSettingsSet',
                'PutSettingsSet',
                'StartProvisioning',
                'UpdateSettingsSet'
            ],
            Read: [
                'DescribeAdditionalNode',
                'DescribeProvisionedApp',
                'DescribeProvisioningEvents',
                'DescribeSettingsSet',
                'GetDeployment',
                'GetInfrastructureSuggestion',
                'GetIpAddress',
                'GetResourceCostEstimate',
                'GetResourceRecommendation',
                'GetSettingsSet',
                'GetWorkload',
                'GetWorkloadAsset',
                'GetWorkloadAssets',
                'GetWorkloadDeploymentPattern',
                'ListTagsForResource'
            ],
            List: [
                'ListAdditionalNodes',
                'ListAllowedResources',
                'ListDeploymentEvents',
                'ListDeployments',
                'ListProvisionedApps',
                'ListResourceCostEstimates',
                'ListSettingsSets',
                'ListWorkloadDeploymentOptions',
                'ListWorkloadDeploymentPatterns',
                'ListWorkloads'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Launchwizard = Launchwizard;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibGF1bmNod2l6YXJkLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsibGF1bmNod2l6YXJkLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUNBLHlDQUF5RDtBQUd6RDs7OztHQUlHO0FBQ0gsTUFBYSxZQUFhLFNBQVEsd0JBQWU7SUFHL0M7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFrQixDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxXQUFXO1FBQ2hCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxXQUFXLENBQUMsQ0FBQztJQUM5QixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0JBQXdCO1FBQzdCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO0lBQzNDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw0QkFBNEI7UUFDakMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDRCQUE0QixDQUFDLENBQUM7SUFDL0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDZCQUE2QjtRQUNsQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLENBQUMsQ0FBQztJQUNoRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsY0FBYyxDQUFDLENBQUM7SUFDakMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLENBQUMsQ0FBQztJQUM1QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMkJBQTJCO1FBQ2hDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO0lBQzlDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ2hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFrQixDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksOEJBQThCO1FBQ25DLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO0lBQ2pELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGlCQUFpQixDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMkJBQTJCO1FBQ2hDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO0lBQzlDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFrQixDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksK0JBQStCO1FBQ3BDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO0lBQ2xELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQ0FBZ0M7UUFDckMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdDQUFnQyxDQUFDLENBQUM7SUFDbkQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGVBQWU7UUFDcEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxDQUFDLENBQUM7SUFDaEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGVBQWU7UUFDcEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQWtERDs7Ozs7Ozs7OztPQVVHO0lBQ0ksWUFBWSxDQUFDLFlBQW9CLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDN0YsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsaUJBQWtCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxlQUFnQixZQUFhLEVBQUUsQ0FBQyxDQUFDO0lBQ2hMLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxlQUFlLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBNEI7UUFDM0YsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFtQixNQUFPLEVBQUUsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2hGLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FpQkc7SUFDSSxnQkFBZ0IsQ0FBQyxNQUFjLEVBQUUsS0FBd0IsRUFBRSxRQUE0QjtRQUM1RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW9CLE1BQU8sRUFBRSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDakYsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLFlBQVksQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ3hFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRSxDQUFDO0lBRUQ7OztPQUdHO0lBQ0gsWUFBWSxLQUFnQztRQUMxQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7UUFyaEJSLGtCQUFhLEdBQUcsY0FBYyxDQUFDO1FBeVo1QixvQkFBZSxHQUFvQjtZQUMzQyxLQUFLLEVBQUU7Z0JBQ0wsc0JBQXNCO2dCQUN0QixrQkFBa0I7Z0JBQ2xCLG1CQUFtQjtnQkFDbkIsc0JBQXNCO2dCQUN0QixXQUFXO2dCQUNYLGtCQUFrQjtnQkFDbEIsbUJBQW1CO2dCQUNuQixnQkFBZ0I7Z0JBQ2hCLG1CQUFtQjtnQkFDbkIsbUJBQW1CO2FBQ3BCO1lBQ0QsSUFBSSxFQUFFO2dCQUNKLHdCQUF3QjtnQkFDeEIsd0JBQXdCO2dCQUN4Qiw0QkFBNEI7Z0JBQzVCLHFCQUFxQjtnQkFDckIsZUFBZTtnQkFDZiw2QkFBNkI7Z0JBQzdCLGNBQWM7Z0JBQ2QseUJBQXlCO2dCQUN6QiwyQkFBMkI7Z0JBQzNCLGdCQUFnQjtnQkFDaEIsYUFBYTtnQkFDYixrQkFBa0I7Z0JBQ2xCLG1CQUFtQjtnQkFDbkIsOEJBQThCO2dCQUM5QixxQkFBcUI7YUFDdEI7WUFDRCxJQUFJLEVBQUU7Z0JBQ0oscUJBQXFCO2dCQUNyQixzQkFBc0I7Z0JBQ3RCLHNCQUFzQjtnQkFDdEIsaUJBQWlCO2dCQUNqQixxQkFBcUI7Z0JBQ3JCLDJCQUEyQjtnQkFDM0Isa0JBQWtCO2dCQUNsQiwrQkFBK0I7Z0JBQy9CLGdDQUFnQztnQkFDaEMsZUFBZTthQUNoQjtZQUNELE9BQU8sRUFBRTtnQkFDUCxhQUFhO2dCQUNiLGVBQWU7YUFDaEI7U0FDRixDQUFDO0lBK0VGLENBQUM7Q0FDRjtBQXhoQkQsb0NBd2hCQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEFjY2Vzc0xldmVsTGlzdCB9IGZyb20gJy4uLy4uL3NoYXJlZC9hY2Nlc3MtbGV2ZWwnO1xuaW1wb3J0IHsgUG9saWN5U3RhdGVtZW50LCBPcGVyYXRvciB9IGZyb20gJy4uLy4uL3NoYXJlZCc7XG5pbXBvcnQgeyBhd3NfaWFtIGFzIGlhbSB9IGZyb20gXCJhd3MtY2RrLWxpYlwiO1xuXG4vKipcbiAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbbGF1bmNod2l6YXJkXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hd3NsYXVuY2h3aXphcmQuaHRtbCkuXG4gKlxuICogQHBhcmFtIHNpZCBbU0lEXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX3NpZC5odG1sKSBvZiB0aGUgc3RhdGVtZW50XG4gKi9cbmV4cG9ydCBjbGFzcyBMYXVuY2h3aXphcmQgZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnQge1xuICBwdWJsaWMgc2VydmljZVByZWZpeCA9ICdsYXVuY2h3aXphcmQnO1xuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYW4gYWRkaXRpb25hbCBub2RlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xhdW5jaHdpemFyZC9cbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUFkZGl0aW9uYWxOb2RlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVBZGRpdGlvbmFsTm9kZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIGRlcGxveW1lbnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGF1bmNod2l6YXJkL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZURlcGxveW1lbnQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlRGVwbG95bWVudCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlRGVwbG95bWVudCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhbiBhcHBsaWNhdGlvbiBzZXR0aW5ncyBzZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGF1bmNod2l6YXJkL1xuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlU2V0dGluZ3NTZXQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZVNldHRpbmdzU2V0Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGFuIGFkZGl0aW9uYWwgbm9kZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9sYXVuY2h3aXphcmQvXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVBZGRpdGlvbmFsTm9kZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlQWRkaXRpb25hbE5vZGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYW4gYXBwbGljYXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGF1bmNod2l6YXJkL1xuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlQXBwKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVBcHAnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBkZXBsb3ltZW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xhdW5jaHdpemFyZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVEZXBsb3ltZW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZURlcGxveW1lbnQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZURlcGxveW1lbnQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBzZXR0aW5ncyBzZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGF1bmNod2l6YXJkL1xuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlU2V0dGluZ3NTZXQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZVNldHRpbmdzU2V0Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVzY3JpYmUgYW4gYWRkaXRpb25hbCBub2RlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGF1bmNod2l6YXJkL1xuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVBZGRpdGlvbmFsTm9kZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVBZGRpdGlvbmFsTm9kZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlc2NyaWJlIHByb3Zpc2lvbmluZyBhcHBsaWNhdGlvbnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9sYXVuY2h3aXphcmQvXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZVByb3Zpc2lvbmVkQXBwKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZVByb3Zpc2lvbmVkQXBwJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVzY3JpYmUgcHJvdmlzaW9uaW5nIGV2ZW50c1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xhdW5jaHdpemFyZC9cbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlUHJvdmlzaW9uaW5nRXZlbnRzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZVByb3Zpc2lvbmluZ0V2ZW50cycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlc2NyaWJlIGFuIGFwcGxpY2F0aW9uIHNldHRpbmdzIHNldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xhdW5jaHdpemFyZC9cbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlU2V0dGluZ3NTZXQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlU2V0dGluZ3NTZXQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgYSBkZXBsb3ltZW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGF1bmNod2l6YXJkL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldERlcGxveW1lbnQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0RGVwbG95bWVudCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0RGVwbG95bWVudCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBpbmZyYXN0cnVjdHVyZSBzdWdnZXN0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGF1bmNod2l6YXJkL1xuICAgKi9cbiAgcHVibGljIHRvR2V0SW5mcmFzdHJ1Y3R1cmVTdWdnZXN0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRJbmZyYXN0cnVjdHVyZVN1Z2dlc3Rpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgY3VzdG9tZXIncyBpcCBhZGRyZXNzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGF1bmNod2l6YXJkL1xuICAgKi9cbiAgcHVibGljIHRvR2V0SXBBZGRyZXNzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRJcEFkZHJlc3MnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgcmVzb3VyY2UgY29zdCBlc3RpbWF0ZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xhdW5jaHdpemFyZC9cbiAgICovXG4gIHB1YmxpYyB0b0dldFJlc291cmNlQ29zdEVzdGltYXRlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRSZXNvdXJjZUNvc3RFc3RpbWF0ZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCByZWNvbW1lbmRhdGlvbiBmb3IgYSByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xhdW5jaHdpemFyZC9cbiAgICovXG4gIHB1YmxpYyB0b0dldFJlc291cmNlUmVjb21tZW5kYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldFJlc291cmNlUmVjb21tZW5kYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgYSBzZXR0aW5ncyBzZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9sYXVuY2h3aXphcmQvXG4gICAqL1xuICBwdWJsaWMgdG9HZXRTZXR0aW5nc1NldCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0U2V0dGluZ3NTZXQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgYSB3b3JrbG9hZFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xhdW5jaHdpemFyZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXRXb3JrbG9hZC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRXb3JrbG9hZCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0V29ya2xvYWQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgYSB3b3JrbG9hZCdzIGFzc2V0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGF1bmNod2l6YXJkL1xuICAgKi9cbiAgcHVibGljIHRvR2V0V29ya2xvYWRBc3NldCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0V29ya2xvYWRBc3NldCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCB3b3JrbG9hZCBhc3NldHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9sYXVuY2h3aXphcmQvXG4gICAqL1xuICBwdWJsaWMgdG9HZXRXb3JrbG9hZEFzc2V0cygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0V29ya2xvYWRBc3NldHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgYSBkZXBsb3ltZW50IHBhdHRlcm5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9sYXVuY2h3aXphcmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0V29ya2xvYWREZXBsb3ltZW50UGF0dGVybi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRXb3JrbG9hZERlcGxveW1lbnRQYXR0ZXJuKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRXb3JrbG9hZERlcGxveW1lbnRQYXR0ZXJuJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBhZGRpdGlvbmFsIG5vZGVzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGF1bmNod2l6YXJkL1xuICAgKi9cbiAgcHVibGljIHRvTGlzdEFkZGl0aW9uYWxOb2RlcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdEFkZGl0aW9uYWxOb2RlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgdGhlIGFsbG93ZWQgcmVzb3VyY2VzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGF1bmNod2l6YXJkL1xuICAgKi9cbiAgcHVibGljIHRvTGlzdEFsbG93ZWRSZXNvdXJjZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RBbGxvd2VkUmVzb3VyY2VzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCB0aGUgZXZlbnRzIHRoYXQgb2NjdXJlZCBkdXJpbmcgYSBkZXBsb3ltZW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGF1bmNod2l6YXJkL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3REZXBsb3ltZW50RXZlbnRzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3REZXBsb3ltZW50RXZlbnRzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0RGVwbG95bWVudEV2ZW50cycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgZGVwbG95bWVudHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9sYXVuY2h3aXphcmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdERlcGxveW1lbnRzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3REZXBsb3ltZW50cygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdERlcGxveW1lbnRzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBwcm92aXNpb25pbmcgYXBwbGljYXRpb25zXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGF1bmNod2l6YXJkL1xuICAgKi9cbiAgcHVibGljIHRvTGlzdFByb3Zpc2lvbmVkQXBwcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFByb3Zpc2lvbmVkQXBwcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgdGhlIGNvc3QgZXN0aW1hdGVzIG9mIHJlc291cmNlc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xhdW5jaHdpemFyZC9cbiAgICovXG4gIHB1YmxpYyB0b0xpc3RSZXNvdXJjZUNvc3RFc3RpbWF0ZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RSZXNvdXJjZUNvc3RFc3RpbWF0ZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHNldHRpbmdzIHNldHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9sYXVuY2h3aXphcmQvXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0U2V0dGluZ3NTZXRzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0U2V0dGluZ3NTZXRzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCB0YWdzIGZvciBhIExhdW5jaFdpemFyZCByZXNvdXJjZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9sYXVuY2h3aXphcmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdFRhZ3NGb3JSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0VGFnc0ZvclJlc291cmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0VGFnc0ZvclJlc291cmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBkZXBsb3ltZW50IG9wdGlvbnMgb2YgYSBnaXZlbiB3b3JrbG9hZFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xhdW5jaHdpemFyZC9cbiAgICovXG4gIHB1YmxpYyB0b0xpc3RXb3JrbG9hZERlcGxveW1lbnRPcHRpb25zKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0V29ya2xvYWREZXBsb3ltZW50T3B0aW9ucycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgdGhlIGRlcGxveW1lbnQgcGF0dGVybnMgb2YgYSB3b3JrbG9hZFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xhdW5jaHdpemFyZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0V29ya2xvYWREZXBsb3ltZW50UGF0dGVybnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFdvcmtsb2FkRGVwbG95bWVudFBhdHRlcm5zKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0V29ya2xvYWREZXBsb3ltZW50UGF0dGVybnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHdvcmtsb2Fkc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xhdW5jaHdpemFyZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0V29ya2xvYWRzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RXb3JrbG9hZHMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RXb3JrbG9hZHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBzZXR0aW5ncyBzZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGF1bmNod2l6YXJkL1xuICAgKi9cbiAgcHVibGljIHRvUHV0U2V0dGluZ3NTZXQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1B1dFNldHRpbmdzU2V0Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3RhcnQgYSBwcm92aXNpb25pbmdcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGF1bmNod2l6YXJkL1xuICAgKi9cbiAgcHVibGljIHRvU3RhcnRQcm92aXNpb25pbmcoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1N0YXJ0UHJvdmlzaW9uaW5nJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdGFnIGEgTGF1bmNoV2l6YXJkIHJlc291cmNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xhdW5jaHdpemFyZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9UYWdSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9UYWdSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVGFnUmVzb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1bnRhZyBhIExhdW5jaFdpemFyZCByZXNvdXJjZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9sYXVuY2h3aXphcmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVW50YWdSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VbnRhZ1Jlc291cmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVbnRhZ1Jlc291cmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGFuIGFwcGxpY2F0aW9uIHNldHRpbmdzIHNldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9sYXVuY2h3aXphcmQvXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVTZXR0aW5nc1NldCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlU2V0dGluZ3NTZXQnKTtcbiAgfVxuXG4gIHByb3RlY3RlZCBhY2Nlc3NMZXZlbExpc3Q6IEFjY2Vzc0xldmVsTGlzdCA9IHtcbiAgICBXcml0ZTogW1xuICAgICAgJ0NyZWF0ZUFkZGl0aW9uYWxOb2RlJyxcbiAgICAgICdDcmVhdGVEZXBsb3ltZW50JyxcbiAgICAgICdDcmVhdGVTZXR0aW5nc1NldCcsXG4gICAgICAnRGVsZXRlQWRkaXRpb25hbE5vZGUnLFxuICAgICAgJ0RlbGV0ZUFwcCcsXG4gICAgICAnRGVsZXRlRGVwbG95bWVudCcsXG4gICAgICAnRGVsZXRlU2V0dGluZ3NTZXQnLFxuICAgICAgJ1B1dFNldHRpbmdzU2V0JyxcbiAgICAgICdTdGFydFByb3Zpc2lvbmluZycsXG4gICAgICAnVXBkYXRlU2V0dGluZ3NTZXQnXG4gICAgXSxcbiAgICBSZWFkOiBbXG4gICAgICAnRGVzY3JpYmVBZGRpdGlvbmFsTm9kZScsXG4gICAgICAnRGVzY3JpYmVQcm92aXNpb25lZEFwcCcsXG4gICAgICAnRGVzY3JpYmVQcm92aXNpb25pbmdFdmVudHMnLFxuICAgICAgJ0Rlc2NyaWJlU2V0dGluZ3NTZXQnLFxuICAgICAgJ0dldERlcGxveW1lbnQnLFxuICAgICAgJ0dldEluZnJhc3RydWN0dXJlU3VnZ2VzdGlvbicsXG4gICAgICAnR2V0SXBBZGRyZXNzJyxcbiAgICAgICdHZXRSZXNvdXJjZUNvc3RFc3RpbWF0ZScsXG4gICAgICAnR2V0UmVzb3VyY2VSZWNvbW1lbmRhdGlvbicsXG4gICAgICAnR2V0U2V0dGluZ3NTZXQnLFxuICAgICAgJ0dldFdvcmtsb2FkJyxcbiAgICAgICdHZXRXb3JrbG9hZEFzc2V0JyxcbiAgICAgICdHZXRXb3JrbG9hZEFzc2V0cycsXG4gICAgICAnR2V0V29ya2xvYWREZXBsb3ltZW50UGF0dGVybicsXG4gICAgICAnTGlzdFRhZ3NGb3JSZXNvdXJjZSdcbiAgICBdLFxuICAgIExpc3Q6IFtcbiAgICAgICdMaXN0QWRkaXRpb25hbE5vZGVzJyxcbiAgICAgICdMaXN0QWxsb3dlZFJlc291cmNlcycsXG4gICAgICAnTGlzdERlcGxveW1lbnRFdmVudHMnLFxuICAgICAgJ0xpc3REZXBsb3ltZW50cycsXG4gICAgICAnTGlzdFByb3Zpc2lvbmVkQXBwcycsXG4gICAgICAnTGlzdFJlc291cmNlQ29zdEVzdGltYXRlcycsXG4gICAgICAnTGlzdFNldHRpbmdzU2V0cycsXG4gICAgICAnTGlzdFdvcmtsb2FkRGVwbG95bWVudE9wdGlvbnMnLFxuICAgICAgJ0xpc3RXb3JrbG9hZERlcGxveW1lbnRQYXR0ZXJucycsXG4gICAgICAnTGlzdFdvcmtsb2FkcydcbiAgICBdLFxuICAgIFRhZ2dpbmc6IFtcbiAgICAgICdUYWdSZXNvdXJjZScsXG4gICAgICAnVW50YWdSZXNvdXJjZSdcbiAgICBdXG4gIH07XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGRlcGxveW1lbnQgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBAcGFyYW0gZGVwbG95bWVudElkIC0gSWRlbnRpZmllciBmb3IgdGhlIGRlcGxveW1lbnRJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uRGVwbG95bWVudChkZXBsb3ltZW50SWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpsYXVuY2h3aXphcmQ6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmRlcGxveW1lbnQvJHsgZGVwbG95bWVudElkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBiYXNlZCBvbiB0aGUgcHJlc2VuY2Ugb2YgdGFnIGtleS12YWx1ZSBwYWlycyBpbiB0aGUgcmVxdWVzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtcmVxdWVzdHRhZ1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQ3JlYXRlRGVwbG95bWVudCgpXG4gICAqIC0gLnRvVGFnUmVzb3VyY2UoKVxuICAgKlxuICAgKiBAcGFyYW0gdGFnS2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzUmVxdWVzdFRhZyh0YWdLZXk6IHN0cmluZywgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpSZXF1ZXN0VGFnLyR7IHRhZ0tleSB9YCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYmFzZWQgb24gdGFnIGtleS12YWx1ZSBwYWlycyBhdHRhY2hlZCB0byB0aGUgcmVzb3VyY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXJlc291cmNldGFnXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9EZWxldGVEZXBsb3ltZW50KClcbiAgICogLSAudG9HZXREZXBsb3ltZW50KClcbiAgICogLSAudG9MaXN0VGFnc0ZvclJlc291cmNlKClcbiAgICogLSAudG9UYWdSZXNvdXJjZSgpXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gZGVwbG95bWVudFxuICAgKlxuICAgKiBAcGFyYW0gdGFnS2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzUmVzb3VyY2VUYWcodGFnS2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6UmVzb3VyY2VUYWcvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBiYXNlZCBvbiB0aGUgcHJlc2VuY2Ugb2YgdGFnIGtleXMgaW4gdGhlIHJlcXVlc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXRhZ2tleXNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0NyZWF0ZURlcGxveW1lbnQoKVxuICAgKiAtIC50b1RhZ1Jlc291cmNlKClcbiAgICogLSAudG9VbnRhZ1Jlc291cmNlKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NUYWdLZXlzKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6VGFnS2V5c2AsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbbGF1bmNod2l6YXJkXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hd3NsYXVuY2h3aXphcmQuaHRtbCkuXG4gICAqXG4gICAqL1xuICBjb25zdHJ1Y3Rvcihwcm9wcz86IGlhbS5Qb2xpY3lTdGF0ZW1lbnRQcm9wcykge1xuICAgIHN1cGVyKHByb3BzKTtcbiAgfVxufVxuIl19