"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LexV2 = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [lex-v2](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonlexv2.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class LexV2 extends shared_1.PolicyStatement {
    /**
     * Grants permission to create new items in an existing custom vocabulary
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_BatchCreateCustomVocabularyItem.html
     */
    toBatchCreateCustomVocabularyItem() {
        return this.to('BatchCreateCustomVocabularyItem');
    }
    /**
     * Grants permission to delete existing items in an existing custom vocabulary
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_BatchDeleteCustomVocabularyItem.html
     */
    toBatchDeleteCustomVocabularyItem() {
        return this.to('BatchDeleteCustomVocabularyItem');
    }
    /**
     * Grants permission to update existing items in an existing custom vocabulary
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_BatchUpdateCustomVocabularyItem.html
     */
    toBatchUpdateCustomVocabularyItem() {
        return this.to('BatchUpdateCustomVocabularyItem');
    }
    /**
     * Grants permission to build an existing bot locale in a bot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_BuildBotLocale.html
     */
    toBuildBotLocale() {
        return this.to('BuildBotLocale');
    }
    /**
     * Grants permission to create a new bot and a test bot alias pointing to the DRAFT bot version
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_CreateBot.html
     */
    toCreateBot() {
        return this.to('CreateBot');
    }
    /**
     * Grants permission to create a new bot alias in a bot
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_CreateBotAlias.html
     */
    toCreateBotAlias() {
        return this.to('CreateBotAlias');
    }
    /**
     * Grants permission to create a bot channel in an existing bot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/dg/deploying-messaging-platform.html
     */
    toCreateBotChannel() {
        return this.to('CreateBotChannel');
    }
    /**
     * Grants permission to create a new bot locale in an existing bot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_CreateBotLocale.html
     */
    toCreateBotLocale() {
        return this.to('CreateBotLocale');
    }
    /**
     * Grants permission to create bot replica for a bot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_CreateBotReplica.html
     */
    toCreateBotReplica() {
        return this.to('CreateBotReplica');
    }
    /**
     * Grants permission to create a new version of an existing bot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_CreateBotVersion.html
     */
    toCreateBotVersion() {
        return this.to('CreateBotVersion');
    }
    /**
     * Grants permission to create a new custom vocabulary in an existing bot locale
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/dg/vocab.html
     */
    toCreateCustomVocabulary() {
        return this.to('CreateCustomVocabulary');
    }
    /**
     * Grants permission to create an export for an existing resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_CreateExport.html
     */
    toCreateExport() {
        return this.to('CreateExport');
    }
    /**
     * Grants permission to create a new intent in an existing bot locale
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_CreateIntent.html
     */
    toCreateIntent() {
        return this.to('CreateIntent');
    }
    /**
     * Grants permission to create a new resource policy for a Lex resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_CreateResourcePolicy.html
     */
    toCreateResourcePolicy() {
        return this.to('CreateResourcePolicy');
    }
    /**
     * Grants permission to create a new slot in an intent
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_CreateSlot.html
     */
    toCreateSlot() {
        return this.to('CreateSlot');
    }
    /**
     * Grants permission to create a new slot type in an existing bot locale
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_CreateSlotType.html
     */
    toCreateSlotType() {
        return this.to('CreateSlotType');
    }
    /**
     * Grants permission to import a new test-set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/dg/create-test-set-from-CSV.html
     */
    toCreateTestSet() {
        return this.to('CreateTestSet');
    }
    /**
     * Grants permission to create a test set discrepancy report
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_CreateTestSetDiscrepancyReport.html
     */
    toCreateTestSetDiscrepancyReport() {
        return this.to('CreateTestSetDiscrepancyReport');
    }
    /**
     * Grants permission to create an upload url for import file
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_CreateUploadUrl.html
     */
    toCreateUploadUrl() {
        return this.to('CreateUploadUrl');
    }
    /**
     * Grants permission to delete an existing bot
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - lex:DeleteBotAlias
     * - lex:DeleteBotChannel
     * - lex:DeleteBotLocale
     * - lex:DeleteBotVersion
     * - lex:DeleteIntent
     * - lex:DeleteSlot
     * - lex:DeleteSlotType
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DeleteBot.html
     */
    toDeleteBot() {
        return this.to('DeleteBot');
    }
    /**
     * Grants permission to delete an existing bot alias in a bot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DeleteBotAlias.html
     */
    toDeleteBotAlias() {
        return this.to('DeleteBotAlias');
    }
    /**
     * Grants permission to delete an existing bot channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/dg/deploying-messaging-platform.html
     */
    toDeleteBotChannel() {
        return this.to('DeleteBotChannel');
    }
    /**
     * Grants permission to delete an existing bot locale in a bot
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - lex:DeleteIntent
     * - lex:DeleteSlot
     * - lex:DeleteSlotType
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DeleteBotLocale.html
     */
    toDeleteBotLocale() {
        return this.to('DeleteBotLocale');
    }
    /**
     * Grants permission to delete an existing bot replica
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DeleteBotReplica.html
     */
    toDeleteBotReplica() {
        return this.to('DeleteBotReplica');
    }
    /**
     * Grants permission to delete an existing bot version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DeleteBotVersion.html
     */
    toDeleteBotVersion() {
        return this.to('DeleteBotVersion');
    }
    /**
     * Grants permission to delete an existing custom vocabulary in a bot locale
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DeleteCustomVocabulary.html
     */
    toDeleteCustomVocabulary() {
        return this.to('DeleteCustomVocabulary');
    }
    /**
     * Grants permission to delete an existing export
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DeleteExport.html
     */
    toDeleteExport() {
        return this.to('DeleteExport');
    }
    /**
     * Grants permission to delete an existing import
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DeleteImport.html
     */
    toDeleteImport() {
        return this.to('DeleteImport');
    }
    /**
     * Grants permission to delete an existing intent in a bot locale
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DeleteIntent.html
     */
    toDeleteIntent() {
        return this.to('DeleteIntent');
    }
    /**
     * Grants permission to delete an existing resource policy for a Lex resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to delete session information for a bot alias and user ID
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_runtime_DeleteSession.html
     */
    toDeleteSession() {
        return this.to('DeleteSession');
    }
    /**
     * Grants permission to delete an existing slot in an intent
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DeleteSlot.html
     */
    toDeleteSlot() {
        return this.to('DeleteSlot');
    }
    /**
     * Grants permission to delete an existing slot type in a bot locale
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DeleteSlotType.html
     */
    toDeleteSlotType() {
        return this.to('DeleteSlotType');
    }
    /**
     * Grants permission to delete an existing test set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DeleteTestSet.html
     */
    toDeleteTestSet() {
        return this.to('DeleteTestSet');
    }
    /**
     * Grants permission to delete utterance data for a bot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DeleteUtterances.html
     */
    toDeleteUtterances() {
        return this.to('DeleteUtterances');
    }
    /**
     * Grants permission to retrieve an existing bot
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeBot.html
     */
    toDescribeBot() {
        return this.to('DescribeBot');
    }
    /**
     * Grants permission to retrieve an existing bot alias
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeBotAlias.html
     */
    toDescribeBotAlias() {
        return this.to('DescribeBotAlias');
    }
    /**
     * Grants permission to retrieve an existing bot channel
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/dg/deploying-messaging-platform.html
     */
    toDescribeBotChannel() {
        return this.to('DescribeBotChannel');
    }
    /**
     * Grants permission to retrieve an existing bot locale
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeBotLocale.html
     */
    toDescribeBotLocale() {
        return this.to('DescribeBotLocale');
    }
    /**
     * Grants permission to retrieve metadata information about a bot recommendation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeBotRecommendation.html
     */
    toDescribeBotRecommendation() {
        return this.to('DescribeBotRecommendation');
    }
    /**
     * Grants permission to retrieve an existing bot replica
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeBotReplica.html
     */
    toDescribeBotReplica() {
        return this.to('DescribeBotReplica');
    }
    /**
     * Grants permission to retrieve metadata information for a bot resource generation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeBotResourceGeneration.html
     */
    toDescribeBotResourceGeneration() {
        return this.to('DescribeBotResourceGeneration');
    }
    /**
     * Grants permission to retrieve an existing bot version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeBotVersion.html
     */
    toDescribeBotVersion() {
        return this.to('DescribeBotVersion');
    }
    /**
     * Grants permission to retrieve an existing custom vocabulary
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/dg/vocab.html
     */
    toDescribeCustomVocabulary() {
        return this.to('DescribeCustomVocabulary');
    }
    /**
     * Grants permission to retrieve metadata of an existing custom vocabulary
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeCustomVocabularyMetadata.html
     */
    toDescribeCustomVocabularyMetadata() {
        return this.to('DescribeCustomVocabularyMetadata');
    }
    /**
     * Grants permission to retrieve an existing export
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - lex:DescribeBot
     * - lex:DescribeBotLocale
     * - lex:DescribeIntent
     * - lex:DescribeSlot
     * - lex:DescribeSlotType
     * - lex:ListBotLocales
     * - lex:ListIntents
     * - lex:ListSlotTypes
     * - lex:ListSlots
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeExport.html
     */
    toDescribeExport() {
        return this.to('DescribeExport');
    }
    /**
     * Grants permission to retrieve an existing import
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeImport.html
     */
    toDescribeImport() {
        return this.to('DescribeImport');
    }
    /**
     * Grants permission to retrieve an existing intent
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeIntent.html
     */
    toDescribeIntent() {
        return this.to('DescribeIntent');
    }
    /**
     * Grants permission to retrieve an existing resource policy for a Lex resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeResourcePolicy.html
     */
    toDescribeResourcePolicy() {
        return this.to('DescribeResourcePolicy');
    }
    /**
     * Grants permission to retrieve an existing slot
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeSlot.html
     */
    toDescribeSlot() {
        return this.to('DescribeSlot');
    }
    /**
     * Grants permission to retrieve an existing slot type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeSlotType.html
     */
    toDescribeSlotType() {
        return this.to('DescribeSlotType');
    }
    /**
     * Grants permission to retrieve test execution metadata
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeTestExecution.html
     */
    toDescribeTestExecution() {
        return this.to('DescribeTestExecution');
    }
    /**
     * Grants permission to retrieve an existing test set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeTestSet.html
     */
    toDescribeTestSet() {
        return this.to('DescribeTestSet');
    }
    /**
     * Grants permission to retrieve test set discrepancy report metadata
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeTestSetDiscrepancyReport.html
     */
    toDescribeTestSetDiscrepancyReport() {
        return this.to('DescribeTestSetDiscrepancyReport');
    }
    /**
     * Grants permission to retrieve test set generation metadata
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeTestSetGeneration.html
     */
    toDescribeTestSetGeneration() {
        return this.to('DescribeTestSetGeneration');
    }
    /**
     * Grants permission to generate supported fields or elements for a bot
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_GenerateBotElement.html
     */
    toGenerateBotElement() {
        return this.to('GenerateBotElement');
    }
    /**
     * Grants permission to retrieve session information for a bot alias and user ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_runtime_GetSession.html
     */
    toGetSession() {
        return this.to('GetSession');
    }
    /**
     * Grants permission to retrieve artifacts URL for a test execution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_GetTestExecutionArtifactsUrl.html
     */
    toGetTestExecutionArtifactsUrl() {
        return this.to('GetTestExecutionArtifactsUrl');
    }
    /**
     * Grants permission to list utterances and statistics for a bot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListAggregatedUtterances.html
     */
    toListAggregatedUtterances() {
        return this.to('ListAggregatedUtterances');
    }
    /**
     * Grants permission to list alias replicas in a bot replica
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListBotAliasReplicas.html
     */
    toListBotAliasReplicas() {
        return this.to('ListBotAliasReplicas');
    }
    /**
     * Grants permission to list bot aliases in an bot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListBotAliases.html
     */
    toListBotAliases() {
        return this.to('ListBotAliases');
    }
    /**
     * Grants permission to list bot channels
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/dg/deploying-messaging-platform.html
     */
    toListBotChannels() {
        return this.to('ListBotChannels');
    }
    /**
     * Grants permission to list bot locales in a bot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListBotLocales.html
     */
    toListBotLocales() {
        return this.to('ListBotLocales');
    }
    /**
     * Grants permission to get a list of bot recommendations that meet the specified criteria
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListBotRecommendations.html
     */
    toListBotRecommendations() {
        return this.to('ListBotRecommendations');
    }
    /**
     * Grants permission to list replicas of a bot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListBotReplicas.html
     */
    toListBotReplicas() {
        return this.to('ListBotReplicas');
    }
    /**
     * Grants permission to list the resource generations for a bot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListBotResourceGenerations.html
     */
    toListBotResourceGenerations() {
        return this.to('ListBotResourceGenerations');
    }
    /**
     * Grants permission to list version replicas in a bot replica
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListBotVersionReplicas.html
     */
    toListBotVersionReplicas() {
        return this.to('ListBotVersionReplicas');
    }
    /**
     * Grants permission to list existing bot versions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListBotVersions.html
     */
    toListBotVersions() {
        return this.to('ListBotVersions');
    }
    /**
     * Grants permission to list existing bots
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListBots.html
     */
    toListBots() {
        return this.to('ListBots');
    }
    /**
     * Grants permission to list built-in intents
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListBuiltInIntents.html
     */
    toListBuiltInIntents() {
        return this.to('ListBuiltInIntents');
    }
    /**
     * Grants permission to list built-in slot types
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListBuiltInSlotTypes.html
     */
    toListBuiltInSlotTypes() {
        return this.to('ListBuiltInSlotTypes');
    }
    /**
     * Grants permission to list items of an existing custom vocabulary
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListCustomVocabularyItems.html
     */
    toListCustomVocabularyItems() {
        return this.to('ListCustomVocabularyItems');
    }
    /**
     * Grants permission to list existing exports
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListExports.html
     */
    toListExports() {
        return this.to('ListExports');
    }
    /**
     * Grants permission to list existing imports
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListImports.html
     */
    toListImports() {
        return this.to('ListImports');
    }
    /**
     * Grants permission to list intent analytics metrics for a bot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListIntentMetrics.html
     */
    toListIntentMetrics() {
        return this.to('ListIntentMetrics');
    }
    /**
     * Grants permission to list intent path analytics for a bot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListIntentPaths.html
     */
    toListIntentPaths() {
        return this.to('ListIntentPaths');
    }
    /**
     * Grants permission to list intentStage analytics metrics for a bot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListIntentStageMetrics.html
     */
    toListIntentStageMetrics() {
        return this.to('ListIntentStageMetrics');
    }
    /**
     * Grants permission to list intents in a bot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListIntents.html
     */
    toListIntents() {
        return this.to('ListIntents');
    }
    /**
     * Grants permission to get a list of recommended intents provided by the bot recommendation
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListRecommendedIntents.html
     */
    toListRecommendedIntents() {
        return this.to('ListRecommendedIntents');
    }
    /**
     * Grants permission to list session analytics data for a bot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListSessionAnalyticsData.html
     */
    toListSessionAnalyticsData() {
        return this.to('ListSessionAnalyticsData');
    }
    /**
     * Grants permission to list session analytics metrics for a bot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListSessionMetrics.html
     */
    toListSessionMetrics() {
        return this.to('ListSessionMetrics');
    }
    /**
     * Grants permission to list slot types in a bot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListSlotTypes.html
     */
    toListSlotTypes() {
        return this.to('ListSlotTypes');
    }
    /**
     * Grants permission to list slots in an intent
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListSlots.html
     */
    toListSlots() {
        return this.to('ListSlots');
    }
    /**
     * Grants permission to lists tags for a Lex resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to retrieve test results data for a test execution
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - lex:ListTestSetRecords
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListTestExecutionResultItems.html
     */
    toListTestExecutionResultItems() {
        return this.to('ListTestExecutionResultItems');
    }
    /**
     * Grants permission to list test executions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListTestExecutions.html
     */
    toListTestExecutions() {
        return this.to('ListTestExecutions');
    }
    /**
     * Grants permission to retrieve records inside an existing test set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListTestSetRecords.html
     */
    toListTestSetRecords() {
        return this.to('ListTestSetRecords');
    }
    /**
     * Grants permission to list test sets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListTestSets.html
     */
    toListTestSets() {
        return this.to('ListTestSets');
    }
    /**
     * Grants permission to create a new session or modify an existing session for a bot alias and user ID
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_runtime_PutSession.html
     */
    toPutSession() {
        return this.to('PutSession');
    }
    /**
     * Grants permission to send user input (text-only) to an bot alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_runtime_RecognizeText.html
     */
    toRecognizeText() {
        return this.to('RecognizeText');
    }
    /**
     * Grants permission to send user input (text or speech) to an bot alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_runtime_RecognizeUtterance.html
     */
    toRecognizeUtterance() {
        return this.to('RecognizeUtterance');
    }
    /**
     * Grants permission to search for associated transcripts that meet the specified criteria
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_SearchAssociatedTranscripts.html
     */
    toSearchAssociatedTranscripts() {
        return this.to('SearchAssociatedTranscripts');
    }
    /**
     * Grants permission to start a bot recommendation for an existing bot locale
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_StartBotRecommendation.html
     */
    toStartBotRecommendation() {
        return this.to('StartBotRecommendation');
    }
    /**
     * Grants permission to start a resource generation for an existing bot locale
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_StartBotResourceGeneration.html
     */
    toStartBotResourceGeneration() {
        return this.to('StartBotResourceGeneration');
    }
    /**
     * Grants permission to stream user input (speech/text/DTMF) to a bot alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_runtime_StartConversation.html
     */
    toStartConversation() {
        return this.to('StartConversation');
    }
    /**
     * Grants permission to start a new import with the uploaded import file
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - lex:CreateBot
     * - lex:CreateBotLocale
     * - lex:CreateCustomVocabulary
     * - lex:CreateIntent
     * - lex:CreateSlot
     * - lex:CreateSlotType
     * - lex:CreateTestSet
     * - lex:DeleteBotLocale
     * - lex:DeleteCustomVocabulary
     * - lex:DeleteIntent
     * - lex:DeleteSlot
     * - lex:DeleteSlotType
     * - lex:UpdateBot
     * - lex:UpdateBotLocale
     * - lex:UpdateCustomVocabulary
     * - lex:UpdateIntent
     * - lex:UpdateSlot
     * - lex:UpdateSlotType
     * - lex:UpdateTestSet
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_StartImport.html
     */
    toStartImport() {
        return this.to('StartImport');
    }
    /**
     * Grants permission to start a test execution using a test set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_StartTestExecution.html
     */
    toStartTestExecution() {
        return this.to('StartTestExecution');
    }
    /**
     * Grants permission to generate a test set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_StartTestSetGeneration.html
     */
    toStartTestSetGeneration() {
        return this.to('StartTestSetGeneration');
    }
    /**
     * Grants permission to stop a bot recommendation for an existing bot locale
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_StopBotRecommendation.html
     */
    toStopBotRecommendation() {
        return this.to('StopBotRecommendation');
    }
    /**
     * Grants permission to add or overwrite tags of a Lex resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a Lex resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an existing bot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_UpdateBot.html
     */
    toUpdateBot() {
        return this.to('UpdateBot');
    }
    /**
     * Grants permission to update an existing bot alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_UpdateBotAlias.html
     */
    toUpdateBotAlias() {
        return this.to('UpdateBotAlias');
    }
    /**
     * Grants permission to update an existing bot locale
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_UpdateBotLocale.html
     */
    toUpdateBotLocale() {
        return this.to('UpdateBotLocale');
    }
    /**
     * Grants permission to update an existing bot recommendation request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_UpdateBotRecommendation.html
     */
    toUpdateBotRecommendation() {
        return this.to('UpdateBotRecommendation');
    }
    /**
     * Grants permission to update an existing custom vocabulary
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/dg/vocab.html
     */
    toUpdateCustomVocabulary() {
        return this.to('UpdateCustomVocabulary');
    }
    /**
     * Grants permission to update an existing export
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_UpdateExport.html
     */
    toUpdateExport() {
        return this.to('UpdateExport');
    }
    /**
     * Grants permission to update an existing intent
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_UpdateIntent.html
     */
    toUpdateIntent() {
        return this.to('UpdateIntent');
    }
    /**
     * Grants permission to update an existing resource policy for a Lex resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_UpdateResourcePolicy.html
     */
    toUpdateResourcePolicy() {
        return this.to('UpdateResourcePolicy');
    }
    /**
     * Grants permission to update an existing slot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_UpdateSlot.html
     */
    toUpdateSlot() {
        return this.to('UpdateSlot');
    }
    /**
     * Grants permission to update an existing slot type
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_UpdateSlotType.html
     */
    toUpdateSlotType() {
        return this.to('UpdateSlotType');
    }
    /**
     * Grants permission to update an existing test set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_UpdateTestSet.html
     */
    toUpdateTestSet() {
        return this.to('UpdateTestSet');
    }
    /**
     * Adds a resource of type bot to the statement
     *
     * https://docs.aws.amazon.com/lexv2/latest/dg/how-it-works.html
     *
     * @param botId - Identifier for the botId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBot(botId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:lex:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:bot/${botId}`);
    }
    /**
     * Adds a resource of type bot alias to the statement
     *
     * https://docs.aws.amazon.com/lexv2/latest/dg/how-it-works.html
     *
     * @param botId - Identifier for the botId.
     * @param botAliasId - Identifier for the botAliasId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBotAlias(botId, botAliasId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:lex:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:bot-alias/${botId}/${botAliasId}`);
    }
    /**
     * Adds a resource of type test set to the statement
     *
     * https://docs.aws.amazon.com/lexv2/latest/dg/test-workbench.html
     *
     * @param testSetId - Identifier for the testSetId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTestSet(testSetId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:lex:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:test-set/${testSetId}`);
    }
    /**
     * Filters access by the tags in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateBot()
     * - .toCreateBotAlias()
     * - .toStartImport()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags attached to a Lex resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - bot
     * - bot alias
     * - test set
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the set of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateBot()
     * - .toCreateBotAlias()
     * - .toStartImport()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [lex-v2](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonlexv2.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'lex';
        this.accessLevelList = {
            Write: [
                'BatchCreateCustomVocabularyItem',
                'BatchDeleteCustomVocabularyItem',
                'BatchUpdateCustomVocabularyItem',
                'BuildBotLocale',
                'CreateBot',
                'CreateBotAlias',
                'CreateBotChannel',
                'CreateBotLocale',
                'CreateBotReplica',
                'CreateBotVersion',
                'CreateCustomVocabulary',
                'CreateExport',
                'CreateIntent',
                'CreateResourcePolicy',
                'CreateSlot',
                'CreateSlotType',
                'CreateTestSet',
                'CreateTestSetDiscrepancyReport',
                'CreateUploadUrl',
                'DeleteBot',
                'DeleteBotAlias',
                'DeleteBotChannel',
                'DeleteBotLocale',
                'DeleteBotReplica',
                'DeleteBotVersion',
                'DeleteCustomVocabulary',
                'DeleteExport',
                'DeleteImport',
                'DeleteIntent',
                'DeleteResourcePolicy',
                'DeleteSession',
                'DeleteSlot',
                'DeleteSlotType',
                'DeleteTestSet',
                'DeleteUtterances',
                'PutSession',
                'RecognizeText',
                'RecognizeUtterance',
                'StartBotRecommendation',
                'StartBotResourceGeneration',
                'StartConversation',
                'StartImport',
                'StartTestExecution',
                'StartTestSetGeneration',
                'StopBotRecommendation',
                'UpdateBot',
                'UpdateBotAlias',
                'UpdateBotLocale',
                'UpdateBotRecommendation',
                'UpdateCustomVocabulary',
                'UpdateExport',
                'UpdateIntent',
                'UpdateResourcePolicy',
                'UpdateSlot',
                'UpdateSlotType',
                'UpdateTestSet'
            ],
            Read: [
                'DescribeBot',
                'DescribeBotAlias',
                'DescribeBotChannel',
                'DescribeBotLocale',
                'DescribeBotRecommendation',
                'DescribeBotReplica',
                'DescribeBotResourceGeneration',
                'DescribeBotVersion',
                'DescribeCustomVocabulary',
                'DescribeCustomVocabularyMetadata',
                'DescribeExport',
                'DescribeImport',
                'DescribeIntent',
                'DescribeResourcePolicy',
                'DescribeSlot',
                'DescribeSlotType',
                'DescribeTestExecution',
                'DescribeTestSet',
                'DescribeTestSetDiscrepancyReport',
                'DescribeTestSetGeneration',
                'GenerateBotElement',
                'GetSession',
                'GetTestExecutionArtifactsUrl',
                'ListTagsForResource',
                'ListTestExecutionResultItems',
                'ListTestSetRecords'
            ],
            List: [
                'ListAggregatedUtterances',
                'ListBotAliasReplicas',
                'ListBotAliases',
                'ListBotChannels',
                'ListBotLocales',
                'ListBotRecommendations',
                'ListBotReplicas',
                'ListBotResourceGenerations',
                'ListBotVersionReplicas',
                'ListBotVersions',
                'ListBots',
                'ListBuiltInIntents',
                'ListBuiltInSlotTypes',
                'ListCustomVocabularyItems',
                'ListExports',
                'ListImports',
                'ListIntentMetrics',
                'ListIntentPaths',
                'ListIntentStageMetrics',
                'ListIntents',
                'ListRecommendedIntents',
                'ListSessionAnalyticsData',
                'ListSessionMetrics',
                'ListSlotTypes',
                'ListSlots',
                'ListTestExecutions',
                'ListTestSets',
                'SearchAssociatedTranscripts'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.LexV2 = LexV2;
//# sourceMappingURL=data:application/json;base64,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