"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Geo = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [geo](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonlocation.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Geo extends shared_1.PolicyStatement {
    /**
     * Grants permission to create an association between a geofence-collection and a tracker resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_AssociateTrackerConsumer.html
     */
    toAssociateTrackerConsumer() {
        return this.to('AssociateTrackerConsumer');
    }
    /**
     * Grants permission to delete a batch of device position histories from a tracker resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDeviceIds()
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_BatchDeleteDevicePositionHistory.html
     */
    toBatchDeleteDevicePositionHistory() {
        return this.to('BatchDeleteDevicePositionHistory');
    }
    /**
     * Grants permission to delete a batch of geofences from a geofence collection
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifGeofenceIds()
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_BatchDeleteGeofence.html
     */
    toBatchDeleteGeofence() {
        return this.to('BatchDeleteGeofence');
    }
    /**
     * Grants permission to evaluate device positions against the position of geofences in a given geofence collection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_BatchEvaluateGeofences.html
     */
    toBatchEvaluateGeofences() {
        return this.to('BatchEvaluateGeofences');
    }
    /**
     * Grants permission to send a batch request to retrieve device positions
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDeviceIds()
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_BatchGetDevicePosition.html
     */
    toBatchGetDevicePosition() {
        return this.to('BatchGetDevicePosition');
    }
    /**
     * Grants permission to send a batch request for adding geofences into a given geofence collection
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifGeofenceIds()
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_BatchPutGeofence.html
     */
    toBatchPutGeofence() {
        return this.to('BatchPutGeofence');
    }
    /**
     * Grants permission to upload a position update for one or more devices to a tracker resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDeviceIds()
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_BatchUpdateDevicePosition.html
     */
    toBatchUpdateDevicePosition() {
        return this.to('BatchUpdateDevicePosition');
    }
    /**
     * Grants permission to calculate routes using a given route calculator resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_CalculateRoute.html
     */
    toCalculateRoute() {
        return this.to('CalculateRoute');
    }
    /**
     * Grants permission to calculate a route matrix using a given route calculator resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_CalculateRouteMatrix.html
     */
    toCalculateRouteMatrix() {
        return this.to('CalculateRouteMatrix');
    }
    /**
     * Grants permission to create a geofence-collection
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_CreateGeofenceCollection.html
     */
    toCreateGeofenceCollection() {
        return this.to('CreateGeofenceCollection');
    }
    /**
     * Grants permission to create an API key resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_CreateKey.html
     */
    toCreateKey() {
        return this.to('CreateKey');
    }
    /**
     * Grants permission to create a map resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_CreateMap.html
     */
    toCreateMap() {
        return this.to('CreateMap');
    }
    /**
     * Grants permission to create a place index resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_CreatePlaceIndex.html
     */
    toCreatePlaceIndex() {
        return this.to('CreatePlaceIndex');
    }
    /**
     * Grants permission to create a route calculator resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_CreateRouteCalculator.html
     */
    toCreateRouteCalculator() {
        return this.to('CreateRouteCalculator');
    }
    /**
     * Grants permission to create a tracker resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_CreateTracker.html
     */
    toCreateTracker() {
        return this.to('CreateTracker');
    }
    /**
     * Grants permission to delete a geofence-collection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_DeleteGeofenceCollection.html
     */
    toDeleteGeofenceCollection() {
        return this.to('DeleteGeofenceCollection');
    }
    /**
     * Grants permission to delete an API key resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_DeleteKey.html
     */
    toDeleteKey() {
        return this.to('DeleteKey');
    }
    /**
     * Grants permission to delete a map resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_DeleteMap.html
     */
    toDeleteMap() {
        return this.to('DeleteMap');
    }
    /**
     * Grants permission to delete a place index resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_DeletePlaceIndex.html
     */
    toDeletePlaceIndex() {
        return this.to('DeletePlaceIndex');
    }
    /**
     * Grants permission to delete a route calculator resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_DeleteRouteCalculator.html
     */
    toDeleteRouteCalculator() {
        return this.to('DeleteRouteCalculator');
    }
    /**
     * Grants permission to delete a tracker resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_DeleteTracker.html
     */
    toDeleteTracker() {
        return this.to('DeleteTracker');
    }
    /**
     * Grants permission to retrieve geofence collection details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_DescribeGeofenceCollection.html
     */
    toDescribeGeofenceCollection() {
        return this.to('DescribeGeofenceCollection');
    }
    /**
     * Grants permission to retrieve API key resource details and secret
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_DescribeKey.html
     */
    toDescribeKey() {
        return this.to('DescribeKey');
    }
    /**
     * Grants permission to retrieve map resource details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_DescribeMap.html
     */
    toDescribeMap() {
        return this.to('DescribeMap');
    }
    /**
     * Grants permission to retrieve place-index resource details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_DescribePlaceIndex.html
     */
    toDescribePlaceIndex() {
        return this.to('DescribePlaceIndex');
    }
    /**
     * Grants permission to retrieve route calculator resource details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_DescribeRouteCalculator.html
     */
    toDescribeRouteCalculator() {
        return this.to('DescribeRouteCalculator');
    }
    /**
     * Grants permission to retrieve a tracker resource details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_DescribeTracker.html
     */
    toDescribeTracker() {
        return this.to('DescribeTracker');
    }
    /**
     * Grants permission to remove the association between a tracker resource and a geofence-collection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_DisassociateTrackerConsumer.html
     */
    toDisassociateTrackerConsumer() {
        return this.to('DisassociateTrackerConsumer');
    }
    /**
     * Grants permission to forecast events for geofences stored in a given geofence collection
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_ForecastGeofenceEvents.html
     */
    toForecastGeofenceEvents() {
        return this.to('ForecastGeofenceEvents');
    }
    /**
     * Grants permission to retrieve the latest device position
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDeviceIds()
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_GetDevicePosition.html
     */
    toGetDevicePosition() {
        return this.to('GetDevicePosition');
    }
    /**
     * Grants permission to retrieve the device position history
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDeviceIds()
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_GetDevicePositionHistory.html
     */
    toGetDevicePositionHistory() {
        return this.to('GetDevicePositionHistory');
    }
    /**
     * Grants permission to retrieve the geofence details from a geofence-collection
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifGeofenceIds()
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_GetGeofence.html
     */
    toGetGeofence() {
        return this.to('GetGeofence');
    }
    /**
     * Grants permission to retrieve the glyph file for a map resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_GetMapGlyphs.html
     */
    toGetMapGlyphs() {
        return this.to('GetMapGlyphs');
    }
    /**
     * Grants permission to retrieve the sprite file for a map resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_GetMapSprites.html
     */
    toGetMapSprites() {
        return this.to('GetMapSprites');
    }
    /**
     * Grants permission to retrieve the map style descriptor from a map resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_GetMapStyleDescriptor.html
     */
    toGetMapStyleDescriptor() {
        return this.to('GetMapStyleDescriptor');
    }
    /**
     * Grants permission to retrieve the map tile from the map resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_GetMapTile.html
     */
    toGetMapTile() {
        return this.to('GetMapTile');
    }
    /**
     * Grants permission to find a place by its unique ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_GetPlace.html
     */
    toGetPlace() {
        return this.to('GetPlace');
    }
    /**
     * Grants permission to retrieve a list of devices and their latest positions from the given tracker resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_ListDevicePositions.html
     */
    toListDevicePositions() {
        return this.to('ListDevicePositions');
    }
    /**
     * Grants permission to lists geofence-collections
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_ListGeofenceCollections.html
     */
    toListGeofenceCollections() {
        return this.to('ListGeofenceCollections');
    }
    /**
     * Grants permission to list geofences stored in a given geofence collection
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_ListGeofences.html
     */
    toListGeofences() {
        return this.to('ListGeofences');
    }
    /**
     * Grants permission to list API key resources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_ListKeys.html
     */
    toListKeys() {
        return this.to('ListKeys');
    }
    /**
     * Grants permission to list map resources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_ListMaps.html
     */
    toListMaps() {
        return this.to('ListMaps');
    }
    /**
     * Grants permission to return a list of place index resources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_ListPlaceIndexes.html
     */
    toListPlaceIndexes() {
        return this.to('ListPlaceIndexes');
    }
    /**
     * Grants permission to return a list of route calculator resources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_ListRouteCalculators.html
     */
    toListRouteCalculators() {
        return this.to('ListRouteCalculators');
    }
    /**
     * Grants permission to list the tags (metadata) which you have assigned to the resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to retrieve a list of geofence collections currently associated to the given tracker resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_ListTrackerConsumers.html
     */
    toListTrackerConsumers() {
        return this.to('ListTrackerConsumers');
    }
    /**
     * Grants permission to return a list of tracker resources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_ListTrackers.html
     */
    toListTrackers() {
        return this.to('ListTrackers');
    }
    /**
     * Grants permission to add a new geofence or update an existing geofence to a given geofence-collection
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifGeofenceIds()
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_PutGeofence.html
     */
    toPutGeofence() {
        return this.to('PutGeofence');
    }
    /**
     * Grants permission to reverse geocodes a given coordinate
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_SearchPlaceIndexForPosition.html
     */
    toSearchPlaceIndexForPosition() {
        return this.to('SearchPlaceIndexForPosition');
    }
    /**
     * Grants permission to generate suggestions for addresses and points of interest based on partial or misspelled free-form text
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_SearchPlaceIndexForSuggestions.html
     */
    toSearchPlaceIndexForSuggestions() {
        return this.to('SearchPlaceIndexForSuggestions');
    }
    /**
     * Grants permission to geocode free-form text, such as an address, name, city or region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_SearchPlaceIndexForText.html
     */
    toSearchPlaceIndexForText() {
        return this.to('SearchPlaceIndexForText');
    }
    /**
     * Grants permission to adds to or modifies the tags of the given resource. Tags are metadata which can be used to manage a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove the given tags (metadata) from the resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a geofence collection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_UpdateGeofenceCollection.html
     */
    toUpdateGeofenceCollection() {
        return this.to('UpdateGeofenceCollection');
    }
    /**
     * Grants permission to update an API key resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_UpdateKey.html
     */
    toUpdateKey() {
        return this.to('UpdateKey');
    }
    /**
     * Grants permission to update a map resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_UpdateMap.html
     */
    toUpdateMap() {
        return this.to('UpdateMap');
    }
    /**
     * Grants permission to update a place index resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_UpdatePlaceIndex.html
     */
    toUpdatePlaceIndex() {
        return this.to('UpdatePlaceIndex');
    }
    /**
     * Grants permission to update a route calculator resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_UpdateRouteCalculator.html
     */
    toUpdateRouteCalculator() {
        return this.to('UpdateRouteCalculator');
    }
    /**
     * Grants permission to update a tracker resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_UpdateTracker.html
     */
    toUpdateTracker() {
        return this.to('UpdateTracker');
    }
    /**
     * Grants permission to verify a device position
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDeviceIds()
     *
     * https://docs.aws.amazon.com/location/latest/APIReference/API_VerifyDevicePosition.html
     */
    toVerifyDevicePosition() {
        return this.to('VerifyDevicePosition');
    }
    /**
     * Adds a resource of type api-key to the statement
     *
     * https://docs.aws.amazon.com/location/latest/developerguide/using-apikeys.html
     *
     * @param keyName - Identifier for the keyName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onApiKey(keyName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:geo:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:api-key/${keyName}`);
    }
    /**
     * Adds a resource of type geofence-collection to the statement
     *
     * https://docs.aws.amazon.com/location/latest/developerguide/geofence-tracker-concepts.html
     *
     * @param geofenceCollectionName - Identifier for the geofenceCollectionName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifGeofenceIds()
     */
    onGeofenceCollection(geofenceCollectionName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:geo:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:geofence-collection/${geofenceCollectionName}`);
    }
    /**
     * Adds a resource of type map to the statement
     *
     * https://docs.aws.amazon.com/location/latest/developerguide/map-concepts.html
     *
     * @param mapName - Identifier for the mapName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMap(mapName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:geo:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:map/${mapName}`);
    }
    /**
     * Adds a resource of type place-index to the statement
     *
     * https://docs.aws.amazon.com/location/latest/developerguide/places-concepts.html
     *
     * @param indexName - Identifier for the indexName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPlaceIndex(indexName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:geo:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:place-index/${indexName}`);
    }
    /**
     * Adds a resource of type route-calculator to the statement
     *
     * https://docs.aws.amazon.com/location/latest/developerguide/route-concepts.html
     *
     * @param calculatorName - Identifier for the calculatorName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRouteCalculator(calculatorName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:geo:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:route-calculator/${calculatorName}`);
    }
    /**
     * Adds a resource of type tracker to the statement
     *
     * https://docs.aws.amazon.com/location/latest/developerguide/geofence-tracker-concepts.html
     *
     * @param trackerName - Identifier for the trackerName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifDeviceIds()
     */
    onTracker(trackerName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:geo:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:tracker/${trackerName}`);
    }
    /**
     * Filters access by a tag's key and value in a request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateGeofenceCollection()
     * - .toCreateKey()
     * - .toCreateMap()
     * - .toCreatePlaceIndex()
     * - .toCreateRouteCalculator()
     * - .toCreateTracker()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - api-key
     * - geofence-collection
     * - map
     * - place-index
     * - route-calculator
     * - tracker
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys in a request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateGeofenceCollection()
     * - .toCreateKey()
     * - .toCreateMap()
     * - .toCreatePlaceIndex()
     * - .toCreateRouteCalculator()
     * - .toCreateTracker()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of device ids in the request
     *
     * https://docs.aws.amazon.com/location/latest/developerguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * Applies to actions:
     * - .toBatchDeleteDevicePositionHistory()
     * - .toBatchGetDevicePosition()
     * - .toBatchUpdateDevicePosition()
     * - .toGetDevicePosition()
     * - .toGetDevicePositionHistory()
     * - .toVerifyDevicePosition()
     *
     * Applies to resource types:
     * - tracker
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDeviceIds(value, operator) {
        return this.if(`DeviceIds`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of geofence ids in the request
     *
     * https://docs.aws.amazon.com/location/latest/developerguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * Applies to actions:
     * - .toBatchDeleteGeofence()
     * - .toBatchPutGeofence()
     * - .toGetGeofence()
     * - .toPutGeofence()
     *
     * Applies to resource types:
     * - geofence-collection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifGeofenceIds(value, operator) {
        return this.if(`GeofenceIds`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [geo](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonlocation.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'geo';
        this.accessLevelList = {
            Write: [
                'AssociateTrackerConsumer',
                'BatchDeleteDevicePositionHistory',
                'BatchDeleteGeofence',
                'BatchEvaluateGeofences',
                'BatchPutGeofence',
                'BatchUpdateDevicePosition',
                'CreateGeofenceCollection',
                'CreateKey',
                'CreateMap',
                'CreatePlaceIndex',
                'CreateRouteCalculator',
                'CreateTracker',
                'DeleteGeofenceCollection',
                'DeleteKey',
                'DeleteMap',
                'DeletePlaceIndex',
                'DeleteRouteCalculator',
                'DeleteTracker',
                'DisassociateTrackerConsumer',
                'PutGeofence',
                'UpdateGeofenceCollection',
                'UpdateKey',
                'UpdateMap',
                'UpdatePlaceIndex',
                'UpdateRouteCalculator',
                'UpdateTracker'
            ],
            Read: [
                'BatchGetDevicePosition',
                'CalculateRoute',
                'CalculateRouteMatrix',
                'DescribeGeofenceCollection',
                'DescribeKey',
                'DescribeMap',
                'DescribePlaceIndex',
                'DescribeRouteCalculator',
                'DescribeTracker',
                'ForecastGeofenceEvents',
                'GetDevicePosition',
                'GetDevicePositionHistory',
                'GetGeofence',
                'GetMapGlyphs',
                'GetMapSprites',
                'GetMapStyleDescriptor',
                'GetMapTile',
                'GetPlace',
                'ListDevicePositions',
                'ListGeofences',
                'ListTagsForResource',
                'ListTrackerConsumers',
                'SearchPlaceIndexForPosition',
                'SearchPlaceIndexForSuggestions',
                'SearchPlaceIndexForText',
                'VerifyDevicePosition'
            ],
            List: [
                'ListGeofenceCollections',
                'ListKeys',
                'ListMaps',
                'ListPlaceIndexes',
                'ListRouteCalculators',
                'ListTrackers'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Geo = Geo;
//# sourceMappingURL=data:application/json;base64,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