"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Lookoutmetrics = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [lookoutmetrics](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonlookoutformetrics.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Lookoutmetrics extends shared_1.PolicyStatement {
    /**
     * Grants permission to activate an anomaly detector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_ActivateAnomalyDetector.html
     */
    toActivateAnomalyDetector() {
        return this.to('ActivateAnomalyDetector');
    }
    /**
     * Grants permission to run a backtest with an anomaly detector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_BackTestAnomalyDetector.html
     */
    toBackTestAnomalyDetector() {
        return this.to('BackTestAnomalyDetector');
    }
    /**
     * Grants permission to create an alert for an anomaly detector
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_CreateAlert.html
     */
    toCreateAlert() {
        return this.to('CreateAlert');
    }
    /**
     * Grants permission to create an anomaly detector
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_CreateAnomalyDetector.html
     */
    toCreateAnomalyDetector() {
        return this.to('CreateAnomalyDetector');
    }
    /**
     * Grants permission to create a dataset
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_CreateMetricSet.html
     */
    toCreateMetricSet() {
        return this.to('CreateMetricSet');
    }
    /**
     * Grants permission to deactivate an anomaly detector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_DeactivateAnomalyDetector.html
     */
    toDeactivateAnomalyDetector() {
        return this.to('DeactivateAnomalyDetector');
    }
    /**
     * Grants permission to delete an alert
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_DeleteAlert.html
     */
    toDeleteAlert() {
        return this.to('DeleteAlert');
    }
    /**
     * Grants permission to delete an anomaly detector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_DeleteAnomalyDetector.html
     */
    toDeleteAnomalyDetector() {
        return this.to('DeleteAnomalyDetector');
    }
    /**
     * Grants permission to get details about an alert
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_DescribeAlert.html
     */
    toDescribeAlert() {
        return this.to('DescribeAlert');
    }
    /**
     * Grants permission to get information about an anomaly detection job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_DescribeAnomalyDetectionExecutions.html
     */
    toDescribeAnomalyDetectionExecutions() {
        return this.to('DescribeAnomalyDetectionExecutions');
    }
    /**
     * Grants permission to get details about an anomaly detector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_DescribeAnomalyDetector.html
     */
    toDescribeAnomalyDetector() {
        return this.to('DescribeAnomalyDetector');
    }
    /**
     * Grants permission to get details about a dataset
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_DescribeMetricSet.html
     */
    toDescribeMetricSet() {
        return this.to('DescribeMetricSet');
    }
    /**
     * Grants permission to detect metric set config from data source
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_DetectMetricSetConfig.html
     */
    toDetectMetricSetConfig() {
        return this.to('DetectMetricSetConfig');
    }
    /**
     * Grants permission to get details about a group of affected metrics
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_GetAnomalyGroup.html
     */
    toGetAnomalyGroup() {
        return this.to('GetAnomalyGroup');
    }
    /**
     * Grants permission to get data quality metrics for an anomaly detector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_GetDataQualityMetrics.html
     */
    toGetDataQualityMetrics() {
        return this.to('GetDataQualityMetrics');
    }
    /**
     * Grants permission to get feedback on affected metrics for an anomaly group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_GetFeedback.html
     */
    toGetFeedback() {
        return this.to('GetFeedback');
    }
    /**
     * Grants permission to get a selection of sample records from an Amazon S3 datasource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_GetSampleData.html
     */
    toGetSampleData() {
        return this.to('GetSampleData');
    }
    /**
     * Grants permission to get a list of alerts for a detector
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_ListAlerts.html
     */
    toListAlerts() {
        return this.to('ListAlerts');
    }
    /**
     * Grants permission to get a list of anomaly detectors
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_ListAnomalyDetectors.html
     */
    toListAnomalyDetectors() {
        return this.to('ListAnomalyDetectors');
    }
    /**
     * Grants permission to get a list of related measures in an anomaly group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_ListAnomalyGroupRelatedMetrics.html
     */
    toListAnomalyGroupRelatedMetrics() {
        return this.to('ListAnomalyGroupRelatedMetrics');
    }
    /**
     * Grants permission to get a list of anomaly groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_ListAnomalyGroupSummaries.html
     */
    toListAnomalyGroupSummaries() {
        return this.to('ListAnomalyGroupSummaries');
    }
    /**
     * Grants permission to get a list of affected metrics for a measure in an anomaly group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_ListAnomalyGroupTimeSeries.html
     */
    toListAnomalyGroupTimeSeries() {
        return this.to('ListAnomalyGroupTimeSeries');
    }
    /**
     * Grants permission to get a list of datasets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_ListMetricSets.html
     */
    toListMetricSets() {
        return this.to('ListMetricSets');
    }
    /**
     * Grants permission to get a list of tags for a detector, dataset, or alert
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to add feedback for an affected metric in an anomaly group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_PutFeedback.html
     */
    toPutFeedback() {
        return this.to('PutFeedback');
    }
    /**
     * Grants permission to add tags to a detector, dataset, or alert
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a detector, dataset, or alert
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an alert for an anomaly detector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_UpdateAlert.html
     */
    toUpdateAlert() {
        return this.to('UpdateAlert');
    }
    /**
     * Grants permission to update an anomaly detector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_UpdateAnomalyDetector.html
     */
    toUpdateAnomalyDetector() {
        return this.to('UpdateAnomalyDetector');
    }
    /**
     * Grants permission to update a dataset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_UpdateMetricSet.html
     */
    toUpdateMetricSet() {
        return this.to('UpdateMetricSet');
    }
    /**
     * Adds a resource of type AnomalyDetector to the statement
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_AnomalyDetectorSummary.html
     *
     * @param anomalyDetectorName - Identifier for the anomalyDetectorName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAnomalyDetector(anomalyDetectorName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:lookoutmetrics:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:AnomalyDetector:${anomalyDetectorName}`);
    }
    /**
     * Adds a resource of type MetricSet to the statement
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_MetricSetSummary.html
     *
     * @param anomalyDetectorName - Identifier for the anomalyDetectorName.
     * @param metricSetName - Identifier for the metricSetName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMetricSet(anomalyDetectorName, metricSetName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:lookoutmetrics:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:MetricSet/${anomalyDetectorName}/${metricSetName}`);
    }
    /**
     * Adds a resource of type Alert to the statement
     *
     * https://docs.aws.amazon.com/lookoutmetrics/latest/api/API_AlertSummary.html
     *
     * @param alertName - Identifier for the alertName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAlert(alertName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:lookoutmetrics:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:Alert:${alertName}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateAlert()
     * - .toCreateAnomalyDetector()
     * - .toCreateMetricSet()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toTagResource()
     *
     * Applies to resource types:
     * - AnomalyDetector
     * - MetricSet
     * - Alert
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateAlert()
     * - .toCreateAnomalyDetector()
     * - .toCreateMetricSet()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [lookoutmetrics](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonlookoutformetrics.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'lookoutmetrics';
        this.accessLevelList = {
            Write: [
                'ActivateAnomalyDetector',
                'BackTestAnomalyDetector',
                'CreateAlert',
                'CreateAnomalyDetector',
                'CreateMetricSet',
                'DeactivateAnomalyDetector',
                'DeleteAlert',
                'DeleteAnomalyDetector',
                'DetectMetricSetConfig',
                'PutFeedback',
                'UpdateAlert',
                'UpdateAnomalyDetector',
                'UpdateMetricSet'
            ],
            Read: [
                'DescribeAlert',
                'DescribeAnomalyDetectionExecutions',
                'DescribeAnomalyDetector',
                'DescribeMetricSet',
                'GetAnomalyGroup',
                'GetDataQualityMetrics',
                'GetFeedback',
                'GetSampleData',
                'ListTagsForResource'
            ],
            List: [
                'ListAlerts',
                'ListAnomalyDetectors',
                'ListAnomalyGroupRelatedMetrics',
                'ListAnomalyGroupSummaries',
                'ListAnomalyGroupTimeSeries',
                'ListMetricSets'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Lookoutmetrics = Lookoutmetrics;
//# sourceMappingURL=data:application/json;base64,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