"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Macie2 = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [macie2](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmacie.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Macie2 extends shared_1.PolicyStatement {
    /**
     * Grants permission to accept an Amazon Macie membership invitation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/invitations-accept.html
     */
    toAcceptInvitation() {
        return this.to('AcceptInvitation');
    }
    /**
     * Grants permission to retrieve information about one or more custom data identifiers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers-get.html
     */
    toBatchGetCustomDataIdentifiers() {
        return this.to('BatchGetCustomDataIdentifiers');
    }
    /**
     * Grants permission to an Amazon Macie administrator to change the status of automated sensitive data discovery for one or more accounts in their organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/automated-discovery-accounts.html
     */
    toBatchUpdateAutomatedDiscoveryAccounts() {
        return this.to('BatchUpdateAutomatedDiscoveryAccounts');
    }
    /**
     * Grants permission to create and define the settings for an allow list
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/allow-lists.html
     */
    toCreateAllowList() {
        return this.to('CreateAllowList');
    }
    /**
     * Grants permission to create and define the settings for a sensitive data discovery job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/jobs.html
     */
    toCreateClassificationJob() {
        return this.to('CreateClassificationJob');
    }
    /**
     * Grants permission to create and define the settings for a custom data identifier
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers.html
     */
    toCreateCustomDataIdentifier() {
        return this.to('CreateCustomDataIdentifier');
    }
    /**
     * Grants permission to create and define the settings for a findings filter
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findingsfilters.html
     */
    toCreateFindingsFilter() {
        return this.to('CreateFindingsFilter');
    }
    /**
     * Grants permission to send an Amazon Macie membership invitation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/invitations.html
     */
    toCreateInvitations() {
        return this.to('CreateInvitations');
    }
    /**
     * Grants permission to associate an account with an Amazon Macie administrator account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/members.html
     */
    toCreateMember() {
        return this.to('CreateMember');
    }
    /**
     * Grants permission to create sample findings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findings-sample.html
     */
    toCreateSampleFindings() {
        return this.to('CreateSampleFindings');
    }
    /**
     * Grants permission to decline Amazon Macie membership invitations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/invitations-decline.html
     */
    toDeclineInvitations() {
        return this.to('DeclineInvitations');
    }
    /**
     * Grants permission to delete an allow list
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/allow-lists-id.html
     */
    toDeleteAllowList() {
        return this.to('DeleteAllowList');
    }
    /**
     * Grants permission to delete a custom data identifier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers-id.html
     */
    toDeleteCustomDataIdentifier() {
        return this.to('DeleteCustomDataIdentifier');
    }
    /**
     * Grants permission to delete a findings filter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findingsfilters-id.html
     */
    toDeleteFindingsFilter() {
        return this.to('DeleteFindingsFilter');
    }
    /**
     * Grants permission to delete Amazon Macie membership invitations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/invitations-delete.html
     */
    toDeleteInvitations() {
        return this.to('DeleteInvitations');
    }
    /**
     * Grants permission to delete the association between an Amazon Macie administrator account and an account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/members-id.html
     */
    toDeleteMember() {
        return this.to('DeleteMember');
    }
    /**
     * Grants permission to retrieve statistical data and other information about S3 buckets that Amazon Macie monitors and analyzes
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/datasources-s3.html
     */
    toDescribeBuckets() {
        return this.to('DescribeBuckets');
    }
    /**
     * Grants permission to retrieve information about the status and settings for a sensitive data discovery job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/jobs-jobid.html
     */
    toDescribeClassificationJob() {
        return this.to('DescribeClassificationJob');
    }
    /**
     * Grants permission to retrieve information about the Amazon Macie configuration settings for an AWS organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/admin-configuration.html
     */
    toDescribeOrganizationConfiguration() {
        return this.to('DescribeOrganizationConfiguration');
    }
    /**
     * Grants permission to disable an Amazon Macie account, which also deletes Macie resources for the account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/macie.html
     */
    toDisableMacie() {
        return this.to('DisableMacie');
    }
    /**
     * Grants permission to disable an account as the delegated Amazon Macie administrator account for an AWS organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/admin.html
     */
    toDisableOrganizationAdminAccount() {
        return this.to('DisableOrganizationAdminAccount');
    }
    /**
     * Grants permission to an Amazon Macie member account to disassociate from its Macie administrator account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/administrator-disassociate.html
     */
    toDisassociateFromAdministratorAccount() {
        return this.to('DisassociateFromAdministratorAccount');
    }
    /**
     * Grants permission to an Amazon Macie member account to disassociate from its Macie administrator account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/master-disassociate.html
     */
    toDisassociateFromMasterAccount() {
        return this.to('DisassociateFromMasterAccount');
    }
    /**
     * Grants permission to an Amazon Macie administrator account to disassociate from a Macie member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/members-disassociate-id.html
     */
    toDisassociateMember() {
        return this.to('DisassociateMember');
    }
    /**
     * Grants permission to enable and specify the configuration settings for a new Amazon Macie account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/macie.html
     */
    toEnableMacie() {
        return this.to('EnableMacie');
    }
    /**
     * Grants permission to enable an account as the delegated Amazon Macie administrator account for an AWS organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/admin.html
     */
    toEnableOrganizationAdminAccount() {
        return this.to('EnableOrganizationAdminAccount');
    }
    /**
     * Grants permission to retrieve information about the Amazon Macie administrator account for an account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/administrator.html
     */
    toGetAdministratorAccount() {
        return this.to('GetAdministratorAccount');
    }
    /**
     * Grants permission to retrieve the settings and status of an allow list
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/allow-lists-id.html
     */
    toGetAllowList() {
        return this.to('GetAllowList');
    }
    /**
     * Grants permission to retrieve the configuration settings and status of automated sensitive data discovery for an Amazon Macie administrator account, organization, or standalone account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/automated-discovery-configuration.html
     */
    toGetAutomatedDiscoveryConfiguration() {
        return this.to('GetAutomatedDiscoveryConfiguration');
    }
    /**
     * Grants permission to retrieve aggregated statistical data for all the S3 buckets that Amazon Macie monitors and analyzes
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/datasources-s3-statistics.html
     */
    toGetBucketStatistics() {
        return this.to('GetBucketStatistics');
    }
    /**
     * Grants permission to retrieve the settings for exporting sensitive data discovery results
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/classification-export-configuration.html
     */
    toGetClassificationExportConfiguration() {
        return this.to('GetClassificationExportConfiguration');
    }
    /**
     * Grants permission to retrieve the classification scope settings for an account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/classification-scopes-id.html
     */
    toGetClassificationScope() {
        return this.to('GetClassificationScope');
    }
    /**
     * Grants permission to retrieve information about the settings for a custom data identifier
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers-id.html
     */
    toGetCustomDataIdentifier() {
        return this.to('GetCustomDataIdentifier');
    }
    /**
     * Grants permission to retrieve aggregated statistical data about findings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findings-statistics.html
     */
    toGetFindingStatistics() {
        return this.to('GetFindingStatistics');
    }
    /**
     * Grants permission to retrieve the details of one or more findings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findings-describe.html
     */
    toGetFindings() {
        return this.to('GetFindings');
    }
    /**
     * Grants permission to retrieve information about the settings for a findings filter
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findingsfilters-id.html
     */
    toGetFindingsFilter() {
        return this.to('GetFindingsFilter');
    }
    /**
     * Grants permission to retrieve the configuration settings for publishing findings to AWS Security Hub
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findings-publication-configuration.html
     */
    toGetFindingsPublicationConfiguration() {
        return this.to('GetFindingsPublicationConfiguration');
    }
    /**
     * Grants permission to retrieve the count of Amazon Macie membership invitations that were received by an account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/invitations-count.html
     */
    toGetInvitationsCount() {
        return this.to('GetInvitationsCount');
    }
    /**
     * Grants permission to retrieve information about the status and configuration settings for an Amazon Macie account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/macie.html
     */
    toGetMacieSession() {
        return this.to('GetMacieSession');
    }
    /**
     * Grants permission to retrieve information about the Amazon Macie administrator account for an account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/master.html
     */
    toGetMasterAccount() {
        return this.to('GetMasterAccount');
    }
    /**
     * Grants permission to retrieve information about an account that's associated with an Amazon Macie administrator account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/members-id.html
     */
    toGetMember() {
        return this.to('GetMember');
    }
    /**
     * Grants permission to retrieve sensitive data discovery statistics and the sensitivity score for an S3 bucket
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/resource-profiles.html
     */
    toGetResourceProfile() {
        return this.to('GetResourceProfile');
    }
    /**
     * Grants permission to retrieve the status and configuration settings for retrieving occurrences of sensitive data reported by findings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/reveal-configuration.html
     */
    toGetRevealConfiguration() {
        return this.to('GetRevealConfiguration');
    }
    /**
     * Grants permission to retrieve occurrences of sensitive data reported by a finding
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findings-findingid-reveal.html
     */
    toGetSensitiveDataOccurrences() {
        return this.to('GetSensitiveDataOccurrences');
    }
    /**
     * Grants permission to check whether occurrences of sensitive data can be retrieved for a finding
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findings-findingid-reveal-availability.html
     */
    toGetSensitiveDataOccurrencesAvailability() {
        return this.to('GetSensitiveDataOccurrencesAvailability');
    }
    /**
     * Grants permission to retrieve the sensitivity inspection template settings for an account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/templates-sensitivity-inspections-id.html
     */
    toGetSensitivityInspectionTemplate() {
        return this.to('GetSensitivityInspectionTemplate');
    }
    /**
     * Grants permission to retrieve quotas and aggregated usage data for one or more accounts
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/usage-statistics.html
     */
    toGetUsageStatistics() {
        return this.to('GetUsageStatistics');
    }
    /**
     * Grants permission to retrieve aggregated usage data for an account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/usage.html
     */
    toGetUsageTotals() {
        return this.to('GetUsageTotals');
    }
    /**
     * Grants permission to retrieve a subset of information about all the allow lists for an account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/allow-lists.html
     */
    toListAllowLists() {
        return this.to('ListAllowLists');
    }
    /**
     * Grants permission to retrieve the status of automated sensitive data discovery for an account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/automated-discovery-accounts.html
     */
    toListAutomatedDiscoveryAccounts() {
        return this.to('ListAutomatedDiscoveryAccounts');
    }
    /**
     * Grants permission to retrieve a subset of information about the status and settings for one or more sensitive data discovery jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/jobs-list.html
     */
    toListClassificationJobs() {
        return this.to('ListClassificationJobs');
    }
    /**
     * Grants permission to retrieve a subset of information about the classification scope for an account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/classification-scopes.html
     */
    toListClassificationScopes() {
        return this.to('ListClassificationScopes');
    }
    /**
     * Grants permission to retrieve information about all custom data identifiers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers-list.html
     */
    toListCustomDataIdentifiers() {
        return this.to('ListCustomDataIdentifiers');
    }
    /**
     * Grants permission to retrieve a subset of information about one or more findings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findings.html
     */
    toListFindings() {
        return this.to('ListFindings');
    }
    /**
     * Grants permission to retrieve information about all findings filters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findingsfilters.html
     */
    toListFindingsFilters() {
        return this.to('ListFindingsFilters');
    }
    /**
     * Grants permission to retrieve information about all the Amazon Macie membership invitations that were received by an account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/invitations.html
     */
    toListInvitations() {
        return this.to('ListInvitations');
    }
    /**
     * Grants permission to retrieve information about managed data identifiers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/managed-data-identifiers-list.html
     */
    toListManagedDataIdentifiers() {
        return this.to('ListManagedDataIdentifiers');
    }
    /**
     * Grants permission to retrieve information about the Amazon Macie member accounts that are associated with a Macie administrator account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/members.html
     */
    toListMembers() {
        return this.to('ListMembers');
    }
    /**
     * Grants permission to retrieve information about the delegated Amazon Macie administrator account for an AWS organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/admin.html
     */
    toListOrganizationAdminAccounts() {
        return this.to('ListOrganizationAdminAccounts');
    }
    /**
     * Grants permission to retrieve information about objects that Amazon Macie selected from an S3 bucket for automated sensitive data discovery
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/resource-profiles-artifacts.html
     */
    toListResourceProfileArtifacts() {
        return this.to('ListResourceProfileArtifacts');
    }
    /**
     * Grants permission to retrieve information about the types and amount of sensitive data that Amazon Macie found in an S3 bucket
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/resource-profiles-detections.html
     */
    toListResourceProfileDetections() {
        return this.to('ListResourceProfileDetections');
    }
    /**
     * Grants permission to retrieve a subset of information about the sensitivity inspection template for an account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/templates-sensitivity-inspections.html
     */
    toListSensitivityInspectionTemplates() {
        return this.to('ListSensitivityInspectionTemplates');
    }
    /**
     * Grants permission to retrieve the tags for an Amazon Macie resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/tags-resourcearn.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to create or update the settings for storing sensitive data discovery results
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/classification-export-configuration.html
     */
    toPutClassificationExportConfiguration() {
        return this.to('PutClassificationExportConfiguration');
    }
    /**
     * Grants permission to update the configuration settings for publishing findings to AWS Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findings-publication-configuration.html
     */
    toPutFindingsPublicationConfiguration() {
        return this.to('PutFindingsPublicationConfiguration');
    }
    /**
     * Grants permission to retrieve statistical data and other information about AWS resources that Amazon Macie monitors and analyzes
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/datasources-search-resources.html
     */
    toSearchResources() {
        return this.to('SearchResources');
    }
    /**
     * Grants permission to add or update the tags for an Amazon Macie resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/tags-resourcearn.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to test a custom data identifier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/custom-data-identifiers-test.html
     */
    toTestCustomDataIdentifier() {
        return this.to('TestCustomDataIdentifier');
    }
    /**
     * Grants permission to remove tags from an Amazon Macie resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/tags-resourcearn.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the settings for an allow list
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/allow-lists-id.html
     */
    toUpdateAllowList() {
        return this.to('UpdateAllowList');
    }
    /**
     * Grants permission to change the status of automated sensitive data discovery for an Amazon Macie administrator account, organization, or standalone account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/automated-discovery-configuration.html
     */
    toUpdateAutomatedDiscoveryConfiguration() {
        return this.to('UpdateAutomatedDiscoveryConfiguration');
    }
    /**
     * Grants permission to change the status of a sensitive data discovery job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/jobs-jobid.html
     */
    toUpdateClassificationJob() {
        return this.to('UpdateClassificationJob');
    }
    /**
     * Grants permission to update the classification scope settings for an account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/classification-scopes-id.html
     */
    toUpdateClassificationScope() {
        return this.to('UpdateClassificationScope');
    }
    /**
     * Grants permission to update the settings for a findings filter
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/findingsfilters-id.html
     */
    toUpdateFindingsFilter() {
        return this.to('UpdateFindingsFilter');
    }
    /**
     * Grants permission to an Amazon Macie administrator account to suspend or re-enable Macie for a member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/macie.html
     */
    toUpdateMacieSession() {
        return this.to('UpdateMacieSession');
    }
    /**
     * Grants permission to an Amazon Macie administrator account to suspend or re-enable a Macie member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/macie-members-id.html
     */
    toUpdateMemberSession() {
        return this.to('UpdateMemberSession');
    }
    /**
     * Grants permission to update Amazon Macie configuration settings for an AWS organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/admin-configuration.html
     */
    toUpdateOrganizationConfiguration() {
        return this.to('UpdateOrganizationConfiguration');
    }
    /**
     * Grants permission to update the sensitivity score for an S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/resource-profiles.html
     */
    toUpdateResourceProfile() {
        return this.to('UpdateResourceProfile');
    }
    /**
     * Grants permission to update the sensitivity scoring settings for an S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/resource-profiles-detections.html
     */
    toUpdateResourceProfileDetections() {
        return this.to('UpdateResourceProfileDetections');
    }
    /**
     * Grants permission to update the status and configuration settings for retrieving occurrences of sensitive data reported by findings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/reveal-configuration.html
     */
    toUpdateRevealConfiguration() {
        return this.to('UpdateRevealConfiguration');
    }
    /**
     * Grants permission to update the sensitivity inspection template settings for an account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/latest/APIReference/templates-sensitivity-inspections-id.html
     */
    toUpdateSensitivityInspectionTemplate() {
        return this.to('UpdateSensitivityInspectionTemplate');
    }
    /**
     * Adds a resource of type AllowList to the statement
     *
     * https://docs.aws.amazon.com/macie/latest/user/what-is-macie.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAllowList(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:macie2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:allow-list/${resourceId}`);
    }
    /**
     * Adds a resource of type ClassificationJob to the statement
     *
     * https://docs.aws.amazon.com/macie/latest/user/what-is-macie.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onClassificationJob(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:macie2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:classification-job/${resourceId}`);
    }
    /**
     * Adds a resource of type CustomDataIdentifier to the statement
     *
     * https://docs.aws.amazon.com/macie/latest/user/what-is-macie.html.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCustomDataIdentifier(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:macie2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:custom-data-identifier/${resourceId}`);
    }
    /**
     * Adds a resource of type FindingsFilter to the statement
     *
     * https://docs.aws.amazon.com/macie/latest/user/what-is-macie.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFindingsFilter(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:macie2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:findings-filter/${resourceId}`);
    }
    /**
     * Adds a resource of type Member to the statement
     *
     * https://docs.aws.amazon.com/macie/latest/user/what-is-macie.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMember(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:macie2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:member/${resourceId}`);
    }
    /**
     * Filters access by a tag key and value pair that is allowed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateAllowList()
     * - .toCreateClassificationJob()
     * - .toCreateCustomDataIdentifier()
     * - .toCreateFindingsFilter()
     * - .toCreateMember()
     * - .toTagResource()
     * - .toUpdateClassificationJob()
     * - .toUpdateFindingsFilter()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a tag key and value pair of a resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - AllowList
     * - ClassificationJob
     * - CustomDataIdentifier
     * - FindingsFilter
     * - Member
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateAllowList()
     * - .toCreateClassificationJob()
     * - .toCreateCustomDataIdentifier()
     * - .toCreateFindingsFilter()
     * - .toCreateMember()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateClassificationJob()
     * - .toUpdateFindingsFilter()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [macie2](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmacie.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'macie2';
        this.accessLevelList = {
            Write: [
                'AcceptInvitation',
                'BatchUpdateAutomatedDiscoveryAccounts',
                'CreateAllowList',
                'CreateClassificationJob',
                'CreateCustomDataIdentifier',
                'CreateFindingsFilter',
                'CreateInvitations',
                'CreateMember',
                'CreateSampleFindings',
                'DeclineInvitations',
                'DeleteAllowList',
                'DeleteCustomDataIdentifier',
                'DeleteFindingsFilter',
                'DeleteInvitations',
                'DeleteMember',
                'DisableMacie',
                'DisableOrganizationAdminAccount',
                'DisassociateFromAdministratorAccount',
                'DisassociateFromMasterAccount',
                'DisassociateMember',
                'EnableMacie',
                'EnableOrganizationAdminAccount',
                'PutClassificationExportConfiguration',
                'PutFindingsPublicationConfiguration',
                'TestCustomDataIdentifier',
                'UpdateAllowList',
                'UpdateAutomatedDiscoveryConfiguration',
                'UpdateClassificationJob',
                'UpdateClassificationScope',
                'UpdateFindingsFilter',
                'UpdateMacieSession',
                'UpdateMemberSession',
                'UpdateOrganizationConfiguration',
                'UpdateResourceProfile',
                'UpdateResourceProfileDetections',
                'UpdateRevealConfiguration',
                'UpdateSensitivityInspectionTemplate'
            ],
            Read: [
                'BatchGetCustomDataIdentifiers',
                'DescribeBuckets',
                'DescribeClassificationJob',
                'DescribeOrganizationConfiguration',
                'GetAdministratorAccount',
                'GetAllowList',
                'GetAutomatedDiscoveryConfiguration',
                'GetBucketStatistics',
                'GetClassificationExportConfiguration',
                'GetClassificationScope',
                'GetCustomDataIdentifier',
                'GetFindingStatistics',
                'GetFindings',
                'GetFindingsFilter',
                'GetFindingsPublicationConfiguration',
                'GetInvitationsCount',
                'GetMacieSession',
                'GetMasterAccount',
                'GetMember',
                'GetResourceProfile',
                'GetRevealConfiguration',
                'GetSensitiveDataOccurrences',
                'GetSensitiveDataOccurrencesAvailability',
                'GetSensitivityInspectionTemplate',
                'GetUsageStatistics',
                'GetUsageTotals',
                'ListTagsForResource',
                'SearchResources'
            ],
            List: [
                'ListAllowLists',
                'ListAutomatedDiscoveryAccounts',
                'ListClassificationJobs',
                'ListClassificationScopes',
                'ListCustomDataIdentifiers',
                'ListFindings',
                'ListFindingsFilters',
                'ListInvitations',
                'ListManagedDataIdentifiers',
                'ListMembers',
                'ListOrganizationAdminAccounts',
                'ListResourceProfileArtifacts',
                'ListResourceProfileDetections',
                'ListSensitivityInspectionTemplates'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Macie2 = Macie2;
//# sourceMappingURL=data:application/json;base64,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