import { AccessLevelList } from '../../shared/access-level';
import { PolicyStatement, Operator } from '../../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [grafana](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmanagedgrafana.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class Grafana extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to upgrade a workspace with a license
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - aws-marketplace:ViewSubscriptions
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toAssociateLicense(): this;
    /**
     * Grants permission to create a workspace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - ec2:GetManagedPrefixListEntries
     * - iam:CreateServiceLinkedRole
     * - organizations:DescribeOrganization
     * - sso:CreateManagedApplicationInstance
     * - sso:DescribeRegisteredRegions
     * - sso:GetSharedSsoConfiguration
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toCreateWorkspace(): this;
    /**
     * Grants permission to create API keys for a workspace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toCreateWorkspaceApiKey(): this;
    /**
     * Grants permission to create service accounts for a workspace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toCreateWorkspaceServiceAccount(): this;
    /**
     * Grants permission to create service account tokens for a workspace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toCreateWorkspaceServiceAccountToken(): this;
    /**
     * Grants permission to delete a workspace
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - sso:DeleteManagedApplicationInstance
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toDeleteWorkspace(): this;
    /**
     * Grants permission to delete API keys from a workspace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toDeleteWorkspaceApiKey(): this;
    /**
     * Grants permission to delete service accounts for a workspace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toDeleteWorkspaceServiceAccount(): this;
    /**
     * Grants permission to delete service account tokens for a workspace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toDeleteWorkspaceServiceAccountToken(): this;
    /**
     * Grants permission to describe a workspace
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toDescribeWorkspace(): this;
    /**
     * Grants permission to describe authentication providers on a workspace
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toDescribeWorkspaceAuthentication(): this;
    /**
     * Grants permission to describe the current configuration string for the given workspace
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/grafana/latest/APIReference/API_DescribeWorkspaceConfiguration.html
     */
    toDescribeWorkspaceConfiguration(): this;
    /**
     * Grants permission to remove a license from a workspace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toDisassociateLicense(): this;
    /**
     * Grants permission to list the permissions on a wokspace
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toListPermissions(): this;
    /**
     * Grants permission to list tags associated with a workspace
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/grafana/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource(): this;
    /**
     * Grants permission to list all available supported Grafana versions. Optionally, include a workspace to list the versions to which it can be upgraded
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/grafana/latest/APIReference/API_ListVersions.html
     */
    toListVersions(): this;
    /**
     * Grants permission to list service account tokens for a workspace
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toListWorkspaceServiceAccountTokens(): this;
    /**
     * Grants permission to list service accounts for a workspace
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toListWorkspaceServiceAccounts(): this;
    /**
     * Grants permission to list workspaces
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toListWorkspaces(): this;
    /**
     * Grants permission to add tags to, or update tag values of, a workspace
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/grafana/latest/APIReference/API_TagResource.html
     */
    toTagResource(): this;
    /**
     * Grants permission to remove tags from a workspace
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/grafana/latest/APIReference/API_UntagResource.html
     */
    toUntagResource(): this;
    /**
     * Grants permission to modify the permissions on a workspace
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toUpdatePermissions(): this;
    /**
     * Grants permission to modify a workspace
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - ec2:GetManagedPrefixListEntries
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toUpdateWorkspace(): this;
    /**
     * Grants permission to modify authentication providers on a workspace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toUpdateWorkspaceAuthentication(): this;
    /**
     * Grants permission to update the configuration string for the given workspace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/grafana/latest/APIReference/API_UpdateWorkspaceConfiguration.html
     */
    toUpdateWorkspaceConfiguration(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type workspace to the statement
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/security-iam.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorkspace(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access by actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-tags
     *
     * Applies to actions:
     * - .toCreateWorkspace()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-tags
     *
     * Applies to resource types:
     * - workspace
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-tags
     *
     * Applies to actions:
     * - .toCreateWorkspace()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [grafana](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmanagedgrafana.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
