"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Grafana = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [grafana](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmanagedgrafana.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Grafana extends shared_1.PolicyStatement {
    /**
     * Grants permission to upgrade a workspace with a license
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - aws-marketplace:ViewSubscriptions
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toAssociateLicense() {
        return this.to('AssociateLicense');
    }
    /**
     * Grants permission to create a workspace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - ec2:GetManagedPrefixListEntries
     * - iam:CreateServiceLinkedRole
     * - organizations:DescribeOrganization
     * - sso:CreateManagedApplicationInstance
     * - sso:DescribeRegisteredRegions
     * - sso:GetSharedSsoConfiguration
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toCreateWorkspace() {
        return this.to('CreateWorkspace');
    }
    /**
     * Grants permission to create API keys for a workspace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toCreateWorkspaceApiKey() {
        return this.to('CreateWorkspaceApiKey');
    }
    /**
     * Grants permission to create service accounts for a workspace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toCreateWorkspaceServiceAccount() {
        return this.to('CreateWorkspaceServiceAccount');
    }
    /**
     * Grants permission to create service account tokens for a workspace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toCreateWorkspaceServiceAccountToken() {
        return this.to('CreateWorkspaceServiceAccountToken');
    }
    /**
     * Grants permission to delete a workspace
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - sso:DeleteManagedApplicationInstance
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toDeleteWorkspace() {
        return this.to('DeleteWorkspace');
    }
    /**
     * Grants permission to delete API keys from a workspace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toDeleteWorkspaceApiKey() {
        return this.to('DeleteWorkspaceApiKey');
    }
    /**
     * Grants permission to delete service accounts for a workspace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toDeleteWorkspaceServiceAccount() {
        return this.to('DeleteWorkspaceServiceAccount');
    }
    /**
     * Grants permission to delete service account tokens for a workspace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toDeleteWorkspaceServiceAccountToken() {
        return this.to('DeleteWorkspaceServiceAccountToken');
    }
    /**
     * Grants permission to describe a workspace
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toDescribeWorkspace() {
        return this.to('DescribeWorkspace');
    }
    /**
     * Grants permission to describe authentication providers on a workspace
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toDescribeWorkspaceAuthentication() {
        return this.to('DescribeWorkspaceAuthentication');
    }
    /**
     * Grants permission to describe the current configuration string for the given workspace
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/grafana/latest/APIReference/API_DescribeWorkspaceConfiguration.html
     */
    toDescribeWorkspaceConfiguration() {
        return this.to('DescribeWorkspaceConfiguration');
    }
    /**
     * Grants permission to remove a license from a workspace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toDisassociateLicense() {
        return this.to('DisassociateLicense');
    }
    /**
     * Grants permission to list the permissions on a wokspace
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toListPermissions() {
        return this.to('ListPermissions');
    }
    /**
     * Grants permission to list tags associated with a workspace
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/grafana/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list all available supported Grafana versions. Optionally, include a workspace to list the versions to which it can be upgraded
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/grafana/latest/APIReference/API_ListVersions.html
     */
    toListVersions() {
        return this.to('ListVersions');
    }
    /**
     * Grants permission to list service account tokens for a workspace
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toListWorkspaceServiceAccountTokens() {
        return this.to('ListWorkspaceServiceAccountTokens');
    }
    /**
     * Grants permission to list service accounts for a workspace
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toListWorkspaceServiceAccounts() {
        return this.to('ListWorkspaceServiceAccounts');
    }
    /**
     * Grants permission to list workspaces
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toListWorkspaces() {
        return this.to('ListWorkspaces');
    }
    /**
     * Grants permission to add tags to, or update tag values of, a workspace
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/grafana/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a workspace
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/grafana/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to modify the permissions on a workspace
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toUpdatePermissions() {
        return this.to('UpdatePermissions');
    }
    /**
     * Grants permission to modify a workspace
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - ec2:GetManagedPrefixListEntries
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toUpdateWorkspace() {
        return this.to('UpdateWorkspace');
    }
    /**
     * Grants permission to modify authentication providers on a workspace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/AMG-and-IAM.html
     */
    toUpdateWorkspaceAuthentication() {
        return this.to('UpdateWorkspaceAuthentication');
    }
    /**
     * Grants permission to update the configuration string for the given workspace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/grafana/latest/APIReference/API_UpdateWorkspaceConfiguration.html
     */
    toUpdateWorkspaceConfiguration() {
        return this.to('UpdateWorkspaceConfiguration');
    }
    /**
     * Adds a resource of type workspace to the statement
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/security-iam.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorkspace(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:grafana:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:/workspaces/${resourceId}`);
    }
    /**
     * Filters access by actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-tags
     *
     * Applies to actions:
     * - .toCreateWorkspace()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-tags
     *
     * Applies to resource types:
     * - workspace
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/grafana/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-tags
     *
     * Applies to actions:
     * - .toCreateWorkspace()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [grafana](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmanagedgrafana.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'grafana';
        this.accessLevelList = {
            Write: [
                'AssociateLicense',
                'CreateWorkspace',
                'CreateWorkspaceApiKey',
                'CreateWorkspaceServiceAccount',
                'CreateWorkspaceServiceAccountToken',
                'DeleteWorkspace',
                'DeleteWorkspaceApiKey',
                'DeleteWorkspaceServiceAccount',
                'DeleteWorkspaceServiceAccountToken',
                'DisassociateLicense',
                'UpdateWorkspace',
                'UpdateWorkspaceAuthentication',
                'UpdateWorkspaceConfiguration'
            ],
            Read: [
                'DescribeWorkspace',
                'DescribeWorkspaceAuthentication',
                'DescribeWorkspaceConfiguration',
                'ListTagsForResource',
                'ListWorkspaceServiceAccountTokens',
                'ListWorkspaceServiceAccounts',
                'ListWorkspaces'
            ],
            List: [
                'ListPermissions',
                'ListVersions'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ],
            'Permissions management': [
                'UpdatePermissions'
            ]
        };
    }
}
exports.Grafana = Grafana;
//# sourceMappingURL=data:application/json;base64,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