"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Aps = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [aps](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmanagedserviceforprometheus.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Aps extends shared_1.PolicyStatement {
    /**
     * Grants permission to create alerts
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-CreateAlertManagerAlerts
     */
    toCreateAlertManagerAlerts() {
        return this.to('CreateAlertManagerAlerts');
    }
    /**
     * Grants permission to create an alert manager definition
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-CreateAlertManagerDefinition
     */
    toCreateAlertManagerDefinition() {
        return this.to('CreateAlertManagerDefinition');
    }
    /**
     * Grants permission to create a logging configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-CreateLoggingConfiguration
     */
    toCreateLoggingConfiguration() {
        return this.to('CreateLoggingConfiguration');
    }
    /**
     * Grants permission to create a rule groups namespace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-CreateRuleGroupsNamespace
     */
    toCreateRuleGroupsNamespace() {
        return this.to('CreateRuleGroupsNamespace');
    }
    /**
     * Grants permission to create a scraper
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - aps:TagResource
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - eks:DescribeCluster
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-CreateScraper
     */
    toCreateScraper() {
        return this.to('CreateScraper');
    }
    /**
     * Grants permission to create a workspace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-CreateWorkspace
     */
    toCreateWorkspace() {
        return this.to('CreateWorkspace');
    }
    /**
     * Grants permission to delete an alert manager definition
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-DeleteAlertManagerDefinition
     */
    toDeleteAlertManagerDefinition() {
        return this.to('DeleteAlertManagerDefinition');
    }
    /**
     * Grants permission to delete a silence
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-DeleteAlertManagerSilence
     */
    toDeleteAlertManagerSilence() {
        return this.to('DeleteAlertManagerSilence');
    }
    /**
     * Grants permission to delete a logging configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-DeleteLoggingConfiguration
     */
    toDeleteLoggingConfiguration() {
        return this.to('DeleteLoggingConfiguration');
    }
    /**
     * Grants permission to delete a rule groups namespace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-DeleteRuleGroupsNamespace
     */
    toDeleteRuleGroupsNamespace() {
        return this.to('DeleteRuleGroupsNamespace');
    }
    /**
     * Grants permission to delete a scraper
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-DeleteScraper
     */
    toDeleteScraper() {
        return this.to('DeleteScraper');
    }
    /**
     * Grants permission to delete a workspace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-DeleteWorkspace
     */
    toDeleteWorkspace() {
        return this.to('DeleteWorkspace');
    }
    /**
     * Grants permission to describe an alert manager definition
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-DescribeAlertManagerDefinition
     */
    toDescribeAlertManagerDefinition() {
        return this.to('DescribeAlertManagerDefinition');
    }
    /**
     * Grants permission to describe a logging configuration
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-DescribeLoggingConfiguration
     */
    toDescribeLoggingConfiguration() {
        return this.to('DescribeLoggingConfiguration');
    }
    /**
     * Grants permission to describe a rule groups namespace
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-DescribeRuleGroupsNamespace
     */
    toDescribeRuleGroupsNamespace() {
        return this.to('DescribeRuleGroupsNamespace');
    }
    /**
     * Grants permission to describe a scraper
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-DescribeScraper
     */
    toDescribeScraper() {
        return this.to('DescribeScraper');
    }
    /**
     * Grants permission to describe a workspace
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-DescribeWorkspace
     */
    toDescribeWorkspace() {
        return this.to('DescribeWorkspace');
    }
    /**
     * Grants permission to describe workspace configuration
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-DescribeWorkspaceConfiguration
     */
    toDescribeWorkspaceConfiguration() {
        return this.to('DescribeWorkspaceConfiguration');
    }
    /**
     * Grants permission to get a silence
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-GetAlertManagerSilence
     */
    toGetAlertManagerSilence() {
        return this.to('GetAlertManagerSilence');
    }
    /**
     * Grants permission to get current status of an alertmanager
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-GetAlertManagerStatus
     */
    toGetAlertManagerStatus() {
        return this.to('GetAlertManagerStatus');
    }
    /**
     * Grants permission to get default scraper configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-GetDefaultScraperConfiguration
     */
    toGetDefaultScraperConfiguration() {
        return this.to('GetDefaultScraperConfiguration');
    }
    /**
     * Grants permission to retrieve AMP workspace labels
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-GetLabels
     */
    toGetLabels() {
        return this.to('GetLabels');
    }
    /**
     * Grants permission to retrieve the metadata for AMP workspace metrics
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-GetMetricMetadata
     */
    toGetMetricMetadata() {
        return this.to('GetMetricMetadata');
    }
    /**
     * Grants permission to retrieve AMP workspace time series data
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-GetSeries
     */
    toGetSeries() {
        return this.to('GetSeries');
    }
    /**
     * Grants permission to list groups
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-ListAlertManagerAlertGroups
     */
    toListAlertManagerAlertGroups() {
        return this.to('ListAlertManagerAlertGroups');
    }
    /**
     * Grants permission to list alerts
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-ListAlertManagerAlerts
     */
    toListAlertManagerAlerts() {
        return this.to('ListAlertManagerAlerts');
    }
    /**
     * Grants permission to list receivers
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-ListAlertManagerReceivers
     */
    toListAlertManagerReceivers() {
        return this.to('ListAlertManagerReceivers');
    }
    /**
     * Grants permission to list silences
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-ListAlertManagerSilences
     */
    toListAlertManagerSilences() {
        return this.to('ListAlertManagerSilences');
    }
    /**
     * Grants permission to list active alerts
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-ListAlerts
     */
    toListAlerts() {
        return this.to('ListAlerts');
    }
    /**
     * Grants permission to list rule groups namespaces
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-ListRuleGroupsNamespaces
     */
    toListRuleGroupsNamespaces() {
        return this.to('ListRuleGroupsNamespaces');
    }
    /**
     * Grants permission to list alerting and recording rules
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-ListRules
     */
    toListRules() {
        return this.to('ListRules');
    }
    /**
     * Grants permission to list scrapers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-ListScrapers
     */
    toListScrapers() {
        return this.to('ListScrapers');
    }
    /**
     * Grants permission to list tags on an AMP resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-ListTagsForResource
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list workspaces
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-ListWorkspaces
     */
    toListWorkspaces() {
        return this.to('ListWorkspaces');
    }
    /**
     * Grants permission to update an alert manager definition
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-PutAlertManagerDefinition
     */
    toPutAlertManagerDefinition() {
        return this.to('PutAlertManagerDefinition');
    }
    /**
     * Grants permission to create or update a silence
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-PutAlertManagerSilences
     */
    toPutAlertManagerSilences() {
        return this.to('PutAlertManagerSilences');
    }
    /**
     * Grants permission to update a rule groups namespace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-PutRuleGroupsNamespace
     */
    toPutRuleGroupsNamespace() {
        return this.to('PutRuleGroupsNamespace');
    }
    /**
     * Grants permission to run a query on AMP workspace metrics
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-QueryMetrics
     */
    toQueryMetrics() {
        return this.to('QueryMetrics');
    }
    /**
     * Grants permission to perform a remote write operation to initiate the streaming of metrics to AMP workspace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-RemoteWrite
     */
    toRemoteWrite() {
        return this.to('RemoteWrite');
    }
    /**
     * Grants permission to tag an AMP resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-TagResource
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag an AMP resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-UntagResource
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a logging configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-UpdateLoggingConfiguration
     */
    toUpdateLoggingConfiguration() {
        return this.to('UpdateLoggingConfiguration');
    }
    /**
     * Grants permission to update a scraper
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - aps:CreateScraper
     * - aps:TagResource
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-UpdateScraper
     */
    toUpdateScraper() {
        return this.to('UpdateScraper');
    }
    /**
     * Grants permission to modify the alias of existing AMP workspace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-UpdateWorkspaceAlias
     */
    toUpdateWorkspaceAlias() {
        return this.to('UpdateWorkspaceAlias');
    }
    /**
     * Grants permission to update workspace configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference.html#AMP-APIReference-UpdateWorkspaceConfiguration
     */
    toUpdateWorkspaceConfiguration() {
        return this.to('UpdateWorkspaceConfiguration');
    }
    /**
     * Adds a resource of type workspace to the statement
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/security-iam.html
     *
     * @param workspaceId - Identifier for the workspaceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     */
    onWorkspace(workspaceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:aps:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workspace/${workspaceId}`);
    }
    /**
     * Adds a resource of type rulegroupsnamespace to the statement
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/security-iam.html
     *
     * @param workspaceId - Identifier for the workspaceId.
     * @param namespace - Identifier for the namespace.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     */
    onRulegroupsnamespace(workspaceId, namespace, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:aps:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:rulegroupsnamespace/${workspaceId}/${namespace}`);
    }
    /**
     * Adds a resource of type scraper to the statement
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/security-iam.html
     *
     * @param scraperId - Identifier for the scraperId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     */
    onScraper(scraperId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:aps:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:scraper/${scraperId}`);
    }
    /**
     * Adds a resource of type cluster to the statement
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/clusters.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCluster(clusterName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:eks:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:cluster/${clusterName}`);
    }
    /**
     * Filters access based on the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateRuleGroupsNamespace()
     * - .toCreateScraper()
     * - .toCreateWorkspace()
     * - .toListTagsForResource()
     * - .toTagResource()
     *
     * Applies to resource types:
     * - workspace
     * - rulegroupsnamespace
     * - scraper
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toCreateAlertManagerAlerts()
     * - .toCreateAlertManagerDefinition()
     * - .toCreateLoggingConfiguration()
     * - .toCreateScraper()
     * - .toDeleteAlertManagerDefinition()
     * - .toDeleteAlertManagerSilence()
     * - .toDeleteLoggingConfiguration()
     * - .toDeleteRuleGroupsNamespace()
     * - .toDeleteScraper()
     * - .toDeleteWorkspace()
     * - .toDescribeAlertManagerDefinition()
     * - .toDescribeLoggingConfiguration()
     * - .toDescribeRuleGroupsNamespace()
     * - .toDescribeScraper()
     * - .toDescribeWorkspace()
     * - .toDescribeWorkspaceConfiguration()
     * - .toGetAlertManagerSilence()
     * - .toGetAlertManagerStatus()
     * - .toGetLabels()
     * - .toGetMetricMetadata()
     * - .toGetSeries()
     * - .toListAlertManagerAlertGroups()
     * - .toListAlertManagerAlerts()
     * - .toListAlertManagerReceivers()
     * - .toListAlertManagerSilences()
     * - .toListAlerts()
     * - .toListRuleGroupsNamespaces()
     * - .toListRules()
     * - .toPutAlertManagerDefinition()
     * - .toPutAlertManagerSilences()
     * - .toPutRuleGroupsNamespace()
     * - .toQueryMetrics()
     * - .toRemoteWrite()
     * - .toUpdateLoggingConfiguration()
     * - .toUpdateScraper()
     * - .toUpdateWorkspaceAlias()
     * - .toUpdateWorkspaceConfiguration()
     *
     * Applies to resource types:
     * - workspace
     * - rulegroupsnamespace
     * - scraper
     * - cluster
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateRuleGroupsNamespace()
     * - .toCreateScraper()
     * - .toCreateWorkspace()
     * - .toListTagsForResource()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * Applies to resource types:
     * - workspace
     * - rulegroupsnamespace
     * - scraper
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [aps](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmanagedserviceforprometheus.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'aps';
        this.accessLevelList = {
            Write: [
                'CreateAlertManagerAlerts',
                'CreateAlertManagerDefinition',
                'CreateLoggingConfiguration',
                'CreateRuleGroupsNamespace',
                'CreateScraper',
                'CreateWorkspace',
                'DeleteAlertManagerDefinition',
                'DeleteAlertManagerSilence',
                'DeleteLoggingConfiguration',
                'DeleteRuleGroupsNamespace',
                'DeleteScraper',
                'DeleteWorkspace',
                'PutAlertManagerDefinition',
                'PutAlertManagerSilences',
                'PutRuleGroupsNamespace',
                'RemoteWrite',
                'UpdateLoggingConfiguration',
                'UpdateScraper',
                'UpdateWorkspaceAlias',
                'UpdateWorkspaceConfiguration'
            ],
            Read: [
                'DescribeAlertManagerDefinition',
                'DescribeLoggingConfiguration',
                'DescribeRuleGroupsNamespace',
                'DescribeScraper',
                'DescribeWorkspace',
                'DescribeWorkspaceConfiguration',
                'GetAlertManagerSilence',
                'GetAlertManagerStatus',
                'GetDefaultScraperConfiguration',
                'GetLabels',
                'GetMetricMetadata',
                'GetSeries',
                'ListAlertManagerAlertGroups',
                'ListAlertManagerAlerts',
                'ListAlertManagerReceivers',
                'ListAlertManagerSilences',
                'ListAlerts',
                'ListRules',
                'ListTagsForResource',
                'QueryMetrics'
            ],
            List: [
                'ListRuleGroupsNamespaces',
                'ListScrapers',
                'ListWorkspaces'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Aps = Aps;
//# sourceMappingURL=data:application/json;base64,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