import { AccessLevelList } from '../../shared/access-level';
import { PolicyStatement, Operator } from '../../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [airflow](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmanagedworkflowsforapacheairflow.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class Airflow extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to create a short-lived token that allows a user to invoke Airflow CLI via an endpoint on the Apache Airflow Webserver
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mwaa/latest/API/API_CreateCliToken.html
     */
    toCreateCliToken(): this;
    /**
     * Grants permission to create an Amazon MWAA environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mwaa/latest/API/API_CreateEnvironment.html
     */
    toCreateEnvironment(): this;
    /**
     * Grants permission to create a short-lived token that allows a user to log into Apache Airflow web UI
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mwaa/latest/API/API_CreateWebLoginToken.html
     */
    toCreateWebLoginToken(): this;
    /**
     * Grants permission to delete an Amazon MWAA environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/mwaa/latest/API/API_DeleteEnvironment.html
     */
    toDeleteEnvironment(): this;
    /**
     * Grants permission to view details about an Amazon MWAA environment
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/mwaa/latest/API/API_GetEnvironment.html
     */
    toGetEnvironment(): this;
    /**
     * Grants permission to invoke Airflow REST API via an endpoint on the Apache Airflow Webserver
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mwaa/latest/API/API_InvokeRestApi.html
     */
    toInvokeRestApi(): this;
    /**
     * Grants permission to list the Amazon MWAA environments in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mwaa/latest/API/API_ListEnvironments.html
     */
    toListEnvironments(): this;
    /**
     * Grants permission to lists tag for an Amazon MWAA environment
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/mwaa/latest/API/API_ListTagsForResource.html
     */
    toListTagsForResource(): this;
    /**
     * Grants permission to publish metrics for an Amazon MWAA environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mwaa/latest/API/API_PublishMetrics.html
     */
    toPublishMetrics(): this;
    /**
     * Grants permission to tag an Amazon MWAA environment
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/mwaa/latest/API/API_TagResource.html
     */
    toTagResource(): this;
    /**
     * Grants permission to untag an Amazon MWAA environment
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/mwaa/latest/API/API_UntagResource.html
     */
    toUntagResource(): this;
    /**
     * Grants permission to modify an Amazon MWAA environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/mwaa/latest/API/API_UpdateEnvironment.html
     */
    toUpdateEnvironment(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type environment to the statement
     *
     * https://docs.aws.amazon.com/mwaa/latest/userguide/using-mwaa.html
     *
     * @param environmentName - Identifier for the environmentName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onEnvironment(environmentName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type rbac-role to the statement
     *
     * https://docs.aws.amazon.com/mwaa/latest/userguide/access-policies.html
     *
     * @param environmentName - Identifier for the environmentName.
     * @param roleName - Identifier for the roleName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onRbacRole(environmentName: string, roleName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateEnvironment()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toCreateEnvironment()
     * - .toDeleteEnvironment()
     * - .toGetEnvironment()
     * - .toListTagsForResource()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateEnvironment()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateEnvironment()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [airflow](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmanagedworkflowsforapacheairflow.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
