"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NeptuneGraph = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [neptune-graph](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonneptuneanalytics.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class NeptuneGraph extends shared_1.PolicyStatement {
    /**
     * Grants permission to cancel an ongoing export task
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_CancelExportTask.html
     */
    toCancelExportTask() {
        return this.to('CancelExportTask');
    }
    /**
     * Grants permission to cancel an ongoing import task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_CancelImportTask.html
     */
    toCancelImportTask() {
        return this.to('CancelImportTask');
    }
    /**
     * Grants permission to cancel a query
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_CancelQuery.html
     */
    toCancelQuery() {
        return this.to('CancelQuery');
    }
    /**
     * Grants permission to create a new graph
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifPublicConnectivity()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - kms:CreateGrant
     * - kms:Decrypt
     * - kms:DescribeKey
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_CreateGraph.html
     */
    toCreateGraph() {
        return this.to('CreateGraph');
    }
    /**
     * Grants permission to create a new snapshot from an existing graph
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_CreateGraphSnapshot.html
     */
    toCreateGraphSnapshot() {
        return this.to('CreateGraphSnapshot');
    }
    /**
     * Grants permission to create a new graph while importing data into the new graph
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifPublicConnectivity()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - iam:PassRole
     * - kms:CreateGrant
     * - kms:Decrypt
     * - kms:DescribeKey
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_CreateGraphUsingImportTask.html
     */
    toCreateGraphUsingImportTask() {
        return this.to('CreateGraphUsingImportTask');
    }
    /**
     * Grants permission to create a new private graph endpoint to access the graph from within a vpc
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:CreateVpcEndpoint
     * - ec2:DescribeAvailabilityZones
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcAttribute
     * - ec2:DescribeVpcEndpoints
     * - ec2:DescribeVpcs
     * - ec2:ModifyVpcEndpoint
     * - route53:AssociateVPCWithHostedZone
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_CreatePrivateGraphEndpoint.html
     */
    toCreatePrivateGraphEndpoint() {
        return this.to('CreatePrivateGraphEndpoint');
    }
    /**
     * Grants permission to delete data via query APIs on the graph
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_ExecuteQuery.html
     */
    toDeleteDataViaQuery() {
        return this.to('DeleteDataViaQuery');
    }
    /**
     * Grants permission to delete a graph
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_DeleteGraph.html
     */
    toDeleteGraph() {
        return this.to('DeleteGraph');
    }
    /**
     * Grants permission to delete a snapshot
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_DeleteGraphSnapshot.html
     */
    toDeleteGraphSnapshot() {
        return this.to('DeleteGraphSnapshot');
    }
    /**
     * Grants permission to delete a private graph endpoint of a graph
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - ec2:DeleteVpcEndpoints
     * - ec2:DescribeAvailabilityZones
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcAttribute
     * - ec2:DescribeVpcEndpoints
     * - ec2:DescribeVpcs
     * - ec2:ModifyVpcEndpoint
     * - route53:DisassociateVPCFromHostedZone
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_DeletePrivateGraphEndpoint.html
     */
    toDeletePrivateGraphEndpoint() {
        return this.to('DeletePrivateGraphEndpoint');
    }
    /**
     * Grants permission to get the engine status of the graph
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/#GetEngineStatus
     */
    toGetEngineStatus() {
        return this.to('GetEngineStatus');
    }
    /**
     * Grants permission to get details about an export task
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_GetExportTask.html
     */
    toGetExportTask() {
        return this.to('GetExportTask');
    }
    /**
     * Grants permission to get details about a graph
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_GetGraph.html
     */
    toGetGraph() {
        return this.to('GetGraph');
    }
    /**
     * Grants permission to get details about a snapshot
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_GetGraphSnapshot.html
     */
    toGetGraphSnapshot() {
        return this.to('GetGraphSnapshot');
    }
    /**
     * Grants permission to get the summary for the data in the graph
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_GetGraphSummary.html
     */
    toGetGraphSummary() {
        return this.to('GetGraphSummary');
    }
    /**
     * Grants permission to get details about an import task
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_GetImportTask.html
     */
    toGetImportTask() {
        return this.to('GetImportTask');
    }
    /**
     * Grants permission to get details about a private graph endpoint of a graph
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_GetPrivateGraphEndpoint.html
     */
    toGetPrivateGraphEndpoint() {
        return this.to('GetPrivateGraphEndpoint');
    }
    /**
     * Grants permission to check the status of a given query
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_GetQuery.html
     */
    toGetQueryStatus() {
        return this.to('GetQueryStatus');
    }
    /**
     * Grants permission to get the statistics for the data in the graph
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/#GetStatisticsStatus
     */
    toGetStatisticsStatus() {
        return this.to('GetStatisticsStatus');
    }
    /**
     * Grants permission to list the export tasks in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_ListExportTasks.html
     */
    toListExportTasks() {
        return this.to('ListExportTasks');
    }
    /**
     * Grants permission to list the snapshots in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_ListGraphSnapshots.html
     */
    toListGraphSnapshots() {
        return this.to('ListGraphSnapshots');
    }
    /**
     * Grants permission to list the graphs in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_ListGraphs.html
     */
    toListGraphs() {
        return this.to('ListGraphs');
    }
    /**
     * Grants permission to list the import tasks in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_ListImportTasks.html
     */
    toListImportTasks() {
        return this.to('ListImportTasks');
    }
    /**
     * Grants permission to list the private graph endpoints for a given graph
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_ListPrivateGraphEndpoints.html
     */
    toListPrivateGraphEndpoints() {
        return this.to('ListPrivateGraphEndpoints');
    }
    /**
     * Grants permission to check the status of all active queries
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_ListQueries.html
     */
    toListQueries() {
        return this.to('ListQueries');
    }
    /**
     * Grants permission to lists tag for a Neptune Analytics resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to read data via query APIs on the graph
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_ExecuteQuery.html
     */
    toReadDataViaQuery() {
        return this.to('ReadDataViaQuery');
    }
    /**
     * Grants permission to reset a graph which deletes all data within the graph
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_ResetGraph.html
     */
    toResetGraph() {
        return this.to('ResetGraph');
    }
    /**
     * Grants permission to create a new graph from an existing snapshot
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifPublicConnectivity()
     *
     * Dependent actions:
     * - kms:CreateGrant
     * - kms:Decrypt
     * - kms:DescribeKey
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_RestoreGraphFromSnapshot.html
     */
    toRestoreGraphFromSnapshot() {
        return this.to('RestoreGraphFromSnapshot');
    }
    /**
     * Grants permission to export data from an existing graph
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_StartExportTask.html
     */
    toStartExportTask() {
        return this.to('StartExportTask');
    }
    /**
     * Grants permission to import data into an existing graph
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_StartImportTask.html
     */
    toStartImportTask() {
        return this.to('StartImportTask');
    }
    /**
     * Grants permission to tag a Neptune Analytics resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a Neptune Analytics resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to modify a graph
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifPublicConnectivity()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_UpdateGraph.html
     */
    toUpdateGraph() {
        return this.to('UpdateGraph');
    }
    /**
     * Grants permission to write data via query APIs on the graph
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_ExecuteQuery.html
     */
    toWriteDataViaQuery() {
        return this.to('WriteDataViaQuery');
    }
    /**
     * Adds a resource of type graph to the statement
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/userguide/iam-resources.html#graph
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGraph(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:neptune-graph:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:graph/${resourceId}`);
    }
    /**
     * Adds a resource of type graph-snapshot to the statement
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/userguide/iam-resources.html#graph-snapshot
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGraphSnapshot(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:neptune-graph:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:graph-snapshot/${resourceId}`);
    }
    /**
     * Adds a resource of type import-task to the statement
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/userguide/iam-resources.html#import-task
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onImportTask(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:neptune-graph:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:import-task/${resourceId}`);
    }
    /**
     * Adds a resource of type export-task to the statement
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/userguide/iam-resources.html#export-task
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onExportTask(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:neptune-graph:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:export-task/${resourceId}`);
    }
    /**
     * Filters access by a tag's key and value in a request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateGraph()
     * - .toCreateGraphSnapshot()
     * - .toCreateGraphUsingImportTask()
     * - .toCreatePrivateGraphEndpoint()
     * - .toRestoreGraphFromSnapshot()
     * - .toStartExportTask()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toCancelExportTask()
     * - .toCancelQuery()
     * - .toCreateGraphSnapshot()
     * - .toCreatePrivateGraphEndpoint()
     * - .toDeleteDataViaQuery()
     * - .toDeleteGraph()
     * - .toDeleteGraphSnapshot()
     * - .toDeletePrivateGraphEndpoint()
     * - .toGetEngineStatus()
     * - .toGetExportTask()
     * - .toGetGraph()
     * - .toGetGraphSnapshot()
     * - .toGetGraphSummary()
     * - .toGetPrivateGraphEndpoint()
     * - .toGetQueryStatus()
     * - .toGetStatisticsStatus()
     * - .toListPrivateGraphEndpoints()
     * - .toListQueries()
     * - .toListTagsForResource()
     * - .toReadDataViaQuery()
     * - .toResetGraph()
     * - .toRestoreGraphFromSnapshot()
     * - .toStartExportTask()
     * - .toUpdateGraph()
     * - .toWriteDataViaQuery()
     *
     * Applies to resource types:
     * - graph
     * - graph-snapshot
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys in a request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateGraph()
     * - .toCreateGraphSnapshot()
     * - .toCreateGraphUsingImportTask()
     * - .toCreatePrivateGraphEndpoint()
     * - .toRestoreGraphFromSnapshot()
     * - .toStartExportTask()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the value of the public connectivity parameter provided in the request or its default value, if unspecified. All access to graphs is IAM authenticated
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/userguide/iam-condition-keys.html#publicconnectivity
     *
     * Applies to actions:
     * - .toCreateGraph()
     * - .toCreateGraphUsingImportTask()
     * - .toRestoreGraphFromSnapshot()
     * - .toUpdateGraph()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifPublicConnectivity(value) {
        return this.if(`PublicConnectivity`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Statement provider for service [neptune-graph](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonneptuneanalytics.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'neptune-graph';
        this.accessLevelList = {
            Write: [
                'CancelExportTask',
                'CancelImportTask',
                'CancelQuery',
                'CreateGraph',
                'CreateGraphSnapshot',
                'CreateGraphUsingImportTask',
                'CreatePrivateGraphEndpoint',
                'DeleteDataViaQuery',
                'DeleteGraph',
                'DeleteGraphSnapshot',
                'DeletePrivateGraphEndpoint',
                'ResetGraph',
                'RestoreGraphFromSnapshot',
                'StartExportTask',
                'StartImportTask',
                'UpdateGraph',
                'WriteDataViaQuery'
            ],
            Read: [
                'GetEngineStatus',
                'GetExportTask',
                'GetGraph',
                'GetGraphSnapshot',
                'GetGraphSummary',
                'GetImportTask',
                'GetPrivateGraphEndpoint',
                'GetQueryStatus',
                'GetStatisticsStatus',
                'ListExportTasks',
                'ListGraphSnapshots',
                'ListGraphs',
                'ListImportTasks',
                'ListPrivateGraphEndpoints',
                'ListQueries',
                'ListTagsForResource',
                'ReadDataViaQuery'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.NeptuneGraph = NeptuneGraph;
//# sourceMappingURL=data:application/json;base64,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