"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Networkmanager = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [networkmanager](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsnetworkmanager.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Networkmanager extends shared_1.PolicyStatement {
    /**
     * Grants permission to accept creation of an attachment between a source and destination in a core network
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DescribeRegions
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_AcceptAttachment.html
     */
    toAcceptAttachment() {
        return this.to('AcceptAttachment');
    }
    /**
     * Grants permission to associate a Connect Peer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_AssociateConnectPeer.html
     */
    toAssociateConnectPeer() {
        return this.to('AssociateConnectPeer');
    }
    /**
     * Grants permission to associate a customer gateway to a device
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCgwArn()
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_AssociateCustomerGateway.html
     */
    toAssociateCustomerGateway() {
        return this.to('AssociateCustomerGateway');
    }
    /**
     * Grants permission to associate a link to a device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_AssociateLink.html
     */
    toAssociateLink() {
        return this.to('AssociateLink');
    }
    /**
     * Grants permission to associate a transit gateway connect peer to a device
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTgwConnectPeerArn()
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_AssociateTransitGatewayConnectPeer.html
     */
    toAssociateTransitGatewayConnectPeer() {
        return this.to('AssociateTransitGatewayConnectPeer');
    }
    /**
     * Grants permission to create a Connect attachment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:DescribeRegions
     * - networkmanager:TagResource
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_CreateConnectAttachment.html
     */
    toCreateConnectAttachment() {
        return this.to('CreateConnectAttachment');
    }
    /**
     * Grants permission to create a Connect Peer connection
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:DescribeRegions
     * - networkmanager:TagResource
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_CreateConnectPeer.html
     */
    toCreateConnectPeer() {
        return this.to('CreateConnectPeer');
    }
    /**
     * Grants permission to create a new connection
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - networkmanager:TagResource
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_CreateConnection.html
     */
    toCreateConnection() {
        return this.to('CreateConnection');
    }
    /**
     * Grants permission to create a new core network
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:DescribeRegions
     * - networkmanager:TagResource
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_CreateCoreNetwork.html
     */
    toCreateCoreNetwork() {
        return this.to('CreateCoreNetwork');
    }
    /**
     * Grants permission to create a new device
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - networkmanager:TagResource
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_CreateDevice.html
     */
    toCreateDevice() {
        return this.to('CreateDevice');
    }
    /**
     * Grants permission to create a Direct Connect gateway attachment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifDirectConnectGatewayArn()
     * - .ifEdgeLocations()
     *
     * Dependent actions:
     * - ec2:DescribeRegions
     * - networkmanager:TagResource
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_CreateDirectConnectGatewayAttachment.html
     */
    toCreateDirectConnectGatewayAttachment() {
        return this.to('CreateDirectConnectGatewayAttachment');
    }
    /**
     * Grants permission to create a new global network
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - networkmanager:TagResource
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_CreateGlobalNetwork.html
     */
    toCreateGlobalNetwork() {
        return this.to('CreateGlobalNetwork');
    }
    /**
     * Grants permission to create a new link
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - networkmanager:TagResource
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_CreateLink.html
     */
    toCreateLink() {
        return this.to('CreateLink');
    }
    /**
     * Grants permission to create a new site
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - networkmanager:TagResource
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_CreateSite.html
     */
    toCreateSite() {
        return this.to('CreateSite');
    }
    /**
     * Grants permission to create a site-to-site VPN attachment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifVpnConnectionArn()
     *
     * Dependent actions:
     * - ec2:DescribeRegions
     * - networkmanager:TagResource
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_CreateSiteToSiteVpnAttachment.html
     */
    toCreateSiteToSiteVpnAttachment() {
        return this.to('CreateSiteToSiteVpnAttachment');
    }
    /**
     * Grants permission to create a Transit Gateway peering
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifTgwArn()
     *
     * Dependent actions:
     * - ec2:DescribeRegions
     * - networkmanager:TagResource
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_CreateTransitGatewayPeering.html
     */
    toCreateTransitGatewayPeering() {
        return this.to('CreateTransitGatewayPeering');
    }
    /**
     * Grants permission to create a TGW RTB attachment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifTgwRtbArn()
     *
     * Dependent actions:
     * - ec2:DescribeRegions
     * - networkmanager:TagResource
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_CreateTransitGatewayRouteTableAttachment.html
     */
    toCreateTransitGatewayRouteTableAttachment() {
        return this.to('CreateTransitGatewayRouteTableAttachment');
    }
    /**
     * Grants permission to create a VPC attachment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifVpcArn()
     * - .ifSubnetArns()
     *
     * Dependent actions:
     * - ec2:DescribeRegions
     * - networkmanager:TagResource
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_CreateVpcAttachment.html
     */
    toCreateVpcAttachment() {
        return this.to('CreateVpcAttachment');
    }
    /**
     * Grants permission to delete an attachment
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DescribeRegions
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DeleteAttachment.html
     */
    toDeleteAttachment() {
        return this.to('DeleteAttachment');
    }
    /**
     * Grants permission to delete a Connect Peer
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DescribeRegions
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DeleteConnectPeer.html
     */
    toDeleteConnectPeer() {
        return this.to('DeleteConnectPeer');
    }
    /**
     * Grants permission to delete a connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DeleteConnection.html
     */
    toDeleteConnection() {
        return this.to('DeleteConnection');
    }
    /**
     * Grants permission to delete a core network
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DescribeRegions
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DeleteCoreNetwork.html
     */
    toDeleteCoreNetwork() {
        return this.to('DeleteCoreNetwork');
    }
    /**
     * Grants permission to delete the core network policy version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DeleteCoreNetworkPolicyVersion.html
     */
    toDeleteCoreNetworkPolicyVersion() {
        return this.to('DeleteCoreNetworkPolicyVersion');
    }
    /**
     * Grants permission to delete a device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DeleteDevice.html
     */
    toDeleteDevice() {
        return this.to('DeleteDevice');
    }
    /**
     * Grants permission to delete a global network
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DeleteGlobalNetwork.html
     */
    toDeleteGlobalNetwork() {
        return this.to('DeleteGlobalNetwork');
    }
    /**
     * Grants permission to delete a link
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DeleteLink.html
     */
    toDeleteLink() {
        return this.to('DeleteLink');
    }
    /**
     * Grants permission to delete a peering
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DescribeRegions
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DeletePeering.html
     */
    toDeletePeering() {
        return this.to('DeletePeering');
    }
    /**
     * Grants permission to delete a resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to delete a site
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DeleteSite.html
     */
    toDeleteSite() {
        return this.to('DeleteSite');
    }
    /**
     * Grants permission to deregister a transit gateway from a global network
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTgwArn()
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DeregisterTransitGateway.html
     */
    toDeregisterTransitGateway() {
        return this.to('DeregisterTransitGateway');
    }
    /**
     * Grants permission to describe global networks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DescribeGlobalNetworks.html
     */
    toDescribeGlobalNetworks() {
        return this.to('DescribeGlobalNetworks');
    }
    /**
     * Grants permission to disassociate a Connect Peer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DisassociateConnectPeer.html
     */
    toDisassociateConnectPeer() {
        return this.to('DisassociateConnectPeer');
    }
    /**
     * Grants permission to disassociate a customer gateway from a device
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCgwArn()
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DisassociateCustomerGateway.html
     */
    toDisassociateCustomerGateway() {
        return this.to('DisassociateCustomerGateway');
    }
    /**
     * Grants permission to disassociate a link from a device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DisassociateLink.html
     */
    toDisassociateLink() {
        return this.to('DisassociateLink');
    }
    /**
     * Grants permission to disassociate a transit gateway connect peer from a device
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTgwConnectPeerArn()
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DisassociateTransitGatewayConnectPeer.html
     */
    toDisassociateTransitGatewayConnectPeer() {
        return this.to('DisassociateTransitGatewayConnectPeer');
    }
    /**
     * Grants permission to apply changes to the core network
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DescribeRegions
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_ExecuteCoreNetworkChangeSet.html
     */
    toExecuteCoreNetworkChangeSet() {
        return this.to('ExecuteCoreNetworkChangeSet');
    }
    /**
     * Grants permission to retrieve a Connect attachment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetConnectAttachment.html
     */
    toGetConnectAttachment() {
        return this.to('GetConnectAttachment');
    }
    /**
     * Grants permission to retrieve a Connect Peer
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetConnectPeer.html
     */
    toGetConnectPeer() {
        return this.to('GetConnectPeer');
    }
    /**
     * Grants permission to describe Connect Peer associations
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetConnectPeerAssociations.html
     */
    toGetConnectPeerAssociations() {
        return this.to('GetConnectPeerAssociations');
    }
    /**
     * Grants permission to describe connections
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetConnections.html
     */
    toGetConnections() {
        return this.to('GetConnections');
    }
    /**
     * Grants permission to retrieve a core network
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetCoreNetwork.html
     */
    toGetCoreNetwork() {
        return this.to('GetCoreNetwork');
    }
    /**
     * Grants permission to retrieve a list of core network change events
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetCoreNetworkChangeEvents.html
     */
    toGetCoreNetworkChangeEvents() {
        return this.to('GetCoreNetworkChangeEvents');
    }
    /**
     * Grants permission to retrieve a list of core network change sets
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetCoreNetworkChangeSet.html
     */
    toGetCoreNetworkChangeSet() {
        return this.to('GetCoreNetworkChangeSet');
    }
    /**
     * Grants permission to retrieve core network policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetCoreNetworkPolicy.html
     */
    toGetCoreNetworkPolicy() {
        return this.to('GetCoreNetworkPolicy');
    }
    /**
     * Grants permission to describe customer gateway associations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetCustomerGatewayAssociations.html
     */
    toGetCustomerGatewayAssociations() {
        return this.to('GetCustomerGatewayAssociations');
    }
    /**
     * Grants permission to describe devices
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetDevices.html
     */
    toGetDevices() {
        return this.to('GetDevices');
    }
    /**
     * Grants permission to retrieve a Direct Connect gateway attachment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetDirectConnectGatewayAttachment.html
     */
    toGetDirectConnectGatewayAttachment() {
        return this.to('GetDirectConnectGatewayAttachment');
    }
    /**
     * Grants permission to describe link associations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetLinkAssociations.html
     */
    toGetLinkAssociations() {
        return this.to('GetLinkAssociations');
    }
    /**
     * Grants permission to describe links
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetLinks.html
     */
    toGetLinks() {
        return this.to('GetLinks');
    }
    /**
     * Grants permission to return the number of resources for a global network grouped by type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetNetworkResourceCounts.html
     */
    toGetNetworkResourceCounts() {
        return this.to('GetNetworkResourceCounts');
    }
    /**
     * Grants permission to retrieve related resources for a resource within the global network
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetNetworkResourceRelationships.html
     */
    toGetNetworkResourceRelationships() {
        return this.to('GetNetworkResourceRelationships');
    }
    /**
     * Grants permission to retrieve a global network resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetNetworkResources.html
     */
    toGetNetworkResources() {
        return this.to('GetNetworkResources');
    }
    /**
     * Grants permission to retrieve routes for a route table within the global network
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetNetworkRoutes.html
     */
    toGetNetworkRoutes() {
        return this.to('GetNetworkRoutes');
    }
    /**
     * Grants permission to retrieve network telemetry objects for the global network
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetNetworkTelemetry.html
     */
    toGetNetworkTelemetry() {
        return this.to('GetNetworkTelemetry');
    }
    /**
     * Grants permission to retrieve a resource policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetResourcePolicy.html
     */
    toGetResourcePolicy() {
        return this.to('GetResourcePolicy');
    }
    /**
     * Grants permission to retrieve a route analysis configuration and result
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetRouteAnalysis.html
     */
    toGetRouteAnalysis() {
        return this.to('GetRouteAnalysis');
    }
    /**
     * Grants permission to retrieve a site-to-site VPN attachment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetSiteToSiteVpnAttachment.html
     */
    toGetSiteToSiteVpnAttachment() {
        return this.to('GetSiteToSiteVpnAttachment');
    }
    /**
     * Grants permission to describe global networks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetSites.html
     */
    toGetSites() {
        return this.to('GetSites');
    }
    /**
     * Grants permission to describe transit gateway connect peer associations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetTransitGatewayConnectPeerAssociations.html
     */
    toGetTransitGatewayConnectPeerAssociations() {
        return this.to('GetTransitGatewayConnectPeerAssociations');
    }
    /**
     * Grants permission to retrieve a Transit Gateway peering
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetTransitGatewayPeering.html
     */
    toGetTransitGatewayPeering() {
        return this.to('GetTransitGatewayPeering');
    }
    /**
     * Grants permission to describe transit gateway registrations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetTransitGatewayRegistrations.html
     */
    toGetTransitGatewayRegistrations() {
        return this.to('GetTransitGatewayRegistrations');
    }
    /**
     * Grants permission to retrieve a TGW RTB attachment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetTransitGatewayRouteTableAttachment.html
     */
    toGetTransitGatewayRouteTableAttachment() {
        return this.to('GetTransitGatewayRouteTableAttachment');
    }
    /**
     * Grants permission to retrieve a VPC attachment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetVpcAttachment.html
     */
    toGetVpcAttachment() {
        return this.to('GetVpcAttachment');
    }
    /**
     * Grants permission to describe attachments
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_ListAttachments.html
     */
    toListAttachments() {
        return this.to('ListAttachments');
    }
    /**
     * Grants permission to describe Connect Peers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_ListConnectPeers.html
     */
    toListConnectPeers() {
        return this.to('ListConnectPeers');
    }
    /**
     * Grants permission to list core network policy versions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_ListCoreNetworkPolicyVersions.html
     */
    toListCoreNetworkPolicyVersions() {
        return this.to('ListCoreNetworkPolicyVersions');
    }
    /**
     * Grants permission to list core networks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_ListCoreNetworks.html
     */
    toListCoreNetworks() {
        return this.to('ListCoreNetworks');
    }
    /**
     * Grants permission to list organization service access status
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_ListOrganizationServiceAccessStatus.html
     */
    toListOrganizationServiceAccessStatus() {
        return this.to('ListOrganizationServiceAccessStatus');
    }
    /**
     * Grants permission to describe peerings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_ListPeerings.html
     */
    toListPeerings() {
        return this.to('ListPeerings');
    }
    /**
     * Grants permission to list tags for a Network Manager resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to create a core network policy
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DescribeRegions
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_PutCoreNetworkPolicy.html
     */
    toPutCoreNetworkPolicy() {
        return this.to('PutCoreNetworkPolicy');
    }
    /**
     * Grants permission to create or update a resource policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_PutResourcePolicy.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to register a transit gateway to a global network
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTgwArn()
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_RegisterTransitGateway.html
     */
    toRegisterTransitGateway() {
        return this.to('RegisterTransitGateway');
    }
    /**
     * Grants permission to reject attachment request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_RejectAttachment.html
     */
    toRejectAttachment() {
        return this.to('RejectAttachment');
    }
    /**
     * Grants permission to restore the core network policy to a previous version
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DescribeRegions
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_RestoreCoreNetworkPolicyVersion.html
     */
    toRestoreCoreNetworkPolicyVersion() {
        return this.to('RestoreCoreNetworkPolicyVersion');
    }
    /**
     * Grants permission to start organization service access update
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_StartOrganizationServiceAccessUpdate.html
     */
    toStartOrganizationServiceAccessUpdate() {
        return this.to('StartOrganizationServiceAccessUpdate');
    }
    /**
     * Grants permission to start a route analysis and stores analysis configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_StartRouteAnalysis.html
     */
    toStartRouteAnalysis() {
        return this.to('StartRouteAnalysis');
    }
    /**
     * Grants permission to tag a Network Manager resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a Network Manager resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_UpdateConnection.html
     */
    toUpdateConnection() {
        return this.to('UpdateConnection');
    }
    /**
     * Grants permission to update a core network
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_UpdateCoreNetwork.html
     */
    toUpdateCoreNetwork() {
        return this.to('UpdateCoreNetwork');
    }
    /**
     * Grants permission to update a device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_UpdateDevice.html
     */
    toUpdateDevice() {
        return this.to('UpdateDevice');
    }
    /**
     * Grants permission to update a Direct Connect gateway attachment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifEdgeLocations()
     *
     * Dependent actions:
     * - ec2:DescribeRegions
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_UpdateDirectConnectGatewayAttachment.html
     */
    toUpdateDirectConnectGatewayAttachment() {
        return this.to('UpdateDirectConnectGatewayAttachment');
    }
    /**
     * Grants permission to update a global network
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_UpdateGlobalNetwork.html
     */
    toUpdateGlobalNetwork() {
        return this.to('UpdateGlobalNetwork');
    }
    /**
     * Grants permission to update a link
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_UpdateLink.html
     */
    toUpdateLink() {
        return this.to('UpdateLink');
    }
    /**
     * Grants permission to add or update metadata key/value pairs on network resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_UpdateNetworkResourceMetadata.html
     */
    toUpdateNetworkResourceMetadata() {
        return this.to('UpdateNetworkResourceMetadata');
    }
    /**
     * Grants permission to update a site
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_UpdateSite.html
     */
    toUpdateSite() {
        return this.to('UpdateSite');
    }
    /**
     * Grants permission to update a VPC attachment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifSubnetArns()
     *
     * Dependent actions:
     * - ec2:DescribeRegions
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_UpdateVpcAttachment.html
     */
    toUpdateVpcAttachment() {
        return this.to('UpdateVpcAttachment');
    }
    /**
     * Adds a resource of type global-network to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/what-is-network-manager.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGlobalNetwork(resourceId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:networkmanager::${account ?? this.defaultAccount}:global-network/${resourceId}`);
    }
    /**
     * Adds a resource of type site to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/what-is-network-manager.html
     *
     * @param globalNetworkId - Identifier for the globalNetworkId.
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSite(globalNetworkId, resourceId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:networkmanager::${account ?? this.defaultAccount}:site/${globalNetworkId}/${resourceId}`);
    }
    /**
     * Adds a resource of type link to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/what-is-network-manager.html
     *
     * @param globalNetworkId - Identifier for the globalNetworkId.
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLink(globalNetworkId, resourceId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:networkmanager::${account ?? this.defaultAccount}:link/${globalNetworkId}/${resourceId}`);
    }
    /**
     * Adds a resource of type device to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/what-is-network-manager.html
     *
     * @param globalNetworkId - Identifier for the globalNetworkId.
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDevice(globalNetworkId, resourceId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:networkmanager::${account ?? this.defaultAccount}:device/${globalNetworkId}/${resourceId}`);
    }
    /**
     * Adds a resource of type connection to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/what-is-network-manager.html
     *
     * @param globalNetworkId - Identifier for the globalNetworkId.
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConnection(globalNetworkId, resourceId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:networkmanager::${account ?? this.defaultAccount}:connection/${globalNetworkId}/${resourceId}`);
    }
    /**
     * Adds a resource of type core-network to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/what-is-network-manager.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCoreNetwork(resourceId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:networkmanager::${account ?? this.defaultAccount}:core-network/${resourceId}`);
    }
    /**
     * Adds a resource of type attachment to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/what-is-network-manager.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAttachment(resourceId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:networkmanager::${account ?? this.defaultAccount}:attachment/${resourceId}`);
    }
    /**
     * Adds a resource of type connect-peer to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/what-is-network-manager.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConnectPeer(resourceId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:networkmanager::${account ?? this.defaultAccount}:connect-peer/${resourceId}`);
    }
    /**
     * Adds a resource of type peering to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/what-is-network-manager.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPeering(resourceId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:networkmanager::${account ?? this.defaultAccount}:peering/${resourceId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateConnectAttachment()
     * - .toCreateConnectPeer()
     * - .toCreateConnection()
     * - .toCreateCoreNetwork()
     * - .toCreateDevice()
     * - .toCreateDirectConnectGatewayAttachment()
     * - .toCreateGlobalNetwork()
     * - .toCreateLink()
     * - .toCreateSite()
     * - .toCreateSiteToSiteVpnAttachment()
     * - .toCreateTransitGatewayPeering()
     * - .toCreateTransitGatewayRouteTableAttachment()
     * - .toCreateVpcAttachment()
     * - .toTagResource()
     * - .toUpdateDirectConnectGatewayAttachment()
     * - .toUpdateVpcAttachment()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toListTagsForResource()
     * - .toTagResource()
     *
     * Applies to resource types:
     * - global-network
     * - site
     * - link
     * - device
     * - connection
     * - core-network
     * - attachment
     * - connect-peer
     * - peering
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateConnectAttachment()
     * - .toCreateConnectPeer()
     * - .toCreateConnection()
     * - .toCreateCoreNetwork()
     * - .toCreateDevice()
     * - .toCreateDirectConnectGatewayAttachment()
     * - .toCreateGlobalNetwork()
     * - .toCreateLink()
     * - .toCreateSite()
     * - .toCreateSiteToSiteVpnAttachment()
     * - .toCreateTransitGatewayPeering()
     * - .toCreateTransitGatewayRouteTableAttachment()
     * - .toCreateVpcAttachment()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateDirectConnectGatewayAttachment()
     * - .toUpdateVpcAttachment()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by which customer gateways can be associated or disassociated
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/nm-security-iam.html
     *
     * Applies to actions:
     * - .toAssociateCustomerGateway()
     * - .toDisassociateCustomerGateway()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifCgwArn(value, operator) {
        return this.if(`cgwArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by which Direct Connect gateway can be used to a create/update attachment
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/nm-security-iam.html
     *
     * Applies to actions:
     * - .toCreateDirectConnectGatewayAttachment()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifDirectConnectGatewayArn(value, operator) {
        return this.if(`directConnectGatewayArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by which edge locations can be added or removed from a Direct Connect gateway attachment
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/nm-security-iam.html
     *
     * Applies to actions:
     * - .toCreateDirectConnectGatewayAttachment()
     * - .toUpdateDirectConnectGatewayAttachment()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEdgeLocations(value, operator) {
        return this.if(`edgeLocations`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by which VPC subnets can be added or removed from a VPC attachment
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/nm-security-iam.html
     *
     * Applies to actions:
     * - .toCreateVpcAttachment()
     * - .toUpdateVpcAttachment()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifSubnetArns(value, operator) {
        return this.if(`subnetArns`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by which transit gateways can be registered, deregistered, or peered
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/nm-security-iam.html
     *
     * Applies to actions:
     * - .toCreateTransitGatewayPeering()
     * - .toDeregisterTransitGateway()
     * - .toRegisterTransitGateway()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifTgwArn(value, operator) {
        return this.if(`tgwArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by which transit gateway connect peers can be associated or disassociated
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/nm-security-iam.html
     *
     * Applies to actions:
     * - .toAssociateTransitGatewayConnectPeer()
     * - .toDisassociateTransitGatewayConnectPeer()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifTgwConnectPeerArn(value, operator) {
        return this.if(`tgwConnectPeerArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by which Transit Gateway Route Table can be used to create an attachment
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/nm-security-iam.html
     *
     * Applies to actions:
     * - .toCreateTransitGatewayRouteTableAttachment()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifTgwRtbArn(value, operator) {
        return this.if(`tgwRtbArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by which VPC can be used to a create/update attachment
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/nm-security-iam.html
     *
     * Applies to actions:
     * - .toCreateVpcAttachment()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifVpcArn(value, operator) {
        return this.if(`vpcArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by which Site-to-Site VPN can be used to a create/update attachment
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/nm-security-iam.html
     *
     * Applies to actions:
     * - .toCreateSiteToSiteVpnAttachment()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifVpnConnectionArn(value, operator) {
        return this.if(`vpnConnectionArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Statement provider for service [networkmanager](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsnetworkmanager.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'networkmanager';
        this.accessLevelList = {
            Write: [
                'AcceptAttachment',
                'AssociateConnectPeer',
                'AssociateCustomerGateway',
                'AssociateLink',
                'AssociateTransitGatewayConnectPeer',
                'CreateConnectAttachment',
                'CreateConnectPeer',
                'CreateConnection',
                'CreateCoreNetwork',
                'CreateDevice',
                'CreateDirectConnectGatewayAttachment',
                'CreateGlobalNetwork',
                'CreateLink',
                'CreateSite',
                'CreateSiteToSiteVpnAttachment',
                'CreateTransitGatewayPeering',
                'CreateTransitGatewayRouteTableAttachment',
                'CreateVpcAttachment',
                'DeleteAttachment',
                'DeleteConnectPeer',
                'DeleteConnection',
                'DeleteCoreNetwork',
                'DeleteCoreNetworkPolicyVersion',
                'DeleteDevice',
                'DeleteGlobalNetwork',
                'DeleteLink',
                'DeletePeering',
                'DeleteResourcePolicy',
                'DeleteSite',
                'DeregisterTransitGateway',
                'DisassociateConnectPeer',
                'DisassociateCustomerGateway',
                'DisassociateLink',
                'DisassociateTransitGatewayConnectPeer',
                'ExecuteCoreNetworkChangeSet',
                'PutCoreNetworkPolicy',
                'PutResourcePolicy',
                'RegisterTransitGateway',
                'RejectAttachment',
                'RestoreCoreNetworkPolicyVersion',
                'StartOrganizationServiceAccessUpdate',
                'StartRouteAnalysis',
                'UpdateConnection',
                'UpdateCoreNetwork',
                'UpdateDevice',
                'UpdateDirectConnectGatewayAttachment',
                'UpdateGlobalNetwork',
                'UpdateLink',
                'UpdateNetworkResourceMetadata',
                'UpdateSite',
                'UpdateVpcAttachment'
            ],
            List: [
                'DescribeGlobalNetworks',
                'GetConnections',
                'GetCustomerGatewayAssociations',
                'GetDevices',
                'GetLinkAssociations',
                'GetLinks',
                'GetSites',
                'GetTransitGatewayConnectPeerAssociations',
                'GetTransitGatewayRegistrations',
                'ListAttachments',
                'ListConnectPeers',
                'ListCoreNetworkPolicyVersions',
                'ListCoreNetworks',
                'ListOrganizationServiceAccessStatus',
                'ListPeerings'
            ],
            Read: [
                'GetConnectAttachment',
                'GetConnectPeer',
                'GetConnectPeerAssociations',
                'GetCoreNetwork',
                'GetCoreNetworkChangeEvents',
                'GetCoreNetworkChangeSet',
                'GetCoreNetworkPolicy',
                'GetDirectConnectGatewayAttachment',
                'GetNetworkResourceCounts',
                'GetNetworkResourceRelationships',
                'GetNetworkResources',
                'GetNetworkRoutes',
                'GetNetworkTelemetry',
                'GetResourcePolicy',
                'GetRouteAnalysis',
                'GetSiteToSiteVpnAttachment',
                'GetTransitGatewayPeering',
                'GetTransitGatewayRouteTableAttachment',
                'GetVpcAttachment',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Networkmanager = Networkmanager;
//# sourceMappingURL=data:application/json;base64,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