"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Outposts = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [outposts](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsoutposts.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Outposts extends shared_1.PolicyStatement {
    /**
     * Grants permission to cancel a capacity task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_CancelCapacityTask.html
     */
    toCancelCapacityTask() {
        return this.to('CancelCapacityTask');
    }
    /**
     * Grants permission to cancel an order
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_CancelOrder.html
     */
    toCancelOrder() {
        return this.to('CancelOrder');
    }
    /**
     * Grants permission to create an order
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_CreateOrder.html
     */
    toCreateOrder() {
        return this.to('CreateOrder');
    }
    /**
     * Grants permission to create an Outpost
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_CreateOutpost.html
     */
    toCreateOutpost() {
        return this.to('CreateOutpost');
    }
    /**
     * Grants permission to create a private connectivity configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/how-outposts-works.html#private-connectivity
     */
    toCreatePrivateConnectivityConfig() {
        return this.to('CreatePrivateConnectivityConfig');
    }
    /**
     * Grants permission to create a site
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_CreateSite.html
     */
    toCreateSite() {
        return this.to('CreateSite');
    }
    /**
     * Grants permission to delete an Outpost
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_DeleteOutpost.html
     */
    toDeleteOutpost() {
        return this.to('DeleteOutpost');
    }
    /**
     * Grants permission to delete a site
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_DeleteSite.html
     */
    toDeleteSite() {
        return this.to('DeleteSite');
    }
    /**
     * Grants permission to get information about the specified capacity task
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_GetCapacityTask.html
     */
    toGetCapacityTask() {
        return this.to('GetCapacityTask');
    }
    /**
     * Grants permission to get a catalog item
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_GetCatalogItem.html
     */
    toGetCatalogItem() {
        return this.to('GetCatalogItem');
    }
    /**
     * Grants permission to get information about the connection for your Outpost server
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_GetConnection.html
     */
    toGetConnection() {
        return this.to('GetConnection');
    }
    /**
     * Grants permission to get information about an order
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_GetOrder.html
     */
    toGetOrder() {
        return this.to('GetOrder');
    }
    /**
     * Grants permission to get information about the specified Outpost
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_GetOutpost.html
     */
    toGetOutpost() {
        return this.to('GetOutpost');
    }
    /**
     * Grants permission to get the instance types for the specified Outpost
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_GetOutpostInstanceTypes.html
     */
    toGetOutpostInstanceTypes() {
        return this.to('GetOutpostInstanceTypes');
    }
    /**
     * Grants permission to get the supported instance types for the specified Outpost
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_GetOutpostSupportedInstanceTypes.html
     */
    toGetOutpostSupportedInstanceTypes() {
        return this.to('GetOutpostSupportedInstanceTypes');
    }
    /**
     * Grants permission to get a private connectivity configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/how-outposts-works.html#private-connectivity
     */
    toGetPrivateConnectivityConfig() {
        return this.to('GetPrivateConnectivityConfig');
    }
    /**
     * Grants permission to get a site
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_GetSite.html
     */
    toGetSite() {
        return this.to('GetSite');
    }
    /**
     * Grants permission to get a site address
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_GetSiteAddress.html
     */
    toGetSiteAddress() {
        return this.to('GetSiteAddress');
    }
    /**
     * Grants permission to list all running instances for the specified Outpost
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_ListAssetInstances.html
     */
    toListAssetInstances() {
        return this.to('ListAssetInstances');
    }
    /**
     * Grants permission to list the assets for your Outpost
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_ListAssets.html
     */
    toListAssets() {
        return this.to('ListAssets');
    }
    /**
     * Grants permission to list all running instances that are blocking the capacity task from running for the specified Outpost
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_ListBlockingInstancesForCapacityTask.html
     */
    toListBlockingInstancesForCapacityTask() {
        return this.to('ListBlockingInstancesForCapacityTask');
    }
    /**
     * Grants permission to list the capacity tasks for your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_ListCapacityTasks.html
     */
    toListCapacityTasks() {
        return this.to('ListCapacityTasks');
    }
    /**
     * Grants permission to list all catalog items
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_ListCatalogItems.html
     */
    toListCatalogItems() {
        return this.to('ListCatalogItems');
    }
    /**
     * Grants permission to list the orders for your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_ListOrders.html
     */
    toListOrders() {
        return this.to('ListOrders');
    }
    /**
     * Grants permission to list the Outposts for your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_ListOutposts.html
     */
    toListOutposts() {
        return this.to('ListOutposts');
    }
    /**
     * Grants permission to list the sites for your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_ListSites.html
     */
    toListSites() {
        return this.to('ListSites');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to create a capacity task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_StartCapacityTask.html
     */
    toStartCapacityTask() {
        return this.to('StartCapacityTask');
    }
    /**
     * Grants permission to start a connection for your Outpost server
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_StartConnection.html
     */
    toStartConnection() {
        return this.to('StartConnection');
    }
    /**
     * Grants permission to tag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an Outpost
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_UpdateOutpost.html
     */
    toUpdateOutpost() {
        return this.to('UpdateOutpost');
    }
    /**
     * Grants permission to update a site
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_UpdateSite.html
     */
    toUpdateSite() {
        return this.to('UpdateSite');
    }
    /**
     * Grants permission to update the site address
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_UpdateSiteAddress.html
     */
    toUpdateSiteAddress() {
        return this.to('UpdateSiteAddress');
    }
    /**
     * Grants permission to update the physical properties of a rack at a site
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_UpdateSiteRackPhysicalProperties.html
     */
    toUpdateSiteRackPhysicalProperties() {
        return this.to('UpdateSiteRackPhysicalProperties');
    }
    /**
     * Adds a resource of type outpost to the statement
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/what-is-outposts.html
     *
     * @param outpostId - Identifier for the outpostId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onOutpost(outpostId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:outposts:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:outpost/${outpostId}`);
    }
    /**
     * Adds a resource of type site to the statement
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/what-is-outposts.html
     *
     * @param siteId - Identifier for the siteId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSite(siteId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:outposts:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:site/${siteId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/identity-access-management.html
     *
     * Applies to actions:
     * - .toCreateOutpost()
     * - .toCreateSite()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/identity-access-management.html
     *
     * Applies to resource types:
     * - outpost
     * - site
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/identity-access-management.html
     *
     * Applies to actions:
     * - .toCreateOutpost()
     * - .toCreateSite()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [outposts](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsoutposts.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'outposts';
        this.accessLevelList = {
            Write: [
                'CancelCapacityTask',
                'CancelOrder',
                'CreateOrder',
                'CreateOutpost',
                'CreatePrivateConnectivityConfig',
                'CreateSite',
                'DeleteOutpost',
                'DeleteSite',
                'StartCapacityTask',
                'StartConnection',
                'UpdateOutpost',
                'UpdateSite',
                'UpdateSiteAddress',
                'UpdateSiteRackPhysicalProperties'
            ],
            Read: [
                'GetCapacityTask',
                'GetCatalogItem',
                'GetConnection',
                'GetOrder',
                'GetOutpost',
                'GetOutpostInstanceTypes',
                'GetOutpostSupportedInstanceTypes',
                'GetPrivateConnectivityConfig',
                'GetSite',
                'GetSiteAddress',
                'ListTagsForResource'
            ],
            List: [
                'ListAssetInstances',
                'ListAssets',
                'ListBlockingInstancesForCapacityTask',
                'ListCapacityTasks',
                'ListCatalogItems',
                'ListOrders',
                'ListOutposts',
                'ListSites'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Outposts = Outposts;
//# sourceMappingURL=data:application/json;base64,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