"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Partnercentral = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [partnercentral](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awspartnercentralselling.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Partnercentral extends shared_1.PolicyStatement {
    /**
     * Grants permission to accept Engagement Invitations on AWS Partner Central
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_AcceptEngagementInvitation.html
     */
    toAcceptEngagementInvitation() {
        return this.to('AcceptEngagementInvitation');
    }
    /**
     * Grants permission to assign Opportunities on AWS Partner Central
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_AssignOpportunity.html
     */
    toAssignOpportunity() {
        return this.to('AssignOpportunity');
    }
    /**
     * Grants permission to associate Opportunities on AWS Partner Central with other entities
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCatalog()
     * - .ifRelatedEntityType()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_AssociateOpportunity.html
     */
    toAssociateOpportunity() {
        return this.to('AssociateOpportunity');
    }
    /**
     * Grants permission to creating engagements in AWS Partner Central
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_CreateEngagement.html
     */
    toCreateEngagement() {
        return this.to('CreateEngagement');
    }
    /**
     * Grants permission to creating engagement invitations in AWS Partner Central
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_CreateEngagementInvitation.html
     */
    toCreateEngagementInvitation() {
        return this.to('CreateEngagementInvitation');
    }
    /**
     * Grants permission to create new Opportunities on AWS Partner Central
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_CreateOpportunity.html
     */
    toCreateOpportunity() {
        return this.to('CreateOpportunity');
    }
    /**
     * Grants permission to creating resource snapshots in AWS Partner Central
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_CreateResourceSnapshot.html
     */
    toCreateResourceSnapshot() {
        return this.to('CreateResourceSnapshot');
    }
    /**
     * Grants permission to creating resource snapshot jobs in AWS Partner Central
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_CreateResourceSnapshotJob.html
     */
    toCreateResourceSnapshotJob() {
        return this.to('CreateResourceSnapshotJob');
    }
    /**
     * Grants permission to deleting resource snapshot jobs on AWS Partner Central
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_DeleteResourceSnapshotJob.html
     */
    toDeleteResourceSnapshotJob() {
        return this.to('DeleteResourceSnapshotJob');
    }
    /**
     * Grants permission to disassociate Opportunities on AWS Partner Central from other entities
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCatalog()
     * - .ifRelatedEntityType()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_DisassociateOpportunity.html
     */
    toDisassociateOpportunity() {
        return this.to('DisassociateOpportunity');
    }
    /**
     * Grants permission to retrieve AWS Opportunity Summaries for Opportunities on AWS Partner Central
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_GetAwsOpportunitySummary.html
     */
    toGetAwsOpportunitySummary() {
        return this.to('GetAwsOpportunitySummary');
    }
    /**
     * Grants permission to retrieval of engagement details in AWS Partner Central
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_GetEngagement.html
     */
    toGetEngagement() {
        return this.to('GetEngagement');
    }
    /**
     * Grants permission to retrieve details of Engagement Invitations on AWS Partner Central
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_GetEngagementInvitation.html
     */
    toGetEngagementInvitation() {
        return this.to('GetEngagementInvitation');
    }
    /**
     * Grants permission to retrieve details of Opportunities on AWS Partner Central
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_GetOpportunity.html
     */
    toGetOpportunity() {
        return this.to('GetOpportunity');
    }
    /**
     * Grants permission to retrieving resource snapshot details in AWS Partner Central
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_GetResourceSnapshot.html
     */
    toGetResourceSnapshot() {
        return this.to('GetResourceSnapshot');
    }
    /**
     * Grants permission to retrieving resource snapshot job details in AWS Partner Central
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_GetResourceSnapshotJob.html
     */
    toGetResourceSnapshotJob() {
        return this.to('GetResourceSnapshotJob');
    }
    /**
     * Grants permission to retrieving system settings settings in AWS Partner Central
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_GetSellingSystemSettings.html
     */
    toGetSellingSystemSettings() {
        return this.to('GetSellingSystemSettings');
    }
    /**
     * Grants permission to listing engagements by accepting invitation tasks in AWS Partner Central
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_ListEngagementByAcceptingInvitationTasks.html
     */
    toListEngagementByAcceptingInvitationTasks() {
        return this.to('ListEngagementByAcceptingInvitationTasks');
    }
    /**
     * Grants permission to listing engagements from opportunity tasks in AWS Partner Central
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_ListEngagementFromOpportunityTasks.html
     */
    toListEngagementFromOpportunityTasks() {
        return this.to('ListEngagementFromOpportunityTasks');
    }
    /**
     * Grants permission to list Engagement Invitations on AWS Partner Central
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_ListEngagementInvitations.html
     */
    toListEngagementInvitations() {
        return this.to('ListEngagementInvitations');
    }
    /**
     * Grants permission to listing engagement members in AWS Partner Central
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_ListEngagementMembers.html
     */
    toListEngagementMembers() {
        return this.to('ListEngagementMembers');
    }
    /**
     * Grants permission to listing engagement resource associations in AWS Partner Central
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_ListEngagementResourceAssociations.html
     */
    toListEngagementResourceAssociations() {
        return this.to('ListEngagementResourceAssociations');
    }
    /**
     * Grants permission to listing engagements in AWS Partner Central
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_ListEngagements.html
     */
    toListEngagements() {
        return this.to('ListEngagements');
    }
    /**
     * Grants permission to list Opportunities on AWS Partner Central
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_ListOpportunities.html
     */
    toListOpportunities() {
        return this.to('ListOpportunities');
    }
    /**
     * Grants permission to listing resource snapshot jobs in AWS Partner Central
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_ListResourceSnapshotJobs.html
     */
    toListResourceSnapshotJobs() {
        return this.to('ListResourceSnapshotJobs');
    }
    /**
     * Grants permission to listing resource snapshots in AWS Partner Central
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_ListResourceSnapshots.html
     */
    toListResourceSnapshots() {
        return this.to('ListResourceSnapshots');
    }
    /**
     * Grants permission to list Solutions on AWS Partner Central
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_ListSolutions.html
     */
    toListSolutions() {
        return this.to('ListSolutions');
    }
    /**
     * Grants permission to add lists tags to a resource. Supported resource: ResourceSnapshotJob
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to put system settings settings in AWS Partner Central
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_PutSellingSystemSettings.html
     */
    toPutSellingSystemSettings() {
        return this.to('PutSellingSystemSettings');
    }
    /**
     * Grants permission to reject Engagement Invitations on AWS Partner Central
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_RejectEngagementInvitation.html
     */
    toRejectEngagementInvitation() {
        return this.to('RejectEngagementInvitation');
    }
    /**
     * Grants permission to initiate tasks that start Engagements on AWS Partner Central by accepting an Engagement Invitation
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifCatalog()
     *
     * Dependent actions:
     * - partnercentral:AcceptEngagementInvitation
     * - partnercentral:CreateOpportunity
     * - partnercentral:CreateResourceSnapshotJob
     * - partnercentral:GetEngagementInvitation
     * - partnercentral:StartResourceSnapshotJob
     * - partnercentral:SubmitOpportunity
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_StartEngagementByAcceptingInvitationTask.html
     */
    toStartEngagementByAcceptingInvitationTask() {
        return this.to('StartEngagementByAcceptingInvitationTask');
    }
    /**
     * Grants permission to initiate tasks that start Engagements from Opportunities on AWS Partner Central
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifCatalog()
     *
     * Dependent actions:
     * - partnercentral:CreateEngagement
     * - partnercentral:CreateEngagementInvitation
     * - partnercentral:CreateResourceSnapshotJob
     * - partnercentral:GetOpportunity
     * - partnercentral:StartResourceSnapshotJob
     * - partnercentral:SubmitOpportunity
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_StartEngagementFromOpportunityTask.html
     */
    toStartEngagementFromOpportunityTask() {
        return this.to('StartEngagementFromOpportunityTask');
    }
    /**
     * Grants permission to starting resource snapshot jobs in AWS Partner Central
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_StartResourceSnapshotJob.html
     */
    toStartResourceSnapshotJob() {
        return this.to('StartResourceSnapshotJob');
    }
    /**
     * Grants permission to stopping resource snapshot jobs in AWS Partner Central
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_StopResourceSnapshotJob.html
     */
    toStopResourceSnapshotJob() {
        return this.to('StopResourceSnapshotJob');
    }
    /**
     * Grants permission to submit Opportunities on AWS Partner Central
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_SubmitOpportunity.html
     */
    toSubmitOpportunity() {
        return this.to('SubmitOpportunity');
    }
    /**
     * Grants permission to add new tags to a resource. Supported resource: ResourceSnapshotJob
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a resource. Supported resource: ResourceSnapshotJob
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update Opportunities on AWS Partner Central
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_UpdateOpportunity.html
     */
    toUpdateOpportunity() {
        return this.to('UpdateOpportunity');
    }
    /**
     * Adds a resource of type Engagement to the statement
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/working-with-multi-partner-opportunities.html
     *
     * @param catalog - Identifier for the catalog.
     * @param identifier - Identifier for the identifier.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onEngagement(catalog, identifier, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:partnercentral:${region ?? this.defaultRegion}::catalog/${catalog}/engagement/${identifier}`);
    }
    /**
     * Adds a resource of type engagement-by-accepting-invitation-task to the statement
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/working-with-multi-partner-opportunities.html
     *
     * @param catalog - Identifier for the catalog.
     * @param taskId - Identifier for the taskId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onEngagementByAcceptingInvitationTask(catalog, taskId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:partnercentral:${region ?? this.defaultRegion}::catalog/${catalog}/engagement-by-accepting-invitation-task/${taskId}`);
    }
    /**
     * Adds a resource of type engagement-from-opportunity-task to the statement
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/working-with-multi-partner-opportunities.html
     *
     * @param catalog - Identifier for the catalog.
     * @param taskId - Identifier for the taskId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onEngagementFromOpportunityTask(catalog, taskId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:partnercentral:${region ?? this.defaultRegion}::catalog/${catalog}/engagement-from-opportunity-task/${taskId}`);
    }
    /**
     * Adds a resource of type engagement-invitation to the statement
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/working-with-multi-partner-opportunities.html
     *
     * @param catalog - Identifier for the catalog.
     * @param identifier - Identifier for the identifier.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onEngagementInvitation(catalog, identifier, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:partnercentral:${region ?? this.defaultRegion}::catalog/${catalog}/engagement-invitation/${identifier}`);
    }
    /**
     * Adds a resource of type Opportunity to the statement
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/working-with-your-opportunities.html
     *
     * @param catalog - Identifier for the catalog.
     * @param identifier - Identifier for the identifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onOpportunity(catalog, identifier, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:partnercentral:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:catalog/${catalog}/opportunity/${identifier}`);
    }
    /**
     * Adds a resource of type resource-snapshot-job to the statement
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/working-with-multi-partner-opportunities.html
     *
     * @param catalog - Identifier for the catalog.
     * @param identifier - Identifier for the identifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onResourceSnapshotJob(catalog, identifier, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:partnercentral:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:catalog/${catalog}/resource-snapshot-job/${identifier}`);
    }
    /**
     * Adds a resource of type ResourceSnapshot to the statement
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/working-with-multi-partner-opportunities.html
     *
     * @param catalog - Identifier for the catalog.
     * @param engagementIdentifier - Identifier for the engagementIdentifier.
     * @param resourceType - Identifier for the resourceType.
     * @param resourceIdentifier - Identifier for the resourceIdentifier.
     * @param templateIdentifier - Identifier for the templateIdentifier.
     * @param snapshotRevision - Identifier for the snapshotRevision.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onResourceSnapshot(catalog, engagementIdentifier, resourceType, resourceIdentifier, templateIdentifier, snapshotRevision, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:partnercentral:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:catalog/${catalog}/engagement/${engagementIdentifier}/resource/${resourceType}/${resourceIdentifier}/template/${templateIdentifier}/resource-snapshot/${snapshotRevision}`);
    }
    /**
     * Adds a resource of type Solution to the statement
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_ListSolutions.html
     *
     * @param catalog - Identifier for the catalog.
     * @param identifier - Identifier for the identifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onSolution(catalog, identifier, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:partnercentral:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:catalog/${catalog}/solution/${identifier}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateResourceSnapshotJob()
     * - .toStartEngagementByAcceptingInvitationTask()
     * - .toStartEngagementFromOpportunityTask()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toDeleteResourceSnapshotJob()
     * - .toGetResourceSnapshotJob()
     * - .toListTagsForResource()
     * - .toStartResourceSnapshotJob()
     * - .toStopResourceSnapshotJob()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateResourceSnapshotJob()
     * - .toDeleteResourceSnapshotJob()
     * - .toGetResourceSnapshotJob()
     * - .toListTagsForResource()
     * - .toStartEngagementByAcceptingInvitationTask()
     * - .toStartEngagementFromOpportunityTask()
     * - .toStartResourceSnapshotJob()
     * - .toStopResourceSnapshotJob()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a specific Catalog. Accepted values: [AWS, Sandbox]
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/access-control.html#managing-permissions-with-condition-keyspage.html#Catalog
     *
     * Applies to actions:
     * - .toAcceptEngagementInvitation()
     * - .toAssignOpportunity()
     * - .toAssociateOpportunity()
     * - .toCreateEngagement()
     * - .toCreateEngagementInvitation()
     * - .toCreateOpportunity()
     * - .toCreateResourceSnapshot()
     * - .toCreateResourceSnapshotJob()
     * - .toDeleteResourceSnapshotJob()
     * - .toDisassociateOpportunity()
     * - .toGetAwsOpportunitySummary()
     * - .toGetEngagement()
     * - .toGetEngagementInvitation()
     * - .toGetOpportunity()
     * - .toGetResourceSnapshot()
     * - .toGetResourceSnapshotJob()
     * - .toGetSellingSystemSettings()
     * - .toListEngagementByAcceptingInvitationTasks()
     * - .toListEngagementFromOpportunityTasks()
     * - .toListEngagementInvitations()
     * - .toListEngagementMembers()
     * - .toListEngagementResourceAssociations()
     * - .toListEngagements()
     * - .toListOpportunities()
     * - .toListResourceSnapshotJobs()
     * - .toListResourceSnapshots()
     * - .toListSolutions()
     * - .toListTagsForResource()
     * - .toRejectEngagementInvitation()
     * - .toStartEngagementByAcceptingInvitationTask()
     * - .toStartEngagementFromOpportunityTask()
     * - .toStartResourceSnapshotJob()
     * - .toStopResourceSnapshotJob()
     * - .toSubmitOpportunity()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateOpportunity()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCatalog(value, operator) {
        return this.if(`Catalog`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by entity types for Opportunity association. Accepted values: [Solutions, AwsProducts, AwsMarketplaceOffers]
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/access-control.html#managing-permissions-with-condition-keyspage.html#RelatedEntityType
     *
     * Applies to actions:
     * - .toAssociateOpportunity()
     * - .toDisassociateOpportunity()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRelatedEntityType(value, operator) {
        return this.if(`RelatedEntityType`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [partnercentral](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awspartnercentralselling.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'partnercentral';
        this.accessLevelList = {
            Write: [
                'AcceptEngagementInvitation',
                'AssignOpportunity',
                'AssociateOpportunity',
                'CreateEngagement',
                'CreateEngagementInvitation',
                'CreateOpportunity',
                'CreateResourceSnapshot',
                'CreateResourceSnapshotJob',
                'DeleteResourceSnapshotJob',
                'DisassociateOpportunity',
                'PutSellingSystemSettings',
                'RejectEngagementInvitation',
                'StartEngagementByAcceptingInvitationTask',
                'StartEngagementFromOpportunityTask',
                'StartResourceSnapshotJob',
                'StopResourceSnapshotJob',
                'SubmitOpportunity',
                'UpdateOpportunity'
            ],
            Read: [
                'GetAwsOpportunitySummary',
                'GetEngagement',
                'GetEngagementInvitation',
                'GetOpportunity',
                'GetResourceSnapshot',
                'GetResourceSnapshotJob',
                'GetSellingSystemSettings',
                'ListEngagementMembers',
                'ListEngagementResourceAssociations',
                'ListResourceSnapshots',
                'ListTagsForResource'
            ],
            List: [
                'ListEngagementByAcceptingInvitationTasks',
                'ListEngagementFromOpportunityTasks',
                'ListEngagementInvitations',
                'ListEngagements',
                'ListOpportunities',
                'ListResourceSnapshotJobs',
                'ListSolutions'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Partnercentral = Partnercentral;
//# sourceMappingURL=data:application/json;base64,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