"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PaymentCryptography = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [payment-cryptography](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awspaymentcryptography.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class PaymentCryptography extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a user-friendly name for a Key
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_CreateAlias.html
     */
    toCreateAlias() {
        return this.to('CreateAlias');
    }
    /**
     * Grants permission to create a unique customer managed key in the caller's AWS account and region
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - payment-cryptography:TagResource
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_CreateKey.html
     */
    toCreateKey() {
        return this.to('CreateKey');
    }
    /**
     * Grants permission to decrypt ciphertext data to plaintext using symmetric, asymmetric or DUKPT data encryption key
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/DataAPIReference/API_DecryptData.html
     */
    toDecryptData() {
        return this.to('DecryptData');
    }
    /**
     * Grants permission to delete the specified alias
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_DeleteAlias.html
     */
    toDeleteAlias() {
        return this.to('DeleteAlias');
    }
    /**
     * Grants permission to schedule the deletion of a Key
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_DeleteKey.html
     */
    toDeleteKey() {
        return this.to('DeleteKey');
    }
    /**
     * Grants permission to encrypt plaintext data to ciphertext using symmetric, asymmetric or DUKPT data encryption key
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/DataAPIReference/API_EncryptData.html
     */
    toEncryptData() {
        return this.to('EncryptData');
    }
    /**
     * Grants permission to export a key from the service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ExportKey.html
     */
    toExportKey() {
        return this.to('ExportKey');
    }
    /**
     * Grants permission to generate card-related data using algorithms such as Card Verification Values (CVV/CVV2), Dynamic Card Verification Values (dCVV/dCVV2) or Card Security Codes (CSC) that check the validity of a magnetic stripe card
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/DataAPIReference/API_GenerateCardValidationData.html
     */
    toGenerateCardValidationData() {
        return this.to('GenerateCardValidationData');
    }
    /**
     * Grants permission to generate a MAC (Message Authentication Code) cryptogram
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/DataAPIReference/API_GenerateMac.html
     */
    toGenerateMac() {
        return this.to('GenerateMac');
    }
    /**
     * Grants permission to generate a MAC (Message Authentication Code) cryptogram
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/DataAPIReference/API_GenerateMacEmvPinChange.html
     */
    toGenerateMacEmvPinChange() {
        return this.to('GenerateMacEmvPinChange');
    }
    /**
     * Grants permission to generate pin-related data such as PIN, PIN Verification Value (PVV), PIN Block and PIN Offset during new card issuance or card re-issuance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/DataAPIReference/API_GeneratePinData.html
     */
    toGeneratePinData() {
        return this.to('GeneratePinData');
    }
    /**
     * Grants permission to return the keyArn associated with an aliasName
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_GetAlias.html
     */
    toGetAlias() {
        return this.to('GetAlias');
    }
    /**
     * Grants permission to return the detailed information about the specified key
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_GetKey.html
     */
    toGetKey() {
        return this.to('GetKey');
    }
    /**
     * Grants permission to get the export token and the signing key certificate to initiate a TR-34 key export
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_GetParametersForExport.html
     */
    toGetParametersForExport() {
        return this.to('GetParametersForExport');
    }
    /**
     * Grants permission to get the import token and the wrapping key certificate to initiate a TR-34 key import
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_GetParametersForImport.html
     */
    toGetParametersForImport() {
        return this.to('GetParametersForImport');
    }
    /**
     * Grants permission to return the public key from a key of class PUBLIC_KEY
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_GetPublicKeyCertificate.html
     */
    toGetPublicKeyCertificate() {
        return this.to('GetPublicKeyCertificate');
    }
    /**
     * Grants permission to imports keys and public key certificates
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - payment-cryptography:TagResource
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ImportKey.html
     */
    toImportKey() {
        return this.to('ImportKey');
    }
    /**
     * Grants permission to return a list of aliases created for all keys in the caller's AWS account and Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ListAliases.html
     */
    toListAliases() {
        return this.to('ListAliases');
    }
    /**
     * Grants permission to return a list of keys created in the caller's AWS account and Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ListKeys.html
     */
    toListKeys() {
        return this.to('ListKeys');
    }
    /**
     * Grants permission to return a list of tags created in the caller's AWS account and Region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to re-encrypt ciphertext using DUKPT, Symmetric and Asymmetric Data Encryption Keys
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/DataAPIReference/API_ReEncryptData.html
     */
    toReEncryptData() {
        return this.to('ReEncryptData');
    }
    /**
     * Grants permission to cancel a scheduled key deletion if at any point during the waiting period a Key needs to be revived
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_RestoreKey.html
     */
    toRestoreKey() {
        return this.to('RestoreKey');
    }
    /**
     * Grants permission to enable a disabled Key
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_StartKeyUsage.html
     */
    toStartKeyUsage() {
        return this.to('StartKeyUsage');
    }
    /**
     * Grants permission to disable an enabled Key
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_StopKeyUsage.html
     */
    toStopKeyUsage() {
        return this.to('StopKeyUsage');
    }
    /**
     * Grants permission to add or overwrites one or more tags for the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to translate encrypted PIN block from and to ISO 9564 formats 0,1,3,4
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/DataAPIReference/API_TranslatePinData.html
     */
    toTranslatePinData() {
        return this.to('TranslatePinData');
    }
    /**
     * Grants permission to remove the specified tag or tags from the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to change the key to which an alias is assigned, or unassign it from its current key
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_UpdateAlias.html
     */
    toUpdateAlias() {
        return this.to('UpdateAlias');
    }
    /**
     * Grants permission to verify Authorization Request Cryptogram (ARQC) for a EMV chip payment card authorization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/DataAPIReference/API_VerifyAuthRequestCryptogram.html
     */
    toVerifyAuthRequestCryptogram() {
        return this.to('VerifyAuthRequestCryptogram');
    }
    /**
     * Grants permission to verify card-related validation data using algorithms such as Card Verification Values (CVV/CVV2), Dynamic Card Verification Values (dCVV/dCVV2) and Card Security Codes (CSC)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/DataAPIReference/API_VerifyCardValidationData.html
     */
    toVerifyCardValidationData() {
        return this.to('VerifyCardValidationData');
    }
    /**
     * Grants permission to verify MAC (Message Authentication Code) of input data against a provided MAC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/DataAPIReference/API_VerifyMac.html
     */
    toVerifyMac() {
        return this.to('VerifyMac');
    }
    /**
     * Grants permission to verify pin-related data such as PIN and PIN Offset using algorithms including VISA PVV and IBM3624
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/DataAPIReference/API_VerifyPinData.html
     */
    toVerifyPinData() {
        return this.to('VerifyPinData');
    }
    /**
     * Adds a resource of type key to the statement
     *
     * @param keyId - Identifier for the keyId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceAliases()
     */
    onKey(keyId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:payment-cryptography:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:key/${keyId}`);
    }
    /**
     * Adds a resource of type alias to the statement
     *
     * @param alias - Identifier for the alias.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifResourceAliases()
     */
    onAlias(alias, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:payment-cryptography:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:alias/${alias}`);
    }
    /**
     * Filters access by both the key and value of the tag in the request for the specified operation
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateKey()
     * - .toDeleteAlias()
     * - .toGetAlias()
     * - .toImportKey()
     * - .toTagResource()
     * - .toUpdateAlias()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tags assigned to a key for the specified operation
     *
     * https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/security_iam_service-with-iam.html#security_iam_service-with-iam-tags
     *
     * Applies to resource types:
     * - key
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys in the request for the specified operation
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateKey()
     * - .toDeleteAlias()
     * - .toGetAlias()
     * - .toImportKey()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateAlias()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the CertificateAuthorityPublicKeyIdentifier specified in the request or the ImportKey, and ExportKey operations
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCertificateAuthorityPublicKeyIdentifier(value, operator) {
        return this.if(`CertificateAuthorityPublicKeyIdentifier`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the type of key material being imported [RootCertificatePublicKey, TrustedCertificatePublicKey, Tr34KeyBlock, Tr31KeyBlock] for the ImportKey operation
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifImportKeyMaterial(value, operator) {
        return this.if(`ImportKeyMaterial`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by KeyAlgorithm specified in the request for the CreateKey operation
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifKeyAlgorithm(value, operator) {
        return this.if(`KeyAlgorithm`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by KeyClass specified in the request for the CreateKey operation
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifKeyClass(value, operator) {
        return this.if(`KeyClass`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by KeyClass specified in the request or associated with a key for the CreateKey operation
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifKeyUsage(value, operator) {
        return this.if(`KeyUsage`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by aliases in the request for the specified operation
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestAlias(value, operator) {
        return this.if(`RequestAlias`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by aliases associated with a key for the specified operation
     *
     * Applies to resource types:
     * - key
     * - alias
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceAliases(value, operator) {
        return this.if(`ResourceAliases`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the WrappingKeyIdentifier specified in the request for the ImportKey, and ExportKey operations
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifWrappingKeyIdentifier(value, operator) {
        return this.if(`WrappingKeyIdentifier`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [payment-cryptography](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awspaymentcryptography.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'payment-cryptography';
        this.accessLevelList = {
            Write: [
                'CreateAlias',
                'CreateKey',
                'DecryptData',
                'DeleteAlias',
                'DeleteKey',
                'EncryptData',
                'ExportKey',
                'GenerateCardValidationData',
                'GenerateMac',
                'GenerateMacEmvPinChange',
                'GeneratePinData',
                'ImportKey',
                'ReEncryptData',
                'RestoreKey',
                'StartKeyUsage',
                'StopKeyUsage',
                'TranslatePinData',
                'UpdateAlias',
                'VerifyAuthRequestCryptogram',
                'VerifyCardValidationData',
                'VerifyMac',
                'VerifyPinData'
            ],
            Read: [
                'GetAlias',
                'GetKey',
                'GetParametersForExport',
                'GetParametersForImport',
                'GetPublicKeyCertificate',
                'ListTagsForResource'
            ],
            List: [
                'ListAliases',
                'ListKeys'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.PaymentCryptography = PaymentCryptography;
//# sourceMappingURL=data:application/json;base64,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