"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Polly = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [polly](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonpolly.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Polly extends shared_1.PolicyStatement {
    /**
     * Grants permission to delete the specified pronunciation lexicon stored in an AWS Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_DeleteLexicon.html
     */
    toDeleteLexicon() {
        return this.to('DeleteLexicon');
    }
    /**
     * Grants permission to describe the list of voices that are available for use when requesting speech synthesis
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_DescribeVoices.html
     */
    toDescribeVoices() {
        return this.to('DescribeVoices');
    }
    /**
     * Grants permission to retrieve the content of the specified pronunciation lexicon stored in an AWS Region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_GetLexicon.html
     */
    toGetLexicon() {
        return this.to('GetLexicon');
    }
    /**
     * Grants permission to get information about specific speech synthesis task
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_GetSpeechSynthesisTask.html
     */
    toGetSpeechSynthesisTask() {
        return this.to('GetSpeechSynthesisTask');
    }
    /**
     * Grants permission to list the pronunciation lexicons stored in an AWS Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_ListLexicons.html
     */
    toListLexicons() {
        return this.to('ListLexicons');
    }
    /**
     * Grants permission to list requested speech synthesis tasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_ListSpeechSynthesisTasks.html
     */
    toListSpeechSynthesisTasks() {
        return this.to('ListSpeechSynthesisTasks');
    }
    /**
     * Grants permission to store a pronunciation lexicon in an AWS Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_PutLexicon.html
     */
    toPutLexicon() {
        return this.to('PutLexicon');
    }
    /**
     * Grants permission to synthesize long inputs to the provided S3 location
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_StartSpeechSynthesisTask.html
     */
    toStartSpeechSynthesisTask() {
        return this.to('StartSpeechSynthesisTask');
    }
    /**
     * Grants permission to synthesize speech
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/polly/latest/dg/API_SynthesizeSpeech.html
     */
    toSynthesizeSpeech() {
        return this.to('SynthesizeSpeech');
    }
    /**
     * Adds a resource of type lexicon to the statement
     *
     * https://docs.aws.amazon.com/polly/latest/dg/managing-lexicons.html
     *
     * @param lexiconName - Identifier for the lexiconName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onLexicon(lexiconName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:polly:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:lexicon/${lexiconName}`);
    }
    /**
     * Statement provider for service [polly](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonpolly.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'polly';
        this.accessLevelList = {
            Write: [
                'DeleteLexicon',
                'PutLexicon',
                'StartSpeechSynthesisTask'
            ],
            List: [
                'DescribeVoices',
                'ListLexicons',
                'ListSpeechSynthesisTasks'
            ],
            Read: [
                'GetLexicon',
                'GetSpeechSynthesisTask',
                'SynthesizeSpeech'
            ]
        };
    }
}
exports.Polly = Polly;
//# sourceMappingURL=data:application/json;base64,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