"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Q = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [q](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonq.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Q extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate an AWS resource with an Amazon Q connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toAssociateConnectorResource() {
        return this.to('AssociateConnectorResource');
    }
    /**
     * Grants permission to create a user or group assignment for an Amazon Q Developer Profile
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifIdentitystoreUserId()
     * - .ifIdentitystoreGroupId()
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toCreateAssignment() {
        return this.to('CreateAssignment');
    }
    /**
     * Grants permission to create OAuth user in Amazon Q
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     * - kms:GenerateDataKeyWithoutPlaintext
     * - kms:ReEncryptFrom
     * - kms:ReEncryptTo
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toCreateAuthGrant() {
        return this.to('CreateAuthGrant');
    }
    /**
     * Grants permission to register an OAuth application in Amazon Q
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     * - kms:GenerateDataKeyWithoutPlaintext
     * - kms:ReEncryptFrom
     * - kms:ReEncryptTo
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toCreateOAuthAppConnection() {
        return this.to('CreateOAuthAppConnection');
    }
    /**
     * Grants permission to create and configure a third party plugin in Amazon Q
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toCreatePlugin() {
        return this.to('CreatePlugin');
    }
    /**
     * Grants permission to delete a user or group assignment for an Amazon Q Developer Profile
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifIdentitystoreUserId()
     * - .ifIdentitystoreGroupId()
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toDeleteAssignment() {
        return this.to('DeleteAssignment');
    }
    /**
     * Grants permission to delete a conversation with Amazon Q
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toDeleteConversation() {
        return this.to('DeleteConversation');
    }
    /**
     * Grants permission to delete an OAuth application in Amazon Q
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     * - kms:GenerateDataKeyWithoutPlaintext
     * - kms:ReEncryptFrom
     * - kms:ReEncryptTo
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toDeleteOAuthAppConnection() {
        return this.to('DeleteOAuthAppConnection');
    }
    /**
     * Grants permission to delete a configured plugin in Amazon Q
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toDeletePlugin() {
        return this.to('DeletePlugin');
    }
    /**
     * Grants permission to generate code from CLI commands in Amazon Q
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toGenerateCodeFromCommands() {
        return this.to('GenerateCodeFromCommands');
    }
    /**
     * Grants permission to generate code recommendations in Amazon Q
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toGenerateCodeRecommendations() {
        return this.to('GenerateCodeRecommendations');
    }
    /**
     * Grants permission to view information about a specific Amazon Q connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toGetConnector() {
        return this.to('GetConnector');
    }
    /**
     * Grants permission to get individual messages associated with a specific conversation with Amazon Q
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toGetConversation() {
        return this.to('GetConversation');
    }
    /**
     * Grants permission to Amazon Q to get the identity metadata
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toGetIdentityMetadata() {
        return this.to('GetIdentityMetadata');
    }
    /**
     * Grants permission to view information about a specific configured Amazon Q plugin
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toGetPlugin() {
        return this.to('GetPlugin');
    }
    /**
     * Grants permission to get troubleshooting results with Amazon Q
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toGetTroubleshootingResults() {
        return this.to('GetTroubleshootingResults');
    }
    /**
     * Grants permission to list individual conversations associated with a specific Amazon Q user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toListConversations() {
        return this.to('ListConversations');
    }
    /**
     * Grants permission to read metrics to populate Amazon Q dashboard
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toListDashboardMetrics() {
        return this.to('ListDashboardMetrics');
    }
    /**
     * Grants permission to list available plugins in Amazon Q
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toListPluginProviders() {
        return this.to('ListPluginProviders');
    }
    /**
     * Grants permission to list configured plugins in Amazon Q
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toListPlugins() {
        return this.to('ListPlugins');
    }
    /**
     * Grants permission to list all tags associated with an Amazon Q resource
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to allow Amazon Q to perform actions on your behalf
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toPassRequest() {
        return this.to('PassRequest');
    }
    /**
     * Grants permission to reject a connection request for an Amazon Q connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toRejectConnector() {
        return this.to('RejectConnector');
    }
    /**
     * Grants permission to trigger asynchronous Amazon Q actions
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     * - kms:GenerateDataKeyWithoutPlaintext
     * - kms:ReEncryptFrom
     * - kms:ReEncryptTo
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toSendEvent() {
        return this.to('SendEvent');
    }
    /**
     * Grants permission to send a message to Amazon Q
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toSendMessage() {
        return this.to('SendMessage');
    }
    /**
     * Grants permission to start a conversation with Amazon Q
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toStartConversation() {
        return this.to('StartConversation');
    }
    /**
     * Grants permission to start a troubleshooting analysis with Amazon Q
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toStartTroubleshootingAnalysis() {
        return this.to('StartTroubleshootingAnalysis');
    }
    /**
     * Grants permission to start a troubleshooting resolution explanation with Amazon Q
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toStartTroubleshootingResolutionExplanation() {
        return this.to('StartTroubleshootingResolutionExplanation');
    }
    /**
     * Grants permission to associate tags with an Amazon Q resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags associated with an Amazon Q resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update OAuth user in Amazon Q
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     * - kms:GenerateDataKeyWithoutPlaintext
     * - kms:ReEncryptFrom
     * - kms:ReEncryptTo
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toUpdateAuthGrant() {
        return this.to('UpdateAuthGrant');
    }
    /**
     * Grants permission to update a conversation with Amazon Q
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toUpdateConversation() {
        return this.to('UpdateConversation');
    }
    /**
     * Grants permission to update an OAuth application in Amazon Q
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     * - kms:GenerateDataKeyWithoutPlaintext
     * - kms:ReEncryptFrom
     * - kms:ReEncryptTo
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toUpdateOAuthAppConnection() {
        return this.to('UpdateOAuthAppConnection');
    }
    /**
     * Grants permission to update a third party plugin in Amazon Q
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toUpdatePlugin() {
        return this.to('UpdatePlugin');
    }
    /**
     * Grants permission to update a troubleshooting command result with Amazon Q
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toUpdateTroubleshootingCommandResult() {
        return this.to('UpdateTroubleshootingCommandResult');
    }
    /**
     * Grants permission to use Amazon Q plugins
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toUsePlugin() {
        return this.to('UsePlugin');
    }
    /**
     * Grants permission to verify an OAuth application in Amazon Q
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     * - kms:GenerateDataKeyWithoutPlaintext
     * - kms:ReEncryptFrom
     * - kms:ReEncryptTo
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security_iam_manage-access-with-policies.html
     */
    toVerifyOAuthAppConnection() {
        return this.to('VerifyOAuthAppConnection');
    }
    /**
     * Adds a resource of type profile to the statement
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/as-whisper-admin.html#about-profiles
     *
     * @param identifier - Identifier for the identifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onProfile(identifier, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codewhisperer:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:profile/${identifier}`);
    }
    /**
     * Adds a resource of type plugin to the statement
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/plugins.html
     *
     * @param identifier - Identifier for the identifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPlugin(identifier, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:qdeveloper:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:plugin/${identifier}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security-iam-service-with-iam.html
     *
     * Applies to actions:
     * - .toCreatePlugin()
     * - .toTagResource()
     * - .toUpdatePlugin()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the Amazon Q resource
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security-iam-service-with-iam.html
     *
     * Applies to actions:
     * - .toDeletePlugin()
     * - .toGetPlugin()
     * - .toListTagsForResource()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * Applies to resource types:
     * - plugin
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security-iam-service-with-iam.html
     *
     * Applies to actions:
     * - .toCreatePlugin()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdatePlugin()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by IAM Identity Center Group ID
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security-iam-service-with-iam.html
     *
     * Applies to actions:
     * - .toCreateAssignment()
     * - .toDeleteAssignment()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifIdentitystoreGroupId(value, operator) {
        return this.if(`identitystore:GroupId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by IAM Identity Center User ID
     *
     * https://docs.aws.amazon.com/amazonq/latest/qdeveloper-ug/security-iam-service-with-iam.html
     *
     * Applies to actions:
     * - .toCreateAssignment()
     * - .toDeleteAssignment()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifIdentitystoreUserId(value, operator) {
        return this.if(`identitystore:UserId`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [q](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonq.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'q';
        this.accessLevelList = {
            Write: [
                'AssociateConnectorResource',
                'CreateAssignment',
                'CreateAuthGrant',
                'CreateOAuthAppConnection',
                'CreatePlugin',
                'DeleteAssignment',
                'DeleteConversation',
                'DeleteOAuthAppConnection',
                'DeletePlugin',
                'PassRequest',
                'RejectConnector',
                'SendEvent',
                'SendMessage',
                'StartConversation',
                'StartTroubleshootingAnalysis',
                'StartTroubleshootingResolutionExplanation',
                'UpdateAuthGrant',
                'UpdateConversation',
                'UpdateOAuthAppConnection',
                'UpdatePlugin',
                'UpdateTroubleshootingCommandResult',
                'UsePlugin',
                'VerifyOAuthAppConnection'
            ],
            Read: [
                'GenerateCodeFromCommands',
                'GenerateCodeRecommendations',
                'GetConnector',
                'GetConversation',
                'GetIdentityMetadata',
                'GetPlugin',
                'GetTroubleshootingResults',
                'ListConversations'
            ],
            List: [
                'ListDashboardMetrics',
                'ListPluginProviders',
                'ListPlugins',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Q = Q;
//# sourceMappingURL=data:application/json;base64,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