"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Wisdom = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [wisdom](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonqinconnect.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Wisdom extends shared_1.PolicyStatement {
    /**
     * Grants permission to activate a message template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ActivateMessageTemplate.html
     */
    toActivateMessageTemplate() {
        return this.to('ActivateMessageTemplate');
    }
    /**
     * Grants permission to configure vended log delivery for an assistant
     *
     * Access Level: Permissions management
     */
    toAllowVendedLogDeliveryForResource() {
        return this.to('AllowVendedLogDeliveryForResource');
    }
    /**
     * Grants permission to create an ai agent
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateAIAgent.html
     */
    toCreateAIAgent() {
        return this.to('CreateAIAgent');
    }
    /**
     * Grants permission to create an ai agent version
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateAIAgentVersion.html
     */
    toCreateAIAgentVersion() {
        return this.to('CreateAIAgentVersion');
    }
    /**
     * Grants permission to create an ai guardrail
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateAIGuardrail.html
     */
    toCreateAIGuardrail() {
        return this.to('CreateAIGuardrail');
    }
    /**
     * Grants permission to create an ai guardrail version
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateAIGuardrailVersion.html
     */
    toCreateAIGuardrailVersion() {
        return this.to('CreateAIGuardrailVersion');
    }
    /**
     * Grants permission to create an ai prompt
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateAIPrompt.html
     */
    toCreateAIPrompt() {
        return this.to('CreateAIPrompt');
    }
    /**
     * Grants permission to create an ai prompt version
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateAIPromptVersion.html
     */
    toCreateAIPromptVersion() {
        return this.to('CreateAIPromptVersion');
    }
    /**
     * Grants permission to create an assistant
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateAssistant.html
     */
    toCreateAssistant() {
        return this.to('CreateAssistant');
    }
    /**
     * Grants permission to create an association between an assistant and another resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateAssistantAssociation.html
     */
    toCreateAssistantAssociation() {
        return this.to('CreateAssistantAssociation');
    }
    /**
     * Grants permission to create content
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateContent.html
     */
    toCreateContent() {
        return this.to('CreateContent');
    }
    /**
     * Grants permission to create a content association
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateContentAssociation.html
     */
    toCreateContentAssociation() {
        return this.to('CreateContentAssociation');
    }
    /**
     * Grants permission to create a knowledge base
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateKnowledgeBase.html
     */
    toCreateKnowledgeBase() {
        return this.to('CreateKnowledgeBase');
    }
    /**
     * Grants permission to create a message template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateMessageTemplate.html
     */
    toCreateMessageTemplate() {
        return this.to('CreateMessageTemplate');
    }
    /**
     * Grants permission to create an attachment to a message template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateMessageTemplateAttachment.html
     */
    toCreateMessageTemplateAttachment() {
        return this.to('CreateMessageTemplateAttachment');
    }
    /**
     * Grants permission to create a version of a message template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateMessageTemplateVersion.html
     */
    toCreateMessageTemplateVersion() {
        return this.to('CreateMessageTemplateVersion');
    }
    /**
     * Grants permission to create quick response
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateQuickResponse.html
     */
    toCreateQuickResponse() {
        return this.to('CreateQuickResponse');
    }
    /**
     * Grants permission to create a session
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateSession.html
     */
    toCreateSession() {
        return this.to('CreateSession');
    }
    /**
     * Grants permission to deactivate a message template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeactivateMessageTemplate.html
     */
    toDeactivateMessageTemplate() {
        return this.to('DeactivateMessageTemplate');
    }
    /**
     * Grants permission to delete an ai agent
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeleteAIAgent.html
     */
    toDeleteAIAgent() {
        return this.to('DeleteAIAgent');
    }
    /**
     * Grants permission to delete an ai agent version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeleteAIAgentVersion.html
     */
    toDeleteAIAgentVersion() {
        return this.to('DeleteAIAgentVersion');
    }
    /**
     * Grants permission to delete an ai guardrail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeleteAIGuardrail.html
     */
    toDeleteAIGuardrail() {
        return this.to('DeleteAIGuardrail');
    }
    /**
     * Grants permission to delete an ai guardrail version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeleteAIGuardrailVersion.html
     */
    toDeleteAIGuardrailVersion() {
        return this.to('DeleteAIGuardrailVersion');
    }
    /**
     * Grants permission to delete an ai prompt
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeleteAIPrompt.html
     */
    toDeleteAIPrompt() {
        return this.to('DeleteAIPrompt');
    }
    /**
     * Grants permission to delete an ai prompt version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeleteAIPromptVersion.html
     */
    toDeleteAIPromptVersion() {
        return this.to('DeleteAIPromptVersion');
    }
    /**
     * Grants permission to delete an assistant
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeleteAssistant.html
     */
    toDeleteAssistant() {
        return this.to('DeleteAssistant');
    }
    /**
     * Grants permission to delete an assistant association
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeleteAssistantAssociation.html
     */
    toDeleteAssistantAssociation() {
        return this.to('DeleteAssistantAssociation');
    }
    /**
     * Grants permission to delete content
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeleteContent.html
     */
    toDeleteContent() {
        return this.to('DeleteContent');
    }
    /**
     * Grants permission to delete a content association
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeleteContentAssociation.html
     */
    toDeleteContentAssociation() {
        return this.to('DeleteContentAssociation');
    }
    /**
     * Grants permission to delete a import job of a knowledge base
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeleteImportJob.html
     */
    toDeleteImportJob() {
        return this.to('DeleteImportJob');
    }
    /**
     * Grants permission to delete a knowledge base
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeleteKnowledgeBase.html
     */
    toDeleteKnowledgeBase() {
        return this.to('DeleteKnowledgeBase');
    }
    /**
     * Grants permission to delete a message template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeleteMessageTemplate.html
     */
    toDeleteMessageTemplate() {
        return this.to('DeleteMessageTemplate');
    }
    /**
     * Grants permission to delete an attachment from a message template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeleteMessageTemplateAttachment.html
     */
    toDeleteMessageTemplateAttachment() {
        return this.to('DeleteMessageTemplateAttachment');
    }
    /**
     * Grants permission to delete quick response
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeleteQuickResponse.html
     */
    toDeleteQuickResponse() {
        return this.to('DeleteQuickResponse');
    }
    /**
     * Grants permission to retrieve information about an ai agent
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_GetAIAgent.html
     */
    toGetAIAgent() {
        return this.to('GetAIAgent');
    }
    /**
     * Grants permission to retrieve information about an ai guardrail
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_GetAIGuardrail.html
     */
    toGetAIGuardrail() {
        return this.to('GetAIGuardrail');
    }
    /**
     * Grants permission to retrieve information about an ai prompt
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_GetAIPrompt.html
     */
    toGetAIPrompt() {
        return this.to('GetAIPrompt');
    }
    /**
     * Grants permission to retrieve information about an assistant
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_GetAssistant.html
     */
    toGetAssistant() {
        return this.to('GetAssistant');
    }
    /**
     * Grants permission to retrieve information about an assistant association
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_GetAssistantAssociation.html
     */
    toGetAssistantAssociation() {
        return this.to('GetAssistantAssociation');
    }
    /**
     * Grants permission to retrieve content, including a pre-signed URL to download the content
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_GetContent.html
     */
    toGetContent() {
        return this.to('GetContent');
    }
    /**
     * Grants permission to retrieve information about a content association
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_GetContentAssociation.html
     */
    toGetContentAssociation() {
        return this.to('GetContentAssociation');
    }
    /**
     * Grants permission to retrieve summary information about the content
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_GetContentSummary.html
     */
    toGetContentSummary() {
        return this.to('GetContentSummary');
    }
    /**
     * Grants permission to retrieve information about the import job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_GetImportJob.html
     */
    toGetImportJob() {
        return this.to('GetImportJob');
    }
    /**
     * Grants permission to retrieve information about the knowledge base
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_GetKnowledgeBase.html
     */
    toGetKnowledgeBase() {
        return this.to('GetKnowledgeBase');
    }
    /**
     * Grants permission to retrieve a message template
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifMessageTemplateRoutingProfileArn()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_GetMessageTemplate.html
     */
    toGetMessageTemplate() {
        return this.to('GetMessageTemplate');
    }
    /**
     * Grants permission to retrieve for next message in a session
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_GetNextMessage.html
     */
    toGetNextMessage() {
        return this.to('GetNextMessage');
    }
    /**
     * Grants permission to retrieve content
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_GetQuickResponse.html
     */
    toGetQuickResponse() {
        return this.to('GetQuickResponse');
    }
    /**
     * Grants permission to retrieve recommendations for the specified session
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_GetRecommendations.html
     */
    toGetRecommendations() {
        return this.to('GetRecommendations');
    }
    /**
     * Grants permission to retrieve information for a specified session
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_GetSession.html
     */
    toGetSession() {
        return this.to('GetSession');
    }
    /**
     * Grants permission to list information about ai agent versions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListAIAgentVersions.html
     */
    toListAIAgentVersions() {
        return this.to('ListAIAgentVersions');
    }
    /**
     * Grants permission to list information about ai agents
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListAIAgents.html
     */
    toListAIAgents() {
        return this.to('ListAIAgents');
    }
    /**
     * Grants permission to list information about ai guardrail versions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListAIGuardrailVersions.html
     */
    toListAIGuardrailVersions() {
        return this.to('ListAIGuardrailVersions');
    }
    /**
     * Grants permission to list information about ai guardrails
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListAIGuardrails.html
     */
    toListAIGuardrails() {
        return this.to('ListAIGuardrails');
    }
    /**
     * Grants permission to list information about ai prompt versions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListAIPromptVersions.html
     */
    toListAIPromptVersions() {
        return this.to('ListAIPromptVersions');
    }
    /**
     * Grants permission to list information about ai prompts
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListAIPrompts.html
     */
    toListAIPrompts() {
        return this.to('ListAIPrompts');
    }
    /**
     * Grants permission to list information about assistant associations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListAssistantAssociations.html
     */
    toListAssistantAssociations() {
        return this.to('ListAssistantAssociations');
    }
    /**
     * Grants permission to list information about assistants
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListAssistants.html
     */
    toListAssistants() {
        return this.to('ListAssistants');
    }
    /**
     * Grants permission to list information about content associations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListContentAssociations.html
     */
    toListContentAssociations() {
        return this.to('ListContentAssociations');
    }
    /**
     * Grants permission to list the content with a knowledge base
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListContents.html
     */
    toListContents() {
        return this.to('ListContents');
    }
    /**
     * Grants permission to list information about knowledge bases
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListImportJobs.html
     */
    toListImportJobs() {
        return this.to('ListImportJobs');
    }
    /**
     * Grants permission to list information about knowledge bases
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListKnowledgeBases.html
     */
    toListKnowledgeBases() {
        return this.to('ListKnowledgeBases');
    }
    /**
     * Grants permission to list message template versions for the specified message template
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListMessageTemplateVersions.html
     */
    toListMessageTemplateVersions() {
        return this.to('ListMessageTemplateVersions');
    }
    /**
     * Grants permission to list the message templates for a knowledge base
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListMessageTemplates.html
     */
    toListMessageTemplates() {
        return this.to('ListMessageTemplates');
    }
    /**
     * Grants permission to list messages in a session
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListMessages.html
     */
    toListMessages() {
        return this.to('ListMessages');
    }
    /**
     * Grants permission to list the quick response with a knowledge base
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListQuickResponses.html
     */
    toListQuickResponses() {
        return this.to('ListQuickResponses');
    }
    /**
     * Grants permission to list the tags for the specified resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to remove the specified recommendations from the specified assistant's queue of newly available recommendations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_NotifyRecommendationsReceived.html
     */
    toNotifyRecommendationsReceived() {
        return this.to('NotifyRecommendationsReceived');
    }
    /**
     * Grants permission to submit feedback
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_PutFeedback.html
     */
    toPutFeedback() {
        return this.to('PutFeedback');
    }
    /**
     * Grants permission to perform a manual search against the specified assistant
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_QueryAssistant.html
     */
    toQueryAssistant() {
        return this.to('QueryAssistant');
    }
    /**
     * Grants permission to remove an ai agent from an assistant
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_RemoveAssistantAIAgent.html
     */
    toRemoveAssistantAIAgent() {
        return this.to('RemoveAssistantAIAgent');
    }
    /**
     * Grants permission to remove a URI template from a knowledge base
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_RemoveKnowledgeBaseTemplateUri.html
     */
    toRemoveKnowledgeBaseTemplateUri() {
        return this.to('RemoveKnowledgeBaseTemplateUri');
    }
    /**
     * Grants permission to render a message template
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifMessageTemplateRoutingProfileArn()
     *
     * Dependent actions:
     * - wisdom:GetMessageTemplate
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_RenderMessageTemplate.html
     */
    toRenderMessageTemplate() {
        return this.to('RenderMessageTemplate');
    }
    /**
     * Grants permission to search for content referencing a specified knowledge base. Can be used to get a specific content resource by its name
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_SearchContent.html
     */
    toSearchContent() {
        return this.to('SearchContent');
    }
    /**
     * Grants permission to search for message templates referencing a specified knowledge base
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifSearchFilterRoutingProfileArn()
     * - .ifSearchFilterQualifier()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_SearchMessageTemplates.html
     */
    toSearchMessageTemplates() {
        return this.to('SearchMessageTemplates');
    }
    /**
     * Grants permission to search for quick response referencing a specified knowledge base
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifSearchFilterRoutingProfileArn()
     *
     * Dependent actions:
     * - wisdom:GetQuickResponse
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_SearchQuickResponses.html
     */
    toSearchQuickResponses() {
        return this.to('SearchQuickResponses');
    }
    /**
     * Grants permission to search for sessions referencing a specified assistant. Can be used to et a specific session resource by its name
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_SearchSessions.html
     */
    toSearchSessions() {
        return this.to('SearchSessions');
    }
    /**
     * Grants permission to send a message
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_SendMessage.html
     */
    toSendMessage() {
        return this.to('SendMessage');
    }
    /**
     * Grants permission to get a URL to upload content to a knowledge base
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_StartContentUpload.html
     */
    toStartContentUpload() {
        return this.to('StartContentUpload');
    }
    /**
     * Grants permission to create multiple quick responses
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_StartImportJob.html
     */
    toStartImportJob() {
        return this.to('StartImportJob');
    }
    /**
     * Grants permission to add the specified tags to the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove the specified tags from the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update information about an ai agent
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_UpdateAIAgent.html
     */
    toUpdateAIAgent() {
        return this.to('UpdateAIAgent');
    }
    /**
     * Grants permission to update information about an ai guardrail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_UpdateAIGuardrail.html
     */
    toUpdateAIGuardrail() {
        return this.to('UpdateAIGuardrail');
    }
    /**
     * Grants permission to update information about an ai prompt
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_UpdateAIPrompt.html
     */
    toUpdateAIPrompt() {
        return this.to('UpdateAIPrompt');
    }
    /**
     * Grants permission to update assistant information about an ai agent
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_UpdateAssistantAIAgent.html
     */
    toUpdateAssistantAIAgent() {
        return this.to('UpdateAssistantAIAgent');
    }
    /**
     * Grants permission to update information about the content
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_UpdateContent.html
     */
    toUpdateContent() {
        return this.to('UpdateContent');
    }
    /**
     * Grants permission to update the template URI of a knowledge base
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_UpdateKnowledgeBaseTemplateUri.html
     */
    toUpdateKnowledgeBaseTemplateUri() {
        return this.to('UpdateKnowledgeBaseTemplateUri');
    }
    /**
     * Grants permission to update content of the message template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_UpdateMessageTemplate.html
     */
    toUpdateMessageTemplate() {
        return this.to('UpdateMessageTemplate');
    }
    /**
     * Grants permission to update metadata of the message template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_UpdateMessageTemplateMetadata.html
     */
    toUpdateMessageTemplateMetadata() {
        return this.to('UpdateMessageTemplateMetadata');
    }
    /**
     * Grants permission to update information or content of the quick response
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_UpdateQuickResponse.html
     */
    toUpdateQuickResponse() {
        return this.to('UpdateQuickResponse');
    }
    /**
     * Grants permission to update a session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_UpdateSession.html
     */
    toUpdateSession() {
        return this.to('UpdateSession');
    }
    /**
     * Grants permission to update data stored in a session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_UpdateSessionData.html
     */
    toUpdateSessionData() {
        return this.to('UpdateSessionData');
    }
    /**
     * Adds a resource of type AIAgent to the statement
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_AIAgentData.html
     *
     * @param assistantId - Identifier for the assistantId.
     * @param aIAgentId - Identifier for the aIAgentId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAIAgent(assistantId, aIAgentId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wisdom:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:ai-agent/${assistantId}/${aIAgentId}`);
    }
    /**
     * Adds a resource of type AIPrompt to the statement
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_AIPromptData.html
     *
     * @param assistantId - Identifier for the assistantId.
     * @param aIPromptId - Identifier for the aIPromptId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAIPrompt(assistantId, aIPromptId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wisdom:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:ai-prompt/${assistantId}/${aIPromptId}`);
    }
    /**
     * Adds a resource of type AIGuardrail to the statement
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_AIGuardrailData.html
     *
     * @param assistantId - Identifier for the assistantId.
     * @param aIGuardrailId - Identifier for the aIGuardrailId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAIGuardrail(assistantId, aIGuardrailId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wisdom:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:ai-guardrail/${assistantId}/${aIGuardrailId}`);
    }
    /**
     * Adds a resource of type Assistant to the statement
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_AssistantData.html
     *
     * @param assistantId - Identifier for the assistantId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAssistant(assistantId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wisdom:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:assistant/${assistantId}`);
    }
    /**
     * Adds a resource of type AssistantAssociation to the statement
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_AssistantAssociationData.html
     *
     * @param assistantId - Identifier for the assistantId.
     * @param assistantAssociationId - Identifier for the assistantAssociationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAssistantAssociation(assistantId, assistantAssociationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wisdom:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:association/${assistantId}/${assistantAssociationId}`);
    }
    /**
     * Adds a resource of type Content to the statement
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ContentData.html
     *
     * @param knowledgeBaseId - Identifier for the knowledgeBaseId.
     * @param contentId - Identifier for the contentId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onContent(knowledgeBaseId, contentId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wisdom:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:content/${knowledgeBaseId}/${contentId}`);
    }
    /**
     * Adds a resource of type ContentAssociation to the statement
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ContentAssociationData.html
     *
     * @param knowledgeBaseId - Identifier for the knowledgeBaseId.
     * @param contentId - Identifier for the contentId.
     * @param contentAssociationId - Identifier for the contentAssociationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onContentAssociation(knowledgeBaseId, contentId, contentAssociationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wisdom:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:content-association/${knowledgeBaseId}/${contentId}/${contentAssociationId}`);
    }
    /**
     * Adds a resource of type KnowledgeBase to the statement
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_KnowledgeBaseData.html
     *
     * @param knowledgeBaseId - Identifier for the knowledgeBaseId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onKnowledgeBase(knowledgeBaseId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wisdom:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:knowledge-base/${knowledgeBaseId}`);
    }
    /**
     * Adds a resource of type MessageTemplate to the statement
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_MessageTemplateData.html
     *
     * @param knowledgeBaseId - Identifier for the knowledgeBaseId.
     * @param messageTemplateId - Identifier for the messageTemplateId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifMessageTemplateRoutingProfileArn()
     */
    onMessageTemplate(knowledgeBaseId, messageTemplateId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wisdom:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:message-template/${knowledgeBaseId}/${messageTemplateId}`);
    }
    /**
     * Adds a resource of type Session to the statement
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_SessionData.html
     *
     * @param assistantId - Identifier for the assistantId.
     * @param sessionId - Identifier for the sessionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSession(assistantId, sessionId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wisdom:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:session/${assistantId}/${sessionId}`);
    }
    /**
     * Adds a resource of type QuickResponse to the statement
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_QuickResponseData.html
     *
     * @param knowledgeBaseId - Identifier for the knowledgeBaseId.
     * @param quickResponseId - Identifier for the quickResponseId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onQuickResponse(knowledgeBaseId, quickResponseId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wisdom:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:quick-response/${knowledgeBaseId}/${quickResponseId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateAIAgent()
     * - .toCreateAIAgentVersion()
     * - .toCreateAIGuardrail()
     * - .toCreateAIGuardrailVersion()
     * - .toCreateAIPrompt()
     * - .toCreateAIPromptVersion()
     * - .toCreateAssistant()
     * - .toCreateAssistantAssociation()
     * - .toCreateContent()
     * - .toCreateContentAssociation()
     * - .toCreateKnowledgeBase()
     * - .toCreateMessageTemplate()
     * - .toCreateQuickResponse()
     * - .toCreateSession()
     * - .toStartImportJob()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toTagResource()
     * - .toUntagResource()
     *
     * Applies to resource types:
     * - AIAgent
     * - AIPrompt
     * - AIGuardrail
     * - Assistant
     * - AssistantAssociation
     * - Content
     * - ContentAssociation
     * - KnowledgeBase
     * - MessageTemplate
     * - Session
     * - QuickResponse
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateAIAgent()
     * - .toCreateAIAgentVersion()
     * - .toCreateAIGuardrail()
     * - .toCreateAIGuardrailVersion()
     * - .toCreateAIPrompt()
     * - .toCreateAIPromptVersion()
     * - .toCreateAssistant()
     * - .toCreateAssistantAssociation()
     * - .toCreateContent()
     * - .toCreateContentAssociation()
     * - .toCreateKnowledgeBase()
     * - .toCreateMessageTemplate()
     * - .toCreateQuickResponse()
     * - .toCreateSession()
     * - .toStartImportJob()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the connect routing profile arns associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonconnectwisdom.html#amazonconnectwisdom-policy-keys
     *
     * Applies to actions:
     * - .toGetMessageTemplate()
     * - .toRenderMessageTemplate()
     *
     * Applies to resource types:
     * - MessageTemplate
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifMessageTemplateRoutingProfileArn(value, operator) {
        return this.if(`MessageTemplate/RoutingProfileArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the qualifiers that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonconnectwisdom.html#amazonconnectwisdom-policy-keys
     *
     * Applies to actions:
     * - .toSearchMessageTemplates()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSearchFilterQualifier(value, operator) {
        return this.if(`SearchFilter/Qualifier`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the connect routing profile arn that is passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonconnectwisdom.html#amazonconnectwisdom-policy-keys
     *
     * Applies to actions:
     * - .toSearchMessageTemplates()
     * - .toSearchQuickResponses()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifSearchFilterRoutingProfileArn(value, operator) {
        return this.if(`SearchFilter/RoutingProfileArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Statement provider for service [wisdom](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonqinconnect.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'wisdom';
        this.accessLevelList = {
            Write: [
                'ActivateMessageTemplate',
                'CreateAIAgent',
                'CreateAIAgentVersion',
                'CreateAIGuardrail',
                'CreateAIGuardrailVersion',
                'CreateAIPrompt',
                'CreateAIPromptVersion',
                'CreateAssistant',
                'CreateAssistantAssociation',
                'CreateContent',
                'CreateContentAssociation',
                'CreateKnowledgeBase',
                'CreateMessageTemplate',
                'CreateMessageTemplateAttachment',
                'CreateMessageTemplateVersion',
                'CreateQuickResponse',
                'CreateSession',
                'DeactivateMessageTemplate',
                'DeleteAIAgent',
                'DeleteAIAgentVersion',
                'DeleteAIGuardrail',
                'DeleteAIGuardrailVersion',
                'DeleteAIPrompt',
                'DeleteAIPromptVersion',
                'DeleteAssistant',
                'DeleteAssistantAssociation',
                'DeleteContent',
                'DeleteContentAssociation',
                'DeleteImportJob',
                'DeleteKnowledgeBase',
                'DeleteMessageTemplate',
                'DeleteMessageTemplateAttachment',
                'DeleteQuickResponse',
                'NotifyRecommendationsReceived',
                'PutFeedback',
                'RemoveAssistantAIAgent',
                'RemoveKnowledgeBaseTemplateUri',
                'SendMessage',
                'StartContentUpload',
                'StartImportJob',
                'UpdateAIAgent',
                'UpdateAIGuardrail',
                'UpdateAIPrompt',
                'UpdateAssistantAIAgent',
                'UpdateContent',
                'UpdateKnowledgeBaseTemplateUri',
                'UpdateMessageTemplate',
                'UpdateMessageTemplateMetadata',
                'UpdateQuickResponse',
                'UpdateSession',
                'UpdateSessionData'
            ],
            'Permissions management': [
                'AllowVendedLogDeliveryForResource'
            ],
            Read: [
                'GetAIAgent',
                'GetAIGuardrail',
                'GetAIPrompt',
                'GetAssistant',
                'GetAssistantAssociation',
                'GetContent',
                'GetContentAssociation',
                'GetContentSummary',
                'GetImportJob',
                'GetKnowledgeBase',
                'GetMessageTemplate',
                'GetNextMessage',
                'GetQuickResponse',
                'GetRecommendations',
                'GetSession',
                'ListTagsForResource',
                'QueryAssistant',
                'RenderMessageTemplate',
                'SearchContent',
                'SearchMessageTemplates',
                'SearchQuickResponses',
                'SearchSessions'
            ],
            List: [
                'ListAIAgentVersions',
                'ListAIAgents',
                'ListAIGuardrailVersions',
                'ListAIGuardrails',
                'ListAIPromptVersions',
                'ListAIPrompts',
                'ListAssistantAssociations',
                'ListAssistants',
                'ListContentAssociations',
                'ListContents',
                'ListImportJobs',
                'ListKnowledgeBases',
                'ListMessageTemplateVersions',
                'ListMessageTemplates',
                'ListMessages',
                'ListQuickResponses'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Wisdom = Wisdom;
//# sourceMappingURL=data:application/json;base64,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