"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Rds = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [rds](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonrds.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Rds extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate an Identity and Access Management (IAM) role from an Aurora DB cluster
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_AddRoleToDBCluster.html
     */
    toAddRoleToDBCluster() {
        return this.to('AddRoleToDBCluster');
    }
    /**
     * Grants permission to associate an AWS Identity and Access Management (IAM) role with a DB instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_AddRoleToDBInstance.html
     */
    toAddRoleToDBInstance() {
        return this.to('AddRoleToDBInstance');
    }
    /**
     * Grants permission to add a source identifier to an existing RDS event notification subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_AddSourceIdentifierToSubscription.html
     */
    toAddSourceIdentifierToSubscription() {
        return this.to('AddSourceIdentifierToSubscription');
    }
    /**
     * Grants permission to add metadata tags to an Amazon RDS resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_AddTagsToResource.html
     */
    toAddTagsToResource() {
        return this.to('AddTagsToResource');
    }
    /**
     * Grants permission to apply a pending maintenance action to a resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ApplyPendingMaintenanceAction.html
     */
    toApplyPendingMaintenanceAction() {
        return this.to('ApplyPendingMaintenanceAction');
    }
    /**
     * Grants permission to enable ingress to a DBSecurityGroup using one of two forms of authorization
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_AuthorizeDBSecurityGroupIngress.html
     */
    toAuthorizeDBSecurityGroupIngress() {
        return this.to('AuthorizeDBSecurityGroupIngress');
    }
    /**
     * Grants permission to backtrack a DB cluster to a specific time, without creating a new DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_BacktrackDBCluster.html
     */
    toBacktrackDBCluster() {
        return this.to('BacktrackDBCluster');
    }
    /**
     * Grants permission to cancel an export task in progress
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CancelExportTask.html
     */
    toCancelExportTask() {
        return this.to('CancelExportTask');
    }
    /**
     * Grants permission to copy a custom engine version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonrds.html
     */
    toCopyCustomDBEngineVersion() {
        return this.to('CopyCustomDBEngineVersion');
    }
    /**
     * Grants permission to copy the specified DB cluster parameter group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CopyDBClusterParameterGroup.html
     */
    toCopyDBClusterParameterGroup() {
        return this.to('CopyDBClusterParameterGroup');
    }
    /**
     * Grants permission to create a snapshot of a DB cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CopyDBClusterSnapshot.html
     */
    toCopyDBClusterSnapshot() {
        return this.to('CopyDBClusterSnapshot');
    }
    /**
     * Grants permission to copy the specified DB parameter group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CopyDBParameterGroup.html
     */
    toCopyDBParameterGroup() {
        return this.to('CopyDBParameterGroup');
    }
    /**
     * Grants permission to copy the specified DB snapshot
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifCopyOptionGroup()
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     * - rds:CopyCustomDBEngineVersion
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CopyDBSnapshot.html
     */
    toCopyDBSnapshot() {
        return this.to('CopyDBSnapshot');
    }
    /**
     * Grants permission to copy the specified option group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CopyOptionGroup.html
     */
    toCopyOptionGroup() {
        return this.to('CopyOptionGroup');
    }
    /**
     * Grants permission to create a blue-green deployment for a given source cluster or instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifClusterTag()
     * - .ifClusterPgTag()
     * - .ifDbTag()
     * - .ifPgTag()
     * - .ifReqTag()
     * - .ifDatabaseEngine()
     * - .ifDatabaseName()
     * - .ifStorageEncrypted()
     * - .ifDatabaseClass()
     * - .ifStorageSize()
     * - .ifMultiAz()
     * - .ifPiops()
     * - .ifVpc()
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     * - rds:CreateDBCluster
     * - rds:CreateDBClusterEndpoint
     * - rds:CreateDBInstance
     * - rds:CreateDBInstanceReadReplica
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateBlueGreenDeployment.html
     */
    toCreateBlueGreenDeployment() {
        return this.to('CreateBlueGreenDeployment');
    }
    /**
     * Grants permission to create a custom engine version
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - mediaimport:CreateDatabaseBinarySnapshot
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateCustomDBEngineVersion.html
     */
    toCreateCustomDBEngineVersion() {
        return this.to('CreateCustomDBEngineVersion');
    }
    /**
     * Grants permission to create a new DB cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     * - .ifDatabaseEngine()
     * - .ifDatabaseName()
     * - .ifStorageEncrypted()
     * - .ifDatabaseClass()
     * - .ifStorageSize()
     * - .ifPiops()
     * - .ifManageMasterUserPassword()
     *
     * Dependent actions:
     * - iam:PassRole
     * - kms:CreateGrant
     * - kms:Decrypt
     * - kms:DescribeKey
     * - kms:GenerateDataKey
     * - rds:AddTagsToResource
     * - rds:CreateDBInstance
     * - secretsmanager:CreateSecret
     * - secretsmanager:TagResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBCluster.html
     */
    toCreateDBCluster() {
        return this.to('CreateDBCluster');
    }
    /**
     * Grants permission to create a new custom endpoint and associates it with an Amazon Aurora DB cluster or Amazon DocumentDB cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifEndpointType()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBClusterEndpoint.html
     */
    toCreateDBClusterEndpoint() {
        return this.to('CreateDBClusterEndpoint');
    }
    /**
     * Grants permission to create a new DB cluster parameter group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBClusterParameterGroup.html
     */
    toCreateDBClusterParameterGroup() {
        return this.to('CreateDBClusterParameterGroup');
    }
    /**
     * Grants permission to create a snapshot of a DB cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBClusterSnapshot.html
     */
    toCreateDBClusterSnapshot() {
        return this.to('CreateDBClusterSnapshot');
    }
    /**
     * Grants permission to create a new DB instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifBackupTarget()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     * - .ifManageMasterUserPassword()
     *
     * Dependent actions:
     * - iam:PassRole
     * - kms:CreateGrant
     * - kms:Decrypt
     * - kms:DescribeKey
     * - kms:GenerateDataKey
     * - rds:AddTagsToResource
     * - rds:CreateTenantDatabase
     * - secretsmanager:CreateSecret
     * - secretsmanager:TagResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstance.html
     */
    toCreateDBInstance() {
        return this.to('CreateDBInstance');
    }
    /**
     * Grants permission to create a DB instance that acts as a Read Replica of a source DB instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - iam:PassRole
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstanceReadReplica.html
     */
    toCreateDBInstanceReadReplica() {
        return this.to('CreateDBInstanceReadReplica');
    }
    /**
     * Grants permission to create a new DB parameter group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBParameterGroup.html
     */
    toCreateDBParameterGroup() {
        return this.to('CreateDBParameterGroup');
    }
    /**
     * Grants permission to create a database proxy
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBProxy.html
     */
    toCreateDBProxy() {
        return this.to('CreateDBProxy');
    }
    /**
     * Grants permission to create a database proxy endpoint
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBProxyEndpoint.html
     */
    toCreateDBProxyEndpoint() {
        return this.to('CreateDBProxyEndpoint');
    }
    /**
     * Grants permission to create a new DB security group. DB security groups control access to a DB instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBSecurityGroup.html
     */
    toCreateDBSecurityGroup() {
        return this.to('CreateDBSecurityGroup');
    }
    /**
     * Grants permission to create a new Aurora Limitless Database DB shard group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBShardGroup.html
     */
    toCreateDBShardGroup() {
        return this.to('CreateDBShardGroup');
    }
    /**
     * Grants permission to create a DBSnapshot
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifBackupTarget()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBSnapshot.html
     */
    toCreateDBSnapshot() {
        return this.to('CreateDBSnapshot');
    }
    /**
     * Grants permission to create a new DB subnet group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBSubnetGroup.html
     */
    toCreateDBSubnetGroup() {
        return this.to('CreateDBSubnetGroup');
    }
    /**
     * Grants permission to create an RDS event notification subscription
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateEventSubscription.html
     */
    toCreateEventSubscription() {
        return this.to('CreateEventSubscription');
    }
    /**
     * Grants permission to create an Aurora global database or DocumentDB global database spread across multiple regions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateGlobalCluster.html
     */
    toCreateGlobalCluster() {
        return this.to('CreateGlobalCluster');
    }
    /**
     * Grants permission to create an Aurora zero-ETL integration with Redshift
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - kms:CreateGrant
     * - kms:DescribeKey
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateIntegration.html
     */
    toCreateIntegration() {
        return this.to('CreateIntegration');
    }
    /**
     * Grants permission to create a new option group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateOptionGroup.html
     */
    toCreateOptionGroup() {
        return this.to('CreateOptionGroup');
    }
    /**
     * Grants permission to create a new tenant database
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifTenantDatabaseName()
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateTenantDatabase.html
     */
    toCreateTenantDatabase() {
        return this.to('CreateTenantDatabase');
    }
    /**
     * Grants permission to access a resource in the remote Region when executing cross-Region operations, such as cross-Region snapshot copy or cross-Region read replica creation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     */
    toCrossRegionCommunication() {
        return this.to('CrossRegionCommunication');
    }
    /**
     * Grants permission to delete blue green deployments
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - rds:DeleteDBCluster
     * - rds:DeleteDBClusterEndpoint
     * - rds:DeleteDBInstance
     * - rds:PromoteReadReplica
     * - rds:PromoteReadReplicaDBCluster
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteBlueGreenDeployment.html
     */
    toDeleteBlueGreenDeployment() {
        return this.to('DeleteBlueGreenDeployment');
    }
    /**
     * Grants permission to delete an existing custom engine version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteCustomDBEngineVersion.html
     */
    toDeleteCustomDBEngineVersion() {
        return this.to('DeleteCustomDBEngineVersion');
    }
    /**
     * Grants permission to delete a previously provisioned DB cluster
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     * - rds:CreateDBClusterSnapshot
     * - rds:DeleteDBInstance
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBCluster.html
     */
    toDeleteDBCluster() {
        return this.to('DeleteDBCluster');
    }
    /**
     * Grants permission to delete cluster automated backups based on the source cluster's DbClusterResourceId value or the restorable cluster's resource ID
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBClusterAutomatedBackup.html
     */
    toDeleteDBClusterAutomatedBackup() {
        return this.to('DeleteDBClusterAutomatedBackup');
    }
    /**
     * Grants permission to delete a custom endpoint and removes it from an Amazon Aurora DB cluster or Amazon DocumentDB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBClusterEndpoint.html
     */
    toDeleteDBClusterEndpoint() {
        return this.to('DeleteDBClusterEndpoint');
    }
    /**
     * Grants permission to delete a specified DB cluster parameter group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBClusterParameterGroup.html
     */
    toDeleteDBClusterParameterGroup() {
        return this.to('DeleteDBClusterParameterGroup');
    }
    /**
     * Grants permission to delete a DB cluster snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBClusterSnapshot.html
     */
    toDeleteDBClusterSnapshot() {
        return this.to('DeleteDBClusterSnapshot');
    }
    /**
     * Grants permission to delete a previously provisioned DB instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     * - rds:CreateDBSnapshot
     * - rds:DeleteTenantDatabase
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBInstance.html
     */
    toDeleteDBInstance() {
        return this.to('DeleteDBInstance');
    }
    /**
     * Grants permission to delete automated backups based on the source instance's DbiResourceId value or the restorable instance's resource ID
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBInstanceAutomatedBackup.html
     */
    toDeleteDBInstanceAutomatedBackup() {
        return this.to('DeleteDBInstanceAutomatedBackup');
    }
    /**
     * Grants permission to delete a specified DBParameterGroup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBParameterGroup.html
     */
    toDeleteDBParameterGroup() {
        return this.to('DeleteDBParameterGroup');
    }
    /**
     * Grants permission to delete a database proxy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBProxy.html
     */
    toDeleteDBProxy() {
        return this.to('DeleteDBProxy');
    }
    /**
     * Grants permission to delete a database proxy endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBProxyEndpoint.html
     */
    toDeleteDBProxyEndpoint() {
        return this.to('DeleteDBProxyEndpoint');
    }
    /**
     * Grants permission to delete a DB security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBSecurityGroup.html
     */
    toDeleteDBSecurityGroup() {
        return this.to('DeleteDBSecurityGroup');
    }
    /**
     * Grants permission to delete an Aurora Limitless Database DB shard group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBShardGroup.html
     */
    toDeleteDBShardGroup() {
        return this.to('DeleteDBShardGroup');
    }
    /**
     * Grants permission to delete a DBSnapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBSnapshot.html
     */
    toDeleteDBSnapshot() {
        return this.to('DeleteDBSnapshot');
    }
    /**
     * Grants permission to delete a DB subnet group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBSubnetGroup.html
     */
    toDeleteDBSubnetGroup() {
        return this.to('DeleteDBSubnetGroup');
    }
    /**
     * Grants permission to delete an RDS event notification subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteEventSubscription.html
     */
    toDeleteEventSubscription() {
        return this.to('DeleteEventSubscription');
    }
    /**
     * Grants permission to delete a global database cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteGlobalCluster.html
     */
    toDeleteGlobalCluster() {
        return this.to('DeleteGlobalCluster');
    }
    /**
     * Grants permission to delete an Aurora zero-ETL integration with Redshift
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteIntegration.html
     */
    toDeleteIntegration() {
        return this.to('DeleteIntegration');
    }
    /**
     * Grants permission to delete an existing option group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteOptionGroup.html
     */
    toDeleteOptionGroup() {
        return this.to('DeleteOptionGroup');
    }
    /**
     * Grants permission to delete a tenant database
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     * - rds:CreateDBSnapshot
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteTenantDatabase.html
     */
    toDeleteTenantDatabase() {
        return this.to('DeleteTenantDatabase');
    }
    /**
     * Grants permission to remove targets from a database proxy target group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeregisterDBProxyTargets.html
     */
    toDeregisterDBProxyTargets() {
        return this.to('DeregisterDBProxyTargets');
    }
    /**
     * Grants permission to list all of the attributes for a customer account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeAccountAttributes.html
     */
    toDescribeAccountAttributes() {
        return this.to('DescribeAccountAttributes');
    }
    /**
     * Grants permission to describe blue green deployments
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeBlueGreenDeployments.html
     */
    toDescribeBlueGreenDeployments() {
        return this.to('DescribeBlueGreenDeployments');
    }
    /**
     * Grants permission to list the set of CA certificates provided by Amazon RDS for this AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeCertificates.html
     */
    toDescribeCertificates() {
        return this.to('DescribeCertificates');
    }
    /**
     * Grants permission to return a list of cluster automated backups for both current and deleted clusters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusterAutomatedBackups.html
     */
    toDescribeDBClusterAutomatedBackups() {
        return this.to('DescribeDBClusterAutomatedBackups');
    }
    /**
     * Grants permission to return information about backtracks for a DB cluster
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusterBacktracks.html
     */
    toDescribeDBClusterBacktracks() {
        return this.to('DescribeDBClusterBacktracks');
    }
    /**
     * Grants permission to return information about endpoints for an Amazon Aurora DB cluster
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusterEndpoints.html
     */
    toDescribeDBClusterEndpoints() {
        return this.to('DescribeDBClusterEndpoints');
    }
    /**
     * Grants permission to return a list of DBClusterParameterGroup descriptions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusterParameterGroups.html
     */
    toDescribeDBClusterParameterGroups() {
        return this.to('DescribeDBClusterParameterGroups');
    }
    /**
     * Grants permission to return the detailed parameter list for a particular DB cluster parameter group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusterParameters.html
     */
    toDescribeDBClusterParameters() {
        return this.to('DescribeDBClusterParameters');
    }
    /**
     * Grants permission to return a list of DB cluster snapshot attribute names and values for a manual DB cluster snapshot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusterSnapshotAttributes.html
     */
    toDescribeDBClusterSnapshotAttributes() {
        return this.to('DescribeDBClusterSnapshotAttributes');
    }
    /**
     * Grants permission to return information about DB cluster snapshots
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusterSnapshots.html
     */
    toDescribeDBClusterSnapshots() {
        return this.to('DescribeDBClusterSnapshots');
    }
    /**
     * Grants permission to return information about provisioned Aurora DB clusters or DocumentDB clusters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusters.html
     */
    toDescribeDBClusters() {
        return this.to('DescribeDBClusters');
    }
    /**
     * Grants permission to return a list of the available DB engines
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBEngineVersions.html
     */
    toDescribeDBEngineVersions() {
        return this.to('DescribeDBEngineVersions');
    }
    /**
     * Grants permission to return a list of automated backups for both current and deleted instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBInstanceAutomatedBackups.html
     */
    toDescribeDBInstanceAutomatedBackups() {
        return this.to('DescribeDBInstanceAutomatedBackups');
    }
    /**
     * Grants permission to return information about provisioned RDS instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBInstances.html
     */
    toDescribeDBInstances() {
        return this.to('DescribeDBInstances');
    }
    /**
     * Grants permission to return a list of DB log files for the DB instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBLogFiles.html
     */
    toDescribeDBLogFiles() {
        return this.to('DescribeDBLogFiles');
    }
    /**
     * Grants permission to return a list of DBParameterGroup descriptions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBParameterGroups.html
     */
    toDescribeDBParameterGroups() {
        return this.to('DescribeDBParameterGroups');
    }
    /**
     * Grants permission to return the detailed parameter list for a particular DB parameter group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBParameters.html
     */
    toDescribeDBParameters() {
        return this.to('DescribeDBParameters');
    }
    /**
     * Grants permission to view proxies
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBProxies.html
     */
    toDescribeDBProxies() {
        return this.to('DescribeDBProxies');
    }
    /**
     * Grants permission to view proxy endpoints
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBProxyEndpoints.html
     */
    toDescribeDBProxyEndpoints() {
        return this.to('DescribeDBProxyEndpoints');
    }
    /**
     * Grants permission to view database proxy target group details
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBProxyTargetGroups.html
     */
    toDescribeDBProxyTargetGroups() {
        return this.to('DescribeDBProxyTargetGroups');
    }
    /**
     * Grants permission to view database proxy target details
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBProxyTargets.html
     */
    toDescribeDBProxyTargets() {
        return this.to('DescribeDBProxyTargets');
    }
    /**
     * Grants permission to list recommendation details
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBRecommendations.html
     */
    toDescribeDBRecommendations() {
        return this.to('DescribeDBRecommendations');
    }
    /**
     * Grants permission to return a list of DBSecurityGroup descriptions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBSecurityGroups.html
     */
    toDescribeDBSecurityGroups() {
        return this.to('DescribeDBSecurityGroups');
    }
    /**
     * Grants permission to return information about all Aurora Limitless Database DB shard groups for this account. You can filter by shard group(s)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBShardGroups.html
     */
    toDescribeDBShardGroups() {
        return this.to('DescribeDBShardGroups');
    }
    /**
     * Grants permission to return a list of DB snapshot attribute names and values for a manual DB snapshot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBSnapshotAttributes.html
     */
    toDescribeDBSnapshotAttributes() {
        return this.to('DescribeDBSnapshotAttributes');
    }
    /**
     * Grants permission to return information about tenant databases in DB snapshots. You can filter by Region or snapshot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBSnapshotTenantDatabases.html
     */
    toDescribeDBSnapshotTenantDatabases() {
        return this.to('DescribeDBSnapshotTenantDatabases');
    }
    /**
     * Grants permission to return information about DB snapshots
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBSnapshots.html
     */
    toDescribeDBSnapshots() {
        return this.to('DescribeDBSnapshots');
    }
    /**
     * Grants permission to return a list of DBSubnetGroup descriptions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBSubnetGroups.html
     */
    toDescribeDBSubnetGroups() {
        return this.to('DescribeDBSubnetGroups');
    }
    /**
     * Grants permission to return the default engine and system parameter information for the cluster database engine
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeEngineDefaultClusterParameters.html
     */
    toDescribeEngineDefaultClusterParameters() {
        return this.to('DescribeEngineDefaultClusterParameters');
    }
    /**
     * Grants permission to return the default engine and system parameter information for the specified database engine
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeEngineDefaultParameters.html
     */
    toDescribeEngineDefaultParameters() {
        return this.to('DescribeEngineDefaultParameters');
    }
    /**
     * Grants permission to display a list of categories for all event source types, or, if specified, for a specified source type
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeEventCategories.html
     */
    toDescribeEventCategories() {
        return this.to('DescribeEventCategories');
    }
    /**
     * Grants permission to list all the subscription descriptions for a customer account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeEventSubscriptions.html
     */
    toDescribeEventSubscriptions() {
        return this.to('DescribeEventSubscriptions');
    }
    /**
     * Grants permission to return events related to DB instances, DB security groups, DB snapshots, and DB parameter groups for the past 14 days
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeEvents.html
     */
    toDescribeEvents() {
        return this.to('DescribeEvents');
    }
    /**
     * Grants permission to return information about the export tasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeExportTasks.html
     */
    toDescribeExportTasks() {
        return this.to('DescribeExportTasks');
    }
    /**
     * Grants permission to return information about Aurora global database clusters or DocumentDB global database clusters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeGlobalClusters.html
     */
    toDescribeGlobalClusters() {
        return this.to('DescribeGlobalClusters');
    }
    /**
     * Grants permission to describe an Aurora zero-ETL integration with Redshift
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeIntegrations.html
     */
    toDescribeIntegrations() {
        return this.to('DescribeIntegrations');
    }
    /**
     * Grants permission to describe all available options
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeOptionGroupOptions.html
     */
    toDescribeOptionGroupOptions() {
        return this.to('DescribeOptionGroupOptions');
    }
    /**
     * Grants permission to describe the available option groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeOptionGroups.html
     */
    toDescribeOptionGroups() {
        return this.to('DescribeOptionGroups');
    }
    /**
     * Grants permission to return a list of orderable DB instance options for the specified engine
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeOrderableDBInstanceOptions.html
     */
    toDescribeOrderableDBInstanceOptions() {
        return this.to('DescribeOrderableDBInstanceOptions');
    }
    /**
     * Grants permission to return a list of resources (for example, DB instances) that have at least one pending maintenance action
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribePendingMaintenanceActions.html
     */
    toDescribePendingMaintenanceActions() {
        return this.to('DescribePendingMaintenanceActions');
    }
    /**
     * Grants permission to return information about recommendation groups
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/USER_Recommendations.html
     */
    toDescribeRecommendationGroups() {
        return this.to('DescribeRecommendationGroups');
    }
    /**
     * Grants permission to return information about recommendations
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/USER_Recommendations.html
     */
    toDescribeRecommendations() {
        return this.to('DescribeRecommendations');
    }
    /**
     * Grants permission to return information about reserved DB instances for this account, or about a specified reserved DB instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeReservedDBInstances.html
     */
    toDescribeReservedDBInstances() {
        return this.to('DescribeReservedDBInstances');
    }
    /**
     * Grants permission to list available reserved DB instance offerings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeReservedDBInstancesOfferings.html
     */
    toDescribeReservedDBInstancesOfferings() {
        return this.to('DescribeReservedDBInstancesOfferings');
    }
    /**
     * Grants permission to return a list of the source AWS Regions where the current AWS Region can create a Read Replica or copy a DB snapshot from
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeSourceRegions.html
     */
    toDescribeSourceRegions() {
        return this.to('DescribeSourceRegions');
    }
    /**
     * Grants permission to return information about provisioned tenant databases. You can filter by Region or snapshot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeTenantDatabases.html
     */
    toDescribeTenantDatabases() {
        return this.to('DescribeTenantDatabases');
    }
    /**
     * Grants permission to list available modifications you can make to your DB instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeValidDBInstanceModifications.html
     */
    toDescribeValidDBInstanceModifications() {
        return this.to('DescribeValidDBInstanceModifications');
    }
    /**
     * Grants permission to disable http endpoint for a DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DisableHttpEndpoint.html
     */
    toDisableHttpEndpoint() {
        return this.to('DisableHttpEndpoint');
    }
    /**
     * Grants permission to download specified log file
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/USER_LogAccess.html
     */
    toDownloadCompleteDBLogFile() {
        return this.to('DownloadCompleteDBLogFile');
    }
    /**
     * Grants permission to download all or a portion of the specified log file, up to 1 MB in size
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DownloadDBLogFilePortion.html
     */
    toDownloadDBLogFilePortion() {
        return this.to('DownloadDBLogFilePortion');
    }
    /**
     * Grants permission to enable http endpoint for a DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_EnableHttpEndpoint.html
     */
    toEnableHttpEndpoint() {
        return this.to('EnableHttpEndpoint');
    }
    /**
     * Grants permission to force a failover for a DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_FailoverDBCluster.html
     */
    toFailoverDBCluster() {
        return this.to('FailoverDBCluster');
    }
    /**
     * Grants permission to failover a global cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_FailoverGlobalCluster.html
     */
    toFailoverGlobalCluster() {
        return this.to('FailoverGlobalCluster');
    }
    /**
     * Grants permission to list all tags on an Amazon RDS resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to modify a database activity stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyActivityStream.html
     */
    toModifyActivityStream() {
        return this.to('ModifyActivityStream');
    }
    /**
     * Grants permission to modify the system-default Secure Sockets Layer/Transport Layer Security (SSL/TLS) certificate for Amazon RDS for new DB instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyCertificates.html
     */
    toModifyCertificates() {
        return this.to('ModifyCertificates');
    }
    /**
     * Grants permission to modify current cluster capacity for an Amazon Aurora Serverless DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyCurrentDBClusterCapacity.html
     */
    toModifyCurrentDBClusterCapacity() {
        return this.to('ModifyCurrentDBClusterCapacity');
    }
    /**
     * Grants permission to modify an existing custom engine version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyCustomDBEngineVersion.html
     */
    toModifyCustomDBEngineVersion() {
        return this.to('ModifyCustomDBEngineVersion');
    }
    /**
     * Grants permission to modify a setting for an Amazon Aurora DB cluster or Amazon DocumentDB cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDatabaseClass()
     * - .ifStorageSize()
     * - .ifPiops()
     * - .ifManageMasterUserPassword()
     *
     * Dependent actions:
     * - iam:PassRole
     * - kms:CreateGrant
     * - kms:Decrypt
     * - kms:DescribeKey
     * - kms:GenerateDataKey
     * - rds:ModifyDBInstance
     * - secretsmanager:CreateSecret
     * - secretsmanager:RotateSecret
     * - secretsmanager:TagResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBCluster.html
     */
    toModifyDBCluster() {
        return this.to('ModifyDBCluster');
    }
    /**
     * Grants permission to modify the properties of an endpoint in an Amazon Aurora DB cluster or Amazon DocumentDB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBClusterEndpoint.html
     */
    toModifyDBClusterEndpoint() {
        return this.to('ModifyDBClusterEndpoint');
    }
    /**
     * Grants permission to modify the parameters of a DB cluster parameter group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBClusterParameterGroup.html
     */
    toModifyDBClusterParameterGroup() {
        return this.to('ModifyDBClusterParameterGroup');
    }
    /**
     * Grants permission to add an attribute and values to, or removes an attribute and values from, a manual DB cluster snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBClusterSnapshotAttribute.html
     */
    toModifyDBClusterSnapshotAttribute() {
        return this.to('ModifyDBClusterSnapshotAttribute');
    }
    /**
     * Grants permission to modify settings for a DB instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifManageMasterUserPassword()
     *
     * Dependent actions:
     * - iam:PassRole
     * - kms:CreateGrant
     * - kms:Decrypt
     * - kms:DescribeKey
     * - kms:GenerateDataKey
     * - rds:AddTagsToResource
     * - rds:CreateTenantDatabase
     * - secretsmanager:CreateSecret
     * - secretsmanager:RotateSecret
     * - secretsmanager:TagResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBInstance.html
     */
    toModifyDBInstance() {
        return this.to('ModifyDBInstance');
    }
    /**
     * Grants permission to modify the parameters of a DB parameter group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBParameterGroup.html
     */
    toModifyDBParameterGroup() {
        return this.to('ModifyDBParameterGroup');
    }
    /**
     * Grants permission to modify database proxy
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBProxy.html
     */
    toModifyDBProxy() {
        return this.to('ModifyDBProxy');
    }
    /**
     * Grants permission to modify database proxy endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBProxyEndpoint.html
     */
    toModifyDBProxyEndpoint() {
        return this.to('ModifyDBProxyEndpoint');
    }
    /**
     * Grants permission to modify target group for a database proxy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBProxyTargetGroup.html
     */
    toModifyDBProxyTargetGroup() {
        return this.to('ModifyDBProxyTargetGroup');
    }
    /**
     * Grants permission to modify recommendation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBRecommendation.html
     */
    toModifyDBRecommendation() {
        return this.to('ModifyDBRecommendation');
    }
    /**
     * Grants permission to modify properties of an Aurora Limitless Database DB shard group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBShardGroup.html
     */
    toModifyDBShardGroup() {
        return this.to('ModifyDBShardGroup');
    }
    /**
     * Grants permission to update a manual DB snapshot, which can be encrypted or not encrypted, with a new engine version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBSnapshot.html
     */
    toModifyDBSnapshot() {
        return this.to('ModifyDBSnapshot');
    }
    /**
     * Grants permission to add an attribute and values to, or removes an attribute and values from, a manual DB snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBSnapshotAttribute.html
     */
    toModifyDBSnapshotAttribute() {
        return this.to('ModifyDBSnapshotAttribute');
    }
    /**
     * Grants permission to modify an existing DB subnet group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBSubnetGroup.html
     */
    toModifyDBSubnetGroup() {
        return this.to('ModifyDBSubnetGroup');
    }
    /**
     * Grants permission to modify an existing RDS event notification subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyEventSubscription.html
     */
    toModifyEventSubscription() {
        return this.to('ModifyEventSubscription');
    }
    /**
     * Grants permission to modify a setting for an Amazon Aurora global cluster or Amazon DocumentDB global cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyGlobalCluster.html
     */
    toModifyGlobalCluster() {
        return this.to('ModifyGlobalCluster');
    }
    /**
     * Grants permission to modify an Aurora zero-ETL integration with Redshift
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyIntegration.html
     */
    toModifyIntegration() {
        return this.to('ModifyIntegration');
    }
    /**
     * Grants permission to modify an existing option group
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyOptionGroup.html
     */
    toModifyOptionGroup() {
        return this.to('ModifyOptionGroup');
    }
    /**
     * Grants permission to modify recommendation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/USER_Recommendations.html
     */
    toModifyRecommendation() {
        return this.to('ModifyRecommendation');
    }
    /**
     * Grants permission to modify a tenant database
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTenantDatabaseName()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyTenantDatabase.html
     */
    toModifyTenantDatabase() {
        return this.to('ModifyTenantDatabase');
    }
    /**
     * Grants permission to promote a Read Replica DB instance to a standalone DB instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_PromoteReadReplica.html
     */
    toPromoteReadReplica() {
        return this.to('PromoteReadReplica');
    }
    /**
     * Grants permission to promote a Read Replica DB cluster to a standalone DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_PromoteReadReplicaDBCluster.html
     */
    toPromoteReadReplicaDBCluster() {
        return this.to('PromoteReadReplicaDBCluster');
    }
    /**
     * Grants permission to purchase a reserved DB instance offering
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_PurchaseReservedDBInstancesOffering.html
     */
    toPurchaseReservedDBInstancesOffering() {
        return this.to('PurchaseReservedDBInstancesOffering');
    }
    /**
     * Grants permission to reboot a previously provisioned DB cluster
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - rds:RebootDBInstance
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RebootDBCluster.html
     */
    toRebootDBCluster() {
        return this.to('RebootDBCluster');
    }
    /**
     * Grants permission to restart the database engine service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RebootDBInstance.html
     */
    toRebootDBInstance() {
        return this.to('RebootDBInstance');
    }
    /**
     * Grants permission to reboot an Aurora Limitless Database DB shard group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RebootDBShardGroup.html
     */
    toRebootDBShardGroup() {
        return this.to('RebootDBShardGroup');
    }
    /**
     * Grants permission to add targets to a database proxy target group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RegisterDBProxyTargets.html
     */
    toRegisterDBProxyTargets() {
        return this.to('RegisterDBProxyTargets');
    }
    /**
     * Grants permission to detach an Aurora secondary cluster from an Aurora global database cluster or DocumentDB global cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RemoveFromGlobalCluster.html
     */
    toRemoveFromGlobalCluster() {
        return this.to('RemoveFromGlobalCluster');
    }
    /**
     * Grants permission to disassociate an AWS Identity and Access Management (IAM) role from an Amazon Aurora DB cluster
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RemoveRoleFromDBCluster.html
     */
    toRemoveRoleFromDBCluster() {
        return this.to('RemoveRoleFromDBCluster');
    }
    /**
     * Grants permission to disassociate an AWS Identity and Access Management (IAM) role from a DB instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RemoveRoleFromDBInstance.html
     */
    toRemoveRoleFromDBInstance() {
        return this.to('RemoveRoleFromDBInstance');
    }
    /**
     * Grants permission to remove a source identifier from an existing RDS event notification subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RemoveSourceIdentifierFromSubscription.html
     */
    toRemoveSourceIdentifierFromSubscription() {
        return this.to('RemoveSourceIdentifierFromSubscription');
    }
    /**
     * Grants permission to remove metadata tags from an Amazon RDS resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RemoveTagsFromResource.html
     */
    toRemoveTagsFromResource() {
        return this.to('RemoveTagsFromResource');
    }
    /**
     * Grants permission to modify the parameters of a DB cluster parameter group to the default value
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ResetDBClusterParameterGroup.html
     */
    toResetDBClusterParameterGroup() {
        return this.to('ResetDBClusterParameterGroup');
    }
    /**
     * Grants permission to modify the parameters of a DB parameter group to the engine/system default value
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ResetDBParameterGroup.html
     */
    toResetDBParameterGroup() {
        return this.to('ResetDBParameterGroup');
    }
    /**
     * Grants permission to create an Amazon Aurora DB cluster from data stored in an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     * - .ifDatabaseEngine()
     * - .ifDatabaseName()
     * - .ifStorageEncrypted()
     * - .ifManageMasterUserPassword()
     *
     * Dependent actions:
     * - iam:PassRole
     * - kms:CreateGrant
     * - kms:Decrypt
     * - kms:DescribeKey
     * - kms:GenerateDataKey
     * - rds:AddTagsToResource
     * - secretsmanager:CreateSecret
     * - secretsmanager:TagResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RestoreDBClusterFromS3.html
     */
    toRestoreDBClusterFromS3() {
        return this.to('RestoreDBClusterFromS3');
    }
    /**
     * Grants permission to create a new DB cluster from a DB cluster snapshot
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     * - .ifDatabaseClass()
     * - .ifStorageSize()
     * - .ifPiops()
     *
     * Dependent actions:
     * - iam:PassRole
     * - rds:AddTagsToResource
     * - rds:CreateDBInstance
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RestoreDBClusterFromSnapshot.html
     */
    toRestoreDBClusterFromSnapshot() {
        return this.to('RestoreDBClusterFromSnapshot');
    }
    /**
     * Grants permission to restore a DB cluster to an arbitrary point in time
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     * - .ifDatabaseClass()
     * - .ifStorageSize()
     * - .ifPiops()
     *
     * Dependent actions:
     * - iam:PassRole
     * - rds:AddTagsToResource
     * - rds:CreateDBInstance
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RestoreDBClusterToPointInTime.html
     */
    toRestoreDBClusterToPointInTime() {
        return this.to('RestoreDBClusterToPointInTime');
    }
    /**
     * Grants permission to create a new DB instance from a DB snapshot
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifBackupTarget()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - iam:PassRole
     * - rds:AddTagsToResource
     * - rds:CreateTenantDatabase
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RestoreDBInstanceFromDBSnapshot.html
     */
    toRestoreDBInstanceFromDBSnapshot() {
        return this.to('RestoreDBInstanceFromDBSnapshot');
    }
    /**
     * Grants permission to create a new DB instance from an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     * - .ifManageMasterUserPassword()
     *
     * Dependent actions:
     * - iam:PassRole
     * - kms:CreateGrant
     * - kms:Decrypt
     * - kms:DescribeKey
     * - kms:GenerateDataKey
     * - rds:AddTagsToResource
     * - secretsmanager:CreateSecret
     * - secretsmanager:TagResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RestoreDBInstanceFromS3.html
     */
    toRestoreDBInstanceFromS3() {
        return this.to('RestoreDBInstanceFromS3');
    }
    /**
     * Grants permission to restore a DB instance to an arbitrary point in time
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifBackupTarget()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - iam:PassRole
     * - rds:AddTagsToResource
     * - rds:CreateTenantDatabase
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RestoreDBInstanceToPointInTime.html
     */
    toRestoreDBInstanceToPointInTime() {
        return this.to('RestoreDBInstanceToPointInTime');
    }
    /**
     * Grants permission to revoke ingress from a DBSecurityGroup for previously authorized IP ranges or EC2 or VPC Security Groups
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RevokeDBSecurityGroupIngress.html
     */
    toRevokeDBSecurityGroupIngress() {
        return this.to('RevokeDBSecurityGroupIngress');
    }
    /**
     * Grants permission to start Activity Stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StartActivityStream.html
     */
    toStartActivityStream() {
        return this.to('StartActivityStream');
    }
    /**
     * Grants permission to start the DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StartDBCluster.html
     */
    toStartDBCluster() {
        return this.to('StartDBCluster');
    }
    /**
     * Grants permission to start the DB instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StartDBInstance.html
     */
    toStartDBInstance() {
        return this.to('StartDBInstance');
    }
    /**
     * Grants permission to start replication of automated backups to a different AWS Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StartDBInstanceAutomatedBackupsReplication.html
     */
    toStartDBInstanceAutomatedBackupsReplication() {
        return this.to('StartDBInstanceAutomatedBackupsReplication');
    }
    /**
     * Grants permission to start a new Export task for a DB snapshot
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StartExportTask.html
     */
    toStartExportTask() {
        return this.to('StartExportTask');
    }
    /**
     * Grants permission to stop Activity Stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StopActivityStream.html
     */
    toStopActivityStream() {
        return this.to('StopActivityStream');
    }
    /**
     * Grants permission to stop the DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StopDBCluster.html
     */
    toStopDBCluster() {
        return this.to('StopDBCluster');
    }
    /**
     * Grants permission to stop the DB instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     * - rds:CreateDBSnapshot
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StopDBInstance.html
     */
    toStopDBInstance() {
        return this.to('StopDBInstance');
    }
    /**
     * Grants permission to stop automated backup replication for a DB instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StopDBInstanceAutomatedBackupsReplication.html
     */
    toStopDBInstanceAutomatedBackupsReplication() {
        return this.to('StopDBInstanceAutomatedBackupsReplication');
    }
    /**
     * Grants permission to switch a blue-green deployment from source instance or cluster to target
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - rds:ModifyDBCluster
     * - rds:ModifyDBInstance
     * - rds:PromoteReadReplica
     * - rds:PromoteReadReplicaDBCluster
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_SwitchoverBlueGreenDeployment.html
     */
    toSwitchoverBlueGreenDeployment() {
        return this.to('SwitchoverBlueGreenDeployment');
    }
    /**
     * Grants permission to switchover a global cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_SwitchoverGlobalCluster.html
     */
    toSwitchoverGlobalCluster() {
        return this.to('SwitchoverGlobalCluster');
    }
    /**
     * Grants permission to switch over a read replica, making it the new primary database
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_SwitchoverReadReplica.html
     */
    toSwitchoverReadReplica() {
        return this.to('SwitchoverReadReplica');
    }
    /**
     * Adds a resource of type cluster to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/CHAP_Aurora.html
     *
     * @param dbClusterInstanceName - Identifier for the dbClusterInstanceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifClusterTag()
     */
    onCluster(dbClusterInstanceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rds:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:cluster:${dbClusterInstanceName}`);
    }
    /**
     * Adds a resource of type shardgrp to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Overview.DBShardGroup.html
     *
     * @param dbShardGroupResourceId - Identifier for the dbShardGroupResourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onShardgrp(dbShardGroupResourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rds:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:shard-group:${dbShardGroupResourceId}`);
    }
    /**
     * Adds a resource of type cluster-auto-backup to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Aurora.Managing.Backups.html
     *
     * @param dbClusterAutomatedBackupId - Identifier for the dbClusterAutomatedBackupId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onClusterAutoBackup(dbClusterAutomatedBackupId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rds:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:cluster-auto-backup:${dbClusterAutomatedBackupId}`);
    }
    /**
     * Adds a resource of type auto-backup to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html
     *
     * @param dbInstanceAutomatedBackupId - Identifier for the dbInstanceAutomatedBackupId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAutoBackup(dbInstanceAutomatedBackupId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rds:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:auto-backup:${dbInstanceAutomatedBackupId}`);
    }
    /**
     * Adds a resource of type cluster-endpoint to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Aurora.Overview.Endpoints.html
     *
     * @param dbClusterEndpoint - Identifier for the dbClusterEndpoint.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onClusterEndpoint(dbClusterEndpoint, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rds:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:cluster-endpoint:${dbClusterEndpoint}`);
    }
    /**
     * Adds a resource of type cluster-pg to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_WorkingWithParamGroups.html
     *
     * @param clusterParameterGroupName - Identifier for the clusterParameterGroupName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifClusterPgTag()
     */
    onClusterPg(clusterParameterGroupName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rds:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:cluster-pg:${clusterParameterGroupName}`);
    }
    /**
     * Adds a resource of type cluster-snapshot to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Aurora.Managing.Backups.html
     *
     * @param clusterSnapshotName - Identifier for the clusterSnapshotName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifClusterSnapshotTag()
     */
    onClusterSnapshot(clusterSnapshotName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rds:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:cluster-snapshot:${clusterSnapshotName}`);
    }
    /**
     * Adds a resource of type db to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Overview.DBInstance.html
     *
     * @param dbInstanceName - Identifier for the dbInstanceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifDatabaseClass()
     * - .ifDatabaseEngine()
     * - .ifDatabaseName()
     * - .ifMultiAz()
     * - .ifPiops()
     * - .ifStorageEncrypted()
     * - .ifStorageSize()
     * - .ifVpc()
     * - .ifDbTag()
     */
    onDb(dbInstanceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rds:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:db:${dbInstanceName}`);
    }
    /**
     * Adds a resource of type es to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Events.html
     *
     * @param subscriptionName - Identifier for the subscriptionName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifEsTag()
     */
    onEs(subscriptionName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rds:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:es:${subscriptionName}`);
    }
    /**
     * Adds a resource of type global-cluster to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database.html
     *
     * @param globalCluster - Identifier for the globalCluster.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onGlobalCluster(globalCluster, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rds::${account ?? this.defaultAccount}:global-cluster:${globalCluster}`);
    }
    /**
     * Adds a resource of type og to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithOptionGroups.html
     *
     * @param optionGroupName - Identifier for the optionGroupName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifOgTag()
     */
    onOg(optionGroupName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rds:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:og:${optionGroupName}`);
    }
    /**
     * Adds a resource of type pg to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithParamGroups.html
     *
     * @param parameterGroupName - Identifier for the parameterGroupName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifPgTag()
     */
    onPg(parameterGroupName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rds:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:pg:${parameterGroupName}`);
    }
    /**
     * Adds a resource of type proxy to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-proxy.html
     *
     * @param dbProxyId - Identifier for the dbProxyId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProxy(dbProxyId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rds:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:db-proxy:${dbProxyId}`);
    }
    /**
     * Adds a resource of type proxy-endpoint to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-proxy.html
     *
     * @param dbProxyEndpointId - Identifier for the dbProxyEndpointId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProxyEndpoint(dbProxyEndpointId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rds:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:db-proxy-endpoint:${dbProxyEndpointId}`);
    }
    /**
     * Adds a resource of type ri to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithReservedDBInstances.html
     *
     * @param reservedDbInstanceName - Identifier for the reservedDbInstanceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifRiTag()
     */
    onRi(reservedDbInstanceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rds:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:ri:${reservedDbInstanceName}`);
    }
    /**
     * Adds a resource of type secgrp to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Overview.RDSSecurityGroups.html
     *
     * @param securityGroupName - Identifier for the securityGroupName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifSecgrpTag()
     */
    onSecgrp(securityGroupName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rds:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:secgrp:${securityGroupName}`);
    }
    /**
     * Adds a resource of type snapshot to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html
     *
     * @param snapshotName - Identifier for the snapshotName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifSnapshotTag()
     */
    onSnapshot(snapshotName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rds:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:snapshot:${snapshotName}`);
    }
    /**
     * Adds a resource of type subgrp to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.Scenarios.html#USER_VPC.Scenario1
     *
     * @param subnetGroupName - Identifier for the subnetGroupName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifSubgrpTag()
     */
    onSubgrp(subnetGroupName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rds:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:subgrp:${subnetGroupName}`);
    }
    /**
     * Adds a resource of type target-group to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-proxy.html
     *
     * @param targetGroupId - Identifier for the targetGroupId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTargetGroup(targetGroupId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rds:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:target-group:${targetGroupId}`);
    }
    /**
     * Adds a resource of type cev to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/custom-cev.html
     *
     * @param engine - Identifier for the engine.
     * @param engineVersion - Identifier for the engineVersion.
     * @param customDbEngineVersionId - Identifier for the customDbEngineVersionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCev(engine, engineVersion, customDbEngineVersionId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rds:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:cev:${engine}/${engineVersion}/${customDbEngineVersionId}`);
    }
    /**
     * Adds a resource of type deployment to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/blue-green-deployments.html
     *
     * @param blueGreenDeploymentIdentifier - Identifier for the blueGreenDeploymentIdentifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDeployment(blueGreenDeploymentIdentifier, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rds:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:deployment:${blueGreenDeploymentIdentifier}`);
    }
    /**
     * Adds a resource of type integration to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/zero-etl.html
     *
     * @param integrationIdentifier - Identifier for the integrationIdentifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIntegration(integrationIdentifier, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rds:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:integration:${integrationIdentifier}`);
    }
    /**
     * Adds a resource of type snapshot-tenant-database to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Oracle.Concepts.single-tenant.snapshots.html#br-cdb.db-snapshots
     *
     * @param snapshotName - Identifier for the snapshotName.
     * @param tenantResourceId - Identifier for the tenantResourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSnapshotTenantDatabase(snapshotName, tenantResourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rds:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:snapshot-tenant-database:${snapshotName}:${tenantResourceId}`);
    }
    /**
     * Adds a resource of type tenant-database to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Oracle.Concepts.CDBs.html#multi-tenant-configuration
     *
     * @param tenantResourceId - Identifier for the tenantResourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTenantDatabase(tenantResourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rds:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:tenant-database:${tenantResourceId}`);
    }
    /**
     * Filters access by the set of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toAddTagsToResource()
     * - .toCopyDBClusterParameterGroup()
     * - .toCopyDBClusterSnapshot()
     * - .toCopyDBParameterGroup()
     * - .toCopyDBSnapshot()
     * - .toCopyOptionGroup()
     * - .toCreateBlueGreenDeployment()
     * - .toCreateCustomDBEngineVersion()
     * - .toCreateDBCluster()
     * - .toCreateDBClusterEndpoint()
     * - .toCreateDBClusterParameterGroup()
     * - .toCreateDBClusterSnapshot()
     * - .toCreateDBInstance()
     * - .toCreateDBInstanceReadReplica()
     * - .toCreateDBParameterGroup()
     * - .toCreateDBProxy()
     * - .toCreateDBProxyEndpoint()
     * - .toCreateDBSecurityGroup()
     * - .toCreateDBShardGroup()
     * - .toCreateDBSnapshot()
     * - .toCreateDBSubnetGroup()
     * - .toCreateEventSubscription()
     * - .toCreateIntegration()
     * - .toCreateOptionGroup()
     * - .toCreateTenantDatabase()
     * - .toPurchaseReservedDBInstancesOffering()
     * - .toRemoveTagsFromResource()
     * - .toRestoreDBClusterFromS3()
     * - .toRestoreDBClusterFromSnapshot()
     * - .toRestoreDBClusterToPointInTime()
     * - .toRestoreDBInstanceFromDBSnapshot()
     * - .toRestoreDBInstanceFromS3()
     * - .toRestoreDBInstanceToPointInTime()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the set of tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toCreateBlueGreenDeployment()
     * - .toDeleteBlueGreenDeployment()
     * - .toDescribeIntegrations()
     * - .toSwitchoverBlueGreenDeployment()
     *
     * Applies to resource types:
     * - cluster
     * - shardgrp
     * - cluster-endpoint
     * - cluster-pg
     * - cluster-snapshot
     * - db
     * - es
     * - og
     * - pg
     * - proxy
     * - proxy-endpoint
     * - ri
     * - secgrp
     * - snapshot
     * - subgrp
     * - target-group
     * - cev
     * - deployment
     * - integration
     * - snapshot-tenant-database
     * - tenant-database
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the set of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toAddTagsToResource()
     * - .toCopyDBClusterParameterGroup()
     * - .toCopyDBClusterSnapshot()
     * - .toCopyDBParameterGroup()
     * - .toCopyDBSnapshot()
     * - .toCopyOptionGroup()
     * - .toCreateBlueGreenDeployment()
     * - .toCreateCustomDBEngineVersion()
     * - .toCreateDBCluster()
     * - .toCreateDBClusterEndpoint()
     * - .toCreateDBClusterParameterGroup()
     * - .toCreateDBClusterSnapshot()
     * - .toCreateDBInstance()
     * - .toCreateDBInstanceReadReplica()
     * - .toCreateDBParameterGroup()
     * - .toCreateDBProxy()
     * - .toCreateDBProxyEndpoint()
     * - .toCreateDBSecurityGroup()
     * - .toCreateDBShardGroup()
     * - .toCreateDBSnapshot()
     * - .toCreateDBSubnetGroup()
     * - .toCreateEventSubscription()
     * - .toCreateIntegration()
     * - .toCreateOptionGroup()
     * - .toCreateTenantDatabase()
     * - .toPurchaseReservedDBInstancesOffering()
     * - .toRemoveTagsFromResource()
     * - .toRestoreDBClusterFromS3()
     * - .toRestoreDBClusterFromSnapshot()
     * - .toRestoreDBClusterToPointInTime()
     * - .toRestoreDBInstanceFromDBSnapshot()
     * - .toRestoreDBInstanceFromS3()
     * - .toRestoreDBInstanceToPointInTime()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the type of backup target. One of: region, outposts
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateDBInstance()
     * - .toCreateDBSnapshot()
     * - .toRestoreDBInstanceFromDBSnapshot()
     * - .toRestoreDBInstanceToPointInTime()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifBackupTarget(value, operator) {
        return this.if(`BackupTarget`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the value that specifies whether the CopyDBSnapshot action requires copying the DB option group
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCopyDBSnapshot()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifCopyOptionGroup(value) {
        return this.if(`CopyOptionGroup`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the type of DB instance class
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateBlueGreenDeployment()
     * - .toCreateDBCluster()
     * - .toModifyDBCluster()
     * - .toRestoreDBClusterFromSnapshot()
     * - .toRestoreDBClusterToPointInTime()
     *
     * Applies to resource types:
     * - db
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDatabaseClass(value, operator) {
        return this.if(`DatabaseClass`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the database engine. For possible values refer to the engine parameter in CreateDBInstance API
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateBlueGreenDeployment()
     * - .toCreateDBCluster()
     * - .toRestoreDBClusterFromS3()
     *
     * Applies to resource types:
     * - db
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDatabaseEngine(value, operator) {
        return this.if(`DatabaseEngine`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the user-defined name of the database on the DB instance
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateBlueGreenDeployment()
     * - .toCreateDBCluster()
     * - .toRestoreDBClusterFromS3()
     *
     * Applies to resource types:
     * - db
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDatabaseName(value, operator) {
        return this.if(`DatabaseName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the type of the endpoint. One of: READER, WRITER, CUSTOM
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateDBClusterEndpoint()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEndpointType(value, operator) {
        return this.if(`EndpointType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the value that specifies whether RDS manages master user password in AWS Secrets Manager for the DB instance or cluster
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateDBCluster()
     * - .toCreateDBInstance()
     * - .toModifyDBCluster()
     * - .toModifyDBInstance()
     * - .toRestoreDBClusterFromS3()
     * - .toRestoreDBInstanceFromS3()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifManageMasterUserPassword(value) {
        return this.if(`ManageMasterUserPassword`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the value that specifies whether the DB instance runs in multiple Availability Zones. To indicate that the DB instance is using Multi-AZ, specify true
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateBlueGreenDeployment()
     *
     * Applies to resource types:
     * - db
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifMultiAz(value) {
        return this.if(`MultiAz`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the value that contains the number of Provisioned IOPS (PIOPS) that the instance supports. To indicate a DB instance that does not have PIOPS enabled, specify 0
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateBlueGreenDeployment()
     * - .toCreateDBCluster()
     * - .toModifyDBCluster()
     * - .toRestoreDBClusterFromSnapshot()
     * - .toRestoreDBClusterToPointInTime()
     *
     * Applies to resource types:
     * - db
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifPiops(value, operator) {
        return this.if(`Piops`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by the value that specifies whether the DB instance storage should be encrypted. To enforce storage encryption, specify true
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateBlueGreenDeployment()
     * - .toCreateDBCluster()
     * - .toRestoreDBClusterFromS3()
     *
     * Applies to resource types:
     * - db
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifStorageEncrypted(value) {
        return this.if(`StorageEncrypted`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the storage volume size (in GB)
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateBlueGreenDeployment()
     * - .toCreateDBCluster()
     * - .toModifyDBCluster()
     * - .toRestoreDBClusterFromSnapshot()
     * - .toRestoreDBClusterToPointInTime()
     *
     * Applies to resource types:
     * - db
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifStorageSize(value, operator) {
        return this.if(`StorageSize`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by the tenant database name in CreateTenantDatabase and by the new tenant database name in ModifyTenantDatabase
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateTenantDatabase()
     * - .toModifyTenantDatabase()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTenantDatabaseName(value, operator) {
        return this.if(`TenantDatabaseName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the value that specifies whether the DB instance runs in an Amazon Virtual Private Cloud (Amazon VPC). To indicate that the DB instance runs in an Amazon VPC, specify true
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateBlueGreenDeployment()
     *
     * Applies to resource types:
     * - db
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifVpc(value) {
        return this.if(`Vpc`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the tag attached to a DB cluster parameter group
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateBlueGreenDeployment()
     *
     * Applies to resource types:
     * - cluster-pg
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifClusterPgTag(tagKey, value, operator) {
        return this.if(`cluster-pg-tag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag attached to a DB cluster snapshot
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - cluster-snapshot
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifClusterSnapshotTag(tagKey, value, operator) {
        return this.if(`cluster-snapshot-tag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag attached to a DB cluster
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateBlueGreenDeployment()
     *
     * Applies to resource types:
     * - cluster
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifClusterTag(tagKey, value, operator) {
        return this.if(`cluster-tag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag attached to a DB instance
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateBlueGreenDeployment()
     *
     * Applies to resource types:
     * - db
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDbTag(tagKey, value, operator) {
        return this.if(`db-tag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag attached to an event subscription
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - es
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEsTag(tagKey, value, operator) {
        return this.if(`es-tag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag attached to a DB option group
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - og
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOgTag(tagKey, value, operator) {
        return this.if(`og-tag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag attached to a DB parameter group
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateBlueGreenDeployment()
     *
     * Applies to resource types:
     * - pg
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPgTag(tagKey, value, operator) {
        return this.if(`pg-tag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the set of tag keys and values that can be used to tag a resource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toAddTagsToResource()
     * - .toCreateBlueGreenDeployment()
     * - .toCreateDBCluster()
     * - .toCreateDBClusterParameterGroup()
     * - .toCreateDBClusterSnapshot()
     * - .toCreateDBInstance()
     * - .toCreateDBInstanceReadReplica()
     * - .toCreateDBParameterGroup()
     * - .toCreateDBSecurityGroup()
     * - .toCreateDBSnapshot()
     * - .toCreateDBSubnetGroup()
     * - .toCreateEventSubscription()
     * - .toCreateIntegration()
     * - .toCreateOptionGroup()
     * - .toRemoveTagsFromResource()
     * - .toRestoreDBClusterFromS3()
     * - .toRestoreDBClusterFromSnapshot()
     * - .toRestoreDBClusterToPointInTime()
     * - .toRestoreDBInstanceFromDBSnapshot()
     * - .toRestoreDBInstanceFromS3()
     * - .toRestoreDBInstanceToPointInTime()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReqTag(tagKey, value, operator) {
        return this.if(`req-tag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag attached to a reserved DB instance
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - ri
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRiTag(tagKey, value, operator) {
        return this.if(`ri-tag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag attached to a DB security group
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - secgrp
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecgrpTag(tagKey, value, operator) {
        return this.if(`secgrp-tag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag attached to a DB snapshot
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - snapshot
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSnapshotTag(tagKey, value, operator) {
        return this.if(`snapshot-tag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag attached to a DB subnet group
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - subgrp
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSubgrpTag(tagKey, value, operator) {
        return this.if(`subgrp-tag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [rds](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonrds.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'rds';
        this.accessLevelList = {
            Write: [
                'AddRoleToDBCluster',
                'AddRoleToDBInstance',
                'AddSourceIdentifierToSubscription',
                'ApplyPendingMaintenanceAction',
                'BacktrackDBCluster',
                'CancelExportTask',
                'CopyCustomDBEngineVersion',
                'CopyDBClusterParameterGroup',
                'CopyDBClusterSnapshot',
                'CopyDBParameterGroup',
                'CopyDBSnapshot',
                'CopyOptionGroup',
                'CreateBlueGreenDeployment',
                'CreateCustomDBEngineVersion',
                'CreateDBCluster',
                'CreateDBClusterEndpoint',
                'CreateDBClusterParameterGroup',
                'CreateDBClusterSnapshot',
                'CreateDBInstance',
                'CreateDBInstanceReadReplica',
                'CreateDBParameterGroup',
                'CreateDBProxy',
                'CreateDBProxyEndpoint',
                'CreateDBSecurityGroup',
                'CreateDBShardGroup',
                'CreateDBSnapshot',
                'CreateDBSubnetGroup',
                'CreateEventSubscription',
                'CreateGlobalCluster',
                'CreateIntegration',
                'CreateOptionGroup',
                'CreateTenantDatabase',
                'CrossRegionCommunication',
                'DeleteBlueGreenDeployment',
                'DeleteCustomDBEngineVersion',
                'DeleteDBCluster',
                'DeleteDBClusterAutomatedBackup',
                'DeleteDBClusterEndpoint',
                'DeleteDBClusterParameterGroup',
                'DeleteDBClusterSnapshot',
                'DeleteDBInstance',
                'DeleteDBInstanceAutomatedBackup',
                'DeleteDBParameterGroup',
                'DeleteDBProxy',
                'DeleteDBProxyEndpoint',
                'DeleteDBSecurityGroup',
                'DeleteDBShardGroup',
                'DeleteDBSnapshot',
                'DeleteDBSubnetGroup',
                'DeleteEventSubscription',
                'DeleteGlobalCluster',
                'DeleteIntegration',
                'DeleteOptionGroup',
                'DeleteTenantDatabase',
                'DeregisterDBProxyTargets',
                'DisableHttpEndpoint',
                'EnableHttpEndpoint',
                'FailoverDBCluster',
                'FailoverGlobalCluster',
                'ModifyActivityStream',
                'ModifyCertificates',
                'ModifyCurrentDBClusterCapacity',
                'ModifyCustomDBEngineVersion',
                'ModifyDBCluster',
                'ModifyDBClusterEndpoint',
                'ModifyDBClusterParameterGroup',
                'ModifyDBClusterSnapshotAttribute',
                'ModifyDBInstance',
                'ModifyDBParameterGroup',
                'ModifyDBProxy',
                'ModifyDBProxyEndpoint',
                'ModifyDBProxyTargetGroup',
                'ModifyDBRecommendation',
                'ModifyDBShardGroup',
                'ModifyDBSnapshot',
                'ModifyDBSnapshotAttribute',
                'ModifyDBSubnetGroup',
                'ModifyEventSubscription',
                'ModifyGlobalCluster',
                'ModifyIntegration',
                'ModifyOptionGroup',
                'ModifyRecommendation',
                'ModifyTenantDatabase',
                'PromoteReadReplica',
                'PromoteReadReplicaDBCluster',
                'PurchaseReservedDBInstancesOffering',
                'RebootDBCluster',
                'RebootDBInstance',
                'RebootDBShardGroup',
                'RegisterDBProxyTargets',
                'RemoveFromGlobalCluster',
                'RemoveRoleFromDBCluster',
                'RemoveRoleFromDBInstance',
                'RemoveSourceIdentifierFromSubscription',
                'ResetDBClusterParameterGroup',
                'ResetDBParameterGroup',
                'RestoreDBClusterFromS3',
                'RestoreDBClusterFromSnapshot',
                'RestoreDBClusterToPointInTime',
                'RestoreDBInstanceFromDBSnapshot',
                'RestoreDBInstanceFromS3',
                'RestoreDBInstanceToPointInTime',
                'RevokeDBSecurityGroupIngress',
                'StartActivityStream',
                'StartDBCluster',
                'StartDBInstance',
                'StartDBInstanceAutomatedBackupsReplication',
                'StartExportTask',
                'StopActivityStream',
                'StopDBCluster',
                'StopDBInstance',
                'StopDBInstanceAutomatedBackupsReplication',
                'SwitchoverBlueGreenDeployment',
                'SwitchoverGlobalCluster',
                'SwitchoverReadReplica'
            ],
            Tagging: [
                'AddTagsToResource',
                'RemoveTagsFromResource'
            ],
            'Permissions management': [
                'AuthorizeDBSecurityGroupIngress'
            ],
            List: [
                'DescribeAccountAttributes',
                'DescribeBlueGreenDeployments',
                'DescribeCertificates',
                'DescribeDBClusterAutomatedBackups',
                'DescribeDBClusterBacktracks',
                'DescribeDBClusterEndpoints',
                'DescribeDBClusterParameterGroups',
                'DescribeDBClusterParameters',
                'DescribeDBClusterSnapshotAttributes',
                'DescribeDBClusterSnapshots',
                'DescribeDBClusters',
                'DescribeDBEngineVersions',
                'DescribeDBInstanceAutomatedBackups',
                'DescribeDBInstances',
                'DescribeDBLogFiles',
                'DescribeDBParameterGroups',
                'DescribeDBParameters',
                'DescribeDBProxies',
                'DescribeDBProxyEndpoints',
                'DescribeDBProxyTargetGroups',
                'DescribeDBProxyTargets',
                'DescribeDBRecommendations',
                'DescribeDBSecurityGroups',
                'DescribeDBShardGroups',
                'DescribeDBSnapshotAttributes',
                'DescribeDBSnapshotTenantDatabases',
                'DescribeDBSnapshots',
                'DescribeDBSubnetGroups',
                'DescribeEngineDefaultClusterParameters',
                'DescribeEngineDefaultParameters',
                'DescribeEventCategories',
                'DescribeEventSubscriptions',
                'DescribeEvents',
                'DescribeExportTasks',
                'DescribeGlobalClusters',
                'DescribeIntegrations',
                'DescribeOptionGroupOptions',
                'DescribeOptionGroups',
                'DescribeOrderableDBInstanceOptions',
                'DescribePendingMaintenanceActions',
                'DescribeReservedDBInstances',
                'DescribeReservedDBInstancesOfferings',
                'DescribeSourceRegions',
                'DescribeTenantDatabases',
                'DescribeValidDBInstanceModifications'
            ],
            Read: [
                'DescribeRecommendationGroups',
                'DescribeRecommendations',
                'DownloadCompleteDBLogFile',
                'DownloadDBLogFilePortion',
                'ListTagsForResource'
            ]
        };
    }
}
exports.Rds = Rds;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmRzLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsicmRzLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUNBLHlDQUF5RDtBQUd6RDs7OztHQUlHO0FBQ0gsTUFBYSxHQUFJLFNBQVEsd0JBQWU7SUFHdEM7Ozs7Ozs7OztPQVNHO0lBQ0ksb0JBQW9CO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1DQUFtQztRQUN4QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUNBQW1DLENBQUMsQ0FBQztJQUN0RCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLCtCQUErQjtRQUNwQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsK0JBQStCLENBQUMsQ0FBQztJQUNsRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUNBQWlDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQ0FBaUMsQ0FBQyxDQUFDO0lBQ3BELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9CQUFvQixDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQWtCLENBQUMsQ0FBQztJQUNyQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMkJBQTJCO1FBQ2hDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO0lBQzlDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksNkJBQTZCO1FBQ2xDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO0lBQ2hELENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksdUJBQXVCO1FBQzVCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO0lBQzFDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksc0JBQXNCO1FBQzNCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7O09BZUc7SUFDSSxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxpQkFBaUI7UUFDdEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGlCQUFpQixDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09BK0JHO0lBQ0ksMkJBQTJCO1FBQ2hDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO0lBQzlDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7O09BZUc7SUFDSSw2QkFBNkI7UUFDbEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixDQUFDLENBQUM7SUFDaEQsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQTZCRztJQUNJLGlCQUFpQjtRQUN0QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7O09BY0c7SUFDSSx5QkFBeUI7UUFDOUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7SUFDNUMsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7OztPQWNHO0lBQ0ksK0JBQStCO1FBQ3BDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO0lBQ2xELENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7T0FjRztJQUNJLHlCQUF5QjtRQUM5QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLENBQUMsQ0FBQztJQUM1QyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQXdCRztJQUNJLGtCQUFrQjtRQUN2QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQWtCLENBQUMsQ0FBQztJQUNyQyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7OztPQWVHO0lBQ0ksNkJBQTZCO1FBQ2xDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO0lBQ2hELENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7T0FjRztJQUNJLHdCQUF3QjtRQUM3QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLGVBQWU7UUFDcEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksdUJBQXVCO1FBQzVCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO0lBQzFDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7T0FjRztJQUNJLHVCQUF1QjtRQUM1QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsdUJBQXVCLENBQUMsQ0FBQztJQUMxQyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLG9CQUFvQjtRQUN6QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7OztPQWVHO0lBQ0ksa0JBQWtCO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7T0FjRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7O09BY0c7SUFDSSx5QkFBeUI7UUFDOUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7SUFDNUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7T0FnQkc7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7OztPQWNHO0lBQ0ksbUJBQW1CO1FBQ3hCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7T0FjRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMEJBQTBCO1FBQy9CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7OztPQWdCRztJQUNJLDJCQUEyQjtRQUNoQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMkJBQTJCLENBQUMsQ0FBQztJQUM5QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksNkJBQTZCO1FBQ2xDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO0lBQ2hELENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLGlCQUFpQjtRQUN0QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0NBQWdDO1FBQ3JDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO0lBQ25ELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5QkFBeUI7UUFDOUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7SUFDNUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLCtCQUErQjtRQUNwQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsK0JBQStCLENBQUMsQ0FBQztJQUNsRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUJBQXlCO1FBQzlCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO0lBQzVDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLGtCQUFrQjtRQUN2QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQWtCLENBQUMsQ0FBQztJQUNyQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUNBQWlDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQ0FBaUMsQ0FBQyxDQUFDO0lBQ3BELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx3QkFBd0I7UUFDN0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7SUFDM0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGVBQWU7UUFDcEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsdUJBQXVCLENBQUMsQ0FBQztJQUMxQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFrQixDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUJBQXlCO1FBQzlCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO0lBQzVDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksc0JBQXNCO1FBQzNCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwwQkFBMEI7UUFDL0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDJCQUEyQjtRQUNoQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMkJBQTJCLENBQUMsQ0FBQztJQUM5QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksOEJBQThCO1FBQ25DLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO0lBQ2pELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxzQkFBc0I7UUFDM0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7SUFDekMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1DQUFtQztRQUN4QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUNBQW1DLENBQUMsQ0FBQztJQUN0RCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksNkJBQTZCO1FBQ2xDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO0lBQ2hELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw0QkFBNEI7UUFDakMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDRCQUE0QixDQUFDLENBQUM7SUFDL0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtDQUFrQztRQUN2QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0NBQWtDLENBQUMsQ0FBQztJQUNyRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksNkJBQTZCO1FBQ2xDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO0lBQ2hELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQ0FBcUM7UUFDMUMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFDQUFxQyxDQUFDLENBQUM7SUFDeEQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDRCQUE0QjtRQUNqQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNEJBQTRCLENBQUMsQ0FBQztJQUMvQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwwQkFBMEI7UUFDL0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9DQUFvQztRQUN6QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0NBQW9DLENBQUMsQ0FBQztJQUN2RCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9CQUFvQixDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDJCQUEyQjtRQUNoQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMkJBQTJCLENBQUMsQ0FBQztJQUM5QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDBCQUEwQjtRQUMvQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLENBQUMsQ0FBQztJQUM3QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksNkJBQTZCO1FBQ2xDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO0lBQ2hELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx3QkFBd0I7UUFDN0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7SUFDM0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDJCQUEyQjtRQUNoQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMkJBQTJCLENBQUMsQ0FBQztJQUM5QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMEJBQTBCO1FBQy9CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDhCQUE4QjtRQUNuQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsOEJBQThCLENBQUMsQ0FBQztJQUNqRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUNBQW1DO1FBQ3hDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQ0FBbUMsQ0FBQyxDQUFDO0lBQ3RELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0NBQXdDO1FBQzdDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3Q0FBd0MsQ0FBQyxDQUFDO0lBQzNELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQ0FBaUM7UUFDdEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGlDQUFpQyxDQUFDLENBQUM7SUFDcEQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLENBQUMsQ0FBQztJQUM1QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksNEJBQTRCO1FBQ2pDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO0lBQy9DLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0JBQXdCO1FBQzdCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO0lBQzNDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxzQkFBc0I7UUFDM0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7SUFDekMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDRCQUE0QjtRQUNqQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNEJBQTRCLENBQUMsQ0FBQztJQUMvQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQ0FBb0M7UUFDekMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9DQUFvQyxDQUFDLENBQUM7SUFDdkQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1DQUFtQztRQUN4QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUNBQW1DLENBQUMsQ0FBQztJQUN0RCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksOEJBQThCO1FBQ25DLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO0lBQ2pELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5QkFBeUI7UUFDOUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7SUFDNUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDZCQUE2QjtRQUNsQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLENBQUMsQ0FBQztJQUNoRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0NBQXNDO1FBQzNDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQ0FBc0MsQ0FBQyxDQUFDO0lBQ3pELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLENBQUMsQ0FBQztJQUM1QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0NBQXNDO1FBQzNDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQ0FBc0MsQ0FBQyxDQUFDO0lBQ3pELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDJCQUEyQjtRQUNoQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMkJBQTJCLENBQUMsQ0FBQztJQUM5QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMEJBQTBCO1FBQy9CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9CQUFvQixDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksdUJBQXVCO1FBQzVCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO0lBQzFDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQ0FBZ0M7UUFDckMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdDQUFnQyxDQUFDLENBQUM7SUFDbkQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDZCQUE2QjtRQUNsQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLENBQUMsQ0FBQztJQUNoRCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09BdUJHO0lBQ0ksaUJBQWlCO1FBQ3RCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO0lBQ3BDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5QkFBeUI7UUFDOUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7SUFDNUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLCtCQUErQjtRQUNwQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsK0JBQStCLENBQUMsQ0FBQztJQUNsRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0NBQWtDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO0lBQ3JELENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09BcUJHO0lBQ0ksa0JBQWtCO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx3QkFBd0I7UUFDN0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7SUFDM0MsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLGVBQWU7UUFDcEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDBCQUEwQjtRQUMvQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLENBQUMsQ0FBQztJQUM3QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0JBQXdCO1FBQzdCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO0lBQzNDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9CQUFvQixDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQWtCLENBQUMsQ0FBQztJQUNyQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMkJBQTJCO1FBQ2hDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO0lBQzlDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLENBQUMsQ0FBQztJQUM1QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxzQkFBc0I7UUFDM0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7SUFDekMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksNkJBQTZCO1FBQ2xDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO0lBQ2hELENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0kscUNBQXFDO1FBQzFDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQ0FBcUMsQ0FBQyxDQUFDO0lBQ3hELENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxpQkFBaUI7UUFDdEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGlCQUFpQixDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQWtCLENBQUMsQ0FBQztJQUNyQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx3QkFBd0I7UUFDN0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7SUFDM0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLENBQUMsQ0FBQztJQUM1QyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0kseUJBQXlCO1FBQzlCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO0lBQzVDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSwwQkFBMEI7UUFDL0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdDQUF3QztRQUM3QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0NBQXdDLENBQUMsQ0FBQztJQUMzRCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSx3QkFBd0I7UUFDN0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7SUFDM0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDhCQUE4QjtRQUNuQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsOEJBQThCLENBQUMsQ0FBQztJQUNqRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksdUJBQXVCO1FBQzVCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO0lBQzFDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQXlCRztJQUNJLHdCQUF3QjtRQUM3QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FtQkc7SUFDSSw4QkFBOEI7UUFDbkMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDhCQUE4QixDQUFDLENBQUM7SUFDakQsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7O09BbUJHO0lBQ0ksK0JBQStCO1FBQ3BDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO0lBQ2xELENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FpQkc7SUFDSSxpQ0FBaUM7UUFDdEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGlDQUFpQyxDQUFDLENBQUM7SUFDcEQsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09Bc0JHO0lBQ0kseUJBQXlCO1FBQzlCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO0lBQzVDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FpQkc7SUFDSSxnQ0FBZ0M7UUFDckMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdDQUFnQyxDQUFDLENBQUM7SUFDbkQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDhCQUE4QjtRQUNuQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsOEJBQThCLENBQUMsQ0FBQztJQUNqRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksNENBQTRDO1FBQ2pELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw0Q0FBNEMsQ0FBQyxDQUFDO0lBQy9ELENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxpQkFBaUI7UUFDdEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGlCQUFpQixDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDJDQUEyQztRQUNoRCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMkNBQTJDLENBQUMsQ0FBQztJQUM5RCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7OztPQWVHO0lBQ0ksK0JBQStCO1FBQ3BDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO0lBQ2xELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5QkFBeUI7UUFDOUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7SUFDNUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsdUJBQXVCLENBQUMsQ0FBQztJQUMxQyxDQUFDO0lBdUxEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxTQUFTLENBQUMscUJBQTZCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDbkcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsUUFBUyxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsWUFBYSxxQkFBc0IsRUFBRSxDQUFDLENBQUM7SUFDN0ssQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLFVBQVUsQ0FBQyxzQkFBOEIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNyRyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixRQUFTLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxnQkFBaUIsc0JBQXVCLEVBQUUsQ0FBQyxDQUFDO0lBQ2xMLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxtQkFBbUIsQ0FBQywwQkFBa0MsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNsSCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixRQUFTLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSx3QkFBeUIsMEJBQTJCLEVBQUUsQ0FBQyxDQUFDO0lBQzlMLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxZQUFZLENBQUMsMkJBQW1DLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDNUcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsUUFBUyxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsZ0JBQWlCLDJCQUE0QixFQUFFLENBQUMsQ0FBQztJQUN2TCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksaUJBQWlCLENBQUMsaUJBQXlCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDdkcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsUUFBUyxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUscUJBQXNCLGlCQUFrQixFQUFFLENBQUMsQ0FBQztJQUNsTCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLFdBQVcsQ0FBQyx5QkFBaUMsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUN6RyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixRQUFTLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxlQUFnQix5QkFBMEIsRUFBRSxDQUFDLENBQUM7SUFDcEwsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxpQkFBaUIsQ0FBQyxtQkFBMkIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUN6RyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixRQUFTLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxxQkFBc0IsbUJBQW9CLEVBQUUsQ0FBQyxDQUFDO0lBQ3BMLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09BcUJHO0lBQ0ksSUFBSSxDQUFDLGNBQXNCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDdkYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsUUFBUyxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsT0FBUSxjQUFlLEVBQUUsQ0FBQyxDQUFDO0lBQ2pLLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksSUFBSSxDQUFDLGdCQUF3QixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3pGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLFFBQVMsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLE9BQVEsZ0JBQWlCLEVBQUUsQ0FBQyxDQUFDO0lBQ25LLENBQUM7SUFFRDs7Ozs7Ozs7T0FRRztJQUNJLGVBQWUsQ0FBQyxhQUFxQixFQUFFLE9BQWdCLEVBQUUsU0FBa0I7UUFDaEYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsU0FBVSxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsbUJBQW9CLGFBQWMsRUFBRSxDQUFDLENBQUM7SUFDM0ksQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxJQUFJLENBQUMsZUFBdUIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUN4RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixRQUFTLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxPQUFRLGVBQWdCLEVBQUUsQ0FBQyxDQUFDO0lBQ2xLLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksSUFBSSxDQUFDLGtCQUEwQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQzNGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLFFBQVMsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLE9BQVEsa0JBQW1CLEVBQUUsQ0FBQyxDQUFDO0lBQ3JLLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxPQUFPLENBQUMsU0FBaUIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNyRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixRQUFTLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxhQUFjLFNBQVUsRUFBRSxDQUFDLENBQUM7SUFDbEssQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGVBQWUsQ0FBQyxpQkFBeUIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNyRyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixRQUFTLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxzQkFBdUIsaUJBQWtCLEVBQUUsQ0FBQyxDQUFDO0lBQ25MLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksSUFBSSxDQUFDLHNCQUE4QixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQy9GLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLFFBQVMsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLE9BQVEsc0JBQXVCLEVBQUUsQ0FBQyxDQUFDO0lBQ3pLLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksUUFBUSxDQUFDLGlCQUF5QixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQzlGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLFFBQVMsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLFdBQVksaUJBQWtCLEVBQUUsQ0FBQyxDQUFDO0lBQ3hLLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksVUFBVSxDQUFDLFlBQW9CLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDM0YsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsUUFBUyxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsYUFBYyxZQUFhLEVBQUUsQ0FBQyxDQUFDO0lBQ3JLLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksUUFBUSxDQUFDLGVBQXVCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDNUYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsUUFBUyxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsV0FBWSxlQUFnQixFQUFFLENBQUMsQ0FBQztJQUN0SyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksYUFBYSxDQUFDLGFBQXFCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDL0YsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsUUFBUyxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsaUJBQWtCLGFBQWMsRUFBRSxDQUFDLENBQUM7SUFDMUssQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7OztPQWNHO0lBQ0ksS0FBSyxDQUFDLE1BQWMsRUFBRSxhQUFxQixFQUFFLHVCQUErQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3hJLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLFFBQVMsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLFFBQVMsTUFBTyxJQUFLLGFBQWMsSUFBSyx1QkFBd0IsRUFBRSxDQUFDLENBQUM7SUFDMU0sQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLFlBQVksQ0FBQyw2QkFBcUMsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUM5RyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixRQUFTLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxlQUFnQiw2QkFBOEIsRUFBRSxDQUFDLENBQUM7SUFDeEwsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGFBQWEsQ0FBQyxxQkFBNkIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUN2RyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixRQUFTLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxnQkFBaUIscUJBQXNCLEVBQUUsQ0FBQyxDQUFDO0lBQ2pMLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksd0JBQXdCLENBQUMsWUFBb0IsRUFBRSxnQkFBd0IsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNuSSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixRQUFTLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSw2QkFBOEIsWUFBYSxJQUFLLGdCQUFpQixFQUFFLENBQUMsQ0FBQztJQUMzTSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksZ0JBQWdCLENBQUMsZ0JBQXdCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDckcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsUUFBUyxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsb0JBQXFCLGdCQUFpQixFQUFFLENBQUMsQ0FBQztJQUNoTCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0EyQ0c7SUFDSSxlQUFlLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBNEI7UUFDM0YsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFtQixNQUFPLEVBQUUsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2hGLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQXFDRztJQUNJLGdCQUFnQixDQUFDLE1BQWMsRUFBRSxLQUF3QixFQUFFLFFBQTRCO1FBQzVGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBb0IsTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQTBDRztJQUNJLFlBQVksQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ3hFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLGNBQWMsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQzFFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxjQUFjLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNsRSxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksaUJBQWlCLENBQUMsS0FBZTtRQUN0QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUJBQWlCLEVBQUUsQ0FBQyxPQUFPLEtBQUssS0FBSyxXQUFXLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEVBQUUsTUFBTSxDQUFDLENBQUM7SUFDM0YsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7OztPQWlCRztJQUNJLGVBQWUsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQzNFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNuRSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7OztPQWVHO0lBQ0ksZ0JBQWdCLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUM1RSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNwRSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7OztPQWVHO0lBQ0ksY0FBYyxDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDMUUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGNBQWMsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2xFLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksY0FBYyxDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDMUUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGNBQWMsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2xFLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7T0FjRztJQUNJLDBCQUEwQixDQUFDLEtBQWU7UUFDL0MsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixFQUFFLENBQUMsT0FBTyxLQUFLLEtBQUssV0FBVyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxFQUFFLE1BQU0sQ0FBQyxDQUFDO0lBQ3BHLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxTQUFTLENBQUMsS0FBZTtRQUM5QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsU0FBUyxFQUFFLENBQUMsT0FBTyxLQUFLLEtBQUssV0FBVyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxFQUFFLE1BQU0sQ0FBQyxDQUFDO0lBQ25GLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FpQkc7SUFDSSxPQUFPLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUNuRSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBTyxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksZUFBZSxDQUFDLENBQUM7SUFDOUQsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7OztPQWNHO0lBQ0ksa0JBQWtCLENBQUMsS0FBZTtRQUN2QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQWtCLEVBQUUsQ0FBQyxPQUFPLEtBQUssS0FBSyxXQUFXLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEVBQUUsTUFBTSxDQUFDLENBQUM7SUFDNUYsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7OztPQWlCRztJQUNJLGFBQWEsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ3pFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxlQUFlLENBQUMsQ0FBQztJQUNwRSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxvQkFBb0IsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ2hGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ3hFLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxLQUFLLENBQUMsS0FBZTtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsS0FBSyxFQUFFLENBQUMsT0FBTyxLQUFLLEtBQUssV0FBVyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxFQUFFLE1BQU0sQ0FBQyxDQUFDO0lBQy9FLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7T0FjRztJQUNJLGNBQWMsQ0FBQyxNQUFjLEVBQUUsS0FBd0IsRUFBRSxRQUE0QjtRQUMxRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQW1CLE1BQU8sRUFBRSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDaEYsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksb0JBQW9CLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBNEI7UUFDaEcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF5QixNQUFPLEVBQUUsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ3RGLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7T0FjRztJQUNJLFlBQVksQ0FBQyxNQUFjLEVBQUUsS0FBd0IsRUFBRSxRQUE0QjtRQUN4RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZ0IsTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUM3RSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7O09BY0c7SUFDSSxPQUFPLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBNEI7UUFDbkYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFVBQVcsTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUN4RSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxPQUFPLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBNEI7UUFDbkYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFVBQVcsTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUN4RSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxPQUFPLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBNEI7UUFDbkYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFVBQVcsTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUN4RSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7O09BY0c7SUFDSSxPQUFPLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBNEI7UUFDbkYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFVBQVcsTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUN4RSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0ErQkc7SUFDSSxRQUFRLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBNEI7UUFDcEYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFdBQVksTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUN6RSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxPQUFPLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBNEI7UUFDbkYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFVBQVcsTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUN4RSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxXQUFXLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBNEI7UUFDdkYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGNBQWUsTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUM1RSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxhQUFhLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBNEI7UUFDekYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFpQixNQUFPLEVBQUUsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQzlFLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLFdBQVcsQ0FBQyxNQUFjLEVBQUUsS0FBd0IsRUFBRSxRQUE0QjtRQUN2RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsY0FBZSxNQUFPLEVBQUUsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQzVFLENBQUM7SUFFRDs7O09BR0c7SUFDSCxZQUFZLEtBQWdDO1FBQzFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztRQTU1R1Isa0JBQWEsR0FBRyxLQUFLLENBQUM7UUFvdUVuQixvQkFBZSxHQUFvQjtZQUMzQyxLQUFLLEVBQUU7Z0JBQ0wsb0JBQW9CO2dCQUNwQixxQkFBcUI7Z0JBQ3JCLG1DQUFtQztnQkFDbkMsK0JBQStCO2dCQUMvQixvQkFBb0I7Z0JBQ3BCLGtCQUFrQjtnQkFDbEIsMkJBQTJCO2dCQUMzQiw2QkFBNkI7Z0JBQzdCLHVCQUF1QjtnQkFDdkIsc0JBQXNCO2dCQUN0QixnQkFBZ0I7Z0JBQ2hCLGlCQUFpQjtnQkFDakIsMkJBQTJCO2dCQUMzQiw2QkFBNkI7Z0JBQzdCLGlCQUFpQjtnQkFDakIseUJBQXlCO2dCQUN6QiwrQkFBK0I7Z0JBQy9CLHlCQUF5QjtnQkFDekIsa0JBQWtCO2dCQUNsQiw2QkFBNkI7Z0JBQzdCLHdCQUF3QjtnQkFDeEIsZUFBZTtnQkFDZix1QkFBdUI7Z0JBQ3ZCLHVCQUF1QjtnQkFDdkIsb0JBQW9CO2dCQUNwQixrQkFBa0I7Z0JBQ2xCLHFCQUFxQjtnQkFDckIseUJBQXlCO2dCQUN6QixxQkFBcUI7Z0JBQ3JCLG1CQUFtQjtnQkFDbkIsbUJBQW1CO2dCQUNuQixzQkFBc0I7Z0JBQ3RCLDBCQUEwQjtnQkFDMUIsMkJBQTJCO2dCQUMzQiw2QkFBNkI7Z0JBQzdCLGlCQUFpQjtnQkFDakIsZ0NBQWdDO2dCQUNoQyx5QkFBeUI7Z0JBQ3pCLCtCQUErQjtnQkFDL0IseUJBQXlCO2dCQUN6QixrQkFBa0I7Z0JBQ2xCLGlDQUFpQztnQkFDakMsd0JBQXdCO2dCQUN4QixlQUFlO2dCQUNmLHVCQUF1QjtnQkFDdkIsdUJBQXVCO2dCQUN2QixvQkFBb0I7Z0JBQ3BCLGtCQUFrQjtnQkFDbEIscUJBQXFCO2dCQUNyQix5QkFBeUI7Z0JBQ3pCLHFCQUFxQjtnQkFDckIsbUJBQW1CO2dCQUNuQixtQkFBbUI7Z0JBQ25CLHNCQUFzQjtnQkFDdEIsMEJBQTBCO2dCQUMxQixxQkFBcUI7Z0JBQ3JCLG9CQUFvQjtnQkFDcEIsbUJBQW1CO2dCQUNuQix1QkFBdUI7Z0JBQ3ZCLHNCQUFzQjtnQkFDdEIsb0JBQW9CO2dCQUNwQixnQ0FBZ0M7Z0JBQ2hDLDZCQUE2QjtnQkFDN0IsaUJBQWlCO2dCQUNqQix5QkFBeUI7Z0JBQ3pCLCtCQUErQjtnQkFDL0Isa0NBQWtDO2dCQUNsQyxrQkFBa0I7Z0JBQ2xCLHdCQUF3QjtnQkFDeEIsZUFBZTtnQkFDZix1QkFBdUI7Z0JBQ3ZCLDBCQUEwQjtnQkFDMUIsd0JBQXdCO2dCQUN4QixvQkFBb0I7Z0JBQ3BCLGtCQUFrQjtnQkFDbEIsMkJBQTJCO2dCQUMzQixxQkFBcUI7Z0JBQ3JCLHlCQUF5QjtnQkFDekIscUJBQXFCO2dCQUNyQixtQkFBbUI7Z0JBQ25CLG1CQUFtQjtnQkFDbkIsc0JBQXNCO2dCQUN0QixzQkFBc0I7Z0JBQ3RCLG9CQUFvQjtnQkFDcEIsNkJBQTZCO2dCQUM3QixxQ0FBcUM7Z0JBQ3JDLGlCQUFpQjtnQkFDakIsa0JBQWtCO2dCQUNsQixvQkFBb0I7Z0JBQ3BCLHdCQUF3QjtnQkFDeEIseUJBQXlCO2dCQUN6Qix5QkFBeUI7Z0JBQ3pCLDBCQUEwQjtnQkFDMUIsd0NBQXdDO2dCQUN4Qyw4QkFBOEI7Z0JBQzlCLHVCQUF1QjtnQkFDdkIsd0JBQXdCO2dCQUN4Qiw4QkFBOEI7Z0JBQzlCLCtCQUErQjtnQkFDL0IsaUNBQWlDO2dCQUNqQyx5QkFBeUI7Z0JBQ3pCLGdDQUFnQztnQkFDaEMsOEJBQThCO2dCQUM5QixxQkFBcUI7Z0JBQ3JCLGdCQUFnQjtnQkFDaEIsaUJBQWlCO2dCQUNqQiw0Q0FBNEM7Z0JBQzVDLGlCQUFpQjtnQkFDakIsb0JBQW9CO2dCQUNwQixlQUFlO2dCQUNmLGdCQUFnQjtnQkFDaEIsMkNBQTJDO2dCQUMzQywrQkFBK0I7Z0JBQy9CLHlCQUF5QjtnQkFDekIsdUJBQXVCO2FBQ3hCO1lBQ0QsT0FBTyxFQUFFO2dCQUNQLG1CQUFtQjtnQkFDbkIsd0JBQXdCO2FBQ3pCO1lBQ0Qsd0JBQXdCLEVBQUU7Z0JBQ3hCLGlDQUFpQzthQUNsQztZQUNELElBQUksRUFBRTtnQkFDSiwyQkFBMkI7Z0JBQzNCLDhCQUE4QjtnQkFDOUIsc0JBQXNCO2dCQUN0QixtQ0FBbUM7Z0JBQ25DLDZCQUE2QjtnQkFDN0IsNEJBQTRCO2dCQUM1QixrQ0FBa0M7Z0JBQ2xDLDZCQUE2QjtnQkFDN0IscUNBQXFDO2dCQUNyQyw0QkFBNEI7Z0JBQzVCLG9CQUFvQjtnQkFDcEIsMEJBQTBCO2dCQUMxQixvQ0FBb0M7Z0JBQ3BDLHFCQUFxQjtnQkFDckIsb0JBQW9CO2dCQUNwQiwyQkFBMkI7Z0JBQzNCLHNCQUFzQjtnQkFDdEIsbUJBQW1CO2dCQUNuQiwwQkFBMEI7Z0JBQzFCLDZCQUE2QjtnQkFDN0Isd0JBQXdCO2dCQUN4QiwyQkFBMkI7Z0JBQzNCLDBCQUEwQjtnQkFDMUIsdUJBQXVCO2dCQUN2Qiw4QkFBOEI7Z0JBQzlCLG1DQUFtQztnQkFDbkMscUJBQXFCO2dCQUNyQix3QkFBd0I7Z0JBQ3hCLHdDQUF3QztnQkFDeEMsaUNBQWlDO2dCQUNqQyx5QkFBeUI7Z0JBQ3pCLDRCQUE0QjtnQkFDNUIsZ0JBQWdCO2dCQUNoQixxQkFBcUI7Z0JBQ3JCLHdCQUF3QjtnQkFDeEIsc0JBQXNCO2dCQUN0Qiw0QkFBNEI7Z0JBQzVCLHNCQUFzQjtnQkFDdEIsb0NBQW9DO2dCQUNwQyxtQ0FBbUM7Z0JBQ25DLDZCQUE2QjtnQkFDN0Isc0NBQXNDO2dCQUN0Qyx1QkFBdUI7Z0JBQ3ZCLHlCQUF5QjtnQkFDekIsc0NBQXNDO2FBQ3ZDO1lBQ0QsSUFBSSxFQUFFO2dCQUNKLDhCQUE4QjtnQkFDOUIseUJBQXlCO2dCQUN6QiwyQkFBMkI7Z0JBQzNCLDBCQUEwQjtnQkFDMUIscUJBQXFCO2FBQ3RCO1NBQ0YsQ0FBQztJQXNnQ0YsQ0FBQztDQUNGO0FBLzVHRCxrQkErNUdDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQWNjZXNzTGV2ZWxMaXN0IH0gZnJvbSAnLi4vLi4vc2hhcmVkL2FjY2Vzcy1sZXZlbCc7XG5pbXBvcnQgeyBQb2xpY3lTdGF0ZW1lbnQsIE9wZXJhdG9yIH0gZnJvbSAnLi4vLi4vc2hhcmVkJztcbmltcG9ydCB7IGF3c19pYW0gYXMgaWFtIH0gZnJvbSBcImF3cy1jZGstbGliXCI7XG5cbi8qKlxuICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFtyZHNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXJ2aWNlLWF1dGhvcml6YXRpb24vbGF0ZXN0L3JlZmVyZW5jZS9saXN0X2FtYXpvbnJkcy5odG1sKS5cbiAqXG4gKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAqL1xuZXhwb3J0IGNsYXNzIFJkcyBleHRlbmRzIFBvbGljeVN0YXRlbWVudCB7XG4gIHB1YmxpYyBzZXJ2aWNlUHJlZml4ID0gJ3Jkcyc7XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFzc29jaWF0ZSBhbiBJZGVudGl0eSBhbmQgQWNjZXNzIE1hbmFnZW1lbnQgKElBTSkgcm9sZSBmcm9tIGFuIEF1cm9yYSBEQiBjbHVzdGVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gaWFtOlBhc3NSb2xlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQWRkUm9sZVRvREJDbHVzdGVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0FkZFJvbGVUb0RCQ2x1c3RlcigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQWRkUm9sZVRvREJDbHVzdGVyJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYXNzb2NpYXRlIGFuIEFXUyBJZGVudGl0eSBhbmQgQWNjZXNzIE1hbmFnZW1lbnQgKElBTSkgcm9sZSB3aXRoIGEgREIgaW5zdGFuY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBpYW06UGFzc1JvbGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9BZGRSb2xlVG9EQkluc3RhbmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0FkZFJvbGVUb0RCSW5zdGFuY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0FkZFJvbGVUb0RCSW5zdGFuY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBhZGQgYSBzb3VyY2UgaWRlbnRpZmllciB0byBhbiBleGlzdGluZyBSRFMgZXZlbnQgbm90aWZpY2F0aW9uIHN1YnNjcmlwdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQWRkU291cmNlSWRlbnRpZmllclRvU3Vic2NyaXB0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0FkZFNvdXJjZUlkZW50aWZpZXJUb1N1YnNjcmlwdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQWRkU291cmNlSWRlbnRpZmllclRvU3Vic2NyaXB0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYWRkIG1ldGFkYXRhIHRhZ3MgdG8gYW4gQW1hem9uIFJEUyByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICogLSAuaWZSZXFUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0FkZFRhZ3NUb1Jlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0FkZFRhZ3NUb1Jlc291cmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdBZGRUYWdzVG9SZXNvdXJjZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFwcGx5IGEgcGVuZGluZyBtYWludGVuYW5jZSBhY3Rpb24gdG8gYSByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQXBwbHlQZW5kaW5nTWFpbnRlbmFuY2VBY3Rpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQXBwbHlQZW5kaW5nTWFpbnRlbmFuY2VBY3Rpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0FwcGx5UGVuZGluZ01haW50ZW5hbmNlQWN0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZW5hYmxlIGluZ3Jlc3MgdG8gYSBEQlNlY3VyaXR5R3JvdXAgdXNpbmcgb25lIG9mIHR3byBmb3JtcyBvZiBhdXRob3JpemF0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUGVybWlzc2lvbnMgbWFuYWdlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0F1dGhvcml6ZURCU2VjdXJpdHlHcm91cEluZ3Jlc3MuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQXV0aG9yaXplREJTZWN1cml0eUdyb3VwSW5ncmVzcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQXV0aG9yaXplREJTZWN1cml0eUdyb3VwSW5ncmVzcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGJhY2t0cmFjayBhIERCIGNsdXN0ZXIgdG8gYSBzcGVjaWZpYyB0aW1lLCB3aXRob3V0IGNyZWF0aW5nIGEgbmV3IERCIGNsdXN0ZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0JhY2t0cmFja0RCQ2x1c3Rlci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9CYWNrdHJhY2tEQkNsdXN0ZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0JhY2t0cmFja0RCQ2x1c3RlcicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNhbmNlbCBhbiBleHBvcnQgdGFzayBpbiBwcm9ncmVzc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ2FuY2VsRXhwb3J0VGFzay5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DYW5jZWxFeHBvcnRUYXNrKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDYW5jZWxFeHBvcnRUYXNrJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY29weSBhIGN1c3RvbSBlbmdpbmUgdmVyc2lvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXJ2aWNlLWF1dGhvcml6YXRpb24vbGF0ZXN0L3JlZmVyZW5jZS9saXN0X2FtYXpvbnJkcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9Db3B5Q3VzdG9tREJFbmdpbmVWZXJzaW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDb3B5Q3VzdG9tREJFbmdpbmVWZXJzaW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY29weSB0aGUgc3BlY2lmaWVkIERCIGNsdXN0ZXIgcGFyYW1ldGVyIGdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gcmRzOkFkZFRhZ3NUb1Jlc291cmNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ29weURCQ2x1c3RlclBhcmFtZXRlckdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NvcHlEQkNsdXN0ZXJQYXJhbWV0ZXJHcm91cCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ29weURCQ2x1c3RlclBhcmFtZXRlckdyb3VwJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgc25hcHNob3Qgb2YgYSBEQiBjbHVzdGVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gcmRzOkFkZFRhZ3NUb1Jlc291cmNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ29weURCQ2x1c3RlclNuYXBzaG90Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NvcHlEQkNsdXN0ZXJTbmFwc2hvdCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ29weURCQ2x1c3RlclNuYXBzaG90Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY29weSB0aGUgc3BlY2lmaWVkIERCIHBhcmFtZXRlciBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIHJkczpBZGRUYWdzVG9SZXNvdXJjZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NvcHlEQlBhcmFtZXRlckdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NvcHlEQlBhcmFtZXRlckdyb3VwKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDb3B5REJQYXJhbWV0ZXJHcm91cCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNvcHkgdGhlIHNwZWNpZmllZCBEQiBzbmFwc2hvdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqIC0gLmlmQ29weU9wdGlvbkdyb3VwKClcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gcmRzOkFkZFRhZ3NUb1Jlc291cmNlXG4gICAqIC0gcmRzOkNvcHlDdXN0b21EQkVuZ2luZVZlcnNpb25cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9Db3B5REJTbmFwc2hvdC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9Db3B5REJTbmFwc2hvdCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ29weURCU25hcHNob3QnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjb3B5IHRoZSBzcGVjaWZpZWQgb3B0aW9uIGdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gcmRzOkFkZFRhZ3NUb1Jlc291cmNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ29weU9wdGlvbkdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NvcHlPcHRpb25Hcm91cCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ29weU9wdGlvbkdyb3VwJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgYmx1ZS1ncmVlbiBkZXBsb3ltZW50IGZvciBhIGdpdmVuIHNvdXJjZSBjbHVzdGVyIG9yIGluc3RhbmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqIC0gLmlmQ2x1c3RlclRhZygpXG4gICAqIC0gLmlmQ2x1c3RlclBnVGFnKClcbiAgICogLSAuaWZEYlRhZygpXG4gICAqIC0gLmlmUGdUYWcoKVxuICAgKiAtIC5pZlJlcVRhZygpXG4gICAqIC0gLmlmRGF0YWJhc2VFbmdpbmUoKVxuICAgKiAtIC5pZkRhdGFiYXNlTmFtZSgpXG4gICAqIC0gLmlmU3RvcmFnZUVuY3J5cHRlZCgpXG4gICAqIC0gLmlmRGF0YWJhc2VDbGFzcygpXG4gICAqIC0gLmlmU3RvcmFnZVNpemUoKVxuICAgKiAtIC5pZk11bHRpQXooKVxuICAgKiAtIC5pZlBpb3BzKClcbiAgICogLSAuaWZWcGMoKVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSByZHM6QWRkVGFnc1RvUmVzb3VyY2VcbiAgICogLSByZHM6Q3JlYXRlREJDbHVzdGVyXG4gICAqIC0gcmRzOkNyZWF0ZURCQ2x1c3RlckVuZHBvaW50XG4gICAqIC0gcmRzOkNyZWF0ZURCSW5zdGFuY2VcbiAgICogLSByZHM6Q3JlYXRlREJJbnN0YW5jZVJlYWRSZXBsaWNhXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlQmx1ZUdyZWVuRGVwbG95bWVudC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVCbHVlR3JlZW5EZXBsb3ltZW50KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVCbHVlR3JlZW5EZXBsb3ltZW50Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgY3VzdG9tIGVuZ2luZSB2ZXJzaW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gaWFtOkNyZWF0ZVNlcnZpY2VMaW5rZWRSb2xlXG4gICAqIC0gbWVkaWFpbXBvcnQ6Q3JlYXRlRGF0YWJhc2VCaW5hcnlTbmFwc2hvdFxuICAgKiAtIHJkczpBZGRUYWdzVG9SZXNvdXJjZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZUN1c3RvbURCRW5naW5lVmVyc2lvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVDdXN0b21EQkVuZ2luZVZlcnNpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZUN1c3RvbURCRW5naW5lVmVyc2lvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIG5ldyBEQiBjbHVzdGVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICogLSAuaWZSZXFUYWcoKVxuICAgKiAtIC5pZkRhdGFiYXNlRW5naW5lKClcbiAgICogLSAuaWZEYXRhYmFzZU5hbWUoKVxuICAgKiAtIC5pZlN0b3JhZ2VFbmNyeXB0ZWQoKVxuICAgKiAtIC5pZkRhdGFiYXNlQ2xhc3MoKVxuICAgKiAtIC5pZlN0b3JhZ2VTaXplKClcbiAgICogLSAuaWZQaW9wcygpXG4gICAqIC0gLmlmTWFuYWdlTWFzdGVyVXNlclBhc3N3b3JkKClcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gaWFtOlBhc3NSb2xlXG4gICAqIC0ga21zOkNyZWF0ZUdyYW50XG4gICAqIC0ga21zOkRlY3J5cHRcbiAgICogLSBrbXM6RGVzY3JpYmVLZXlcbiAgICogLSBrbXM6R2VuZXJhdGVEYXRhS2V5XG4gICAqIC0gcmRzOkFkZFRhZ3NUb1Jlc291cmNlXG4gICAqIC0gcmRzOkNyZWF0ZURCSW5zdGFuY2VcbiAgICogLSBzZWNyZXRzbWFuYWdlcjpDcmVhdGVTZWNyZXRcbiAgICogLSBzZWNyZXRzbWFuYWdlcjpUYWdSZXNvdXJjZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZURCQ2x1c3Rlci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVEQkNsdXN0ZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZURCQ2x1c3RlcicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIG5ldyBjdXN0b20gZW5kcG9pbnQgYW5kIGFzc29jaWF0ZXMgaXQgd2l0aCBhbiBBbWF6b24gQXVyb3JhIERCIGNsdXN0ZXIgb3IgQW1hem9uIERvY3VtZW50REIgY2x1c3RlclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmRW5kcG9pbnRUeXBlKClcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gcmRzOkFkZFRhZ3NUb1Jlc291cmNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlREJDbHVzdGVyRW5kcG9pbnQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlREJDbHVzdGVyRW5kcG9pbnQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZURCQ2x1c3RlckVuZHBvaW50Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgbmV3IERCIGNsdXN0ZXIgcGFyYW1ldGVyIGdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICogLSAuaWZSZXFUYWcoKVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSByZHM6QWRkVGFnc1RvUmVzb3VyY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVEQkNsdXN0ZXJQYXJhbWV0ZXJHcm91cC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVEQkNsdXN0ZXJQYXJhbWV0ZXJHcm91cCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlREJDbHVzdGVyUGFyYW1ldGVyR3JvdXAnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBzbmFwc2hvdCBvZiBhIERCIGNsdXN0ZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKiAtIC5pZlJlcVRhZygpXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIHJkczpBZGRUYWdzVG9SZXNvdXJjZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZURCQ2x1c3RlclNuYXBzaG90Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZURCQ2x1c3RlclNuYXBzaG90KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVEQkNsdXN0ZXJTbmFwc2hvdCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIG5ldyBEQiBpbnN0YW5jZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQmFja3VwVGFyZ2V0KClcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICogLSAuaWZSZXFUYWcoKVxuICAgKiAtIC5pZk1hbmFnZU1hc3RlclVzZXJQYXNzd29yZCgpXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGlhbTpQYXNzUm9sZVxuICAgKiAtIGttczpDcmVhdGVHcmFudFxuICAgKiAtIGttczpEZWNyeXB0XG4gICAqIC0ga21zOkRlc2NyaWJlS2V5XG4gICAqIC0ga21zOkdlbmVyYXRlRGF0YUtleVxuICAgKiAtIHJkczpBZGRUYWdzVG9SZXNvdXJjZVxuICAgKiAtIHJkczpDcmVhdGVUZW5hbnREYXRhYmFzZVxuICAgKiAtIHNlY3JldHNtYW5hZ2VyOkNyZWF0ZVNlY3JldFxuICAgKiAtIHNlY3JldHNtYW5hZ2VyOlRhZ1Jlc291cmNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlREJJbnN0YW5jZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVEQkluc3RhbmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVEQkluc3RhbmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgREIgaW5zdGFuY2UgdGhhdCBhY3RzIGFzIGEgUmVhZCBSZXBsaWNhIG9mIGEgc291cmNlIERCIGluc3RhbmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICogLSAuaWZSZXFUYWcoKVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBpYW06UGFzc1JvbGVcbiAgICogLSByZHM6QWRkVGFnc1RvUmVzb3VyY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVEQkluc3RhbmNlUmVhZFJlcGxpY2EuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlREJJbnN0YW5jZVJlYWRSZXBsaWNhKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVEQkluc3RhbmNlUmVhZFJlcGxpY2EnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBuZXcgREIgcGFyYW1ldGVyIGdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICogLSAuaWZSZXFUYWcoKVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSByZHM6QWRkVGFnc1RvUmVzb3VyY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVEQlBhcmFtZXRlckdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZURCUGFyYW1ldGVyR3JvdXAoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZURCUGFyYW1ldGVyR3JvdXAnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBkYXRhYmFzZSBwcm94eVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGlhbTpQYXNzUm9sZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZURCUHJveHkuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlREJQcm94eSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlREJQcm94eScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIGRhdGFiYXNlIHByb3h5IGVuZHBvaW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVEQlByb3h5RW5kcG9pbnQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlREJQcm94eUVuZHBvaW50KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVEQlByb3h5RW5kcG9pbnQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBuZXcgREIgc2VjdXJpdHkgZ3JvdXAuIERCIHNlY3VyaXR5IGdyb3VwcyBjb250cm9sIGFjY2VzcyB0byBhIERCIGluc3RhbmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICogLSAuaWZSZXFUYWcoKVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSByZHM6QWRkVGFnc1RvUmVzb3VyY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVEQlNlY3VyaXR5R3JvdXAuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlREJTZWN1cml0eUdyb3VwKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVEQlNlY3VyaXR5R3JvdXAnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBuZXcgQXVyb3JhIExpbWl0bGVzcyBEYXRhYmFzZSBEQiBzaGFyZCBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIHJkczpBZGRUYWdzVG9SZXNvdXJjZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZURCU2hhcmRHcm91cC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVEQlNoYXJkR3JvdXAoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZURCU2hhcmRHcm91cCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIERCU25hcHNob3RcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkJhY2t1cFRhcmdldCgpXG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqIC0gLmlmUmVxVGFnKClcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gcmRzOkFkZFRhZ3NUb1Jlc291cmNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlREJTbmFwc2hvdC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVEQlNuYXBzaG90KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVEQlNuYXBzaG90Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgbmV3IERCIHN1Ym5ldCBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqIC0gLmlmUmVxVGFnKClcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gcmRzOkFkZFRhZ3NUb1Jlc291cmNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlREJTdWJuZXRHcm91cC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVEQlN1Ym5ldEdyb3VwKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVEQlN1Ym5ldEdyb3VwJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGFuIFJEUyBldmVudCBub3RpZmljYXRpb24gc3Vic2NyaXB0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICogLSAuaWZSZXFUYWcoKVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSByZHM6QWRkVGFnc1RvUmVzb3VyY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVFdmVudFN1YnNjcmlwdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVFdmVudFN1YnNjcmlwdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlRXZlbnRTdWJzY3JpcHRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYW4gQXVyb3JhIGdsb2JhbCBkYXRhYmFzZSBvciBEb2N1bWVudERCIGdsb2JhbCBkYXRhYmFzZSBzcHJlYWQgYWNyb3NzIG11bHRpcGxlIHJlZ2lvbnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZUdsb2JhbENsdXN0ZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlR2xvYmFsQ2x1c3RlcigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlR2xvYmFsQ2x1c3RlcicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhbiBBdXJvcmEgemVyby1FVEwgaW50ZWdyYXRpb24gd2l0aCBSZWRzaGlmdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqIC0gLmlmUmVxVGFnKClcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0ga21zOkNyZWF0ZUdyYW50XG4gICAqIC0ga21zOkRlc2NyaWJlS2V5XG4gICAqIC0gcmRzOkFkZFRhZ3NUb1Jlc291cmNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlSW50ZWdyYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlSW50ZWdyYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZUludGVncmF0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgbmV3IG9wdGlvbiBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqIC0gLmlmUmVxVGFnKClcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gcmRzOkFkZFRhZ3NUb1Jlc291cmNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlT3B0aW9uR3JvdXAuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlT3B0aW9uR3JvdXAoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZU9wdGlvbkdyb3VwJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgbmV3IHRlbmFudCBkYXRhYmFzZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqIC0gLmlmVGVuYW50RGF0YWJhc2VOYW1lKClcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gcmRzOkFkZFRhZ3NUb1Jlc291cmNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlVGVuYW50RGF0YWJhc2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlVGVuYW50RGF0YWJhc2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZVRlbmFudERhdGFiYXNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYWNjZXNzIGEgcmVzb3VyY2UgaW4gdGhlIHJlbW90ZSBSZWdpb24gd2hlbiBleGVjdXRpbmcgY3Jvc3MtUmVnaW9uIG9wZXJhdGlvbnMsIHN1Y2ggYXMgY3Jvc3MtUmVnaW9uIHNuYXBzaG90IGNvcHkgb3IgY3Jvc3MtUmVnaW9uIHJlYWQgcmVwbGljYSBjcmVhdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWwjVXNpbmdXaXRoUkRTLklBTS5Db25kaXRpb25zXG4gICAqL1xuICBwdWJsaWMgdG9Dcm9zc1JlZ2lvbkNvbW11bmljYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Nyb3NzUmVnaW9uQ29tbXVuaWNhdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBibHVlIGdyZWVuIGRlcGxveW1lbnRzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIHJkczpEZWxldGVEQkNsdXN0ZXJcbiAgICogLSByZHM6RGVsZXRlREJDbHVzdGVyRW5kcG9pbnRcbiAgICogLSByZHM6RGVsZXRlREJJbnN0YW5jZVxuICAgKiAtIHJkczpQcm9tb3RlUmVhZFJlcGxpY2FcbiAgICogLSByZHM6UHJvbW90ZVJlYWRSZXBsaWNhREJDbHVzdGVyXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlQmx1ZUdyZWVuRGVwbG95bWVudC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVCbHVlR3JlZW5EZXBsb3ltZW50KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVCbHVlR3JlZW5EZXBsb3ltZW50Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGFuIGV4aXN0aW5nIGN1c3RvbSBlbmdpbmUgdmVyc2lvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlQ3VzdG9tREJFbmdpbmVWZXJzaW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUN1c3RvbURCRW5naW5lVmVyc2lvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlQ3VzdG9tREJFbmdpbmVWZXJzaW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGEgcHJldmlvdXNseSBwcm92aXNpb25lZCBEQiBjbHVzdGVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gcmRzOkFkZFRhZ3NUb1Jlc291cmNlXG4gICAqIC0gcmRzOkNyZWF0ZURCQ2x1c3RlclNuYXBzaG90XG4gICAqIC0gcmRzOkRlbGV0ZURCSW5zdGFuY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVEQkNsdXN0ZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlREJDbHVzdGVyKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVEQkNsdXN0ZXInKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgY2x1c3RlciBhdXRvbWF0ZWQgYmFja3VwcyBiYXNlZCBvbiB0aGUgc291cmNlIGNsdXN0ZXIncyBEYkNsdXN0ZXJSZXNvdXJjZUlkIHZhbHVlIG9yIHRoZSByZXN0b3JhYmxlIGNsdXN0ZXIncyByZXNvdXJjZSBJRFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlREJDbHVzdGVyQXV0b21hdGVkQmFja3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZURCQ2x1c3RlckF1dG9tYXRlZEJhY2t1cCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlREJDbHVzdGVyQXV0b21hdGVkQmFja3VwJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGEgY3VzdG9tIGVuZHBvaW50IGFuZCByZW1vdmVzIGl0IGZyb20gYW4gQW1hem9uIEF1cm9yYSBEQiBjbHVzdGVyIG9yIEFtYXpvbiBEb2N1bWVudERCIGNsdXN0ZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZURCQ2x1c3RlckVuZHBvaW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZURCQ2x1c3RlckVuZHBvaW50KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVEQkNsdXN0ZXJFbmRwb2ludCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIHNwZWNpZmllZCBEQiBjbHVzdGVyIHBhcmFtZXRlciBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlREJDbHVzdGVyUGFyYW1ldGVyR3JvdXAuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlREJDbHVzdGVyUGFyYW1ldGVyR3JvdXAoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZURCQ2x1c3RlclBhcmFtZXRlckdyb3VwJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGEgREIgY2x1c3RlciBzbmFwc2hvdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlREJDbHVzdGVyU25hcHNob3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlREJDbHVzdGVyU25hcHNob3QoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZURCQ2x1c3RlclNuYXBzaG90Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGEgcHJldmlvdXNseSBwcm92aXNpb25lZCBEQiBpbnN0YW5jZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIHJkczpBZGRUYWdzVG9SZXNvdXJjZVxuICAgKiAtIHJkczpDcmVhdGVEQlNuYXBzaG90XG4gICAqIC0gcmRzOkRlbGV0ZVRlbmFudERhdGFiYXNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlREJJbnN0YW5jZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVEQkluc3RhbmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVEQkluc3RhbmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGF1dG9tYXRlZCBiYWNrdXBzIGJhc2VkIG9uIHRoZSBzb3VyY2UgaW5zdGFuY2UncyBEYmlSZXNvdXJjZUlkIHZhbHVlIG9yIHRoZSByZXN0b3JhYmxlIGluc3RhbmNlJ3MgcmVzb3VyY2UgSURcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZURCSW5zdGFuY2VBdXRvbWF0ZWRCYWNrdXAuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlREJJbnN0YW5jZUF1dG9tYXRlZEJhY2t1cCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlREJJbnN0YW5jZUF1dG9tYXRlZEJhY2t1cCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIHNwZWNpZmllZCBEQlBhcmFtZXRlckdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVEQlBhcmFtZXRlckdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZURCUGFyYW1ldGVyR3JvdXAoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZURCUGFyYW1ldGVyR3JvdXAnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBkYXRhYmFzZSBwcm94eVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlREJQcm94eS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVEQlByb3h5KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVEQlByb3h5Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGEgZGF0YWJhc2UgcHJveHkgZW5kcG9pbnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZURCUHJveHlFbmRwb2ludC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVEQlByb3h5RW5kcG9pbnQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZURCUHJveHlFbmRwb2ludCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIERCIHNlY3VyaXR5IGdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVEQlNlY3VyaXR5R3JvdXAuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlREJTZWN1cml0eUdyb3VwKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVEQlNlY3VyaXR5R3JvdXAnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYW4gQXVyb3JhIExpbWl0bGVzcyBEYXRhYmFzZSBEQiBzaGFyZCBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlREJTaGFyZEdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZURCU2hhcmRHcm91cCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlREJTaGFyZEdyb3VwJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGEgREJTbmFwc2hvdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlREJTbmFwc2hvdC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVEQlNuYXBzaG90KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVEQlNuYXBzaG90Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGEgREIgc3VibmV0IGdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVEQlN1Ym5ldEdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZURCU3VibmV0R3JvdXAoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZURCU3VibmV0R3JvdXAnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYW4gUkRTIGV2ZW50IG5vdGlmaWNhdGlvbiBzdWJzY3JpcHRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUV2ZW50U3Vic2NyaXB0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUV2ZW50U3Vic2NyaXB0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVFdmVudFN1YnNjcmlwdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIGdsb2JhbCBkYXRhYmFzZSBjbHVzdGVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVHbG9iYWxDbHVzdGVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUdsb2JhbENsdXN0ZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZUdsb2JhbENsdXN0ZXInKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYW4gQXVyb3JhIHplcm8tRVRMIGludGVncmF0aW9uIHdpdGggUmVkc2hpZnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUludGVncmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUludGVncmF0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVJbnRlZ3JhdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhbiBleGlzdGluZyBvcHRpb24gZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZU9wdGlvbkdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZU9wdGlvbkdyb3VwKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVPcHRpb25Hcm91cCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIHRlbmFudCBkYXRhYmFzZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIHJkczpBZGRUYWdzVG9SZXNvdXJjZVxuICAgKiAtIHJkczpDcmVhdGVEQlNuYXBzaG90XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlVGVuYW50RGF0YWJhc2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlVGVuYW50RGF0YWJhc2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZVRlbmFudERhdGFiYXNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVtb3ZlIHRhcmdldHMgZnJvbSBhIGRhdGFiYXNlIHByb3h5IHRhcmdldCBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVyZWdpc3RlckRCUHJveHlUYXJnZXRzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlcmVnaXN0ZXJEQlByb3h5VGFyZ2V0cygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVyZWdpc3RlckRCUHJveHlUYXJnZXRzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBhbGwgb2YgdGhlIGF0dHJpYnV0ZXMgZm9yIGEgY3VzdG9tZXIgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUFjY291bnRBdHRyaWJ1dGVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlQWNjb3VudEF0dHJpYnV0ZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlQWNjb3VudEF0dHJpYnV0ZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZXNjcmliZSBibHVlIGdyZWVuIGRlcGxveW1lbnRzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlQmx1ZUdyZWVuRGVwbG95bWVudHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVCbHVlR3JlZW5EZXBsb3ltZW50cygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVCbHVlR3JlZW5EZXBsb3ltZW50cycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgdGhlIHNldCBvZiBDQSBjZXJ0aWZpY2F0ZXMgcHJvdmlkZWQgYnkgQW1hem9uIFJEUyBmb3IgdGhpcyBBV1MgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUNlcnRpZmljYXRlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZUNlcnRpZmljYXRlcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVDZXJ0aWZpY2F0ZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gYSBsaXN0IG9mIGNsdXN0ZXIgYXV0b21hdGVkIGJhY2t1cHMgZm9yIGJvdGggY3VycmVudCBhbmQgZGVsZXRlZCBjbHVzdGVyc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZURCQ2x1c3RlckF1dG9tYXRlZEJhY2t1cHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVEQkNsdXN0ZXJBdXRvbWF0ZWRCYWNrdXBzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZURCQ2x1c3RlckF1dG9tYXRlZEJhY2t1cHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gaW5mb3JtYXRpb24gYWJvdXQgYmFja3RyYWNrcyBmb3IgYSBEQiBjbHVzdGVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlREJDbHVzdGVyQmFja3RyYWNrcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZURCQ2x1c3RlckJhY2t0cmFja3MoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlREJDbHVzdGVyQmFja3RyYWNrcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHVybiBpbmZvcm1hdGlvbiBhYm91dCBlbmRwb2ludHMgZm9yIGFuIEFtYXpvbiBBdXJvcmEgREIgY2x1c3RlclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZURCQ2x1c3RlckVuZHBvaW50cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZURCQ2x1c3RlckVuZHBvaW50cygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVEQkNsdXN0ZXJFbmRwb2ludHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gYSBsaXN0IG9mIERCQ2x1c3RlclBhcmFtZXRlckdyb3VwIGRlc2NyaXB0aW9uc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZURCQ2x1c3RlclBhcmFtZXRlckdyb3Vwcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZURCQ2x1c3RlclBhcmFtZXRlckdyb3VwcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVEQkNsdXN0ZXJQYXJhbWV0ZXJHcm91cHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gdGhlIGRldGFpbGVkIHBhcmFtZXRlciBsaXN0IGZvciBhIHBhcnRpY3VsYXIgREIgY2x1c3RlciBwYXJhbWV0ZXIgZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVEQkNsdXN0ZXJQYXJhbWV0ZXJzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlREJDbHVzdGVyUGFyYW1ldGVycygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVEQkNsdXN0ZXJQYXJhbWV0ZXJzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIGEgbGlzdCBvZiBEQiBjbHVzdGVyIHNuYXBzaG90IGF0dHJpYnV0ZSBuYW1lcyBhbmQgdmFsdWVzIGZvciBhIG1hbnVhbCBEQiBjbHVzdGVyIHNuYXBzaG90XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlREJDbHVzdGVyU25hcHNob3RBdHRyaWJ1dGVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlREJDbHVzdGVyU25hcHNob3RBdHRyaWJ1dGVzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZURCQ2x1c3RlclNuYXBzaG90QXR0cmlidXRlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHVybiBpbmZvcm1hdGlvbiBhYm91dCBEQiBjbHVzdGVyIHNuYXBzaG90c1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZURCQ2x1c3RlclNuYXBzaG90cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZURCQ2x1c3RlclNuYXBzaG90cygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVEQkNsdXN0ZXJTbmFwc2hvdHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gaW5mb3JtYXRpb24gYWJvdXQgcHJvdmlzaW9uZWQgQXVyb3JhIERCIGNsdXN0ZXJzIG9yIERvY3VtZW50REIgY2x1c3RlcnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVEQkNsdXN0ZXJzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlREJDbHVzdGVycygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVEQkNsdXN0ZXJzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIGEgbGlzdCBvZiB0aGUgYXZhaWxhYmxlIERCIGVuZ2luZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVEQkVuZ2luZVZlcnNpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlREJFbmdpbmVWZXJzaW9ucygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVEQkVuZ2luZVZlcnNpb25zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIGEgbGlzdCBvZiBhdXRvbWF0ZWQgYmFja3VwcyBmb3IgYm90aCBjdXJyZW50IGFuZCBkZWxldGVkIGluc3RhbmNlc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZURCSW5zdGFuY2VBdXRvbWF0ZWRCYWNrdXBzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlREJJbnN0YW5jZUF1dG9tYXRlZEJhY2t1cHMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlREJJbnN0YW5jZUF1dG9tYXRlZEJhY2t1cHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gaW5mb3JtYXRpb24gYWJvdXQgcHJvdmlzaW9uZWQgUkRTIGluc3RhbmNlc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZURCSW5zdGFuY2VzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlREJJbnN0YW5jZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlREJJbnN0YW5jZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gYSBsaXN0IG9mIERCIGxvZyBmaWxlcyBmb3IgdGhlIERCIGluc3RhbmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlREJMb2dGaWxlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZURCTG9nRmlsZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlREJMb2dGaWxlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHVybiBhIGxpc3Qgb2YgREJQYXJhbWV0ZXJHcm91cCBkZXNjcmlwdGlvbnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVEQlBhcmFtZXRlckdyb3Vwcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZURCUGFyYW1ldGVyR3JvdXBzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZURCUGFyYW1ldGVyR3JvdXBzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIHRoZSBkZXRhaWxlZCBwYXJhbWV0ZXIgbGlzdCBmb3IgYSBwYXJ0aWN1bGFyIERCIHBhcmFtZXRlciBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZURCUGFyYW1ldGVycy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZURCUGFyYW1ldGVycygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVEQlBhcmFtZXRlcnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB2aWV3IHByb3hpZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVEQlByb3hpZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVEQlByb3hpZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlREJQcm94aWVzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdmlldyBwcm94eSBlbmRwb2ludHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVEQlByb3h5RW5kcG9pbnRzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlREJQcm94eUVuZHBvaW50cygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVEQlByb3h5RW5kcG9pbnRzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdmlldyBkYXRhYmFzZSBwcm94eSB0YXJnZXQgZ3JvdXAgZGV0YWlsc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZURCUHJveHlUYXJnZXRHcm91cHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVEQlByb3h5VGFyZ2V0R3JvdXBzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZURCUHJveHlUYXJnZXRHcm91cHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB2aWV3IGRhdGFiYXNlIHByb3h5IHRhcmdldCBkZXRhaWxzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlREJQcm94eVRhcmdldHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVEQlByb3h5VGFyZ2V0cygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVEQlByb3h5VGFyZ2V0cycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgcmVjb21tZW5kYXRpb24gZGV0YWlsc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZURCUmVjb21tZW5kYXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlREJSZWNvbW1lbmRhdGlvbnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlREJSZWNvbW1lbmRhdGlvbnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gYSBsaXN0IG9mIERCU2VjdXJpdHlHcm91cCBkZXNjcmlwdGlvbnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVEQlNlY3VyaXR5R3JvdXBzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlREJTZWN1cml0eUdyb3VwcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVEQlNlY3VyaXR5R3JvdXBzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIGluZm9ybWF0aW9uIGFib3V0IGFsbCBBdXJvcmEgTGltaXRsZXNzIERhdGFiYXNlIERCIHNoYXJkIGdyb3VwcyBmb3IgdGhpcyBhY2NvdW50LiBZb3UgY2FuIGZpbHRlciBieSBzaGFyZCBncm91cChzKVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZURCU2hhcmRHcm91cHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVEQlNoYXJkR3JvdXBzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZURCU2hhcmRHcm91cHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gYSBsaXN0IG9mIERCIHNuYXBzaG90IGF0dHJpYnV0ZSBuYW1lcyBhbmQgdmFsdWVzIGZvciBhIG1hbnVhbCBEQiBzbmFwc2hvdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZURCU25hcHNob3RBdHRyaWJ1dGVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlREJTbmFwc2hvdEF0dHJpYnV0ZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlREJTbmFwc2hvdEF0dHJpYnV0ZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gaW5mb3JtYXRpb24gYWJvdXQgdGVuYW50IGRhdGFiYXNlcyBpbiBEQiBzbmFwc2hvdHMuIFlvdSBjYW4gZmlsdGVyIGJ5IFJlZ2lvbiBvciBzbmFwc2hvdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZURCU25hcHNob3RUZW5hbnREYXRhYmFzZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVEQlNuYXBzaG90VGVuYW50RGF0YWJhc2VzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZURCU25hcHNob3RUZW5hbnREYXRhYmFzZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gaW5mb3JtYXRpb24gYWJvdXQgREIgc25hcHNob3RzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlREJTbmFwc2hvdHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVEQlNuYXBzaG90cygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVEQlNuYXBzaG90cycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHVybiBhIGxpc3Qgb2YgREJTdWJuZXRHcm91cCBkZXNjcmlwdGlvbnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVEQlN1Ym5ldEdyb3Vwcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZURCU3VibmV0R3JvdXBzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZURCU3VibmV0R3JvdXBzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIHRoZSBkZWZhdWx0IGVuZ2luZSBhbmQgc3lzdGVtIHBhcmFtZXRlciBpbmZvcm1hdGlvbiBmb3IgdGhlIGNsdXN0ZXIgZGF0YWJhc2UgZW5naW5lXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlRW5naW5lRGVmYXVsdENsdXN0ZXJQYXJhbWV0ZXJzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlRW5naW5lRGVmYXVsdENsdXN0ZXJQYXJhbWV0ZXJzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZUVuZ2luZURlZmF1bHRDbHVzdGVyUGFyYW1ldGVycycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHVybiB0aGUgZGVmYXVsdCBlbmdpbmUgYW5kIHN5c3RlbSBwYXJhbWV0ZXIgaW5mb3JtYXRpb24gZm9yIHRoZSBzcGVjaWZpZWQgZGF0YWJhc2UgZW5naW5lXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlRW5naW5lRGVmYXVsdFBhcmFtZXRlcnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVFbmdpbmVEZWZhdWx0UGFyYW1ldGVycygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVFbmdpbmVEZWZhdWx0UGFyYW1ldGVycycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRpc3BsYXkgYSBsaXN0IG9mIGNhdGVnb3JpZXMgZm9yIGFsbCBldmVudCBzb3VyY2UgdHlwZXMsIG9yLCBpZiBzcGVjaWZpZWQsIGZvciBhIHNwZWNpZmllZCBzb3VyY2UgdHlwZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUV2ZW50Q2F0ZWdvcmllcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZUV2ZW50Q2F0ZWdvcmllcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVFdmVudENhdGVnb3JpZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGFsbCB0aGUgc3Vic2NyaXB0aW9uIGRlc2NyaXB0aW9ucyBmb3IgYSBjdXN0b21lciBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlRXZlbnRTdWJzY3JpcHRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlRXZlbnRTdWJzY3JpcHRpb25zKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZUV2ZW50U3Vic2NyaXB0aW9ucycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHVybiBldmVudHMgcmVsYXRlZCB0byBEQiBpbnN0YW5jZXMsIERCIHNlY3VyaXR5IGdyb3VwcywgREIgc25hcHNob3RzLCBhbmQgREIgcGFyYW1ldGVyIGdyb3VwcyBmb3IgdGhlIHBhc3QgMTQgZGF5c1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUV2ZW50cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZUV2ZW50cygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVFdmVudHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gaW5mb3JtYXRpb24gYWJvdXQgdGhlIGV4cG9ydCB0YXNrc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUV4cG9ydFRhc2tzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlRXhwb3J0VGFza3MoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlRXhwb3J0VGFza3MnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gaW5mb3JtYXRpb24gYWJvdXQgQXVyb3JhIGdsb2JhbCBkYXRhYmFzZSBjbHVzdGVycyBvciBEb2N1bWVudERCIGdsb2JhbCBkYXRhYmFzZSBjbHVzdGVyc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUdsb2JhbENsdXN0ZXJzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlR2xvYmFsQ2x1c3RlcnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlR2xvYmFsQ2x1c3RlcnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZXNjcmliZSBhbiBBdXJvcmEgemVyby1FVEwgaW50ZWdyYXRpb24gd2l0aCBSZWRzaGlmdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVJbnRlZ3JhdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVJbnRlZ3JhdGlvbnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlSW50ZWdyYXRpb25zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVzY3JpYmUgYWxsIGF2YWlsYWJsZSBvcHRpb25zXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlT3B0aW9uR3JvdXBPcHRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlT3B0aW9uR3JvdXBPcHRpb25zKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZU9wdGlvbkdyb3VwT3B0aW9ucycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlc2NyaWJlIHRoZSBhdmFpbGFibGUgb3B0aW9uIGdyb3Vwc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZU9wdGlvbkdyb3Vwcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZU9wdGlvbkdyb3VwcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVPcHRpb25Hcm91cHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gYSBsaXN0IG9mIG9yZGVyYWJsZSBEQiBpbnN0YW5jZSBvcHRpb25zIGZvciB0aGUgc3BlY2lmaWVkIGVuZ2luZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZU9yZGVyYWJsZURCSW5zdGFuY2VPcHRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlT3JkZXJhYmxlREJJbnN0YW5jZU9wdGlvbnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlT3JkZXJhYmxlREJJbnN0YW5jZU9wdGlvbnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gYSBsaXN0IG9mIHJlc291cmNlcyAoZm9yIGV4YW1wbGUsIERCIGluc3RhbmNlcykgdGhhdCBoYXZlIGF0IGxlYXN0IG9uZSBwZW5kaW5nIG1haW50ZW5hbmNlIGFjdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVBlbmRpbmdNYWludGVuYW5jZUFjdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVQZW5kaW5nTWFpbnRlbmFuY2VBY3Rpb25zKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZVBlbmRpbmdNYWludGVuYW5jZUFjdGlvbnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gaW5mb3JtYXRpb24gYWJvdXQgcmVjb21tZW5kYXRpb24gZ3JvdXBzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9VU0VSX1JlY29tbWVuZGF0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZVJlY29tbWVuZGF0aW9uR3JvdXBzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZVJlY29tbWVuZGF0aW9uR3JvdXBzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIGluZm9ybWF0aW9uIGFib3V0IHJlY29tbWVuZGF0aW9uc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvVVNFUl9SZWNvbW1lbmRhdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVSZWNvbW1lbmRhdGlvbnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlUmVjb21tZW5kYXRpb25zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIGluZm9ybWF0aW9uIGFib3V0IHJlc2VydmVkIERCIGluc3RhbmNlcyBmb3IgdGhpcyBhY2NvdW50LCBvciBhYm91dCBhIHNwZWNpZmllZCByZXNlcnZlZCBEQiBpbnN0YW5jZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVJlc2VydmVkREJJbnN0YW5jZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVSZXNlcnZlZERCSW5zdGFuY2VzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZVJlc2VydmVkREJJbnN0YW5jZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGF2YWlsYWJsZSByZXNlcnZlZCBEQiBpbnN0YW5jZSBvZmZlcmluZ3NcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVSZXNlcnZlZERCSW5zdGFuY2VzT2ZmZXJpbmdzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlUmVzZXJ2ZWREQkluc3RhbmNlc09mZmVyaW5ncygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVSZXNlcnZlZERCSW5zdGFuY2VzT2ZmZXJpbmdzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIGEgbGlzdCBvZiB0aGUgc291cmNlIEFXUyBSZWdpb25zIHdoZXJlIHRoZSBjdXJyZW50IEFXUyBSZWdpb24gY2FuIGNyZWF0ZSBhIFJlYWQgUmVwbGljYSBvciBjb3B5IGEgREIgc25hcHNob3QgZnJvbVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVNvdXJjZVJlZ2lvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVTb3VyY2VSZWdpb25zKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZVNvdXJjZVJlZ2lvbnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gaW5mb3JtYXRpb24gYWJvdXQgcHJvdmlzaW9uZWQgdGVuYW50IGRhdGFiYXNlcy4gWW91IGNhbiBmaWx0ZXIgYnkgUmVnaW9uIG9yIHNuYXBzaG90XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlVGVuYW50RGF0YWJhc2VzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlVGVuYW50RGF0YWJhc2VzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZVRlbmFudERhdGFiYXNlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgYXZhaWxhYmxlIG1vZGlmaWNhdGlvbnMgeW91IGNhbiBtYWtlIHRvIHlvdXIgREIgaW5zdGFuY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVWYWxpZERCSW5zdGFuY2VNb2RpZmljYXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlVmFsaWREQkluc3RhbmNlTW9kaWZpY2F0aW9ucygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVWYWxpZERCSW5zdGFuY2VNb2RpZmljYXRpb25zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGlzYWJsZSBodHRwIGVuZHBvaW50IGZvciBhIERCIGNsdXN0ZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rpc2FibGVIdHRwRW5kcG9pbnQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGlzYWJsZUh0dHBFbmRwb2ludCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGlzYWJsZUh0dHBFbmRwb2ludCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRvd25sb2FkIHNwZWNpZmllZCBsb2cgZmlsZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvVVNFUl9Mb2dBY2Nlc3MuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRG93bmxvYWRDb21wbGV0ZURCTG9nRmlsZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRG93bmxvYWRDb21wbGV0ZURCTG9nRmlsZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRvd25sb2FkIGFsbCBvciBhIHBvcnRpb24gb2YgdGhlIHNwZWNpZmllZCBsb2cgZmlsZSwgdXAgdG8gMSBNQiBpbiBzaXplXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rvd25sb2FkREJMb2dGaWxlUG9ydGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9Eb3dubG9hZERCTG9nRmlsZVBvcnRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rvd25sb2FkREJMb2dGaWxlUG9ydGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGVuYWJsZSBodHRwIGVuZHBvaW50IGZvciBhIERCIGNsdXN0ZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0VuYWJsZUh0dHBFbmRwb2ludC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9FbmFibGVIdHRwRW5kcG9pbnQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0VuYWJsZUh0dHBFbmRwb2ludCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGZvcmNlIGEgZmFpbG92ZXIgZm9yIGEgREIgY2x1c3RlclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRmFpbG92ZXJEQkNsdXN0ZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRmFpbG92ZXJEQkNsdXN0ZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0ZhaWxvdmVyREJDbHVzdGVyJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZmFpbG92ZXIgYSBnbG9iYWwgY2x1c3RlclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRmFpbG92ZXJHbG9iYWxDbHVzdGVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0ZhaWxvdmVyR2xvYmFsQ2x1c3RlcigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRmFpbG92ZXJHbG9iYWxDbHVzdGVyJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBhbGwgdGFncyBvbiBhbiBBbWF6b24gUkRTIHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RUYWdzRm9yUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFRhZ3NGb3JSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFRhZ3NGb3JSZXNvdXJjZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIG1vZGlmeSBhIGRhdGFiYXNlIGFjdGl2aXR5IHN0cmVhbVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTW9kaWZ5QWN0aXZpdHlTdHJlYW0uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTW9kaWZ5QWN0aXZpdHlTdHJlYW0oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ01vZGlmeUFjdGl2aXR5U3RyZWFtJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbW9kaWZ5IHRoZSBzeXN0ZW0tZGVmYXVsdCBTZWN1cmUgU29ja2V0cyBMYXllci9UcmFuc3BvcnQgTGF5ZXIgU2VjdXJpdHkgKFNTTC9UTFMpIGNlcnRpZmljYXRlIGZvciBBbWF6b24gUkRTIGZvciBuZXcgREIgaW5zdGFuY2VzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9Nb2RpZnlDZXJ0aWZpY2F0ZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTW9kaWZ5Q2VydGlmaWNhdGVzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdNb2RpZnlDZXJ0aWZpY2F0ZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBtb2RpZnkgY3VycmVudCBjbHVzdGVyIGNhcGFjaXR5IGZvciBhbiBBbWF6b24gQXVyb3JhIFNlcnZlcmxlc3MgREIgY2x1c3RlclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTW9kaWZ5Q3VycmVudERCQ2x1c3RlckNhcGFjaXR5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b01vZGlmeUN1cnJlbnREQkNsdXN0ZXJDYXBhY2l0eSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTW9kaWZ5Q3VycmVudERCQ2x1c3RlckNhcGFjaXR5Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbW9kaWZ5IGFuIGV4aXN0aW5nIGN1c3RvbSBlbmdpbmUgdmVyc2lvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTW9kaWZ5Q3VzdG9tREJFbmdpbmVWZXJzaW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b01vZGlmeUN1c3RvbURCRW5naW5lVmVyc2lvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTW9kaWZ5Q3VzdG9tREJFbmdpbmVWZXJzaW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbW9kaWZ5IGEgc2V0dGluZyBmb3IgYW4gQW1hem9uIEF1cm9yYSBEQiBjbHVzdGVyIG9yIEFtYXpvbiBEb2N1bWVudERCIGNsdXN0ZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkRhdGFiYXNlQ2xhc3MoKVxuICAgKiAtIC5pZlN0b3JhZ2VTaXplKClcbiAgICogLSAuaWZQaW9wcygpXG4gICAqIC0gLmlmTWFuYWdlTWFzdGVyVXNlclBhc3N3b3JkKClcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gaWFtOlBhc3NSb2xlXG4gICAqIC0ga21zOkNyZWF0ZUdyYW50XG4gICAqIC0ga21zOkRlY3J5cHRcbiAgICogLSBrbXM6RGVzY3JpYmVLZXlcbiAgICogLSBrbXM6R2VuZXJhdGVEYXRhS2V5XG4gICAqIC0gcmRzOk1vZGlmeURCSW5zdGFuY2VcbiAgICogLSBzZWNyZXRzbWFuYWdlcjpDcmVhdGVTZWNyZXRcbiAgICogLSBzZWNyZXRzbWFuYWdlcjpSb3RhdGVTZWNyZXRcbiAgICogLSBzZWNyZXRzbWFuYWdlcjpUYWdSZXNvdXJjZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX01vZGlmeURCQ2x1c3Rlci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9Nb2RpZnlEQkNsdXN0ZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ01vZGlmeURCQ2x1c3RlcicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIG1vZGlmeSB0aGUgcHJvcGVydGllcyBvZiBhbiBlbmRwb2ludCBpbiBhbiBBbWF6b24gQXVyb3JhIERCIGNsdXN0ZXIgb3IgQW1hem9uIERvY3VtZW50REIgY2x1c3RlclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTW9kaWZ5REJDbHVzdGVyRW5kcG9pbnQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTW9kaWZ5REJDbHVzdGVyRW5kcG9pbnQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ01vZGlmeURCQ2x1c3RlckVuZHBvaW50Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbW9kaWZ5IHRoZSBwYXJhbWV0ZXJzIG9mIGEgREIgY2x1c3RlciBwYXJhbWV0ZXIgZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX01vZGlmeURCQ2x1c3RlclBhcmFtZXRlckdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b01vZGlmeURCQ2x1c3RlclBhcmFtZXRlckdyb3VwKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdNb2RpZnlEQkNsdXN0ZXJQYXJhbWV0ZXJHcm91cCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFkZCBhbiBhdHRyaWJ1dGUgYW5kIHZhbHVlcyB0bywgb3IgcmVtb3ZlcyBhbiBhdHRyaWJ1dGUgYW5kIHZhbHVlcyBmcm9tLCBhIG1hbnVhbCBEQiBjbHVzdGVyIHNuYXBzaG90XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9Nb2RpZnlEQkNsdXN0ZXJTbmFwc2hvdEF0dHJpYnV0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9Nb2RpZnlEQkNsdXN0ZXJTbmFwc2hvdEF0dHJpYnV0ZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTW9kaWZ5REJDbHVzdGVyU25hcHNob3RBdHRyaWJ1dGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBtb2RpZnkgc2V0dGluZ3MgZm9yIGEgREIgaW5zdGFuY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZk1hbmFnZU1hc3RlclVzZXJQYXNzd29yZCgpXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGlhbTpQYXNzUm9sZVxuICAgKiAtIGttczpDcmVhdGVHcmFudFxuICAgKiAtIGttczpEZWNyeXB0XG4gICAqIC0ga21zOkRlc2NyaWJlS2V5XG4gICAqIC0ga21zOkdlbmVyYXRlRGF0YUtleVxuICAgKiAtIHJkczpBZGRUYWdzVG9SZXNvdXJjZVxuICAgKiAtIHJkczpDcmVhdGVUZW5hbnREYXRhYmFzZVxuICAgKiAtIHNlY3JldHNtYW5hZ2VyOkNyZWF0ZVNlY3JldFxuICAgKiAtIHNlY3JldHNtYW5hZ2VyOlJvdGF0ZVNlY3JldFxuICAgKiAtIHNlY3JldHNtYW5hZ2VyOlRhZ1Jlc291cmNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTW9kaWZ5REJJbnN0YW5jZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9Nb2RpZnlEQkluc3RhbmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdNb2RpZnlEQkluc3RhbmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbW9kaWZ5IHRoZSBwYXJhbWV0ZXJzIG9mIGEgREIgcGFyYW1ldGVyIGdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9Nb2RpZnlEQlBhcmFtZXRlckdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b01vZGlmeURCUGFyYW1ldGVyR3JvdXAoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ01vZGlmeURCUGFyYW1ldGVyR3JvdXAnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBtb2RpZnkgZGF0YWJhc2UgcHJveHlcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBpYW06UGFzc1JvbGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9Nb2RpZnlEQlByb3h5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b01vZGlmeURCUHJveHkoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ01vZGlmeURCUHJveHknKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBtb2RpZnkgZGF0YWJhc2UgcHJveHkgZW5kcG9pbnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX01vZGlmeURCUHJveHlFbmRwb2ludC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9Nb2RpZnlEQlByb3h5RW5kcG9pbnQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ01vZGlmeURCUHJveHlFbmRwb2ludCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIG1vZGlmeSB0YXJnZXQgZ3JvdXAgZm9yIGEgZGF0YWJhc2UgcHJveHlcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX01vZGlmeURCUHJveHlUYXJnZXRHcm91cC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9Nb2RpZnlEQlByb3h5VGFyZ2V0R3JvdXAoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ01vZGlmeURCUHJveHlUYXJnZXRHcm91cCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIG1vZGlmeSByZWNvbW1lbmRhdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTW9kaWZ5REJSZWNvbW1lbmRhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9Nb2RpZnlEQlJlY29tbWVuZGF0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdNb2RpZnlEQlJlY29tbWVuZGF0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbW9kaWZ5IHByb3BlcnRpZXMgb2YgYW4gQXVyb3JhIExpbWl0bGVzcyBEYXRhYmFzZSBEQiBzaGFyZCBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTW9kaWZ5REJTaGFyZEdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b01vZGlmeURCU2hhcmRHcm91cCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTW9kaWZ5REJTaGFyZEdyb3VwJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGEgbWFudWFsIERCIHNuYXBzaG90LCB3aGljaCBjYW4gYmUgZW5jcnlwdGVkIG9yIG5vdCBlbmNyeXB0ZWQsIHdpdGggYSBuZXcgZW5naW5lIHZlcnNpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX01vZGlmeURCU25hcHNob3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTW9kaWZ5REJTbmFwc2hvdCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTW9kaWZ5REJTbmFwc2hvdCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFkZCBhbiBhdHRyaWJ1dGUgYW5kIHZhbHVlcyB0bywgb3IgcmVtb3ZlcyBhbiBhdHRyaWJ1dGUgYW5kIHZhbHVlcyBmcm9tLCBhIG1hbnVhbCBEQiBzbmFwc2hvdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTW9kaWZ5REJTbmFwc2hvdEF0dHJpYnV0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9Nb2RpZnlEQlNuYXBzaG90QXR0cmlidXRlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdNb2RpZnlEQlNuYXBzaG90QXR0cmlidXRlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbW9kaWZ5IGFuIGV4aXN0aW5nIERCIHN1Ym5ldCBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTW9kaWZ5REJTdWJuZXRHcm91cC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9Nb2RpZnlEQlN1Ym5ldEdyb3VwKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdNb2RpZnlEQlN1Ym5ldEdyb3VwJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbW9kaWZ5IGFuIGV4aXN0aW5nIFJEUyBldmVudCBub3RpZmljYXRpb24gc3Vic2NyaXB0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9Nb2RpZnlFdmVudFN1YnNjcmlwdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9Nb2RpZnlFdmVudFN1YnNjcmlwdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTW9kaWZ5RXZlbnRTdWJzY3JpcHRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBtb2RpZnkgYSBzZXR0aW5nIGZvciBhbiBBbWF6b24gQXVyb3JhIGdsb2JhbCBjbHVzdGVyIG9yIEFtYXpvbiBEb2N1bWVudERCIGdsb2JhbCBjbHVzdGVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9Nb2RpZnlHbG9iYWxDbHVzdGVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b01vZGlmeUdsb2JhbENsdXN0ZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ01vZGlmeUdsb2JhbENsdXN0ZXInKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBtb2RpZnkgYW4gQXVyb3JhIHplcm8tRVRMIGludGVncmF0aW9uIHdpdGggUmVkc2hpZnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX01vZGlmeUludGVncmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b01vZGlmeUludGVncmF0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdNb2RpZnlJbnRlZ3JhdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIG1vZGlmeSBhbiBleGlzdGluZyBvcHRpb24gZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBpYW06UGFzc1JvbGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9Nb2RpZnlPcHRpb25Hcm91cC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9Nb2RpZnlPcHRpb25Hcm91cCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTW9kaWZ5T3B0aW9uR3JvdXAnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBtb2RpZnkgcmVjb21tZW5kYXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9VU0VSX1JlY29tbWVuZGF0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9Nb2RpZnlSZWNvbW1lbmRhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTW9kaWZ5UmVjb21tZW5kYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBtb2RpZnkgYSB0ZW5hbnQgZGF0YWJhc2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZlRlbmFudERhdGFiYXNlTmFtZSgpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTW9kaWZ5VGVuYW50RGF0YWJhc2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTW9kaWZ5VGVuYW50RGF0YWJhc2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ01vZGlmeVRlbmFudERhdGFiYXNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcHJvbW90ZSBhIFJlYWQgUmVwbGljYSBEQiBpbnN0YW5jZSB0byBhIHN0YW5kYWxvbmUgREIgaW5zdGFuY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1Byb21vdGVSZWFkUmVwbGljYS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9Qcm9tb3RlUmVhZFJlcGxpY2EoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1Byb21vdGVSZWFkUmVwbGljYScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHByb21vdGUgYSBSZWFkIFJlcGxpY2EgREIgY2x1c3RlciB0byBhIHN0YW5kYWxvbmUgREIgY2x1c3RlclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUHJvbW90ZVJlYWRSZXBsaWNhREJDbHVzdGVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1Byb21vdGVSZWFkUmVwbGljYURCQ2x1c3RlcigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnUHJvbW90ZVJlYWRSZXBsaWNhREJDbHVzdGVyJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcHVyY2hhc2UgYSByZXNlcnZlZCBEQiBpbnN0YW5jZSBvZmZlcmluZ1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUHVyY2hhc2VSZXNlcnZlZERCSW5zdGFuY2VzT2ZmZXJpbmcuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUHVyY2hhc2VSZXNlcnZlZERCSW5zdGFuY2VzT2ZmZXJpbmcoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1B1cmNoYXNlUmVzZXJ2ZWREQkluc3RhbmNlc09mZmVyaW5nJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVib290IGEgcHJldmlvdXNseSBwcm92aXNpb25lZCBEQiBjbHVzdGVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gcmRzOlJlYm9vdERCSW5zdGFuY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9SZWJvb3REQkNsdXN0ZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUmVib290REJDbHVzdGVyKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdSZWJvb3REQkNsdXN0ZXInKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXN0YXJ0IHRoZSBkYXRhYmFzZSBlbmdpbmUgc2VydmljZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUmVib290REJJbnN0YW5jZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9SZWJvb3REQkluc3RhbmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdSZWJvb3REQkluc3RhbmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVib290IGFuIEF1cm9yYSBMaW1pdGxlc3MgRGF0YWJhc2UgREIgc2hhcmQgZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1JlYm9vdERCU2hhcmRHcm91cC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9SZWJvb3REQlNoYXJkR3JvdXAoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1JlYm9vdERCU2hhcmRHcm91cCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFkZCB0YXJnZXRzIHRvIGEgZGF0YWJhc2UgcHJveHkgdGFyZ2V0IGdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9SZWdpc3RlckRCUHJveHlUYXJnZXRzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1JlZ2lzdGVyREJQcm94eVRhcmdldHMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1JlZ2lzdGVyREJQcm94eVRhcmdldHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZXRhY2ggYW4gQXVyb3JhIHNlY29uZGFyeSBjbHVzdGVyIGZyb20gYW4gQXVyb3JhIGdsb2JhbCBkYXRhYmFzZSBjbHVzdGVyIG9yIERvY3VtZW50REIgZ2xvYmFsIGNsdXN0ZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1JlbW92ZUZyb21HbG9iYWxDbHVzdGVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1JlbW92ZUZyb21HbG9iYWxDbHVzdGVyKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdSZW1vdmVGcm9tR2xvYmFsQ2x1c3RlcicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRpc2Fzc29jaWF0ZSBhbiBBV1MgSWRlbnRpdHkgYW5kIEFjY2VzcyBNYW5hZ2VtZW50IChJQU0pIHJvbGUgZnJvbSBhbiBBbWF6b24gQXVyb3JhIERCIGNsdXN0ZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBpYW06UGFzc1JvbGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9SZW1vdmVSb2xlRnJvbURCQ2x1c3Rlci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9SZW1vdmVSb2xlRnJvbURCQ2x1c3RlcigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnUmVtb3ZlUm9sZUZyb21EQkNsdXN0ZXInKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkaXNhc3NvY2lhdGUgYW4gQVdTIElkZW50aXR5IGFuZCBBY2Nlc3MgTWFuYWdlbWVudCAoSUFNKSByb2xlIGZyb20gYSBEQiBpbnN0YW5jZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGlhbTpQYXNzUm9sZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1JlbW92ZVJvbGVGcm9tREJJbnN0YW5jZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9SZW1vdmVSb2xlRnJvbURCSW5zdGFuY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1JlbW92ZVJvbGVGcm9tREJJbnN0YW5jZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJlbW92ZSBhIHNvdXJjZSBpZGVudGlmaWVyIGZyb20gYW4gZXhpc3RpbmcgUkRTIGV2ZW50IG5vdGlmaWNhdGlvbiBzdWJzY3JpcHRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1JlbW92ZVNvdXJjZUlkZW50aWZpZXJGcm9tU3Vic2NyaXB0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1JlbW92ZVNvdXJjZUlkZW50aWZpZXJGcm9tU3Vic2NyaXB0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdSZW1vdmVTb3VyY2VJZGVudGlmaWVyRnJvbVN1YnNjcmlwdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJlbW92ZSBtZXRhZGF0YSB0YWdzIGZyb20gYW4gQW1hem9uIFJEUyByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICogLSAuaWZSZXFUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1JlbW92ZVRhZ3NGcm9tUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUmVtb3ZlVGFnc0Zyb21SZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnUmVtb3ZlVGFnc0Zyb21SZXNvdXJjZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIG1vZGlmeSB0aGUgcGFyYW1ldGVycyBvZiBhIERCIGNsdXN0ZXIgcGFyYW1ldGVyIGdyb3VwIHRvIHRoZSBkZWZhdWx0IHZhbHVlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9SZXNldERCQ2x1c3RlclBhcmFtZXRlckdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1Jlc2V0REJDbHVzdGVyUGFyYW1ldGVyR3JvdXAoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1Jlc2V0REJDbHVzdGVyUGFyYW1ldGVyR3JvdXAnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBtb2RpZnkgdGhlIHBhcmFtZXRlcnMgb2YgYSBEQiBwYXJhbWV0ZXIgZ3JvdXAgdG8gdGhlIGVuZ2luZS9zeXN0ZW0gZGVmYXVsdCB2YWx1ZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUmVzZXREQlBhcmFtZXRlckdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1Jlc2V0REJQYXJhbWV0ZXJHcm91cCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnUmVzZXREQlBhcmFtZXRlckdyb3VwJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGFuIEFtYXpvbiBBdXJvcmEgREIgY2x1c3RlciBmcm9tIGRhdGEgc3RvcmVkIGluIGFuIEFtYXpvbiBTMyBidWNrZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKiAtIC5pZlJlcVRhZygpXG4gICAqIC0gLmlmRGF0YWJhc2VFbmdpbmUoKVxuICAgKiAtIC5pZkRhdGFiYXNlTmFtZSgpXG4gICAqIC0gLmlmU3RvcmFnZUVuY3J5cHRlZCgpXG4gICAqIC0gLmlmTWFuYWdlTWFzdGVyVXNlclBhc3N3b3JkKClcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gaWFtOlBhc3NSb2xlXG4gICAqIC0ga21zOkNyZWF0ZUdyYW50XG4gICAqIC0ga21zOkRlY3J5cHRcbiAgICogLSBrbXM6RGVzY3JpYmVLZXlcbiAgICogLSBrbXM6R2VuZXJhdGVEYXRhS2V5XG4gICAqIC0gcmRzOkFkZFRhZ3NUb1Jlc291cmNlXG4gICAqIC0gc2VjcmV0c21hbmFnZXI6Q3JlYXRlU2VjcmV0XG4gICAqIC0gc2VjcmV0c21hbmFnZXI6VGFnUmVzb3VyY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9SZXN0b3JlREJDbHVzdGVyRnJvbVMzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1Jlc3RvcmVEQkNsdXN0ZXJGcm9tUzMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1Jlc3RvcmVEQkNsdXN0ZXJGcm9tUzMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBuZXcgREIgY2x1c3RlciBmcm9tIGEgREIgY2x1c3RlciBzbmFwc2hvdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqIC0gLmlmUmVxVGFnKClcbiAgICogLSAuaWZEYXRhYmFzZUNsYXNzKClcbiAgICogLSAuaWZTdG9yYWdlU2l6ZSgpXG4gICAqIC0gLmlmUGlvcHMoKVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBpYW06UGFzc1JvbGVcbiAgICogLSByZHM6QWRkVGFnc1RvUmVzb3VyY2VcbiAgICogLSByZHM6Q3JlYXRlREJJbnN0YW5jZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1Jlc3RvcmVEQkNsdXN0ZXJGcm9tU25hcHNob3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUmVzdG9yZURCQ2x1c3RlckZyb21TbmFwc2hvdCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnUmVzdG9yZURCQ2x1c3RlckZyb21TbmFwc2hvdCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJlc3RvcmUgYSBEQiBjbHVzdGVyIHRvIGFuIGFyYml0cmFyeSBwb2ludCBpbiB0aW1lXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICogLSAuaWZSZXFUYWcoKVxuICAgKiAtIC5pZkRhdGFiYXNlQ2xhc3MoKVxuICAgKiAtIC5pZlN0b3JhZ2VTaXplKClcbiAgICogLSAuaWZQaW9wcygpXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGlhbTpQYXNzUm9sZVxuICAgKiAtIHJkczpBZGRUYWdzVG9SZXNvdXJjZVxuICAgKiAtIHJkczpDcmVhdGVEQkluc3RhbmNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUmVzdG9yZURCQ2x1c3RlclRvUG9pbnRJblRpbWUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUmVzdG9yZURCQ2x1c3RlclRvUG9pbnRJblRpbWUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1Jlc3RvcmVEQkNsdXN0ZXJUb1BvaW50SW5UaW1lJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgbmV3IERCIGluc3RhbmNlIGZyb20gYSBEQiBzbmFwc2hvdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQmFja3VwVGFyZ2V0KClcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICogLSAuaWZSZXFUYWcoKVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBpYW06UGFzc1JvbGVcbiAgICogLSByZHM6QWRkVGFnc1RvUmVzb3VyY2VcbiAgICogLSByZHM6Q3JlYXRlVGVuYW50RGF0YWJhc2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9SZXN0b3JlREJJbnN0YW5jZUZyb21EQlNuYXBzaG90Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1Jlc3RvcmVEQkluc3RhbmNlRnJvbURCU25hcHNob3QoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1Jlc3RvcmVEQkluc3RhbmNlRnJvbURCU25hcHNob3QnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBuZXcgREIgaW5zdGFuY2UgZnJvbSBhbiBBbWF6b24gUzMgYnVja2V0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICogLSAuaWZSZXFUYWcoKVxuICAgKiAtIC5pZk1hbmFnZU1hc3RlclVzZXJQYXNzd29yZCgpXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGlhbTpQYXNzUm9sZVxuICAgKiAtIGttczpDcmVhdGVHcmFudFxuICAgKiAtIGttczpEZWNyeXB0XG4gICAqIC0ga21zOkRlc2NyaWJlS2V5XG4gICAqIC0ga21zOkdlbmVyYXRlRGF0YUtleVxuICAgKiAtIHJkczpBZGRUYWdzVG9SZXNvdXJjZVxuICAgKiAtIHNlY3JldHNtYW5hZ2VyOkNyZWF0ZVNlY3JldFxuICAgKiAtIHNlY3JldHNtYW5hZ2VyOlRhZ1Jlc291cmNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUmVzdG9yZURCSW5zdGFuY2VGcm9tUzMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUmVzdG9yZURCSW5zdGFuY2VGcm9tUzMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1Jlc3RvcmVEQkluc3RhbmNlRnJvbVMzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVzdG9yZSBhIERCIGluc3RhbmNlIHRvIGFuIGFyYml0cmFyeSBwb2ludCBpbiB0aW1lXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZCYWNrdXBUYXJnZXQoKVxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKiAtIC5pZlJlcVRhZygpXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGlhbTpQYXNzUm9sZVxuICAgKiAtIHJkczpBZGRUYWdzVG9SZXNvdXJjZVxuICAgKiAtIHJkczpDcmVhdGVUZW5hbnREYXRhYmFzZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1Jlc3RvcmVEQkluc3RhbmNlVG9Qb2ludEluVGltZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9SZXN0b3JlREJJbnN0YW5jZVRvUG9pbnRJblRpbWUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1Jlc3RvcmVEQkluc3RhbmNlVG9Qb2ludEluVGltZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldm9rZSBpbmdyZXNzIGZyb20gYSBEQlNlY3VyaXR5R3JvdXAgZm9yIHByZXZpb3VzbHkgYXV0aG9yaXplZCBJUCByYW5nZXMgb3IgRUMyIG9yIFZQQyBTZWN1cml0eSBHcm91cHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1Jldm9rZURCU2VjdXJpdHlHcm91cEluZ3Jlc3MuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUmV2b2tlREJTZWN1cml0eUdyb3VwSW5ncmVzcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnUmV2b2tlREJTZWN1cml0eUdyb3VwSW5ncmVzcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHN0YXJ0IEFjdGl2aXR5IFN0cmVhbVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU3RhcnRBY3Rpdml0eVN0cmVhbS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TdGFydEFjdGl2aXR5U3RyZWFtKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdGFydEFjdGl2aXR5U3RyZWFtJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3RhcnQgdGhlIERCIGNsdXN0ZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N0YXJ0REJDbHVzdGVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0YXJ0REJDbHVzdGVyKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdGFydERCQ2x1c3RlcicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHN0YXJ0IHRoZSBEQiBpbnN0YW5jZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU3RhcnREQkluc3RhbmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0YXJ0REJJbnN0YW5jZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnU3RhcnREQkluc3RhbmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3RhcnQgcmVwbGljYXRpb24gb2YgYXV0b21hdGVkIGJhY2t1cHMgdG8gYSBkaWZmZXJlbnQgQVdTIFJlZ2lvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU3RhcnREQkluc3RhbmNlQXV0b21hdGVkQmFja3Vwc1JlcGxpY2F0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0YXJ0REJJbnN0YW5jZUF1dG9tYXRlZEJhY2t1cHNSZXBsaWNhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnU3RhcnREQkluc3RhbmNlQXV0b21hdGVkQmFja3Vwc1JlcGxpY2F0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3RhcnQgYSBuZXcgRXhwb3J0IHRhc2sgZm9yIGEgREIgc25hcHNob3RcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBpYW06UGFzc1JvbGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TdGFydEV4cG9ydFRhc2suaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU3RhcnRFeHBvcnRUYXNrKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdGFydEV4cG9ydFRhc2snKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBzdG9wIEFjdGl2aXR5IFN0cmVhbVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU3RvcEFjdGl2aXR5U3RyZWFtLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0b3BBY3Rpdml0eVN0cmVhbSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnU3RvcEFjdGl2aXR5U3RyZWFtJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3RvcCB0aGUgREIgY2x1c3RlclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU3RvcERCQ2x1c3Rlci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TdG9wREJDbHVzdGVyKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdG9wREJDbHVzdGVyJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3RvcCB0aGUgREIgaW5zdGFuY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSByZHM6QWRkVGFnc1RvUmVzb3VyY2VcbiAgICogLSByZHM6Q3JlYXRlREJTbmFwc2hvdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N0b3BEQkluc3RhbmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0b3BEQkluc3RhbmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdG9wREJJbnN0YW5jZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHN0b3AgYXV0b21hdGVkIGJhY2t1cCByZXBsaWNhdGlvbiBmb3IgYSBEQiBpbnN0YW5jZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU3RvcERCSW5zdGFuY2VBdXRvbWF0ZWRCYWNrdXBzUmVwbGljYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU3RvcERCSW5zdGFuY2VBdXRvbWF0ZWRCYWNrdXBzUmVwbGljYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1N0b3BEQkluc3RhbmNlQXV0b21hdGVkQmFja3Vwc1JlcGxpY2F0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3dpdGNoIGEgYmx1ZS1ncmVlbiBkZXBsb3ltZW50IGZyb20gc291cmNlIGluc3RhbmNlIG9yIGNsdXN0ZXIgdG8gdGFyZ2V0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIHJkczpNb2RpZnlEQkNsdXN0ZXJcbiAgICogLSByZHM6TW9kaWZ5REJJbnN0YW5jZVxuICAgKiAtIHJkczpQcm9tb3RlUmVhZFJlcGxpY2FcbiAgICogLSByZHM6UHJvbW90ZVJlYWRSZXBsaWNhREJDbHVzdGVyXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU3dpdGNob3ZlckJsdWVHcmVlbkRlcGxveW1lbnQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU3dpdGNob3ZlckJsdWVHcmVlbkRlcGxveW1lbnQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1N3aXRjaG92ZXJCbHVlR3JlZW5EZXBsb3ltZW50Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3dpdGNob3ZlciBhIGdsb2JhbCBjbHVzdGVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9Td2l0Y2hvdmVyR2xvYmFsQ2x1c3Rlci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9Td2l0Y2hvdmVyR2xvYmFsQ2x1c3RlcigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnU3dpdGNob3Zlckdsb2JhbENsdXN0ZXInKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBzd2l0Y2ggb3ZlciBhIHJlYWQgcmVwbGljYSwgbWFraW5nIGl0IHRoZSBuZXcgcHJpbWFyeSBkYXRhYmFzZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU3dpdGNob3ZlclJlYWRSZXBsaWNhLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N3aXRjaG92ZXJSZWFkUmVwbGljYSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnU3dpdGNob3ZlclJlYWRSZXBsaWNhJyk7XG4gIH1cblxuICBwcm90ZWN0ZWQgYWNjZXNzTGV2ZWxMaXN0OiBBY2Nlc3NMZXZlbExpc3QgPSB7XG4gICAgV3JpdGU6IFtcbiAgICAgICdBZGRSb2xlVG9EQkNsdXN0ZXInLFxuICAgICAgJ0FkZFJvbGVUb0RCSW5zdGFuY2UnLFxuICAgICAgJ0FkZFNvdXJjZUlkZW50aWZpZXJUb1N1YnNjcmlwdGlvbicsXG4gICAgICAnQXBwbHlQZW5kaW5nTWFpbnRlbmFuY2VBY3Rpb24nLFxuICAgICAgJ0JhY2t0cmFja0RCQ2x1c3RlcicsXG4gICAgICAnQ2FuY2VsRXhwb3J0VGFzaycsXG4gICAgICAnQ29weUN1c3RvbURCRW5naW5lVmVyc2lvbicsXG4gICAgICAnQ29weURCQ2x1c3RlclBhcmFtZXRlckdyb3VwJyxcbiAgICAgICdDb3B5REJDbHVzdGVyU25hcHNob3QnLFxuICAgICAgJ0NvcHlEQlBhcmFtZXRlckdyb3VwJyxcbiAgICAgICdDb3B5REJTbmFwc2hvdCcsXG4gICAgICAnQ29weU9wdGlvbkdyb3VwJyxcbiAgICAgICdDcmVhdGVCbHVlR3JlZW5EZXBsb3ltZW50JyxcbiAgICAgICdDcmVhdGVDdXN0b21EQkVuZ2luZVZlcnNpb24nLFxuICAgICAgJ0NyZWF0ZURCQ2x1c3RlcicsXG4gICAgICAnQ3JlYXRlREJDbHVzdGVyRW5kcG9pbnQnLFxuICAgICAgJ0NyZWF0ZURCQ2x1c3RlclBhcmFtZXRlckdyb3VwJyxcbiAgICAgICdDcmVhdGVEQkNsdXN0ZXJTbmFwc2hvdCcsXG4gICAgICAnQ3JlYXRlREJJbnN0YW5jZScsXG4gICAgICAnQ3JlYXRlREJJbnN0YW5jZVJlYWRSZXBsaWNhJyxcbiAgICAgICdDcmVhdGVEQlBhcmFtZXRlckdyb3VwJyxcbiAgICAgICdDcmVhdGVEQlByb3h5JyxcbiAgICAgICdDcmVhdGVEQlByb3h5RW5kcG9pbnQnLFxuICAgICAgJ0NyZWF0ZURCU2VjdXJpdHlHcm91cCcsXG4gICAgICAnQ3JlYXRlREJTaGFyZEdyb3VwJyxcbiAgICAgICdDcmVhdGVEQlNuYXBzaG90JyxcbiAgICAgICdDcmVhdGVEQlN1Ym5ldEdyb3VwJyxcbiAgICAgICdDcmVhdGVFdmVudFN1YnNjcmlwdGlvbicsXG4gICAgICAnQ3JlYXRlR2xvYmFsQ2x1c3RlcicsXG4gICAgICAnQ3JlYXRlSW50ZWdyYXRpb24nLFxuICAgICAgJ0NyZWF0ZU9wdGlvbkdyb3VwJyxcbiAgICAgICdDcmVhdGVUZW5hbnREYXRhYmFzZScsXG4gICAgICAnQ3Jvc3NSZWdpb25Db21tdW5pY2F0aW9uJyxcbiAgICAgICdEZWxldGVCbHVlR3JlZW5EZXBsb3ltZW50JyxcbiAgICAgICdEZWxldGVDdXN0b21EQkVuZ2luZVZlcnNpb24nLFxuICAgICAgJ0RlbGV0ZURCQ2x1c3RlcicsXG4gICAgICAnRGVsZXRlREJDbHVzdGVyQXV0b21hdGVkQmFja3VwJyxcbiAgICAgICdEZWxldGVEQkNsdXN0ZXJFbmRwb2ludCcsXG4gICAgICAnRGVsZXRlREJDbHVzdGVyUGFyYW1ldGVyR3JvdXAnLFxuICAgICAgJ0RlbGV0ZURCQ2x1c3RlclNuYXBzaG90JyxcbiAgICAgICdEZWxldGVEQkluc3RhbmNlJyxcbiAgICAgICdEZWxldGVEQkluc3RhbmNlQXV0b21hdGVkQmFja3VwJyxcbiAgICAgICdEZWxldGVEQlBhcmFtZXRlckdyb3VwJyxcbiAgICAgICdEZWxldGVEQlByb3h5JyxcbiAgICAgICdEZWxldGVEQlByb3h5RW5kcG9pbnQnLFxuICAgICAgJ0RlbGV0ZURCU2VjdXJpdHlHcm91cCcsXG4gICAgICAnRGVsZXRlREJTaGFyZEdyb3VwJyxcbiAgICAgICdEZWxldGVEQlNuYXBzaG90JyxcbiAgICAgICdEZWxldGVEQlN1Ym5ldEdyb3VwJyxcbiAgICAgICdEZWxldGVFdmVudFN1YnNjcmlwdGlvbicsXG4gICAgICAnRGVsZXRlR2xvYmFsQ2x1c3RlcicsXG4gICAgICAnRGVsZXRlSW50ZWdyYXRpb24nLFxuICAgICAgJ0RlbGV0ZU9wdGlvbkdyb3VwJyxcbiAgICAgICdEZWxldGVUZW5hbnREYXRhYmFzZScsXG4gICAgICAnRGVyZWdpc3RlckRCUHJveHlUYXJnZXRzJyxcbiAgICAgICdEaXNhYmxlSHR0cEVuZHBvaW50JyxcbiAgICAgICdFbmFibGVIdHRwRW5kcG9pbnQnLFxuICAgICAgJ0ZhaWxvdmVyREJDbHVzdGVyJyxcbiAgICAgICdGYWlsb3Zlckdsb2JhbENsdXN0ZXInLFxuICAgICAgJ01vZGlmeUFjdGl2aXR5U3RyZWFtJyxcbiAgICAgICdNb2RpZnlDZXJ0aWZpY2F0ZXMnLFxuICAgICAgJ01vZGlmeUN1cnJlbnREQkNsdXN0ZXJDYXBhY2l0eScsXG4gICAgICAnTW9kaWZ5Q3VzdG9tREJFbmdpbmVWZXJzaW9uJyxcbiAgICAgICdNb2RpZnlEQkNsdXN0ZXInLFxuICAgICAgJ01vZGlmeURCQ2x1c3RlckVuZHBvaW50JyxcbiAgICAgICdNb2RpZnlEQkNsdXN0ZXJQYXJhbWV0ZXJHcm91cCcsXG4gICAgICAnTW9kaWZ5REJDbHVzdGVyU25hcHNob3RBdHRyaWJ1dGUnLFxuICAgICAgJ01vZGlmeURCSW5zdGFuY2UnLFxuICAgICAgJ01vZGlmeURCUGFyYW1ldGVyR3JvdXAnLFxuICAgICAgJ01vZGlmeURCUHJveHknLFxuICAgICAgJ01vZGlmeURCUHJveHlFbmRwb2ludCcsXG4gICAgICAnTW9kaWZ5REJQcm94eVRhcmdldEdyb3VwJyxcbiAgICAgICdNb2RpZnlEQlJlY29tbWVuZGF0aW9uJyxcbiAgICAgICdNb2RpZnlEQlNoYXJkR3JvdXAnLFxuICAgICAgJ01vZGlmeURCU25hcHNob3QnLFxuICAgICAgJ01vZGlmeURCU25hcHNob3RBdHRyaWJ1dGUnLFxuICAgICAgJ01vZGlmeURCU3VibmV0R3JvdXAnLFxuICAgICAgJ01vZGlmeUV2ZW50U3Vic2NyaXB0aW9uJyxcbiAgICAgICdNb2RpZnlHbG9iYWxDbHVzdGVyJyxcbiAgICAgICdNb2RpZnlJbnRlZ3JhdGlvbicsXG4gICAgICAnTW9kaWZ5T3B0aW9uR3JvdXAnLFxuICAgICAgJ01vZGlmeVJlY29tbWVuZGF0aW9uJyxcbiAgICAgICdNb2RpZnlUZW5hbnREYXRhYmFzZScsXG4gICAgICAnUHJvbW90ZVJlYWRSZXBsaWNhJyxcbiAgICAgICdQcm9tb3RlUmVhZFJlcGxpY2FEQkNsdXN0ZXInLFxuICAgICAgJ1B1cmNoYXNlUmVzZXJ2ZWREQkluc3RhbmNlc09mZmVyaW5nJyxcbiAgICAgICdSZWJvb3REQkNsdXN0ZXInLFxuICAgICAgJ1JlYm9vdERCSW5zdGFuY2UnLFxuICAgICAgJ1JlYm9vdERCU2hhcmRHcm91cCcsXG4gICAgICAnUmVnaXN0ZXJEQlByb3h5VGFyZ2V0cycsXG4gICAgICAnUmVtb3ZlRnJvbUdsb2JhbENsdXN0ZXInLFxuICAgICAgJ1JlbW92ZVJvbGVGcm9tREJDbHVzdGVyJyxcbiAgICAgICdSZW1vdmVSb2xlRnJvbURCSW5zdGFuY2UnLFxuICAgICAgJ1JlbW92ZVNvdXJjZUlkZW50aWZpZXJGcm9tU3Vic2NyaXB0aW9uJyxcbiAgICAgICdSZXNldERCQ2x1c3RlclBhcmFtZXRlckdyb3VwJyxcbiAgICAgICdSZXNldERCUGFyYW1ldGVyR3JvdXAnLFxuICAgICAgJ1Jlc3RvcmVEQkNsdXN0ZXJGcm9tUzMnLFxuICAgICAgJ1Jlc3RvcmVEQkNsdXN0ZXJGcm9tU25hcHNob3QnLFxuICAgICAgJ1Jlc3RvcmVEQkNsdXN0ZXJUb1BvaW50SW5UaW1lJyxcbiAgICAgICdSZXN0b3JlREJJbnN0YW5jZUZyb21EQlNuYXBzaG90JyxcbiAgICAgICdSZXN0b3JlREJJbnN0YW5jZUZyb21TMycsXG4gICAgICAnUmVzdG9yZURCSW5zdGFuY2VUb1BvaW50SW5UaW1lJyxcbiAgICAgICdSZXZva2VEQlNlY3VyaXR5R3JvdXBJbmdyZXNzJyxcbiAgICAgICdTdGFydEFjdGl2aXR5U3RyZWFtJyxcbiAgICAgICdTdGFydERCQ2x1c3RlcicsXG4gICAgICAnU3RhcnREQkluc3RhbmNlJyxcbiAgICAgICdTdGFydERCSW5zdGFuY2VBdXRvbWF0ZWRCYWNrdXBzUmVwbGljYXRpb24nLFxuICAgICAgJ1N0YXJ0RXhwb3J0VGFzaycsXG4gICAgICAnU3RvcEFjdGl2aXR5U3RyZWFtJyxcbiAgICAgICdTdG9wREJDbHVzdGVyJyxcbiAgICAgICdTdG9wREJJbnN0YW5jZScsXG4gICAgICAnU3RvcERCSW5zdGFuY2VBdXRvbWF0ZWRCYWNrdXBzUmVwbGljYXRpb24nLFxuICAgICAgJ1N3aXRjaG92ZXJCbHVlR3JlZW5EZXBsb3ltZW50JyxcbiAgICAgICdTd2l0Y2hvdmVyR2xvYmFsQ2x1c3RlcicsXG4gICAgICAnU3dpdGNob3ZlclJlYWRSZXBsaWNhJ1xuICAgIF0sXG4gICAgVGFnZ2luZzogW1xuICAgICAgJ0FkZFRhZ3NUb1Jlc291cmNlJyxcbiAgICAgICdSZW1vdmVUYWdzRnJvbVJlc291cmNlJ1xuICAgIF0sXG4gICAgJ1Blcm1pc3Npb25zIG1hbmFnZW1lbnQnOiBbXG4gICAgICAnQXV0aG9yaXplREJTZWN1cml0eUdyb3VwSW5ncmVzcydcbiAgICBdLFxuICAgIExpc3Q6IFtcbiAgICAgICdEZXNjcmliZUFjY291bnRBdHRyaWJ1dGVzJyxcbiAgICAgICdEZXNjcmliZUJsdWVHcmVlbkRlcGxveW1lbnRzJyxcbiAgICAgICdEZXNjcmliZUNlcnRpZmljYXRlcycsXG4gICAgICAnRGVzY3JpYmVEQkNsdXN0ZXJBdXRvbWF0ZWRCYWNrdXBzJyxcbiAgICAgICdEZXNjcmliZURCQ2x1c3RlckJhY2t0cmFja3MnLFxuICAgICAgJ0Rlc2NyaWJlREJDbHVzdGVyRW5kcG9pbnRzJyxcbiAgICAgICdEZXNjcmliZURCQ2x1c3RlclBhcmFtZXRlckdyb3VwcycsXG4gICAgICAnRGVzY3JpYmVEQkNsdXN0ZXJQYXJhbWV0ZXJzJyxcbiAgICAgICdEZXNjcmliZURCQ2x1c3RlclNuYXBzaG90QXR0cmlidXRlcycsXG4gICAgICAnRGVzY3JpYmVEQkNsdXN0ZXJTbmFwc2hvdHMnLFxuICAgICAgJ0Rlc2NyaWJlREJDbHVzdGVycycsXG4gICAgICAnRGVzY3JpYmVEQkVuZ2luZVZlcnNpb25zJyxcbiAgICAgICdEZXNjcmliZURCSW5zdGFuY2VBdXRvbWF0ZWRCYWNrdXBzJyxcbiAgICAgICdEZXNjcmliZURCSW5zdGFuY2VzJyxcbiAgICAgICdEZXNjcmliZURCTG9nRmlsZXMnLFxuICAgICAgJ0Rlc2NyaWJlREJQYXJhbWV0ZXJHcm91cHMnLFxuICAgICAgJ0Rlc2NyaWJlREJQYXJhbWV0ZXJzJyxcbiAgICAgICdEZXNjcmliZURCUHJveGllcycsXG4gICAgICAnRGVzY3JpYmVEQlByb3h5RW5kcG9pbnRzJyxcbiAgICAgICdEZXNjcmliZURCUHJveHlUYXJnZXRHcm91cHMnLFxuICAgICAgJ0Rlc2NyaWJlREJQcm94eVRhcmdldHMnLFxuICAgICAgJ0Rlc2NyaWJlREJSZWNvbW1lbmRhdGlvbnMnLFxuICAgICAgJ0Rlc2NyaWJlREJTZWN1cml0eUdyb3VwcycsXG4gICAgICAnRGVzY3JpYmVEQlNoYXJkR3JvdXBzJyxcbiAgICAgICdEZXNjcmliZURCU25hcHNob3RBdHRyaWJ1dGVzJyxcbiAgICAgICdEZXNjcmliZURCU25hcHNob3RUZW5hbnREYXRhYmFzZXMnLFxuICAgICAgJ0Rlc2NyaWJlREJTbmFwc2hvdHMnLFxuICAgICAgJ0Rlc2NyaWJlREJTdWJuZXRHcm91cHMnLFxuICAgICAgJ0Rlc2NyaWJlRW5naW5lRGVmYXVsdENsdXN0ZXJQYXJhbWV0ZXJzJyxcbiAgICAgICdEZXNjcmliZUVuZ2luZURlZmF1bHRQYXJhbWV0ZXJzJyxcbiAgICAgICdEZXNjcmliZUV2ZW50Q2F0ZWdvcmllcycsXG4gICAgICAnRGVzY3JpYmVFdmVudFN1YnNjcmlwdGlvbnMnLFxuICAgICAgJ0Rlc2NyaWJlRXZlbnRzJyxcbiAgICAgICdEZXNjcmliZUV4cG9ydFRhc2tzJyxcbiAgICAgICdEZXNjcmliZUdsb2JhbENsdXN0ZXJzJyxcbiAgICAgICdEZXNjcmliZUludGVncmF0aW9ucycsXG4gICAgICAnRGVzY3JpYmVPcHRpb25Hcm91cE9wdGlvbnMnLFxuICAgICAgJ0Rlc2NyaWJlT3B0aW9uR3JvdXBzJyxcbiAgICAgICdEZXNjcmliZU9yZGVyYWJsZURCSW5zdGFuY2VPcHRpb25zJyxcbiAgICAgICdEZXNjcmliZVBlbmRpbmdNYWludGVuYW5jZUFjdGlvbnMnLFxuICAgICAgJ0Rlc2NyaWJlUmVzZXJ2ZWREQkluc3RhbmNlcycsXG4gICAgICAnRGVzY3JpYmVSZXNlcnZlZERCSW5zdGFuY2VzT2ZmZXJpbmdzJyxcbiAgICAgICdEZXNjcmliZVNvdXJjZVJlZ2lvbnMnLFxuICAgICAgJ0Rlc2NyaWJlVGVuYW50RGF0YWJhc2VzJyxcbiAgICAgICdEZXNjcmliZVZhbGlkREJJbnN0YW5jZU1vZGlmaWNhdGlvbnMnXG4gICAgXSxcbiAgICBSZWFkOiBbXG4gICAgICAnRGVzY3JpYmVSZWNvbW1lbmRhdGlvbkdyb3VwcycsXG4gICAgICAnRGVzY3JpYmVSZWNvbW1lbmRhdGlvbnMnLFxuICAgICAgJ0Rvd25sb2FkQ29tcGxldGVEQkxvZ0ZpbGUnLFxuICAgICAgJ0Rvd25sb2FkREJMb2dGaWxlUG9ydGlvbicsXG4gICAgICAnTGlzdFRhZ3NGb3JSZXNvdXJjZSdcbiAgICBdXG4gIH07XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGNsdXN0ZXIgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BdXJvcmFVc2VyR3VpZGUvQ0hBUF9BdXJvcmEuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gZGJDbHVzdGVySW5zdGFuY2VOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGRiQ2x1c3Rlckluc3RhbmNlTmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKiAtIC5pZkNsdXN0ZXJUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uQ2x1c3RlcihkYkNsdXN0ZXJJbnN0YW5jZU5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpyZHM6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmNsdXN0ZXI6JHsgZGJDbHVzdGVySW5zdGFuY2VOYW1lIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBzaGFyZGdycCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0F1cm9yYVVzZXJHdWlkZS9PdmVydmlldy5EQlNoYXJkR3JvdXAuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gZGJTaGFyZEdyb3VwUmVzb3VyY2VJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBkYlNoYXJkR3JvdXBSZXNvdXJjZUlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25TaGFyZGdycChkYlNoYXJkR3JvdXBSZXNvdXJjZUlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06cmRzOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpzaGFyZC1ncm91cDokeyBkYlNoYXJkR3JvdXBSZXNvdXJjZUlkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBjbHVzdGVyLWF1dG8tYmFja3VwIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQXVyb3JhVXNlckd1aWRlL0F1cm9yYS5NYW5hZ2luZy5CYWNrdXBzLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGRiQ2x1c3RlckF1dG9tYXRlZEJhY2t1cElkIC0gSWRlbnRpZmllciBmb3IgdGhlIGRiQ2x1c3RlckF1dG9tYXRlZEJhY2t1cElkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICovXG4gIHB1YmxpYyBvbkNsdXN0ZXJBdXRvQmFja3VwKGRiQ2x1c3RlckF1dG9tYXRlZEJhY2t1cElkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06cmRzOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpjbHVzdGVyLWF1dG8tYmFja3VwOiR7IGRiQ2x1c3RlckF1dG9tYXRlZEJhY2t1cElkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBhdXRvLWJhY2t1cCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L1VzZXJHdWlkZS9VU0VSX1dvcmtpbmdXaXRoQXV0b21hdGVkQmFja3Vwcy5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBkYkluc3RhbmNlQXV0b21hdGVkQmFja3VwSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgZGJJbnN0YW5jZUF1dG9tYXRlZEJhY2t1cElkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICovXG4gIHB1YmxpYyBvbkF1dG9CYWNrdXAoZGJJbnN0YW5jZUF1dG9tYXRlZEJhY2t1cElkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06cmRzOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTphdXRvLWJhY2t1cDokeyBkYkluc3RhbmNlQXV0b21hdGVkQmFja3VwSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGNsdXN0ZXItZW5kcG9pbnQgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9BdXJvcmFVc2VyR3VpZGUvQXVyb3JhLk92ZXJ2aWV3LkVuZHBvaW50cy5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBkYkNsdXN0ZXJFbmRwb2ludCAtIElkZW50aWZpZXIgZm9yIHRoZSBkYkNsdXN0ZXJFbmRwb2ludC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uQ2x1c3RlckVuZHBvaW50KGRiQ2x1c3RlckVuZHBvaW50OiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06cmRzOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpjbHVzdGVyLWVuZHBvaW50OiR7IGRiQ2x1c3RlckVuZHBvaW50IH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBjbHVzdGVyLXBnIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQXVyb3JhVXNlckd1aWRlL1VTRVJfV29ya2luZ1dpdGhQYXJhbUdyb3Vwcy5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBjbHVzdGVyUGFyYW1ldGVyR3JvdXBOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGNsdXN0ZXJQYXJhbWV0ZXJHcm91cE5hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICogLSAuaWZDbHVzdGVyUGdUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uQ2x1c3RlclBnKGNsdXN0ZXJQYXJhbWV0ZXJHcm91cE5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpyZHM6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmNsdXN0ZXItcGc6JHsgY2x1c3RlclBhcmFtZXRlckdyb3VwTmFtZSB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgY2x1c3Rlci1zbmFwc2hvdCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0F1cm9yYVVzZXJHdWlkZS9BdXJvcmEuTWFuYWdpbmcuQmFja3Vwcy5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBjbHVzdGVyU25hcHNob3ROYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGNsdXN0ZXJTbmFwc2hvdE5hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICogLSAuaWZDbHVzdGVyU25hcHNob3RUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uQ2x1c3RlclNuYXBzaG90KGNsdXN0ZXJTbmFwc2hvdE5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpyZHM6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmNsdXN0ZXItc25hcHNob3Q6JHsgY2x1c3RlclNuYXBzaG90TmFtZSB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgZGIgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9Vc2VyR3VpZGUvT3ZlcnZpZXcuREJJbnN0YW5jZS5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBkYkluc3RhbmNlTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBkYkluc3RhbmNlTmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKiAtIC5pZkRhdGFiYXNlQ2xhc3MoKVxuICAgKiAtIC5pZkRhdGFiYXNlRW5naW5lKClcbiAgICogLSAuaWZEYXRhYmFzZU5hbWUoKVxuICAgKiAtIC5pZk11bHRpQXooKVxuICAgKiAtIC5pZlBpb3BzKClcbiAgICogLSAuaWZTdG9yYWdlRW5jcnlwdGVkKClcbiAgICogLSAuaWZTdG9yYWdlU2l6ZSgpXG4gICAqIC0gLmlmVnBjKClcbiAgICogLSAuaWZEYlRhZygpXG4gICAqL1xuICBwdWJsaWMgb25EYihkYkluc3RhbmNlTmFtZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OnJkczokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06ZGI6JHsgZGJJbnN0YW5jZU5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGVzIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvVXNlckd1aWRlL1VTRVJfRXZlbnRzLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHN1YnNjcmlwdGlvbk5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgc3Vic2NyaXB0aW9uTmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKiAtIC5pZkVzVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkVzKHN1YnNjcmlwdGlvbk5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpyZHM6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmVzOiR7IHN1YnNjcmlwdGlvbk5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGdsb2JhbC1jbHVzdGVyIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQXVyb3JhVXNlckd1aWRlL2F1cm9yYS1nbG9iYWwtZGF0YWJhc2UuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gZ2xvYmFsQ2x1c3RlciAtIElkZW50aWZpZXIgZm9yIHRoZSBnbG9iYWxDbHVzdGVyLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqL1xuICBwdWJsaWMgb25HbG9iYWxDbHVzdGVyKGdsb2JhbENsdXN0ZXI6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06cmRzOjokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpnbG9iYWwtY2x1c3RlcjokeyBnbG9iYWxDbHVzdGVyIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBvZyB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L1VzZXJHdWlkZS9VU0VSX1dvcmtpbmdXaXRoT3B0aW9uR3JvdXBzLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIG9wdGlvbkdyb3VwTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBvcHRpb25Hcm91cE5hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICogLSAuaWZPZ1RhZygpXG4gICAqL1xuICBwdWJsaWMgb25PZyhvcHRpb25Hcm91cE5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpyZHM6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9Om9nOiR7IG9wdGlvbkdyb3VwTmFtZSB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgcGcgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9Vc2VyR3VpZGUvVVNFUl9Xb3JraW5nV2l0aFBhcmFtR3JvdXBzLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHBhcmFtZXRlckdyb3VwTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBwYXJhbWV0ZXJHcm91cE5hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICogLSAuaWZQZ1RhZygpXG4gICAqL1xuICBwdWJsaWMgb25QZyhwYXJhbWV0ZXJHcm91cE5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpyZHM6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OnBnOiR7IHBhcmFtZXRlckdyb3VwTmFtZSB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgcHJveHkgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9Vc2VyR3VpZGUvcmRzLXByb3h5Lmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGRiUHJveHlJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBkYlByb3h5SWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblByb3h5KGRiUHJveHlJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OnJkczokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06ZGItcHJveHk6JHsgZGJQcm94eUlkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBwcm94eS1lbmRwb2ludCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L1VzZXJHdWlkZS9yZHMtcHJveHkuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gZGJQcm94eUVuZHBvaW50SWQgLSBJZGVudGlmaWVyIGZvciB0aGUgZGJQcm94eUVuZHBvaW50SWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblByb3h5RW5kcG9pbnQoZGJQcm94eUVuZHBvaW50SWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpyZHM6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmRiLXByb3h5LWVuZHBvaW50OiR7IGRiUHJveHlFbmRwb2ludElkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSByaSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L1VzZXJHdWlkZS9VU0VSX1dvcmtpbmdXaXRoUmVzZXJ2ZWREQkluc3RhbmNlcy5odG1sXG4gICAqXG4gICAqIEBwYXJhbSByZXNlcnZlZERiSW5zdGFuY2VOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc2VydmVkRGJJbnN0YW5jZU5hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICogLSAuaWZSaVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25SaShyZXNlcnZlZERiSW5zdGFuY2VOYW1lOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06cmRzOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpyaTokeyByZXNlcnZlZERiSW5zdGFuY2VOYW1lIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBzZWNncnAgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9Vc2VyR3VpZGUvT3ZlcnZpZXcuUkRTU2VjdXJpdHlHcm91cHMuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gc2VjdXJpdHlHcm91cE5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgc2VjdXJpdHlHcm91cE5hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICogLSAuaWZTZWNncnBUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uU2VjZ3JwKHNlY3VyaXR5R3JvdXBOYW1lOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06cmRzOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpzZWNncnA6JHsgc2VjdXJpdHlHcm91cE5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHNuYXBzaG90IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvVXNlckd1aWRlL1VTRVJfV29ya2luZ1dpdGhBdXRvbWF0ZWRCYWNrdXBzLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHNuYXBzaG90TmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBzbmFwc2hvdE5hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICogLSAuaWZTbmFwc2hvdFRhZygpXG4gICAqL1xuICBwdWJsaWMgb25TbmFwc2hvdChzbmFwc2hvdE5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpyZHM6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OnNuYXBzaG90OiR7IHNuYXBzaG90TmFtZSB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgc3ViZ3JwIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvVXNlckd1aWRlL1VTRVJfVlBDLlNjZW5hcmlvcy5odG1sI1VTRVJfVlBDLlNjZW5hcmlvMVxuICAgKlxuICAgKiBAcGFyYW0gc3VibmV0R3JvdXBOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIHN1Ym5ldEdyb3VwTmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKiAtIC5pZlN1YmdycFRhZygpXG4gICAqL1xuICBwdWJsaWMgb25TdWJncnAoc3VibmV0R3JvdXBOYW1lOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06cmRzOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpzdWJncnA6JHsgc3VibmV0R3JvdXBOYW1lIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSB0YXJnZXQtZ3JvdXAgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9Vc2VyR3VpZGUvcmRzLXByb3h5Lmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHRhcmdldEdyb3VwSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgdGFyZ2V0R3JvdXBJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uVGFyZ2V0R3JvdXAodGFyZ2V0R3JvdXBJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OnJkczokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06dGFyZ2V0LWdyb3VwOiR7IHRhcmdldEdyb3VwSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGNldiB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L1VzZXJHdWlkZS9jdXN0b20tY2V2Lmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGVuZ2luZSAtIElkZW50aWZpZXIgZm9yIHRoZSBlbmdpbmUuXG4gICAqIEBwYXJhbSBlbmdpbmVWZXJzaW9uIC0gSWRlbnRpZmllciBmb3IgdGhlIGVuZ2luZVZlcnNpb24uXG4gICAqIEBwYXJhbSBjdXN0b21EYkVuZ2luZVZlcnNpb25JZCAtIElkZW50aWZpZXIgZm9yIHRoZSBjdXN0b21EYkVuZ2luZVZlcnNpb25JZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uQ2V2KGVuZ2luZTogc3RyaW5nLCBlbmdpbmVWZXJzaW9uOiBzdHJpbmcsIGN1c3RvbURiRW5naW5lVmVyc2lvbklkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06cmRzOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpjZXY6JHsgZW5naW5lIH0vJHsgZW5naW5lVmVyc2lvbiB9LyR7IGN1c3RvbURiRW5naW5lVmVyc2lvbklkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBkZXBsb3ltZW50IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvVXNlckd1aWRlL2JsdWUtZ3JlZW4tZGVwbG95bWVudHMuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gYmx1ZUdyZWVuRGVwbG95bWVudElkZW50aWZpZXIgLSBJZGVudGlmaWVyIGZvciB0aGUgYmx1ZUdyZWVuRGVwbG95bWVudElkZW50aWZpZXIuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkRlcGxveW1lbnQoYmx1ZUdyZWVuRGVwbG95bWVudElkZW50aWZpZXI6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpyZHM6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmRlcGxveW1lbnQ6JHsgYmx1ZUdyZWVuRGVwbG95bWVudElkZW50aWZpZXIgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGludGVncmF0aW9uIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvQXVyb3JhVXNlckd1aWRlL3plcm8tZXRsLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGludGVncmF0aW9uSWRlbnRpZmllciAtIElkZW50aWZpZXIgZm9yIHRoZSBpbnRlZ3JhdGlvbklkZW50aWZpZXIuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkludGVncmF0aW9uKGludGVncmF0aW9uSWRlbnRpZmllcjogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OnJkczokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06aW50ZWdyYXRpb246JHsgaW50ZWdyYXRpb25JZGVudGlmaWVyIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBzbmFwc2hvdC10ZW5hbnQtZGF0YWJhc2UgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9Vc2VyR3VpZGUvT3JhY2xlLkNvbmNlcHRzLnNpbmdsZS10ZW5hbnQuc25hcHNob3RzLmh0bWwjYnItY2RiLmRiLXNuYXBzaG90c1xuICAgKlxuICAgKiBAcGFyYW0gc25hcHNob3ROYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIHNuYXBzaG90TmFtZS5cbiAgICogQHBhcmFtIHRlbmFudFJlc291cmNlSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgdGVuYW50UmVzb3VyY2VJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uU25hcHNob3RUZW5hbnREYXRhYmFzZShzbmFwc2hvdE5hbWU6IHN0cmluZywgdGVuYW50UmVzb3VyY2VJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OnJkczokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06c25hcHNob3QtdGVuYW50LWRhdGFiYXNlOiR7IHNuYXBzaG90TmFtZSB9OiR7IHRlbmFudFJlc291cmNlSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHRlbmFudC1kYXRhYmFzZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L1VzZXJHdWlkZS9PcmFjbGUuQ29uY2VwdHMuQ0RCcy5odG1sI211bHRpLXRlbmFudC1jb25maWd1cmF0aW9uXG4gICAqXG4gICAqIEBwYXJhbSB0ZW5hbnRSZXNvdXJjZUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIHRlbmFudFJlc291cmNlSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblRlbmFudERhdGFiYXNlKHRlbmFudFJlc291cmNlSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpyZHM6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OnRlbmFudC1kYXRhYmFzZTokeyB0ZW5hbnRSZXNvdXJjZUlkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgc2V0IG9mIHRhZyBrZXktdmFsdWUgcGFpcnMgaW4gdGhlIHJlcXVlc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXJlcXVlc3R0YWdcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0FkZFRhZ3NUb1Jlc291cmNlKClcbiAgICogLSAudG9Db3B5REJDbHVzdGVyUGFyYW1ldGVyR3JvdXAoKVxuICAgKiAtIC50b0NvcHlEQkNsdXN0ZXJTbmFwc2hvdCgpXG4gICAqIC0gLnRvQ29weURCUGFyYW1ldGVyR3JvdXAoKVxuICAgKiAtIC50b0NvcHlEQlNuYXBzaG90KClcbiAgICogLSAudG9Db3B5T3B0aW9uR3JvdXAoKVxuICAgKiAtIC50b0NyZWF0ZUJsdWVHcmVlbkRlcGxveW1lbnQoKVxuICAgKiAtIC50b0NyZWF0ZUN1c3RvbURCRW5naW5lVmVyc2lvbigpXG4gICAqIC0gLnRvQ3JlYXRlREJDbHVzdGVyKClcbiAgICogLSAudG9DcmVhdGVEQkNsdXN0ZXJFbmRwb2ludCgpXG4gICAqIC0gLnRvQ3JlYXRlREJDbHVzdGVyUGFyYW1ldGVyR3JvdXAoKVxuICAgKiAtIC50b0NyZWF0ZURCQ2x1c3RlclNuYXBzaG90KClcbiAgICogLSAudG9DcmVhdGVEQkluc3RhbmNlKClcbiAgICogLSAudG9DcmVhdGVEQkluc3RhbmNlUmVhZFJlcGxpY2EoKVxuICAgKiAtIC50b0NyZWF0ZURCUGFyYW1ldGVyR3JvdXAoKVxuICAgKiAtIC50b0NyZWF0ZURCUHJveHkoKVxuICAgKiAtIC50b0NyZWF0ZURCUHJveHlFbmRwb2ludCgpXG4gICAqIC0gLnRvQ3JlYXRlREJTZWN1cml0eUdyb3VwKClcbiAgICogLSAudG9DcmVhdGVEQlNoYXJkR3JvdXAoKVxuICAgKiAtIC50b0NyZWF0ZURCU25hcHNob3QoKVxuICAgKiAtIC50b0NyZWF0ZURCU3VibmV0R3JvdXAoKVxuICAgKiAtIC50b0NyZWF0ZUV2ZW50U3Vic2NyaXB0aW9uKClcbiAgICogLSAudG9DcmVhdGVJbnRlZ3JhdGlvbigpXG4gICAqIC0gLnRvQ3JlYXRlT3B0aW9uR3JvdXAoKVxuICAgKiAtIC50b0NyZWF0ZVRlbmFudERhdGFiYXNlKClcbiAgICogLSAudG9QdXJjaGFzZVJlc2VydmVkREJJbnN0YW5jZXNPZmZlcmluZygpXG4gICAqIC0gLnRvUmVtb3ZlVGFnc0Zyb21SZXNvdXJjZSgpXG4gICAqIC0gLnRvUmVzdG9yZURCQ2x1c3RlckZyb21TMygpXG4gICAqIC0gLnRvUmVzdG9yZURCQ2x1c3RlckZyb21TbmFwc2hvdCgpXG4gICAqIC0gLnRvUmVzdG9yZURCQ2x1c3RlclRvUG9pbnRJblRpbWUoKVxuICAgKiAtIC50b1Jlc3RvcmVEQkluc3RhbmNlRnJvbURCU25hcHNob3QoKVxuICAgKiAtIC50b1Jlc3RvcmVEQkluc3RhbmNlRnJvbVMzKClcbiAgICogLSAudG9SZXN0b3JlREJJbnN0YW5jZVRvUG9pbnRJblRpbWUoKVxuICAgKlxuICAgKiBAcGFyYW0gdGFnS2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzUmVxdWVzdFRhZyh0YWdLZXk6IHN0cmluZywgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpSZXF1ZXN0VGFnLyR7IHRhZ0tleSB9YCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgdGhlIHNldCBvZiB0YWcga2V5LXZhbHVlIHBhaXJzIGF0dGFjaGVkIHRvIHRoZSByZXNvdXJjZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtcmVzb3VyY2V0YWdcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0NyZWF0ZUJsdWVHcmVlbkRlcGxveW1lbnQoKVxuICAgKiAtIC50b0RlbGV0ZUJsdWVHcmVlbkRlcGxveW1lbnQoKVxuICAgKiAtIC50b0Rlc2NyaWJlSW50ZWdyYXRpb25zKClcbiAgICogLSAudG9Td2l0Y2hvdmVyQmx1ZUdyZWVuRGVwbG95bWVudCgpXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gY2x1c3RlclxuICAgKiAtIHNoYXJkZ3JwXG4gICAqIC0gY2x1c3Rlci1lbmRwb2ludFxuICAgKiAtIGNsdXN0ZXItcGdcbiAgICogLSBjbHVzdGVyLXNuYXBzaG90XG4gICAqIC0gZGJcbiAgICogLSBlc1xuICAgKiAtIG9nXG4gICAqIC0gcGdcbiAgICogLSBwcm94eVxuICAgKiAtIHByb3h5LWVuZHBvaW50XG4gICAqIC0gcmlcbiAgICogLSBzZWNncnBcbiAgICogLSBzbmFwc2hvdFxuICAgKiAtIHN1YmdycFxuICAgKiAtIHRhcmdldC1ncm91cFxuICAgKiAtIGNldlxuICAgKiAtIGRlcGxveW1lbnRcbiAgICogLSBpbnRlZ3JhdGlvblxuICAgKiAtIHNuYXBzaG90LXRlbmFudC1kYXRhYmFzZVxuICAgKiAtIHRlbmFudC1kYXRhYmFzZVxuICAgKlxuICAgKiBAcGFyYW0gdGFnS2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzUmVzb3VyY2VUYWcodGFnS2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6UmVzb3VyY2VUYWcvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgc2V0IG9mIHRhZyBrZXlzIGluIHRoZSByZXF1ZXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy10YWdrZXlzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9BZGRUYWdzVG9SZXNvdXJjZSgpXG4gICAqIC0gLnRvQ29weURCQ2x1c3RlclBhcmFtZXRlckdyb3VwKClcbiAgICogLSAudG9Db3B5REJDbHVzdGVyU25hcHNob3QoKVxuICAgKiAtIC50b0NvcHlEQlBhcmFtZXRlckdyb3VwKClcbiAgICogLSAudG9Db3B5REJTbmFwc2hvdCgpXG4gICAqIC0gLnRvQ29weU9wdGlvbkdyb3VwKClcbiAgICogLSAudG9DcmVhdGVCbHVlR3JlZW5EZXBsb3ltZW50KClcbiAgICogLSAudG9DcmVhdGVDdXN0b21EQkVuZ2luZVZlcnNpb24oKVxuICAgKiAtIC50b0NyZWF0ZURCQ2x1c3RlcigpXG4gICAqIC0gLnRvQ3JlYXRlREJDbHVzdGVyRW5kcG9pbnQoKVxuICAgKiAtIC50b0NyZWF0ZURCQ2x1c3RlclBhcmFtZXRlckdyb3VwKClcbiAgICogLSAudG9DcmVhdGVEQkNsdXN0ZXJTbmFwc2hvdCgpXG4gICAqIC0gLnRvQ3JlYXRlREJJbnN0YW5jZSgpXG4gICAqIC0gLnRvQ3JlYXRlREJJbnN0YW5jZVJlYWRSZXBsaWNhKClcbiAgICogLSAudG9DcmVhdGVEQlBhcmFtZXRlckdyb3VwKClcbiAgICogLSAudG9DcmVhdGVEQlByb3h5KClcbiAgICogLSAudG9DcmVhdGVEQlByb3h5RW5kcG9pbnQoKVxuICAgKiAtIC50b0NyZWF0ZURCU2VjdXJpdHlHcm91cCgpXG4gICAqIC0gLnRvQ3JlYXRlREJTaGFyZEdyb3VwKClcbiAgICogLSAudG9DcmVhdGVEQlNuYXBzaG90KClcbiAgICogLSAudG9DcmVhdGVEQlN1Ym5ldEdyb3VwKClcbiAgICogLSAudG9DcmVhdGVFdmVudFN1YnNjcmlwdGlvbigpXG4gICAqIC0gLnRvQ3JlYXRlSW50ZWdyYXRpb24oKVxuICAgKiAtIC50b0NyZWF0ZU9wdGlvbkdyb3VwKClcbiAgICogLSAudG9DcmVhdGVUZW5hbnREYXRhYmFzZSgpXG4gICAqIC0gLnRvUHVyY2hhc2VSZXNlcnZlZERCSW5zdGFuY2VzT2ZmZXJpbmcoKVxuICAgKiAtIC50b1JlbW92ZVRhZ3NGcm9tUmVzb3VyY2UoKVxuICAgKiAtIC50b1Jlc3RvcmVEQkNsdXN0ZXJGcm9tUzMoKVxuICAgKiAtIC50b1Jlc3RvcmVEQkNsdXN0ZXJGcm9tU25hcHNob3QoKVxuICAgKiAtIC50b1Jlc3RvcmVEQkNsdXN0ZXJUb1BvaW50SW5UaW1lKClcbiAgICogLSAudG9SZXN0b3JlREJJbnN0YW5jZUZyb21EQlNuYXBzaG90KClcbiAgICogLSAudG9SZXN0b3JlREJJbnN0YW5jZUZyb21TMygpXG4gICAqIC0gLnRvUmVzdG9yZURCSW5zdGFuY2VUb1BvaW50SW5UaW1lKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NUYWdLZXlzKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6VGFnS2V5c2AsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSB0eXBlIG9mIGJhY2t1cCB0YXJnZXQuIE9uZSBvZjogcmVnaW9uLCBvdXRwb3N0c1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9Vc2VyR3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbCNVc2luZ1dpdGhSRFMuSUFNLkNvbmRpdGlvbnNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0NyZWF0ZURCSW5zdGFuY2UoKVxuICAgKiAtIC50b0NyZWF0ZURCU25hcHNob3QoKVxuICAgKiAtIC50b1Jlc3RvcmVEQkluc3RhbmNlRnJvbURCU25hcHNob3QoKVxuICAgKiAtIC50b1Jlc3RvcmVEQkluc3RhbmNlVG9Qb2ludEluVGltZSgpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQmFja3VwVGFyZ2V0KHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBCYWNrdXBUYXJnZXRgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgdmFsdWUgdGhhdCBzcGVjaWZpZXMgd2hldGhlciB0aGUgQ29weURCU25hcHNob3QgYWN0aW9uIHJlcXVpcmVzIGNvcHlpbmcgdGhlIERCIG9wdGlvbiBncm91cFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9Vc2VyR3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbCNVc2luZ1dpdGhSRFMuSUFNLkNvbmRpdGlvbnNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0NvcHlEQlNuYXBzaG90KClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIGB0cnVlYCBvciBgZmFsc2VgLiAqKkRlZmF1bHQ6KiogYHRydWVgXG4gICAqL1xuICBwdWJsaWMgaWZDb3B5T3B0aW9uR3JvdXAodmFsdWU/OiBib29sZWFuKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYENvcHlPcHRpb25Hcm91cGAsICh0eXBlb2YgdmFsdWUgIT09ICd1bmRlZmluZWQnID8gdmFsdWUgOiB0cnVlKSwgJ0Jvb2wnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgdHlwZSBvZiBEQiBpbnN0YW5jZSBjbGFzc1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9Vc2VyR3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbCNVc2luZ1dpdGhSRFMuSUFNLkNvbmRpdGlvbnNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0NyZWF0ZUJsdWVHcmVlbkRlcGxveW1lbnQoKVxuICAgKiAtIC50b0NyZWF0ZURCQ2x1c3RlcigpXG4gICAqIC0gLnRvTW9kaWZ5REJDbHVzdGVyKClcbiAgICogLSAudG9SZXN0b3JlREJDbHVzdGVyRnJvbVNuYXBzaG90KClcbiAgICogLSAudG9SZXN0b3JlREJDbHVzdGVyVG9Qb2ludEluVGltZSgpXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gZGJcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZEYXRhYmFzZUNsYXNzKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBEYXRhYmFzZUNsYXNzYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgdGhlIGRhdGFiYXNlIGVuZ2luZS4gRm9yIHBvc3NpYmxlIHZhbHVlcyByZWZlciB0byB0aGUgZW5naW5lIHBhcmFtZXRlciBpbiBDcmVhdGVEQkluc3RhbmNlIEFQSVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9Vc2VyR3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbCNVc2luZ1dpdGhSRFMuSUFNLkNvbmRpdGlvbnNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0NyZWF0ZUJsdWVHcmVlbkRlcGxveW1lbnQoKVxuICAgKiAtIC50b0NyZWF0ZURCQ2x1c3RlcigpXG4gICAqIC0gLnRvUmVzdG9yZURCQ2x1c3RlckZyb21TMygpXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gZGJcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZEYXRhYmFzZUVuZ2luZSh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgRGF0YWJhc2VFbmdpbmVgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgdXNlci1kZWZpbmVkIG5hbWUgb2YgdGhlIGRhdGFiYXNlIG9uIHRoZSBEQiBpbnN0YW5jZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9Vc2VyR3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbCNVc2luZ1dpdGhSRFMuSUFNLkNvbmRpdGlvbnNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0NyZWF0ZUJsdWVHcmVlbkRlcGxveW1lbnQoKVxuICAgKiAtIC50b0NyZWF0ZURCQ2x1c3RlcigpXG4gICAqIC0gLnRvUmVzdG9yZURCQ2x1c3RlckZyb21TMygpXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gZGJcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZEYXRhYmFzZU5hbWUodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYERhdGFiYXNlTmFtZWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSB0eXBlIG9mIHRoZSBlbmRwb2ludC4gT25lIG9mOiBSRUFERVIsIFdSSVRFUiwgQ1VTVE9NXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L1VzZXJHdWlkZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sI1VzaW5nV2l0aFJEUy5JQU0uQ29uZGl0aW9uc1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQ3JlYXRlREJDbHVzdGVyRW5kcG9pbnQoKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkVuZHBvaW50VHlwZSh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgRW5kcG9pbnRUeXBlYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgdGhlIHZhbHVlIHRoYXQgc3BlY2lmaWVzIHdoZXRoZXIgUkRTIG1hbmFnZXMgbWFzdGVyIHVzZXIgcGFzc3dvcmQgaW4gQVdTIFNlY3JldHMgTWFuYWdlciBmb3IgdGhlIERCIGluc3RhbmNlIG9yIGNsdXN0ZXJcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvVXNlckd1aWRlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWwjVXNpbmdXaXRoUkRTLklBTS5Db25kaXRpb25zXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9DcmVhdGVEQkNsdXN0ZXIoKVxuICAgKiAtIC50b0NyZWF0ZURCSW5zdGFuY2UoKVxuICAgKiAtIC50b01vZGlmeURCQ2x1c3RlcigpXG4gICAqIC0gLnRvTW9kaWZ5REJJbnN0YW5jZSgpXG4gICAqIC0gLnRvUmVzdG9yZURCQ2x1c3RlckZyb21TMygpXG4gICAqIC0gLnRvUmVzdG9yZURCSW5zdGFuY2VGcm9tUzMoKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgYHRydWVgIG9yIGBmYWxzZWAuICoqRGVmYXVsdDoqKiBgdHJ1ZWBcbiAgICovXG4gIHB1YmxpYyBpZk1hbmFnZU1hc3RlclVzZXJQYXNzd29yZCh2YWx1ZT86IGJvb2xlYW4pIHtcbiAgICByZXR1cm4gdGhpcy5pZihgTWFuYWdlTWFzdGVyVXNlclBhc3N3b3JkYCwgKHR5cGVvZiB2YWx1ZSAhPT0gJ3VuZGVmaW5lZCcgPyB2YWx1ZSA6IHRydWUpLCAnQm9vbCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSB2YWx1ZSB0aGF0IHNwZWNpZmllcyB3aGV0aGVyIHRoZSBEQiBpbnN0YW5jZSBydW5zIGluIG11bHRpcGxlIEF2YWlsYWJpbGl0eSBab25lcy4gVG8gaW5kaWNhdGUgdGhhdCB0aGUgREIgaW5zdGFuY2UgaXMgdXNpbmcgTXVsdGktQVosIHNwZWNpZnkgdHJ1ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9Vc2VyR3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbCNVc2luZ1dpdGhSRFMuSUFNLkNvbmRpdGlvbnNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0NyZWF0ZUJsdWVHcmVlbkRlcGxveW1lbnQoKVxuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIGRiXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBgdHJ1ZWAgb3IgYGZhbHNlYC4gKipEZWZhdWx0OioqIGB0cnVlYFxuICAgKi9cbiAgcHVibGljIGlmTXVsdGlBeih2YWx1ZT86IGJvb2xlYW4pIHtcbiAgICByZXR1cm4gdGhpcy5pZihgTXVsdGlBemAsICh0eXBlb2YgdmFsdWUgIT09ICd1bmRlZmluZWQnID8gdmFsdWUgOiB0cnVlKSwgJ0Jvb2wnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgdmFsdWUgdGhhdCBjb250YWlucyB0aGUgbnVtYmVyIG9mIFByb3Zpc2lvbmVkIElPUFMgKFBJT1BTKSB0aGF0IHRoZSBpbnN0YW5jZSBzdXBwb3J0cy4gVG8gaW5kaWNhdGUgYSBEQiBpbnN0YW5jZSB0aGF0IGRvZXMgbm90IGhhdmUgUElPUFMgZW5hYmxlZCwgc3BlY2lmeSAwXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L1VzZXJHdWlkZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sI1VzaW5nV2l0aFJEUy5JQU0uQ29uZGl0aW9uc1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQ3JlYXRlQmx1ZUdyZWVuRGVwbG95bWVudCgpXG4gICAqIC0gLnRvQ3JlYXRlREJDbHVzdGVyKClcbiAgICogLSAudG9Nb2RpZnlEQkNsdXN0ZXIoKVxuICAgKiAtIC50b1Jlc3RvcmVEQkNsdXN0ZXJGcm9tU25hcHNob3QoKVxuICAgKiAtIC50b1Jlc3RvcmVEQkNsdXN0ZXJUb1BvaW50SW5UaW1lKClcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBkYlxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtudW1lcmljIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19OdW1lcmljKS4gKipEZWZhdWx0OioqIGBOdW1lcmljRXF1YWxzYFxuICAgKi9cbiAgcHVibGljIGlmUGlvcHModmFsdWU6IG51bWJlciB8IG51bWJlcltdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYFBpb3BzYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdOdW1lcmljRXF1YWxzJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgdGhlIHZhbHVlIHRoYXQgc3BlY2lmaWVzIHdoZXRoZXIgdGhlIERCIGluc3RhbmNlIHN0b3JhZ2Ugc2hvdWxkIGJlIGVuY3J5cHRlZC4gVG8gZW5mb3JjZSBzdG9yYWdlIGVuY3J5cHRpb24sIHNwZWNpZnkgdHJ1ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9Vc2VyR3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbCNVc2luZ1dpdGhSRFMuSUFNLkNvbmRpdGlvbnNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0NyZWF0ZUJsdWVHcmVlbkRlcGxveW1lbnQoKVxuICAgKiAtIC50b0NyZWF0ZURCQ2x1c3RlcigpXG4gICAqIC0gLnRvUmVzdG9yZURCQ2x1c3RlckZyb21TMygpXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gZGJcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIGB0cnVlYCBvciBgZmFsc2VgLiAqKkRlZmF1bHQ6KiogYHRydWVgXG4gICAqL1xuICBwdWJsaWMgaWZTdG9yYWdlRW5jcnlwdGVkKHZhbHVlPzogYm9vbGVhbikge1xuICAgIHJldHVybiB0aGlzLmlmKGBTdG9yYWdlRW5jcnlwdGVkYCwgKHR5cGVvZiB2YWx1ZSAhPT0gJ3VuZGVmaW5lZCcgPyB2YWx1ZSA6IHRydWUpLCAnQm9vbCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSBzdG9yYWdlIHZvbHVtZSBzaXplIChpbiBHQilcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvVXNlckd1aWRlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWwjVXNpbmdXaXRoUkRTLklBTS5Db25kaXRpb25zXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9DcmVhdGVCbHVlR3JlZW5EZXBsb3ltZW50KClcbiAgICogLSAudG9DcmVhdGVEQkNsdXN0ZXIoKVxuICAgKiAtIC50b01vZGlmeURCQ2x1c3RlcigpXG4gICAqIC0gLnRvUmVzdG9yZURCQ2x1c3RlckZyb21TbmFwc2hvdCgpXG4gICAqIC0gLnRvUmVzdG9yZURCQ2x1c3RlclRvUG9pbnRJblRpbWUoKVxuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIGRiXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW251bWVyaWMgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX051bWVyaWMpLiAqKkRlZmF1bHQ6KiogYE51bWVyaWNFcXVhbHNgXG4gICAqL1xuICBwdWJsaWMgaWZTdG9yYWdlU2l6ZSh2YWx1ZTogbnVtYmVyIHwgbnVtYmVyW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgU3RvcmFnZVNpemVgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ051bWVyaWNFcXVhbHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgdGVuYW50IGRhdGFiYXNlIG5hbWUgaW4gQ3JlYXRlVGVuYW50RGF0YWJhc2UgYW5kIGJ5IHRoZSBuZXcgdGVuYW50IGRhdGFiYXNlIG5hbWUgaW4gTW9kaWZ5VGVuYW50RGF0YWJhc2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvVXNlckd1aWRlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWwjVXNpbmdXaXRoUkRTLklBTS5Db25kaXRpb25zXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9DcmVhdGVUZW5hbnREYXRhYmFzZSgpXG4gICAqIC0gLnRvTW9kaWZ5VGVuYW50RGF0YWJhc2UoKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlRlbmFudERhdGFiYXNlTmFtZSh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgVGVuYW50RGF0YWJhc2VOYW1lYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgdGhlIHZhbHVlIHRoYXQgc3BlY2lmaWVzIHdoZXRoZXIgdGhlIERCIGluc3RhbmNlIHJ1bnMgaW4gYW4gQW1hem9uIFZpcnR1YWwgUHJpdmF0ZSBDbG91ZCAoQW1hem9uIFZQQykuIFRvIGluZGljYXRlIHRoYXQgdGhlIERCIGluc3RhbmNlIHJ1bnMgaW4gYW4gQW1hem9uIFZQQywgc3BlY2lmeSB0cnVlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L1VzZXJHdWlkZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sI1VzaW5nV2l0aFJEUy5JQU0uQ29uZGl0aW9uc1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQ3JlYXRlQmx1ZUdyZWVuRGVwbG95bWVudCgpXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gZGJcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIGB0cnVlYCBvciBgZmFsc2VgLiAqKkRlZmF1bHQ6KiogYHRydWVgXG4gICAqL1xuICBwdWJsaWMgaWZWcGModmFsdWU/OiBib29sZWFuKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYFZwY2AsICh0eXBlb2YgdmFsdWUgIT09ICd1bmRlZmluZWQnID8gdmFsdWUgOiB0cnVlKSwgJ0Jvb2wnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgdGFnIGF0dGFjaGVkIHRvIGEgREIgY2x1c3RlciBwYXJhbWV0ZXIgZ3JvdXBcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvVXNlckd1aWRlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWwjVXNpbmdXaXRoUkRTLklBTS5Db25kaXRpb25zXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9DcmVhdGVCbHVlR3JlZW5EZXBsb3ltZW50KClcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBjbHVzdGVyLXBnXG4gICAqXG4gICAqIEBwYXJhbSB0YWdLZXkgVGhlIHRhZyBrZXkgdG8gY2hlY2tcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZDbHVzdGVyUGdUYWcodGFnS2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBjbHVzdGVyLXBnLXRhZy8keyB0YWdLZXkgfWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSB0YWcgYXR0YWNoZWQgdG8gYSBEQiBjbHVzdGVyIHNuYXBzaG90XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L1VzZXJHdWlkZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sI1VzaW5nV2l0aFJEUy5JQU0uQ29uZGl0aW9uc1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIGNsdXN0ZXItc25hcHNob3RcbiAgICpcbiAgICogQHBhcmFtIHRhZ0tleSBUaGUgdGFnIGtleSB0byBjaGVja1xuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkNsdXN0ZXJTbmFwc2hvdFRhZyh0YWdLZXk6IHN0cmluZywgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGNsdXN0ZXItc25hcHNob3QtdGFnLyR7IHRhZ0tleSB9YCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgdGhlIHRhZyBhdHRhY2hlZCB0byBhIERCIGNsdXN0ZXJcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvVXNlckd1aWRlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWwjVXNpbmdXaXRoUkRTLklBTS5Db25kaXRpb25zXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9DcmVhdGVCbHVlR3JlZW5EZXBsb3ltZW50KClcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBjbHVzdGVyXG4gICAqXG4gICAqIEBwYXJhbSB0YWdLZXkgVGhlIHRhZyBrZXkgdG8gY2hlY2tcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZDbHVzdGVyVGFnKHRhZ0tleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgY2x1c3Rlci10YWcvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgdGFnIGF0dGFjaGVkIHRvIGEgREIgaW5zdGFuY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvVXNlckd1aWRlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWwjVXNpbmdXaXRoUkRTLklBTS5Db25kaXRpb25zXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9DcmVhdGVCbHVlR3JlZW5EZXBsb3ltZW50KClcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBkYlxuICAgKlxuICAgKiBAcGFyYW0gdGFnS2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmRGJUYWcodGFnS2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBkYi10YWcvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgdGFnIGF0dGFjaGVkIHRvIGFuIGV2ZW50IHN1YnNjcmlwdGlvblxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9Vc2VyR3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbCNVc2luZ1dpdGhSRFMuSUFNLkNvbmRpdGlvbnNcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBlc1xuICAgKlxuICAgKiBAcGFyYW0gdGFnS2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmRXNUYWcodGFnS2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBlcy10YWcvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgdGFnIGF0dGFjaGVkIHRvIGEgREIgb3B0aW9uIGdyb3VwXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L1VzZXJHdWlkZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sI1VzaW5nV2l0aFJEUy5JQU0uQ29uZGl0aW9uc1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIG9nXG4gICAqXG4gICAqIEBwYXJhbSB0YWdLZXkgVGhlIHRhZyBrZXkgdG8gY2hlY2tcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZPZ1RhZyh0YWdLZXk6IHN0cmluZywgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYG9nLXRhZy8keyB0YWdLZXkgfWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSB0YWcgYXR0YWNoZWQgdG8gYSBEQiBwYXJhbWV0ZXIgZ3JvdXBcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvVXNlckd1aWRlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWwjVXNpbmdXaXRoUkRTLklBTS5Db25kaXRpb25zXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9DcmVhdGVCbHVlR3JlZW5EZXBsb3ltZW50KClcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBwZ1xuICAgKlxuICAgKiBAcGFyYW0gdGFnS2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmUGdUYWcodGFnS2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBwZy10YWcvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgc2V0IG9mIHRhZyBrZXlzIGFuZCB2YWx1ZXMgdGhhdCBjYW4gYmUgdXNlZCB0byB0YWcgYSByZXNvdXJjZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9Vc2VyR3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbCNVc2luZ1dpdGhSRFMuSUFNLkNvbmRpdGlvbnNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0FkZFRhZ3NUb1Jlc291cmNlKClcbiAgICogLSAudG9DcmVhdGVCbHVlR3JlZW5EZXBsb3ltZW50KClcbiAgICogLSAudG9DcmVhdGVEQkNsdXN0ZXIoKVxuICAgKiAtIC50b0NyZWF0ZURCQ2x1c3RlclBhcmFtZXRlckdyb3VwKClcbiAgICogLSAudG9DcmVhdGVEQkNsdXN0ZXJTbmFwc2hvdCgpXG4gICAqIC0gLnRvQ3JlYXRlREJJbnN0YW5jZSgpXG4gICAqIC0gLnRvQ3JlYXRlREJJbnN0YW5jZVJlYWRSZXBsaWNhKClcbiAgICogLSAudG9DcmVhdGVEQlBhcmFtZXRlckdyb3VwKClcbiAgICogLSAudG9DcmVhdGVEQlNlY3VyaXR5R3JvdXAoKVxuICAgKiAtIC50b0NyZWF0ZURCU25hcHNob3QoKVxuICAgKiAtIC50b0NyZWF0ZURCU3VibmV0R3JvdXAoKVxuICAgKiAtIC50b0NyZWF0ZUV2ZW50U3Vic2NyaXB0aW9uKClcbiAgICogLSAudG9DcmVhdGVJbnRlZ3JhdGlvbigpXG4gICAqIC0gLnRvQ3JlYXRlT3B0aW9uR3JvdXAoKVxuICAgKiAtIC50b1JlbW92ZVRhZ3NGcm9tUmVzb3VyY2UoKVxuICAgKiAtIC50b1Jlc3RvcmVEQkNsdXN0ZXJGcm9tUzMoKVxuICAgKiAtIC50b1Jlc3RvcmVEQkNsdXN0ZXJGcm9tU25hcHNob3QoKVxuICAgKiAtIC50b1Jlc3RvcmVEQkNsdXN0ZXJUb1BvaW50SW5UaW1lKClcbiAgICogLSAudG9SZXN0b3JlREJJbnN0YW5jZUZyb21EQlNuYXBzaG90KClcbiAgICogLSAudG9SZXN0b3JlREJJbnN0YW5jZUZyb21TMygpXG4gICAqIC0gLnRvUmVzdG9yZURCSW5zdGFuY2VUb1BvaW50SW5UaW1lKClcbiAgICpcbiAgICogQHBhcmFtIHRhZ0tleSBUaGUgdGFnIGtleSB0byBjaGVja1xuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlJlcVRhZyh0YWdLZXk6IHN0cmluZywgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYHJlcS10YWcvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgdGFnIGF0dGFjaGVkIHRvIGEgcmVzZXJ2ZWQgREIgaW5zdGFuY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvVXNlckd1aWRlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWwjVXNpbmdXaXRoUkRTLklBTS5Db25kaXRpb25zXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gcmlcbiAgICpcbiAgICogQHBhcmFtIHRhZ0tleSBUaGUgdGFnIGtleSB0byBjaGVja1xuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlJpVGFnKHRhZ0tleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgcmktdGFnLyR7IHRhZ0tleSB9YCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgdGhlIHRhZyBhdHRhY2hlZCB0byBhIERCIHNlY3VyaXR5IGdyb3VwXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L1VzZXJHdWlkZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sI1VzaW5nV2l0aFJEUy5JQU0uQ29uZGl0aW9uc1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIHNlY2dycFxuICAgKlxuICAgKiBAcGFyYW0gdGFnS2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmU2VjZ3JwVGFnKHRhZ0tleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgc2VjZ3JwLXRhZy8keyB0YWdLZXkgfWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSB0YWcgYXR0YWNoZWQgdG8gYSBEQiBzbmFwc2hvdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9Vc2VyR3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbCNVc2luZ1dpdGhSRFMuSUFNLkNvbmRpdGlvbnNcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBzbmFwc2hvdFxuICAgKlxuICAgKiBAcGFyYW0gdGFnS2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmU25hcHNob3RUYWcodGFnS2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBzbmFwc2hvdC10YWcvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgdGFnIGF0dGFjaGVkIHRvIGEgREIgc3VibmV0IGdyb3VwXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L1VzZXJHdWlkZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sI1VzaW5nV2l0aFJEUy5JQU0uQ29uZGl0aW9uc1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIHN1YmdycFxuICAgKlxuICAgKiBAcGFyYW0gdGFnS2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmU3ViZ3JwVGFnKHRhZ0tleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgc3ViZ3JwLXRhZy8keyB0YWdLZXkgfWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbcmRzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hbWF6b25yZHMuaHRtbCkuXG4gICAqXG4gICAqL1xuICBjb25zdHJ1Y3Rvcihwcm9wcz86IGlhbS5Qb2xpY3lTdGF0ZW1lbnRQcm9wcykge1xuICAgIHN1cGVyKHByb3BzKTtcbiAgfVxufVxuIl19