import { AccessLevelList } from '../../shared/access-level';
import { PolicyStatement, Operator } from '../../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [rds-data](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonrdsdataapi.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class RdsData extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to run a batch SQL statement over an array of data
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/rdsdataservice/latest/APIReference/API_BatchExecuteStatement.html
     */
    toBatchExecuteStatement(): this;
    /**
     * Grants permission to start a SQL transaction
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/rdsdataservice/latest/APIReference/API_BeginTransaction.html
     */
    toBeginTransaction(): this;
    /**
     * Grants permission to end a SQL transaction started with the BeginTransaction operation and commits the changes
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - rds-data:BeginTransaction
     *
     * https://docs.aws.amazon.com/rdsdataservice/latest/APIReference/API_CommitTransaction.html
     */
    toCommitTransaction(): this;
    /**
     * Grants permission to run one or more SQL statements. This operation is deprecated. Use the BatchExecuteStatement or ExecuteStatement operation
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/rdsdataservice/latest/APIReference/API_ExecuteSql.html
     */
    toExecuteSql(): this;
    /**
     * Grants permission to run a SQL statement against a database
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/rdsdataservice/latest/APIReference/API_ExecuteStatement.html
     */
    toExecuteStatement(): this;
    /**
     * Grants permission to perform a rollback of a transaction. Rolling back a transaction cancels its changes
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - rds-data:BeginTransaction
     *
     * https://docs.aws.amazon.com/rdsdataservice/latest/APIReference/API_RollbackTransaction.html
     */
    toRollbackTransaction(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type cluster to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Aurora.Managing.html
     *
     * @param dbClusterInstanceName - Identifier for the dbClusterInstanceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     */
    onCluster(dbClusterInstanceName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toBatchExecuteStatement()
     * - .toBeginTransaction()
     * - .toCommitTransaction()
     * - .toExecuteSql()
     * - .toExecuteStatement()
     * - .toRollbackTransaction()
     *
     * Applies to resource types:
     * - cluster
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the tag keys associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toBatchExecuteStatement()
     * - .toBeginTransaction()
     * - .toCommitTransaction()
     * - .toExecuteSql()
     * - .toExecuteStatement()
     * - .toRollbackTransaction()
     *
     * Applies to resource types:
     * - cluster
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [rds-data](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonrdsdataapi.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
