"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ram = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [ram](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsresourceaccessmanagerram.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Ram extends shared_1.PolicyStatement {
    /**
     * Grants permission to accept the specified resource share invitation
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifShareOwnerAccountId()
     * - .ifResourceShareName()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_AcceptResourceShareInvitation.html
     */
    toAcceptResourceShareInvitation() {
        return this.to('AcceptResourceShareInvitation');
    }
    /**
     * Grants permission to associate resource(s) and/or principal(s) to a resource share
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     * - .ifResourceShareName()
     * - .ifAllowsExternalPrincipals()
     * - .ifPrincipal()
     * - .ifRequestedResourceType()
     * - .ifResourceArn()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_AssociateResourceShare.html
     */
    toAssociateResourceShare() {
        return this.to('AssociateResourceShare');
    }
    /**
     * Grants permission to associate a Permission with a Resource Share
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_AssociateResourceSharePermission.html
     */
    toAssociateResourceSharePermission() {
        return this.to('AssociateResourceSharePermission');
    }
    /**
     * Grants permission to create a Permission that can be associated to a Resource Share
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPermissionArn()
     * - .ifPermissionResourceType()
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ram:TagResource
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_CreatePermission.html
     */
    toCreatePermission() {
        return this.to('CreatePermission');
    }
    /**
     * Grants permission to create a new version of a Permission that can be associated to a Resource Share
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPermissionArn()
     * - .ifPermissionResourceType()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_CreatePermissionVersion.html
     */
    toCreatePermissionVersion() {
        return this.to('CreatePermissionVersion');
    }
    /**
     * Grants permission to create a resource share with provided resource(s) and/or principal(s)
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifRequestedResourceType()
     * - .ifResourceArn()
     * - .ifRequestedAllowsExternalPrincipals()
     * - .ifPrincipal()
     * - .ifAllowsExternalPrincipals()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_CreateResourceShare.html
     */
    toCreateResourceShare() {
        return this.to('CreateResourceShare');
    }
    /**
     * Grants permission to delete a specified Permission
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifPermissionArn()
     * - .ifPermissionResourceType()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_DeletePermission.html
     */
    toDeletePermission() {
        return this.to('DeletePermission');
    }
    /**
     * Grants permission to delete a specified version of a permission
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPermissionArn()
     * - .ifPermissionResourceType()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_DeletePermissionVersion.html
     */
    toDeletePermissionVersion() {
        return this.to('DeletePermissionVersion');
    }
    /**
     * Grants permission to delete resource share
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     * - .ifResourceShareName()
     * - .ifAllowsExternalPrincipals()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_DeleteResourceShare.html
     */
    toDeleteResourceShare() {
        return this.to('DeleteResourceShare');
    }
    /**
     * Grants permission to disassociate resource(s) and/or principal(s) from a resource share
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     * - .ifResourceShareName()
     * - .ifAllowsExternalPrincipals()
     * - .ifPrincipal()
     * - .ifRequestedResourceType()
     * - .ifResourceArn()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_DisassociateResourceShare.html
     */
    toDisassociateResourceShare() {
        return this.to('DisassociateResourceShare');
    }
    /**
     * Grants permission to disassociate a Permission from a Resource Share
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_DisassociateResourceSharePermission.html
     */
    toDisassociateResourceSharePermission() {
        return this.to('DisassociateResourceSharePermission');
    }
    /**
     * Grants permission to access customer's organization and create a SLR in the customer's account
     *
     * Access Level: Permissions management
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - organizations:DescribeOrganization
     * - organizations:EnableAWSServiceAccess
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_EnableSharingWithAwsOrganization.html
     */
    toEnableSharingWithAwsOrganization() {
        return this.to('EnableSharingWithAwsOrganization');
    }
    /**
     * Grants permission to get the contents of an AWS RAM permission
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifPermissionArn()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_GetPermission.html
     */
    toGetPermission() {
        return this.to('GetPermission');
    }
    /**
     * Grants permission to get the policies for the specified resources that you own and have shared
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_GetResourcePolicies.html
     */
    toGetResourcePolicies() {
        return this.to('GetResourcePolicies');
    }
    /**
     * Grants permission to get a set of resource share associations from a provided list or with a specified status of the specified type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_GetResourceShareAssociations.html
     */
    toGetResourceShareAssociations() {
        return this.to('GetResourceShareAssociations');
    }
    /**
     * Grants permission to get resource share invitations by the specified invitation arn or those for the resource share
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_GetResourceShareInvitations.html
     */
    toGetResourceShareInvitations() {
        return this.to('GetResourceShareInvitations');
    }
    /**
     * Grants permission to get a set of resource shares from a provided list or with a specified status
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_GetResourceShares.html
     */
    toGetResourceShares() {
        return this.to('GetResourceShares');
    }
    /**
     * Grants permission to list the resources in a resource share that is shared with you but that the invitation is still pending for
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifResourceShareName()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ListPendingInvitationResources.html
     */
    toListPendingInvitationResources() {
        return this.to('ListPendingInvitationResources');
    }
    /**
     * Grants permission to list information about the permission and any associations
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifPermissionArn()
     * - .ifPermissionResourceType()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ListPermissionAssociations.html
     */
    toListPermissionAssociations() {
        return this.to('ListPermissionAssociations');
    }
    /**
     * Grants permission to list the versions of an AWS RAM permission
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ListPermissionVersions.html
     */
    toListPermissionVersions() {
        return this.to('ListPermissionVersions');
    }
    /**
     * Grants permission to list the AWS RAM permissions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ListPermissions.html
     */
    toListPermissions() {
        return this.to('ListPermissions');
    }
    /**
     * Grants permission to list the principals that you have shared resources with or that have shared resources with you
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ListPrincipals.html
     */
    toListPrincipals() {
        return this.to('ListPrincipals');
    }
    /**
     * Grants permission to retrieve the status of the asynchronous permission replacement
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ListReplacePermissionAssociationsWork.html
     */
    toListReplacePermissionAssociationsWork() {
        return this.to('ListReplacePermissionAssociationsWork');
    }
    /**
     * Grants permission to list the Permissions associated with a Resource Share
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceShareName()
     * - .ifAllowsExternalPrincipals()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ListResourceSharePermissions.html
     */
    toListResourceSharePermissions() {
        return this.to('ListResourceSharePermissions');
    }
    /**
     * Grants permission to list the shareable resource types supported by AWS RAM
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ListResourceTypes.html
     */
    toListResourceTypes() {
        return this.to('ListResourceTypes');
    }
    /**
     * Grants permission to list the resources that you added to resource shares or the resources that are shared with you
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ListResources.html
     */
    toListResources() {
        return this.to('ListResources');
    }
    /**
     * Grants permission to create a separate, fully manageable customer managed permission
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPermissionArn()
     * - .ifPermissionResourceType()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_PromotePermissionCreatedFromPolicy.html
     */
    toPromotePermissionCreatedFromPolicy() {
        return this.to('PromotePermissionCreatedFromPolicy');
    }
    /**
     * Grants permission to promote the specified resource share
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_PromoteResourceShareCreatedFromPolicy.html
     */
    toPromoteResourceShareCreatedFromPolicy() {
        return this.to('PromoteResourceShareCreatedFromPolicy');
    }
    /**
     * Grants permission to reject the specified resource share invitation
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifShareOwnerAccountId()
     * - .ifResourceShareName()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_RejectResourceShareInvitation.html
     */
    toRejectResourceShareInvitation() {
        return this.to('RejectResourceShareInvitation');
    }
    /**
     * Grants permission to update all resource shares to a new permission
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPermissionArn()
     * - .ifPermissionResourceType()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ReplacePermissionAssociations.html
     */
    toReplacePermissionAssociations() {
        return this.to('ReplacePermissionAssociations');
    }
    /**
     * Grants permission to specify a version number as the default version for the respective customer managed permission
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPermissionArn()
     * - .ifPermissionResourceType()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_SetDefaultPermissionVersion.html
     */
    toSetDefaultPermissionVersion() {
        return this.to('SetDefaultPermissionVersion');
    }
    /**
     * Grants permission to tag the specified resource share or permission
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag the specified resource share or permission
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update attributes of the resource share
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     * - .ifResourceShareName()
     * - .ifAllowsExternalPrincipals()
     * - .ifRequestedAllowsExternalPrincipals()
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_UpdateResourceShare.html
     */
    toUpdateResourceShare() {
        return this.to('UpdateResourceShare');
    }
    /**
     * Adds a resource of type resource-share to the statement
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ResourceShare.html
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAllowsExternalPrincipals()
     * - .ifResourceShareName()
     */
    onResourceShare(resourcePath, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ram:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:resource-share/${resourcePath}`);
    }
    /**
     * Adds a resource of type resource-share-invitation to the statement
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ResourceShareInvitation.html
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifShareOwnerAccountId()
     */
    onResourceShareInvitation(resourcePath, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ram:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:resource-share-invitation/${resourcePath}`);
    }
    /**
     * Adds a resource of type permission to the statement
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ResourceSharePermissionDetail.html
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifPermissionArn()
     * - .ifPermissionResourceType()
     */
    onPermission(resourcePath, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ram::${account ?? this.defaultAccount}:permission/${resourcePath}`);
    }
    /**
     * Adds a resource of type customer-managed-permission to the statement
     *
     * https://docs.aws.amazon.com/ram/latest/APIReference/API_ResourceSharePermissionDetail.html
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifPermissionArn()
     * - .ifPermissionResourceType()
     */
    onCustomerManagedPermission(resourcePath, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ram:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:permission/${resourcePath}`);
    }
    /**
     * Filters access by the tags that are passed in the request when creating or tagging a resource share. If users don't pass these specific tags, or if they don't specify tags at all, the request fails
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreatePermission()
     * - .toCreateResourceShare()
     * - .toGetResourceShares()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toAssociateResourceShare()
     * - .toCreatePermission()
     * - .toDeletePermission()
     * - .toDeleteResourceShare()
     * - .toDisassociateResourceShare()
     * - .toListResourceSharePermissions()
     * - .toUpdateResourceShare()
     *
     * Applies to resource types:
     * - resource-share
     * - customer-managed-permission
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed when creating or tagging a resource share
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreatePermission()
     * - .toCreateResourceShare()
     * - .toGetResourceShares()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by resource shares that allow or deny sharing with external principals. For example, specify true if the action can only be performed on resource shares that allow sharing with external principals. External principals are AWS accounts that are outside of its AWS organization
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * Applies to actions:
     * - .toAssociateResourceShare()
     * - .toCreateResourceShare()
     * - .toDeleteResourceShare()
     * - .toDisassociateResourceShare()
     * - .toListResourceSharePermissions()
     * - .toUpdateResourceShare()
     *
     * Applies to resource types:
     * - resource-share
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifAllowsExternalPrincipals(value) {
        return this.if(`AllowsExternalPrincipals`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the specified Permission ARN
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * Applies to actions:
     * - .toCreatePermission()
     * - .toCreatePermissionVersion()
     * - .toDeletePermission()
     * - .toDeletePermissionVersion()
     * - .toGetPermission()
     * - .toListPermissionAssociations()
     * - .toPromotePermissionCreatedFromPolicy()
     * - .toReplacePermissionAssociations()
     * - .toSetDefaultPermissionVersion()
     *
     * Applies to resource types:
     * - permission
     * - customer-managed-permission
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifPermissionArn(value, operator) {
        return this.if(`PermissionArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by permissions of specified resource type
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * Applies to actions:
     * - .toCreatePermission()
     * - .toCreatePermissionVersion()
     * - .toDeletePermission()
     * - .toDeletePermissionVersion()
     * - .toListPermissionAssociations()
     * - .toPromotePermissionCreatedFromPolicy()
     * - .toReplacePermissionAssociations()
     * - .toSetDefaultPermissionVersion()
     *
     * Applies to resource types:
     * - permission
     * - customer-managed-permission
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPermissionResourceType(value, operator) {
        return this.if(`PermissionResourceType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by format of the specified principal
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * Applies to actions:
     * - .toAssociateResourceShare()
     * - .toCreateResourceShare()
     * - .toDisassociateResourceShare()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPrincipal(value, operator) {
        return this.if(`Principal`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the specified value for 'allowExternalPrincipals'. External principals are AWS accounts that are outside of its AWS Organization
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * Applies to actions:
     * - .toCreateResourceShare()
     * - .toUpdateResourceShare()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifRequestedAllowsExternalPrincipals(value) {
        return this.if(`RequestedAllowsExternalPrincipals`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the specified resource type
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * Applies to actions:
     * - .toAssociateResourceShare()
     * - .toCreateResourceShare()
     * - .toDisassociateResourceShare()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestedResourceType(value, operator) {
        return this.if(`RequestedResourceType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the specified ARN
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * Applies to actions:
     * - .toAssociateResourceShare()
     * - .toCreateResourceShare()
     * - .toDisassociateResourceShare()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifResourceArn(value, operator) {
        return this.if(`ResourceArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by a resource share with the specified name
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * Applies to actions:
     * - .toAcceptResourceShareInvitation()
     * - .toAssociateResourceShare()
     * - .toDeleteResourceShare()
     * - .toDisassociateResourceShare()
     * - .toListPendingInvitationResources()
     * - .toListResourceSharePermissions()
     * - .toRejectResourceShareInvitation()
     * - .toUpdateResourceShare()
     *
     * Applies to resource types:
     * - resource-share
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceShareName(value, operator) {
        return this.if(`ResourceShareName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toAssociateResourceShare()
     * - .toDeleteResourceShare()
     * - .toDisassociateResourceShare()
     * - .toUpdateResourceShare()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagKey, value, operator) {
        return this.if(`ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by resource shares owned by a specific account. For example, you can use this condition key to specify which resource share invitations can be accepted or rejected based on the resource share owner's account ID
     *
     * https://docs.aws.amazon.com/ram/latest/userguide/iam-policies.html#iam-policies-condition
     *
     * Applies to actions:
     * - .toAcceptResourceShareInvitation()
     * - .toRejectResourceShareInvitation()
     *
     * Applies to resource types:
     * - resource-share-invitation
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifShareOwnerAccountId(value, operator) {
        return this.if(`ShareOwnerAccountId`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [ram](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsresourceaccessmanagerram.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'ram';
        this.accessLevelList = {
            Write: [
                'AcceptResourceShareInvitation',
                'AssociateResourceShare',
                'AssociateResourceSharePermission',
                'CreatePermission',
                'CreatePermissionVersion',
                'CreateResourceShare',
                'DeletePermission',
                'DeletePermissionVersion',
                'DeleteResourceShare',
                'DisassociateResourceShare',
                'DisassociateResourceSharePermission',
                'PromotePermissionCreatedFromPolicy',
                'PromoteResourceShareCreatedFromPolicy',
                'RejectResourceShareInvitation',
                'ReplacePermissionAssociations',
                'SetDefaultPermissionVersion',
                'UpdateResourceShare'
            ],
            'Permissions management': [
                'EnableSharingWithAwsOrganization'
            ],
            Read: [
                'GetPermission',
                'GetResourcePolicies',
                'GetResourceShareAssociations',
                'GetResourceShareInvitations',
                'GetResourceShares',
                'ListPendingInvitationResources'
            ],
            List: [
                'ListPermissionAssociations',
                'ListPermissionVersions',
                'ListPermissions',
                'ListPrincipals',
                'ListReplacePermissionAssociationsWork',
                'ListResourceSharePermissions',
                'ListResourceTypes',
                'ListResources'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Ram = Ram;
//# sourceMappingURL=data:application/json;base64,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