import { AccessLevelList } from '../../shared/access-level';
import { PolicyStatement, Operator } from '../../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [resource-groups](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsresourcegroups.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class ResourceGroups extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to associate a resource to an Application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/arguide/associate-resources.html
     */
    toAssociateResource(): this;
    /**
     * Grants permission to cancel a tag-sync task for an application group
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - resource-groups:DeleteGroup
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_CancelTagSyncTask.html
     */
    toCancelTagSyncTask(): this;
    /**
     * Grants permission to create a resource group with a specified name, description, and resource query
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - cloudformation:DescribeStacks
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_CreateGroup.html
     */
    toCreateGroup(): this;
    /**
     * Grants permission to delete a specified resource group
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - tag:GetResources
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_DeleteGroup.html
     */
    toDeleteGroup(): this;
    /**
     * Grants permission to delete a resource-based policy for the specified group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/license-manager/latest/userguide/management-role.html#service-linked-role-permissions-management-role
     */
    toDeleteGroupPolicy(): this;
    /**
     * Grants permission to disassociate a resource from an Application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/arguide/associate-resources.html
     */
    toDisassociateResource(): this;
    /**
     * Grants permission to get the current status of optional features in Resource Groups
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_GetAccountSettings.html
     */
    toGetAccountSettings(): this;
    /**
     * Grants permission to get information of a specified resource group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_GetGroup.html
     */
    toGetGroup(): this;
    /**
     * Grants permission to get the service configuration associated with the specified resource group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_GetGroupConfiguration.html
     */
    toGetGroupConfiguration(): this;
    /**
     * Grants permission to get a resource-based policy for the specified group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/license-manager/latest/userguide/management-role.html#service-linked-role-permissions-management-role
     */
    toGetGroupPolicy(): this;
    /**
     * Grants permission to get the query associated with a specified resource group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_GetGroupQuery.html
     */
    toGetGroupQuery(): this;
    /**
     * Grants permission to get information of a specified tag-sync task
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_GetTagSyncTask.html
     */
    toGetTagSyncTask(): this;
    /**
     * Grants permission to get the tags associated with a specified resource group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_GetTags.html
     */
    toGetTags(): this;
    /**
     * Grants permission to add the specified resources to the specified group
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - resource-groups:Tag
     * - tag:TagResources
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_GroupResources.html
     */
    toGroupResources(): this;
    /**
     * Grants permission to list the resources that are members of a specified resource group
     *
     * Access Level: List
     *
     * Dependent actions:
     * - cloudformation:DescribeStacks
     * - cloudformation:ListStackResources
     * - tag:GetResources
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_ListGroupResources.html
     */
    toListGroupResources(): this;
    /**
     * Grants permission to list grouping statuses for a specified application group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_ListGroupingStatuses.html
     */
    toListGroupingStatuses(): this;
    /**
     * Grants permission to list all resource groups in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_ListGroups.html
     */
    toListGroups(): this;
    /**
     * Grants permission to list supported resource types
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ARG/latest/userguide/gettingstarted-query.html
     */
    toListResourceTypes(): this;
    /**
     * Grants permission to list all tag-sync tasks in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_ListTagSyncTasks.html
     */
    toListTagSyncTasks(): this;
    /**
     * Grants permission to put the service configuration associated with the specified resource group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_PutGroupConfiguration.html
     */
    toPutGroupConfiguration(): this;
    /**
     * Grants permission to add a resource-based policy for the specified group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/license-manager/latest/userguide/management-role.html#service-linked-role-permissions-management-role
     */
    toPutGroupPolicy(): this;
    /**
     * Grants permission to search for AWS resources matching the given query
     *
     * Access Level: List
     *
     * Dependent actions:
     * - cloudformation:DescribeStacks
     * - cloudformation:ListStackResources
     * - tag:GetResources
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_SearchResources.html
     */
    toSearchResources(): this;
    /**
     * Grants permission to create a tag-sync task for an application group
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     * - resource-groups:CreateGroup
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_StartTagSyncTask.html
     */
    toStartTagSyncTask(): this;
    /**
     * Grants permission to tag a specified resource group
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_Tag.html
     */
    toTag(): this;
    /**
     * Grants permission to remove the specified resources from the specified group
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - resource-groups:Untag
     * - tag:UntagResources
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_UngroupResources.html
     */
    toUngroupResources(): this;
    /**
     * Grants permission to remove tags associated with a specified resource group
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_Untag.html
     */
    toUntag(): this;
    /**
     * Grants permission to update optional features in Resource Groups
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_UpdateAccountSettings.html
     */
    toUpdateAccountSettings(): this;
    /**
     * Grants permission to update a specified resource group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_UpdateGroup.html
     */
    toUpdateGroup(): this;
    /**
     * Grants permission to update the query associated with a specified resource group
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - cloudformation:DescribeStacks
     *
     * https://docs.aws.amazon.com/ARG/latest/APIReference/API_UpdateGroupQuery.html
     */
    toUpdateGroupQuery(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type group to the statement
     *
     * https://docs.aws.amazon.com/ARG/latest/userguide/resource-groups.html
     *
     * @param groupName - Identifier for the groupName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGroup(groupName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type tagSyncTask to the statement
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/arguide/app-tag-sync.html
     *
     * @param groupName - Identifier for the groupName.
     * @param taskId - Identifier for the taskId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTagSyncTask(groupName: string, taskId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateGroup()
     * - .toTag()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - group
     * - tagSyncTask
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateGroup()
     * - .toTag()
     * - .toUntag()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [resource-groups](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsresourcegroups.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
