"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3express = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [s3express](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazons3express.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class S3express extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a new access point
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifLocationName()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateAccessPoint.html
     */
    toCreateAccessPoint() {
        return this.to('CreateAccessPoint');
    }
    /**
     * Grants permission to create a new bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifLocationName()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_CreateBucket.html
     */
    toCreateBucket() {
        return this.to('CreateBucket');
    }
    /**
     * Grants permission to Create Session token which is used for object APIs such as PutObject, GetObject, ect
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSessionMode()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifXAmzServerSideEncryption()
     * - .ifXAmzServerSideEncryptionAwsKmsKeyId()
     * - .ifAllAccessRestrictedToLocalZoneGroup()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_CreateSession.html
     */
    toCreateSession() {
        return this.to('CreateSession');
    }
    /**
     * Grants permission to delete the access point named in the URI
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPoint.html
     */
    toDeleteAccessPoint() {
        return this.to('DeleteAccessPoint');
    }
    /**
     * Grants permission to delete the policy on a specified access point
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPointPolicy.html
     */
    toDeleteAccessPointPolicy() {
        return this.to('DeleteAccessPointPolicy');
    }
    /**
     * Grants permission to delete the scope configuration on a specified access point
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPointScope.html
     */
    toDeleteAccessPointScope() {
        return this.to('DeleteAccessPointScope');
    }
    /**
     * Grants permission to delete the bucket named in the URI
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucket.html
     */
    toDeleteBucket() {
        return this.to('DeleteBucket');
    }
    /**
     * Grants permission to delete the policy on a specified bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucketPolicy.html
     */
    toDeleteBucketPolicy() {
        return this.to('DeleteBucketPolicy');
    }
    /**
     * Grants permission to return configuration information about the specified access point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPoint.html
     */
    toGetAccessPoint() {
        return this.to('GetAccessPoint');
    }
    /**
     * Grants permission to return the access point policy associated with the specified access point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointPolicy.html
     */
    toGetAccessPointPolicy() {
        return this.to('GetAccessPointPolicy');
    }
    /**
     * Grants permission to return the scope configuration associated with the specified access point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointScope.html
     */
    toGetAccessPointScope() {
        return this.to('GetAccessPointScope');
    }
    /**
     * Grants permission to return the policy of the specified bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketPolicy.html
     */
    toGetBucketPolicy() {
        return this.to('GetBucketPolicy');
    }
    /**
     * Grants permission to return the default encryption configuration for a directory bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketEncryption.html
     */
    toGetEncryptionConfiguration() {
        return this.to('GetEncryptionConfiguration');
    }
    /**
     * Grants permission to return the lifecycle configuration information set on a directory bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLifecycleConfiguration.html
     */
    toGetLifecycleConfiguration() {
        return this.to('GetLifecycleConfiguration');
    }
    /**
     * Grants permission to list access points
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListAccessPointsForDirectoryBuckets.html
     */
    toListAccessPointsForDirectoryBuckets() {
        return this.to('ListAccessPointsForDirectoryBuckets');
    }
    /**
     * Grants permission to list all directory buckets owned by the authenticated sender of the request
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListDirectoryBuckets.html
     */
    toListAllMyDirectoryBuckets() {
        return this.to('ListAllMyDirectoryBuckets');
    }
    /**
     * Grants permission to associate an access policy with a specified access point
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutAccessPointPolicy.html
     */
    toPutAccessPointPolicy() {
        return this.to('PutAccessPointPolicy');
    }
    /**
     * Grants permission to associate an access point with a specified access point scope configuration
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutAccessPointScope.html
     */
    toPutAccessPointScope() {
        return this.to('PutAccessPointScope');
    }
    /**
     * Grants permission to add or replace a bucket policy on a bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketPolicy.html
     */
    toPutBucketPolicy() {
        return this.to('PutBucketPolicy');
    }
    /**
     * Grants permission to set the encryption configuration for a directory bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketEncryption.html
     */
    toPutEncryptionConfiguration() {
        return this.to('PutEncryptionConfiguration');
    }
    /**
     * Grants permission to create a new lifecycle configuration for the directory bucket or replace an existing lifecycle configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketLifecycleConfiguration.html
     */
    toPutLifecycleConfiguration() {
        return this.to('PutLifecycleConfiguration');
    }
    /**
     * Adds a resource of type bucket to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-express-security-iam.html
     *
     * @param bucketName - Identifier for the bucketName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onBucket(bucketName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:s3express:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:bucket/${bucketName}`);
    }
    /**
     * Adds a resource of type accesspoint to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/access-points.html
     *
     * @param accessPointName - Identifier for the accessPointName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAccesspoint(accessPointName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:s3express:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:accesspoint/${accessPointName}`);
    }
    /**
     * Filters access by the network origin (Internet or VPC)
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/creating-access-points.html#access-points-policies
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointScope()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointScope()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointScope()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAccessPointNetworkOrigin(value, operator) {
        return this.if(`AccessPointNetworkOrigin`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by AWS Local Zone network border group(s) provided in this condition key
     *
     * Applies to actions:
     * - .toCreateSession()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAllAccessRestrictedToLocalZoneGroup(value, operator) {
        return this.if(`AllAccessRestrictedToLocalZoneGroup`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the AWS Account ID that owns the access point
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/creating-access-points.html#access-points-policies
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointScope()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointScope()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointScope()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDataAccessPointAccount(value, operator) {
        return this.if(`DataAccessPointAccount`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by an access point Amazon Resource Name (ARN)
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/creating-access-points.html#access-points-policies
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointScope()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointScope()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointScope()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifDataAccessPointArn(value, operator) {
        return this.if(`DataAccessPointArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by a specific Availability Zone ID
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/amazon-s3-express-zonal-policy-keys.html#example-location-name
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLocationName(value, operator) {
        return this.if(`LocationName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the permission requested by Access Point Scope configuration, such as GetObject, PutObject
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPermissions(value, operator) {
        return this.if(`Permissions`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the resource owner AWS account ID
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/amazon-s3-express-zonal-policy-keys.html#example-object-resource-account
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toCreateSession()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointScope()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointScope()
     * - .toGetBucketPolicy()
     * - .toGetEncryptionConfiguration()
     * - .toGetLifecycleConfiguration()
     * - .toListAccessPointsForDirectoryBuckets()
     * - .toListAllMyDirectoryBuckets()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointScope()
     * - .toPutBucketPolicy()
     * - .toPutEncryptionConfiguration()
     * - .toPutLifecycleConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceAccount(value, operator) {
        return this.if(`ResourceAccount`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the permission requested by CreateSession API, such as ReadOnly and ReadWrite
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/amazon-s3-express-zonal-policy-keys.html#example-session-mode
     *
     * Applies to actions:
     * - .toCreateSession()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSessionMode(value, operator) {
        return this.if(`SessionMode`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the TLS version used by the client
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toCreateSession()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointScope()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointScope()
     * - .toGetBucketPolicy()
     * - .toGetEncryptionConfiguration()
     * - .toGetLifecycleConfiguration()
     * - .toListAccessPointsForDirectoryBuckets()
     * - .toListAllMyDirectoryBuckets()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointScope()
     * - .toPutBucketPolicy()
     * - .toPutEncryptionConfiguration()
     * - .toPutLifecycleConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifTlsVersion(value, operator) {
        return this.if(`TlsVersion`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by authentication method
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toCreateSession()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointScope()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointScope()
     * - .toGetBucketPolicy()
     * - .toGetEncryptionConfiguration()
     * - .toGetLifecycleConfiguration()
     * - .toListAccessPointsForDirectoryBuckets()
     * - .toListAllMyDirectoryBuckets()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointScope()
     * - .toPutBucketPolicy()
     * - .toPutEncryptionConfiguration()
     * - .toPutLifecycleConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAuthType(value, operator) {
        return this.if(`authType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the age in milliseconds of the request signature
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toCreateSession()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifSignatureAge(value, operator) {
        return this.if(`signatureAge`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by the AWS Signature Version used on the request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toCreateSession()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointScope()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointScope()
     * - .toGetBucketPolicy()
     * - .toGetEncryptionConfiguration()
     * - .toGetLifecycleConfiguration()
     * - .toListAccessPointsForDirectoryBuckets()
     * - .toListAllMyDirectoryBuckets()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointScope()
     * - .toPutBucketPolicy()
     * - .toPutEncryptionConfiguration()
     * - .toPutLifecycleConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSignatureversion(value, operator) {
        return this.if(`signatureversion`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by unsigned content in your bucket
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toCreateSession()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointScope()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointScope()
     * - .toGetBucketPolicy()
     * - .toGetEncryptionConfiguration()
     * - .toGetLifecycleConfiguration()
     * - .toListAccessPointsForDirectoryBuckets()
     * - .toListAllMyDirectoryBuckets()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointScope()
     * - .toPutBucketPolicy()
     * - .toPutEncryptionConfiguration()
     * - .toPutLifecycleConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzContentSha256(value, operator) {
        return this.if(`x-amz-content-sha256`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by server-side encryption
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-express-data-protection.html
     *
     * Applies to actions:
     * - .toCreateSession()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzServerSideEncryption(value, operator) {
        return this.if(`x-amz-server-side-encryption`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by AWS KMS customer managed key for server-side encryption
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-express-UsingKMSEncryption.html#s3-express-require-sse-kms
     *
     * Applies to actions:
     * - .toCreateSession()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifXAmzServerSideEncryptionAwsKmsKeyId(value, operator) {
        return this.if(`x-amz-server-side-encryption-aws-kms-key-id`, value, operator ?? 'ArnLike');
    }
    /**
     * Statement provider for service [s3express](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazons3express.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 's3express';
        this.accessLevelList = {
            Write: [
                'CreateAccessPoint',
                'CreateBucket',
                'CreateSession',
                'DeleteAccessPoint',
                'DeleteBucket',
                'PutEncryptionConfiguration',
                'PutLifecycleConfiguration'
            ],
            'Permissions management': [
                'DeleteAccessPointPolicy',
                'DeleteAccessPointScope',
                'DeleteBucketPolicy',
                'PutAccessPointPolicy',
                'PutAccessPointScope',
                'PutBucketPolicy'
            ],
            Read: [
                'GetAccessPoint',
                'GetAccessPointPolicy',
                'GetAccessPointScope',
                'GetBucketPolicy',
                'GetEncryptionConfiguration',
                'GetLifecycleConfiguration'
            ],
            List: [
                'ListAccessPointsForDirectoryBuckets',
                'ListAllMyDirectoryBuckets'
            ]
        };
    }
}
exports.S3express = S3express;
//# sourceMappingURL=data:application/json;base64,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