"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3ObjectLambda = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [s3-object-lambda](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazons3objectlambda.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class S3ObjectLambda extends shared_1.PolicyStatement {
    /**
     * Grants permission to abort a multipart upload
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_AbortMultipartUpload.html
     */
    toAbortMultipartUpload() {
        return this.to('AbortMultipartUpload');
    }
    /**
     * Grants permission to remove the null version of an object and insert a delete marker, which becomes the current version of the object
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObject.html
     */
    toDeleteObject() {
        return this.to('DeleteObject');
    }
    /**
     * Grants permission to use the tagging subresource to remove the entire tag set from the specified object
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObjectTagging.html
     */
    toDeleteObjectTagging() {
        return this.to('DeleteObjectTagging');
    }
    /**
     * Grants permission to remove a specific version of an object
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     * - .ifVersionid()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObject.html
     */
    toDeleteObjectVersion() {
        return this.to('DeleteObjectVersion');
    }
    /**
     * Grants permission to remove the entire tag set for a specific version of the object
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     * - .ifVersionid()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObjectTagging.html
     */
    toDeleteObjectVersionTagging() {
        return this.to('DeleteObjectVersionTagging');
    }
    /**
     * Grants permission to retrieve objects from Amazon S3
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObject.html
     */
    toGetObject() {
        return this.to('GetObject');
    }
    /**
     * Grants permission to return the access control list (ACL) of an object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectAcl.html
     */
    toGetObjectAcl() {
        return this.to('GetObjectAcl');
    }
    /**
     * Grants permission to get an object's current Legal Hold status
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectLegalHold.html
     */
    toGetObjectLegalHold() {
        return this.to('GetObjectLegalHold');
    }
    /**
     * Grants permission to retrieve the retention settings for an object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectRetention.html
     */
    toGetObjectRetention() {
        return this.to('GetObjectRetention');
    }
    /**
     * Grants permission to return the tag set of an object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectTagging.html
     */
    toGetObjectTagging() {
        return this.to('GetObjectTagging');
    }
    /**
     * Grants permission to retrieve a specific version of an object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     * - .ifVersionid()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObject.html
     */
    toGetObjectVersion() {
        return this.to('GetObjectVersion');
    }
    /**
     * Grants permission to return the access control list (ACL) of a specific object version
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     * - .ifVersionid()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectAcl.html
     */
    toGetObjectVersionAcl() {
        return this.to('GetObjectVersionAcl');
    }
    /**
     * Grants permission to return the tag set for a specific version of the object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     * - .ifVersionid()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/setting-repl-config-perm-overview.html
     */
    toGetObjectVersionTagging() {
        return this.to('GetObjectVersionTagging');
    }
    /**
     * Grants permission to list some or all of the objects in an Amazon S3 bucket (up to 1000)
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListObjectsV2.html
     */
    toListBucket() {
        return this.to('ListBucket');
    }
    /**
     * Grants permission to list in-progress multipart uploads
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListMultipartUploads.html
     */
    toListBucketMultipartUploads() {
        return this.to('ListBucketMultipartUploads');
    }
    /**
     * Grants permission to list metadata about all the versions of objects in an Amazon S3 bucket
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListObjectVersions.html
     */
    toListBucketVersions() {
        return this.to('ListBucketVersions');
    }
    /**
     * Grants permission to list the parts that have been uploaded for a specific multipart upload
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListParts.html
     */
    toListMultipartUploadParts() {
        return this.to('ListMultipartUploadParts');
    }
    /**
     * Grants permission to add an object to a bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObject.html
     */
    toPutObject() {
        return this.to('PutObject');
    }
    /**
     * Grants permission to set the access control list (ACL) permissions for new or existing objects in an S3 bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectAcl.html
     */
    toPutObjectAcl() {
        return this.to('PutObjectAcl');
    }
    /**
     * Grants permission to apply a Legal Hold configuration to the specified object
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectLegalHold.html
     */
    toPutObjectLegalHold() {
        return this.to('PutObjectLegalHold');
    }
    /**
     * Grants permission to place an Object Retention configuration on an object
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectRetention.html
     */
    toPutObjectRetention() {
        return this.to('PutObjectRetention');
    }
    /**
     * Grants permission to set the supplied tag-set to an object that already exists in a bucket
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectTagging.html
     */
    toPutObjectTagging() {
        return this.to('PutObjectTagging');
    }
    /**
     * Grants permission to use the acl subresource to set the access control list (ACL) permissions for an object that already exists in a bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     * - .ifVersionid()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectAcl.html
     */
    toPutObjectVersionAcl() {
        return this.to('PutObjectVersionAcl');
    }
    /**
     * Grants permission to set the supplied tag-set for a specific version of an object
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     * - .ifVersionid()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectTagging.html
     */
    toPutObjectVersionTagging() {
        return this.to('PutObjectVersionTagging');
    }
    /**
     * Grants permission to restore an archived copy of an object back into Amazon S3
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_RestoreObject.html
     */
    toRestoreObject() {
        return this.to('RestoreObject');
    }
    /**
     * Grants permission to provide data for GetObject requests send to S3 Object Lambda
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_WriteGetObjectResponse.html
     */
    toWriteGetObjectResponse() {
        return this.to('WriteGetObjectResponse');
    }
    /**
     * Adds a resource of type objectlambdaaccesspoint to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/transforming-objects.html
     *
     * @param accessPointName - Identifier for the accessPointName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onObjectlambdaaccesspoint(accessPointName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:s3-object-lambda:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:accesspoint/${accessPointName}`);
    }
    /**
     * Filters access by the TLS version used by the client
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionTagging()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListMultipartUploadParts()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toRestoreObject()
     * - .toWriteGetObjectResponse()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifTlsVersion(value, operator) {
        return this.if(`TlsVersion`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by authentication method
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionTagging()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListMultipartUploadParts()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toRestoreObject()
     * - .toWriteGetObjectResponse()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAuthType(value, operator) {
        return this.if(`authType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the age in milliseconds of the request signature
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionTagging()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListMultipartUploadParts()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toRestoreObject()
     * - .toWriteGetObjectResponse()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifSignatureAge(value, operator) {
        return this.if(`signatureAge`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by a specific object version
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html/#getobjectversion-limit-access-to-specific-version-3
     *
     * Applies to actions:
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVersionid(value, operator) {
        return this.if(`versionid`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [s3-object-lambda](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazons3objectlambda.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 's3-object-lambda';
        this.accessLevelList = {
            Write: [
                'AbortMultipartUpload',
                'DeleteObject',
                'DeleteObjectVersion',
                'PutObject',
                'PutObjectLegalHold',
                'PutObjectRetention',
                'RestoreObject',
                'WriteGetObjectResponse'
            ],
            Tagging: [
                'DeleteObjectTagging',
                'DeleteObjectVersionTagging',
                'PutObjectTagging',
                'PutObjectVersionTagging'
            ],
            Read: [
                'GetObject',
                'GetObjectAcl',
                'GetObjectLegalHold',
                'GetObjectRetention',
                'GetObjectTagging',
                'GetObjectVersion',
                'GetObjectVersionAcl',
                'GetObjectVersionTagging'
            ],
            List: [
                'ListBucket',
                'ListBucketMultipartUploads',
                'ListBucketVersions',
                'ListMultipartUploadParts'
            ],
            'Permissions management': [
                'PutObjectAcl',
                'PutObjectVersionAcl'
            ]
        };
    }
}
exports.S3ObjectLambda = S3ObjectLambda;
//# sourceMappingURL=data:application/json;base64,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