import { AccessLevelList } from '../../shared/access-level';
import { PolicyStatement, Operator } from '../../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [sagemaker-geospatial](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonsagemakergeospatialcapabilities.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class SagemakerGeospatial extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to the DeleteEarthObservationJob operation which deletes an existing earth observation job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_DeleteEarthObservationJob.html
     */
    toDeleteEarthObservationJob(): this;
    /**
     * Grants permission to the DeleteVectorEnrichmentJob operation which deletes an existing vector enrichment job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_DeleteVectorEnrichmentJob.html
     */
    toDeleteVectorEnrichmentJob(): this;
    /**
     * Grants permission to copy results of an earth observation job to an S3 location
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_ExportEarthObservationJob.html
     */
    toExportEarthObservationJob(): this;
    /**
     * Grants permission to copy results of an VectorEnrichmentJob to an S3 location
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_ExportVectorEnrichmentJob.html
     */
    toExportVectorEnrichmentJob(): this;
    /**
     * Grants permission to return details about the earth observation job
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_GetEarthObservationJob.html
     */
    toGetEarthObservationJob(): this;
    /**
     * Grants permission to return details about the raster data collection
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_GetRasterDataCollection.html
     */
    toGetRasterDataCollection(): this;
    /**
     * Grants permission to get the tile of an earth observation job
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_GetTile.html
     */
    toGetTile(): this;
    /**
     * Grants permission to return details about the vector enrichment job
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_GetVectorEnrichmentJob.html
     */
    toGetVectorEnrichmentJob(): this;
    /**
     * Grants permission to return an array of earth observation jobs associated with the current account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_ListEarthObservationJobs.html
     */
    toListEarthObservationJobs(): this;
    /**
     * Grants permission to return an array of aster data collections associated with the given model name
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_ListRasterDataCollections.html
     */
    toListRasterDataCollections(): this;
    /**
     * Grants permission to lists tag for an SageMaker Geospatial resource
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_ListTagsForResource.html
     */
    toListTagsForResource(): this;
    /**
     * Grants permission to return an array of vector enrichment jobs associated with the current account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_ListVectorEnrichmentJobs.html
     */
    toListVectorEnrichmentJobs(): this;
    /**
     * Grants permission to query raster data collections
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_SearchRasterDataCollection.html
     */
    toSearchRasterDataCollection(): this;
    /**
     * Grants permission to the StartEarthObservationJob operation which starts a new earth observation job to your account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     * - sagemaker-geospatial:TagResource
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_StartEarthObservationJob.html
     */
    toStartEarthObservationJob(): this;
    /**
     * Grants permission to the StartVectorEnrichmentJob operation which starts a new vector enrichment job to your account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     * - sagemaker-geospatial:TagResource
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_StartVectorEnrichmentJob.html
     */
    toStartVectorEnrichmentJob(): this;
    /**
     * Grants permission to the StopEarthObservationJob operation which stops an existing earth observation job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_StopEarthObservationJob.html
     */
    toStopEarthObservationJob(): this;
    /**
     * Grants permission to the StopVectorEnrichmentJob operation which stops an existing vector enrichment job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_StopVectorEnrichmentJob.html
     */
    toStopVectorEnrichmentJob(): this;
    /**
     * Grants permission to tag an SageMaker Geospatial resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_TagResource.html
     */
    toTagResource(): this;
    /**
     * Grants permission to untag an SageMaker Geospatial resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_UntagResource.html
     */
    toUntagResource(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type EarthObservationJob to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/geospatial-eoj.html
     *
     * @param jobID - Identifier for the jobID.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEarthObservationJob(jobID: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type RasterDataCollection to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/geospatial-data-collections.html
     *
     * @param collectionID - Identifier for the collectionID.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRasterDataCollection(collectionID: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type VectorEnrichmentJob to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/geospatial-vej.html
     *
     * @param jobID - Identifier for the jobID.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVectorEnrichmentJob(jobID: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toStartEarthObservationJob()
     * - .toStartVectorEnrichmentJob()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toDeleteEarthObservationJob()
     * - .toDeleteVectorEnrichmentJob()
     * - .toExportEarthObservationJob()
     * - .toExportVectorEnrichmentJob()
     * - .toGetEarthObservationJob()
     * - .toGetRasterDataCollection()
     * - .toGetVectorEnrichmentJob()
     * - .toListTagsForResource()
     * - .toStopEarthObservationJob()
     * - .toStopVectorEnrichmentJob()
     * - .toTagResource()
     *
     * Applies to resource types:
     * - EarthObservationJob
     * - RasterDataCollection
     * - VectorEnrichmentJob
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toStartEarthObservationJob()
     * - .toStartVectorEnrichmentJob()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [sagemaker-geospatial](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonsagemakergeospatialcapabilities.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
