"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Secretsmanager = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [secretsmanager](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssecretsmanager.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Secretsmanager extends shared_1.PolicyStatement {
    /**
     * Grants permission to retrieve and decrypt a list of secrets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/apireference/API_BatchGetSecretValue.html
     */
    toBatchGetSecretValue() {
        return this.to('BatchGetSecretValue');
    }
    /**
     * Grants permission to cancel an in-progress secret rotation
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     * - .ifAwsResourceTag()
     * - .ifSecretPrimaryRegion()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/apireference/API_CancelRotateSecret.html
     */
    toCancelRotateSecret() {
        return this.to('CancelRotateSecret');
    }
    /**
     * Grants permission to create a secret that stores encrypted data that can be queried and rotated
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifName()
     * - .ifDescription()
     * - .ifKmsKeyArn()
     * - .ifKmsKeyId()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifResourceTag()
     * - .ifAddReplicaRegions()
     * - .ifForceOverwriteReplicaSecret()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/apireference/API_CreateSecret.html
     */
    toCreateSecret() {
        return this.to('CreateSecret');
    }
    /**
     * Grants permission to delete the resource policy attached to a secret
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     * - .ifAwsResourceTag()
     * - .ifSecretPrimaryRegion()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/apireference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to delete a secret
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifRecoveryWindowInDays()
     * - .ifForceDeleteWithoutRecovery()
     * - .ifResourceTag()
     * - .ifAwsResourceTag()
     * - .ifSecretPrimaryRegion()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/apireference/API_DeleteSecret.html
     */
    toDeleteSecret() {
        return this.to('DeleteSecret');
    }
    /**
     * Grants permission to retrieve the metadata about a secret, but not the encrypted data
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     * - .ifAwsResourceTag()
     * - .ifSecretPrimaryRegion()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/apireference/API_DescribeSecret.html
     */
    toDescribeSecret() {
        return this.to('DescribeSecret');
    }
    /**
     * Grants permission to generate a random string for use in password creation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/apireference/API_GetRandomPassword.html
     */
    toGetRandomPassword() {
        return this.to('GetRandomPassword');
    }
    /**
     * Grants permission to get the resource policy attached to a secret
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     * - .ifAwsResourceTag()
     * - .ifSecretPrimaryRegion()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/apireference/API_GetResourcePolicy.html
     */
    toGetResourcePolicy() {
        return this.to('GetResourcePolicy');
    }
    /**
     * Grants permission to retrieve and decrypt the encrypted data
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifVersionId()
     * - .ifVersionStage()
     * - .ifResource()
     * - .ifResourceTag()
     * - .ifAwsResourceTag()
     * - .ifSecretPrimaryRegion()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/apireference/API_GetSecretValue.html
     */
    toGetSecretValue() {
        return this.to('GetSecretValue');
    }
    /**
     * Grants permission to list the available versions of a secret
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     * - .ifAwsResourceTag()
     * - .ifSecretPrimaryRegion()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/apireference/API_ListSecretVersionIds.html
     */
    toListSecretVersionIds() {
        return this.to('ListSecretVersionIds');
    }
    /**
     * Grants permission to list the available secrets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/apireference/API_ListSecrets.html
     */
    toListSecrets() {
        return this.to('ListSecrets');
    }
    /**
     * Grants permission to attach a resource policy to a secret
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     * - .ifAwsResourceTag()
     * - .ifBlockPublicPolicy()
     * - .ifSecretPrimaryRegion()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/apireference/API_PutResourcePolicy.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to create a new version of the secret with new encrypted data
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     * - .ifAwsResourceTag()
     * - .ifSecretPrimaryRegion()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/apireference/API_PutSecretValue.html
     */
    toPutSecretValue() {
        return this.to('PutSecretValue');
    }
    /**
     * Grants permission to remove regions from replication
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     * - .ifAwsResourceTag()
     * - .ifSecretPrimaryRegion()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/apireference/API_RemoveRegionsFromReplication.html
     */
    toRemoveRegionsFromReplication() {
        return this.to('RemoveRegionsFromReplication');
    }
    /**
     * Grants permission to convert an existing secret to a multi-Region secret and begin replicating the secret to a list of new regions
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     * - .ifAwsResourceTag()
     * - .ifSecretPrimaryRegion()
     * - .ifAddReplicaRegions()
     * - .ifForceOverwriteReplicaSecret()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/apireference/API_ReplicateSecretToRegions.html
     */
    toReplicateSecretToRegions() {
        return this.to('ReplicateSecretToRegions');
    }
    /**
     * Grants permission to cancel deletion of a secret
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     * - .ifAwsResourceTag()
     * - .ifSecretPrimaryRegion()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/apireference/API_RestoreSecret.html
     */
    toRestoreSecret() {
        return this.to('RestoreSecret');
    }
    /**
     * Grants permission to start rotation of a secret
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifRotationLambdaARN()
     * - .ifResource()
     * - .ifResourceTag()
     * - .ifAwsResourceTag()
     * - .ifSecretPrimaryRegion()
     * - .ifModifyRotationRules()
     * - .ifRotateImmediately()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/apireference/API_RotateSecret.html
     */
    toRotateSecret() {
        return this.to('RotateSecret');
    }
    /**
     * Grants permission to remove the secret from replication and promote the secret to a regional secret in the replica Region
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     * - .ifAwsResourceTag()
     * - .ifSecretPrimaryRegion()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/apireference/API_StopReplicationToReplica.html
     */
    toStopReplicationToReplica() {
        return this.to('StopReplicationToReplica');
    }
    /**
     * Grants permission to add tags to a secret
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifResource()
     * - .ifResourceTag()
     * - .ifAwsResourceTag()
     * - .ifSecretPrimaryRegion()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/apireference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a secret
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifAwsTagKeys()
     * - .ifResource()
     * - .ifResourceTag()
     * - .ifAwsResourceTag()
     * - .ifSecretPrimaryRegion()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/apireference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a secret with new metadata or with a new version of the encrypted data
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifDescription()
     * - .ifKmsKeyArn()
     * - .ifKmsKeyId()
     * - .ifResource()
     * - .ifResourceTag()
     * - .ifAwsResourceTag()
     * - .ifSecretPrimaryRegion()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/apireference/API_UpdateSecret.html
     */
    toUpdateSecret() {
        return this.to('UpdateSecret');
    }
    /**
     * Grants permission to move a stage from one secret to another
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifVersionStage()
     * - .ifResource()
     * - .ifResourceTag()
     * - .ifAwsResourceTag()
     * - .ifSecretPrimaryRegion()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/apireference/API_UpdateSecretVersionStage.html
     */
    toUpdateSecretVersionStage() {
        return this.to('UpdateSecretVersionStage');
    }
    /**
     * Grants permission to validate a resource policy before attaching policy
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     * - .ifAwsResourceTag()
     * - .ifSecretPrimaryRegion()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/apireference/API_ValidateResourcePolicy.html
     */
    toValidateResourcePolicy() {
        return this.to('ValidateResourcePolicy');
    }
    /**
     * Adds a resource of type Secret to the statement
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-resources
     *
     * @param secretId - Identifier for the secretId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifResourceTag()
     * - .ifResource()
     */
    onSecret(secretId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:secretsmanager:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:secret:${secretId}`);
    }
    /**
     * Filters access by a key that is present in the request the user makes to the Secrets Manager service
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toCreateSecret()
     * - .toTagResource()
     *
     * Applies to resource types:
     * - Secret
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toCancelRotateSecret()
     * - .toCreateSecret()
     * - .toDeleteResourcePolicy()
     * - .toDeleteSecret()
     * - .toDescribeSecret()
     * - .toGetResourcePolicy()
     * - .toGetSecretValue()
     * - .toListSecretVersionIds()
     * - .toPutResourcePolicy()
     * - .toPutSecretValue()
     * - .toRemoveRegionsFromReplication()
     * - .toReplicateSecretToRegions()
     * - .toRestoreSecret()
     * - .toRotateSecret()
     * - .toStopReplicationToReplica()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateSecret()
     * - .toUpdateSecretVersionStage()
     * - .toValidateResourcePolicy()
     *
     * Applies to resource types:
     * - Secret
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the list of all the tag key names present in the request the user makes to the Secrets Manager service
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toCreateSecret()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * Applies to resource types:
     * - Secret
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the list of Regions in which to replicate the secret
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toCreateSecret()
     * - .toReplicateSecretToRegions()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAddReplicaRegions(value, operator) {
        return this.if(`AddReplicaRegions`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by whether the resource policy blocks broad AWS account access
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toPutResourcePolicy()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifBlockPublicPolicy(value) {
        return this.if(`BlockPublicPolicy`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the description text in the request
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toCreateSecret()
     * - .toUpdateSecret()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDescription(value, operator) {
        return this.if(`Description`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by whether the secret is to be deleted immediately without any recovery window
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toDeleteSecret()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifForceDeleteWithoutRecovery(value) {
        return this.if(`ForceDeleteWithoutRecovery`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by whether to overwrite a secret with the same name in the destination Region
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toCreateSecret()
     * - .toReplicateSecretToRegions()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifForceOverwriteReplicaSecret(value) {
        return this.if(`ForceOverwriteReplicaSecret`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the key ARN of the KMS key in the request
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toCreateSecret()
     * - .toUpdateSecret()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifKmsKeyArn(value, operator) {
        return this.if(`KmsKeyArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the key identifier of the KMS key in the request. Deprecated: Use secretsmanager:KmsKeyArn
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toCreateSecret()
     * - .toUpdateSecret()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifKmsKeyId(value, operator) {
        return this.if(`KmsKeyId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by whether the rotation rules of the secret are to be modified
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toRotateSecret()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifModifyRotationRules(value) {
        return this.if(`ModifyRotationRules`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the friendly name of the secret in the request
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toCreateSecret()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifName(value, operator) {
        return this.if(`Name`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the number of days that Secrets Manager waits before it can delete the secret
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toDeleteSecret()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifRecoveryWindowInDays(value, operator) {
        return this.if(`RecoveryWindowInDays`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by a tag key and value pair
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toCancelRotateSecret()
     * - .toCreateSecret()
     * - .toDeleteResourcePolicy()
     * - .toDeleteSecret()
     * - .toDescribeSecret()
     * - .toGetResourcePolicy()
     * - .toGetSecretValue()
     * - .toListSecretVersionIds()
     * - .toPutResourcePolicy()
     * - .toPutSecretValue()
     * - .toRemoveRegionsFromReplication()
     * - .toReplicateSecretToRegions()
     * - .toRestoreSecret()
     * - .toRotateSecret()
     * - .toStopReplicationToReplica()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateSecret()
     * - .toUpdateSecretVersionStage()
     * - .toValidateResourcePolicy()
     *
     * Applies to resource types:
     * - Secret
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagKey, value, operator) {
        return this.if(`ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by whether the secret is to be rotated immediately
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toRotateSecret()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifRotateImmediately(value) {
        return this.if(`RotateImmediately`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the ARN of the rotation Lambda function in the request
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toRotateSecret()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifRotationLambdaARN(value, operator) {
        return this.if(`RotationLambdaARN`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the SecretID value in the request
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toCancelRotateSecret()
     * - .toDeleteResourcePolicy()
     * - .toDeleteSecret()
     * - .toDescribeSecret()
     * - .toGetResourcePolicy()
     * - .toGetSecretValue()
     * - .toListSecretVersionIds()
     * - .toPutResourcePolicy()
     * - .toPutSecretValue()
     * - .toRemoveRegionsFromReplication()
     * - .toReplicateSecretToRegions()
     * - .toRestoreSecret()
     * - .toRotateSecret()
     * - .toStopReplicationToReplica()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateSecret()
     * - .toUpdateSecretVersionStage()
     * - .toValidateResourcePolicy()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifSecretId(value, operator) {
        return this.if(`SecretId`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by primary region in which the secret is created
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toCancelRotateSecret()
     * - .toDeleteResourcePolicy()
     * - .toDeleteSecret()
     * - .toDescribeSecret()
     * - .toGetResourcePolicy()
     * - .toGetSecretValue()
     * - .toListSecretVersionIds()
     * - .toPutResourcePolicy()
     * - .toPutSecretValue()
     * - .toRemoveRegionsFromReplication()
     * - .toReplicateSecretToRegions()
     * - .toRestoreSecret()
     * - .toRotateSecret()
     * - .toStopReplicationToReplica()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateSecret()
     * - .toUpdateSecretVersionStage()
     * - .toValidateResourcePolicy()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecretPrimaryRegion(value, operator) {
        return this.if(`SecretPrimaryRegion`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the unique identifier of the version of the secret in the request
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toGetSecretValue()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVersionId(value, operator) {
        return this.if(`VersionId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the list of version stages in the request
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toGetSecretValue()
     * - .toUpdateSecretVersionStage()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVersionStage(value, operator) {
        return this.if(`VersionStage`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ARN of the rotation Lambda function associated with the secret
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toCancelRotateSecret()
     * - .toDeleteResourcePolicy()
     * - .toDeleteSecret()
     * - .toDescribeSecret()
     * - .toGetResourcePolicy()
     * - .toGetSecretValue()
     * - .toListSecretVersionIds()
     * - .toPutResourcePolicy()
     * - .toPutSecretValue()
     * - .toRemoveRegionsFromReplication()
     * - .toReplicateSecretToRegions()
     * - .toRestoreSecret()
     * - .toRotateSecret()
     * - .toStopReplicationToReplica()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateSecret()
     * - .toUpdateSecretVersionStage()
     * - .toValidateResourcePolicy()
     *
     * Applies to resource types:
     * - Secret
     *
     * @param allowRotationLambdaArn The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifResource(allowRotationLambdaArn, value, operator) {
        return this.if(`resource/${allowRotationLambdaArn}`, value, operator ?? 'ArnLike');
    }
    /**
     * Statement provider for service [secretsmanager](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssecretsmanager.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'secretsmanager';
        this.accessLevelList = {
            List: [
                'BatchGetSecretValue',
                'ListSecrets'
            ],
            Write: [
                'CancelRotateSecret',
                'CreateSecret',
                'DeleteSecret',
                'PutSecretValue',
                'RemoveRegionsFromReplication',
                'ReplicateSecretToRegions',
                'RestoreSecret',
                'RotateSecret',
                'StopReplicationToReplica',
                'UpdateSecret',
                'UpdateSecretVersionStage'
            ],
            'Permissions management': [
                'DeleteResourcePolicy',
                'PutResourcePolicy',
                'ValidateResourcePolicy'
            ],
            Read: [
                'DescribeSecret',
                'GetRandomPassword',
                'GetResourcePolicy',
                'GetSecretValue',
                'ListSecretVersionIds'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Secretsmanager = Secretsmanager;
//# sourceMappingURL=data:application/json;base64,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