import { AccessLevelList } from '../../shared/access-level';
import { PolicyStatement, Operator } from '../../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [security-ir](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssecurityincidentresponse.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class SecurityIr extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to get member account details in batch
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_BatchGetMemberAccountDetails.html
     */
    toBatchGetMemberAccountDetails(): this;
    /**
     * Grants permission to cancel a membership
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_CancelMembership.html
     */
    toCancelMembership(): this;
    /**
     * Grants permission to close a case
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_CloseCase.html
     */
    toCloseCase(): this;
    /**
     * Grants permission to create a case
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_CreateCase.html
     */
    toCreateCase(): this;
    /**
     * Grants permission to create a case comment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_CreateCaseComment.html
     */
    toCreateCaseComment(): this;
    /**
     * Grants permission to create a membership
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - organizations:DescribeOrganization
     * - organizations:ListDelegatedAdministrators
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_CreateMembership.html
     */
    toCreateMembership(): this;
    /**
     * Grants permission to get a case
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_GetCase.html
     */
    toGetCase(): this;
    /**
     * Grants permission to get a case attachment download URL
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_GetCaseAttachmentDownloadUrl.html
     */
    toGetCaseAttachmentDownloadUrl(): this;
    /**
     * Grants permission to get a case attachment upload URL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_GetCaseAttachmentUploadUrl.html
     */
    toGetCaseAttachmentUploadUrl(): this;
    /**
     * Grants permission to get a membership
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_GetMembership.html
     */
    toGetMembership(): this;
    /**
     * Grants permission to list case edits
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_ListCaseEdits.html
     */
    toListCaseEdits(): this;
    /**
     * Grants permission to list cases
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_ListCases.html
     */
    toListCases(): this;
    /**
     * Grants permission to list case comments
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_ListComments.html
     */
    toListComments(): this;
    /**
     * Grants permission to list memberships
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_ListMemberships.html
     */
    toListMemberships(): this;
    /**
     * Grants permission to list the tags attached to the specified resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource(): this;
    /**
     * Grants permission to add tags to the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_TagResource.html
     */
    toTagResource(): this;
    /**
     * Grants permission to remove tags from the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_UntagResource.html
     */
    toUntagResource(): this;
    /**
     * Grants permission to update a case
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_UpdateCase.html
     */
    toUpdateCase(): this;
    /**
     * Grants permission to update a case comment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_UpdateCaseComment.html
     */
    toUpdateCaseComment(): this;
    /**
     * Grants permission to update a case status
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_UpdateCaseStatus.html
     */
    toUpdateCaseStatus(): this;
    /**
     * Grants permission to update memberships
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_UpdateMembership.html
     */
    toUpdateMembership(): this;
    /**
     * Grants permission to update case resolver type
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_UpdateResolverType.html
     */
    toUpdateResolverType(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type case to the statement
     *
     * https://docs.aws.amazon.com/security-ir/latest/userguide/case.html
     *
     * @param caseId - Identifier for the caseId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCase(caseId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type membership to the statement
     *
     * https://docs.aws.amazon.com/security-ir/latest/userguide/select-a-membership-account.html
     *
     * @param membershipId - Identifier for the membershipId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMembership(membershipId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateCase()
     * - .toCreateMembership()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toListTagsForResource()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * Applies to resource types:
     * - case
     * - membership
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateCase()
     * - .toCreateMembership()
     * - .toListTagsForResource()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [security-ir](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssecurityincidentresponse.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
