"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Securitylake = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [securitylake](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonsecuritylake.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Securitylake extends shared_1.PolicyStatement {
    /**
     * Grants permission to enable any source type in any region for accounts that are either part of a trusted organization or standalone account
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - glue:CreateDatabase
     * - glue:CreateTable
     * - glue:GetDatabase
     * - glue:GetTable
     * - iam:CreateServiceLinkedRole
     * - kms:CreateGrant
     * - kms:DescribeKey
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_CreateAwsLogSource.html
     */
    toCreateAwsLogSource() {
        return this.to('CreateAwsLogSource');
    }
    /**
     * Grants permission to add a custom source
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - glue:CreateCrawler
     * - glue:CreateDatabase
     * - glue:CreateTable
     * - glue:StartCrawlerSchedule
     * - iam:DeleteRolePolicy
     * - iam:GetRole
     * - iam:PassRole
     * - iam:PutRolePolicy
     * - kms:CreateGrant
     * - kms:DescribeKey
     * - kms:GenerateDataKey
     * - lakeformation:GrantPermissions
     * - lakeformation:RegisterResource
     * - s3:ListBucket
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_CreateCustomLogSource.html
     */
    toCreateCustomLogSource() {
        return this.to('CreateCustomLogSource');
    }
    /**
     * Grants permission to create a new security data lake
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - events:PutRule
     * - events:PutTargets
     * - iam:CreateServiceLinkedRole
     * - iam:DeleteRolePolicy
     * - iam:GetRole
     * - iam:ListAttachedRolePolicies
     * - iam:PassRole
     * - iam:PutRolePolicy
     * - kms:CreateGrant
     * - kms:DescribeKey
     * - lakeformation:GetDataLakeSettings
     * - lakeformation:PutDataLakeSettings
     * - lambda:AddPermission
     * - lambda:CreateEventSourceMapping
     * - lambda:CreateFunction
     * - organizations:DescribeOrganization
     * - organizations:ListAccounts
     * - organizations:ListDelegatedServicesForAccount
     * - s3:CreateBucket
     * - s3:GetObject
     * - s3:GetObjectVersion
     * - s3:ListBucket
     * - s3:PutBucketPolicy
     * - s3:PutBucketPublicAccessBlock
     * - s3:PutBucketVersioning
     * - sqs:CreateQueue
     * - sqs:GetQueueAttributes
     * - sqs:SetQueueAttributes
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_CreateDataLake.html
     */
    toCreateDataLake() {
        return this.to('CreateDataLake');
    }
    /**
     * Grants permission to get instant notifications about exceptions. Subscribes to the SNS topics for exception notifications
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_CreateDataLakeExceptionSubscription.html
     */
    toCreateDataLakeExceptionSubscription() {
        return this.to('CreateDataLakeExceptionSubscription');
    }
    /**
     * Grants permission to automatically enable Amazon Security Lake for new member accounts in your organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_CreateDataLakeOrganizationConfiguration.html
     */
    toCreateDataLakeOrganizationConfiguration() {
        return this.to('CreateDataLakeOrganizationConfiguration');
    }
    /**
     * Grants permission to create a subscriber
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:CreateRole
     * - iam:DeleteRolePolicy
     * - iam:GetRole
     * - iam:PutRolePolicy
     * - lakeformation:GrantPermissions
     * - lakeformation:ListPermissions
     * - lakeformation:RegisterResource
     * - lakeformation:RevokePermissions
     * - ram:GetResourceShareAssociations
     * - ram:GetResourceShares
     * - ram:UpdateResourceShare
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_CreateSubscriber.html
     */
    toCreateSubscriber() {
        return this.to('CreateSubscriber');
    }
    /**
     * Grants permission to create a webhook invocation to notify a client when there is new data in the data lake
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - events:CreateApiDestination
     * - events:CreateConnection
     * - events:DescribeRule
     * - events:ListApiDestinations
     * - events:ListConnections
     * - events:PutRule
     * - events:PutTargets
     * - iam:DeleteRolePolicy
     * - iam:GetRole
     * - iam:PassRole
     * - s3:GetBucketNotification
     * - s3:PutBucketNotification
     * - sqs:CreateQueue
     * - sqs:DeleteQueue
     * - sqs:GetQueueAttributes
     * - sqs:GetQueueUrl
     * - sqs:SetQueueAttributes
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_CreateSubscriberNotification.html
     */
    toCreateSubscriberNotification() {
        return this.to('CreateSubscriberNotification');
    }
    /**
     * Grants permission to disable any source type in any region for accounts that are part of a trusted organization or standalone accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_DeleteAwsLogSource.html
     */
    toDeleteAwsLogSource() {
        return this.to('DeleteAwsLogSource');
    }
    /**
     * Grants permission to remove a custom source
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - glue:StopCrawlerSchedule
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_DeleteCustomLogSource.html
     */
    toDeleteCustomLogSource() {
        return this.to('DeleteCustomLogSource');
    }
    /**
     * Grants permission to delete security data lake
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - organizations:DescribeOrganization
     * - organizations:ListDelegatedAdministrators
     * - organizations:ListDelegatedServicesForAccount
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_DeleteDataLake.html
     */
    toDeleteDataLake() {
        return this.to('DeleteDataLake');
    }
    /**
     * Grants permission to unsubscribe from SNS topics for exception notifications. Removes exception notifications for the SNS topic
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_DeleteDataLakeExceptionSubscription.html
     */
    toDeleteDataLakeExceptionSubscription() {
        return this.to('DeleteDataLakeExceptionSubscription');
    }
    /**
     * Grants permission to remove the automatic enablement of Amazon Security Lake access for new organization accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_DeleteDataLakeOrganizationConfiguration.html
     */
    toDeleteDataLakeOrganizationConfiguration() {
        return this.to('DeleteDataLakeOrganizationConfiguration');
    }
    /**
     * Grants permission to delete the specified subscriber
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - events:DeleteApiDestination
     * - events:DeleteConnection
     * - events:DeleteRule
     * - events:DescribeRule
     * - events:ListApiDestinations
     * - events:ListTargetsByRule
     * - events:RemoveTargets
     * - iam:DeleteRole
     * - iam:DeleteRolePolicy
     * - iam:GetRole
     * - iam:ListRolePolicies
     * - lakeformation:ListPermissions
     * - lakeformation:RevokePermissions
     * - sqs:DeleteQueue
     * - sqs:GetQueueUrl
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_DeleteSubscriber.html
     */
    toDeleteSubscriber() {
        return this.to('DeleteSubscriber');
    }
    /**
     * Grants permission to remove a webhook invocation to notify a client when there is new data in the data lake
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - events:DeleteApiDestination
     * - events:DeleteConnection
     * - events:DeleteRule
     * - events:DescribeRule
     * - events:ListApiDestinations
     * - events:ListTargetsByRule
     * - events:RemoveTargets
     * - iam:DeleteRole
     * - iam:DeleteRolePolicy
     * - iam:GetRole
     * - iam:ListRolePolicies
     * - lakeformation:RevokePermissions
     * - sqs:DeleteQueue
     * - sqs:GetQueueUrl
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_DeleteSubscriberNotification.html
     */
    toDeleteSubscriberNotification() {
        return this.to('DeleteSubscriberNotification');
    }
    /**
     * Grants permission to remove the Delegated Administrator account and disable Amazon Security Lake as a service for this organization
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - organizations:DeregisterDelegatedAdministrator
     * - organizations:DescribeOrganization
     * - organizations:ListDelegatedServicesForAccount
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_DeregisterDataLakeDelegatedAdministrator.html
     */
    toDeregisterDataLakeDelegatedAdministrator() {
        return this.to('DeregisterDataLakeDelegatedAdministrator');
    }
    /**
     * Grants permission to query the protocol and endpoint that were provided when subscribing to SNS topics for exception notifications
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_GetDataLakeExceptionSubscription.html
     */
    toGetDataLakeExceptionSubscription() {
        return this.to('GetDataLakeExceptionSubscription');
    }
    /**
     * Grants permission to get an organization's configuration setting for automatically enabling Amazon Security Lake access for new organization accounts
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - organizations:DescribeOrganization
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_GetDataLakeOrganizationConfiguration.html
     */
    toGetDataLakeOrganizationConfiguration() {
        return this.to('GetDataLakeOrganizationConfiguration');
    }
    /**
     * Grants permission to get a static snapshot of the security data lake in the current region. The snapshot includes enabled accounts and log sources
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_GetDataLakeSources.html
     */
    toGetDataLakeSources() {
        return this.to('GetDataLakeSources');
    }
    /**
     * Grants permission to get information about subscriber that is already created
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_GetSubscriber.html
     */
    toGetSubscriber() {
        return this.to('GetSubscriber');
    }
    /**
     * Grants permission to get the list of all non-retryable failures
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_ListDataLakeExceptions.html
     */
    toListDataLakeExceptions() {
        return this.to('ListDataLakeExceptions');
    }
    /**
     * Grants permission to list information about the security data lakes
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_ListDataLakes.html
     */
    toListDataLakes() {
        return this.to('ListDataLakes');
    }
    /**
     * Grants permission to view the enabled accounts. You can view the enabled sources in the enabled regions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_ListLogSources.html
     */
    toListLogSources() {
        return this.to('ListLogSources');
    }
    /**
     * Grants permission to list all subscribers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_ListSubscribers.html
     */
    toListSubscribers() {
        return this.to('ListSubscribers');
    }
    /**
     * Grants permission to list all tags for the resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to designate an account as the Amazon Security Lake administrator account for the organization
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - organizations:DescribeOrganization
     * - organizations:EnableAWSServiceAccess
     * - organizations:ListDelegatedAdministrators
     * - organizations:ListDelegatedServicesForAccount
     * - organizations:RegisterDelegatedAdministrator
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_RegisterDataLakeDelegatedAdministrator.html
     */
    toRegisterDataLakeDelegatedAdministrator() {
        return this.to('RegisterDataLakeDelegatedAdministrator');
    }
    /**
     * Grants permission to add tags to the resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from the resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a security data lake
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - events:PutRule
     * - events:PutTargets
     * - iam:CreateServiceLinkedRole
     * - iam:DeleteRolePolicy
     * - iam:GetRole
     * - iam:ListAttachedRolePolicies
     * - iam:PutRolePolicy
     * - kms:CreateGrant
     * - kms:DescribeKey
     * - lakeformation:GetDataLakeSettings
     * - lakeformation:PutDataLakeSettings
     * - lambda:AddPermission
     * - lambda:CreateEventSourceMapping
     * - lambda:CreateFunction
     * - organizations:DescribeOrganization
     * - organizations:ListDelegatedServicesForAccount
     * - s3:CreateBucket
     * - s3:GetObject
     * - s3:GetObjectVersion
     * - s3:ListBucket
     * - s3:PutBucketPolicy
     * - s3:PutBucketPublicAccessBlock
     * - s3:PutBucketVersioning
     * - sqs:CreateQueue
     * - sqs:GetQueueAttributes
     * - sqs:SetQueueAttributes
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_UpdateDataLake.html
     */
    toUpdateDataLake() {
        return this.to('UpdateDataLake');
    }
    /**
     * Grants permission to update subscriptions to the SNS topics for exception notifications
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_UpdateDataLakeExceptionSubscription.html
     */
    toUpdateDataLakeExceptionSubscription() {
        return this.to('UpdateDataLakeExceptionSubscription');
    }
    /**
     * Grants permission to update subscriber
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - events:CreateApiDestination
     * - events:CreateConnection
     * - events:DescribeRule
     * - events:ListApiDestinations
     * - events:ListConnections
     * - events:PutRule
     * - events:PutTargets
     * - iam:DeleteRolePolicy
     * - iam:GetRole
     * - iam:PutRolePolicy
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_UpdateSubscriber.html
     */
    toUpdateSubscriber() {
        return this.to('UpdateSubscriber');
    }
    /**
     * Grants permission to update a webhook invocation to notify a client when there is new data in the data lake
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - events:CreateApiDestination
     * - events:CreateConnection
     * - events:DescribeRule
     * - events:ListApiDestinations
     * - events:ListConnections
     * - events:PutRule
     * - events:PutTargets
     * - iam:CreateServiceLinkedRole
     * - iam:DeleteRolePolicy
     * - iam:GetRole
     * - iam:PassRole
     * - iam:PutRolePolicy
     * - s3:CreateBucket
     * - s3:GetBucketNotification
     * - s3:ListBucket
     * - s3:PutBucketNotification
     * - s3:PutBucketPolicy
     * - s3:PutBucketPublicAccessBlock
     * - s3:PutBucketVersioning
     * - s3:PutLifecycleConfiguration
     * - sqs:CreateQueue
     * - sqs:DeleteQueue
     * - sqs:GetQueueAttributes
     * - sqs:GetQueueUrl
     * - sqs:SetQueueAttributes
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_UpdateSubscriberNotification.html
     */
    toUpdateSubscriberNotification() {
        return this.to('UpdateSubscriberNotification');
    }
    /**
     * Adds a resource of type data-lake to the statement
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_DataLakeResource.html
     *
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     */
    onDataLake(account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:securitylake:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:data-lake/default`);
    }
    /**
     * Adds a resource of type subscriber to the statement
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_SubscriberResource.html
     *
     * @param subscriberId - Identifier for the subscriberId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     */
    onSubscriber(subscriberId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:securitylake:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:subscriber/${subscriberId}`);
    }
    /**
     * Filters access by tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateDataLake()
     * - .toCreateSubscriber()
     * - .toTagResource()
     *
     * Applies to resource types:
     * - data-lake
     * - subscriber
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a tag key and value pair of a resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - data-lake
     * - subscriber
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateDataLake()
     * - .toCreateSubscriber()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [securitylake](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonsecuritylake.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'securitylake';
        this.accessLevelList = {
            Write: [
                'CreateAwsLogSource',
                'CreateCustomLogSource',
                'CreateDataLake',
                'CreateDataLakeExceptionSubscription',
                'CreateDataLakeOrganizationConfiguration',
                'CreateSubscriber',
                'CreateSubscriberNotification',
                'DeleteAwsLogSource',
                'DeleteCustomLogSource',
                'DeleteDataLake',
                'DeleteDataLakeExceptionSubscription',
                'DeleteDataLakeOrganizationConfiguration',
                'DeleteSubscriber',
                'DeleteSubscriberNotification',
                'DeregisterDataLakeDelegatedAdministrator',
                'RegisterDataLakeDelegatedAdministrator',
                'UpdateDataLake',
                'UpdateDataLakeExceptionSubscription',
                'UpdateSubscriber',
                'UpdateSubscriberNotification'
            ],
            Read: [
                'GetDataLakeExceptionSubscription',
                'GetDataLakeOrganizationConfiguration',
                'GetDataLakeSources',
                'GetSubscriber'
            ],
            List: [
                'ListDataLakeExceptions',
                'ListDataLakes',
                'ListLogSources',
                'ListSubscribers',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Securitylake = Securitylake;
//# sourceMappingURL=data:application/json;base64,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