"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Servicequotas = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [servicequotas](https://docs.aws.amazon.com/service-authorization/latest/reference/list_servicequotas.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Servicequotas extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate the Service Quotas template with your organization
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - organizations:DescribeOrganization
     * - organizations:EnableAWSServiceAccess
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_AssociateServiceQuotaTemplate.html
     */
    toAssociateServiceQuotaTemplate() {
        return this.to('AssociateServiceQuotaTemplate');
    }
    /**
     * Grants permission to remove the specified service quota from the service quota template
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - organizations:DescribeOrganization
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_DeleteServiceQuotaIncreaseRequestFromTemplate.html
     */
    toDeleteServiceQuotaIncreaseRequestFromTemplate() {
        return this.to('DeleteServiceQuotaIncreaseRequestFromTemplate');
    }
    /**
     * Grants permission to disassociate the Service Quotas template from your organization
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - organizations:DescribeOrganization
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_DisassociateServiceQuotaTemplate.html
     */
    toDisassociateServiceQuotaTemplate() {
        return this.to('DisassociateServiceQuotaTemplate');
    }
    /**
     * Grants permission to return the details for the specified service quota, including the AWS default value
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetAWSDefaultServiceQuota.html
     */
    toGetAWSDefaultServiceQuota() {
        return this.to('GetAWSDefaultServiceQuota');
    }
    /**
     * Grants permission to retrieve the ServiceQuotaTemplateAssociationStatus value, which tells you if the Service Quotas template is associated with an organization
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - organizations:DescribeOrganization
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetAssociationForServiceQuotaTemplate.html
     */
    toGetAssociationForServiceQuotaTemplate() {
        return this.to('GetAssociationForServiceQuotaTemplate');
    }
    /**
     * Grants permission to retrieve the details for a particular service quota increase request
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetRequestedServiceQuotaChange.html
     */
    toGetRequestedServiceQuotaChange() {
        return this.to('GetRequestedServiceQuotaChange');
    }
    /**
     * Grants permission to return the details for the specified service quota, including the applied value
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - autoscaling:DescribeAccountLimits
     * - cloudformation:DescribeAccountLimits
     * - dynamodb:DescribeLimits
     * - elasticloadbalancing:DescribeAccountLimits
     * - iam:GetAccountSummary
     * - kinesis:DescribeLimits
     * - rds:DescribeAccountAttributes
     * - route53:GetAccountLimit
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetServiceQuota.html
     */
    toGetServiceQuota() {
        return this.to('GetServiceQuota');
    }
    /**
     * Grants permission to retrieve the details for a service quota increase request from the service quota template
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - organizations:DescribeOrganization
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetServiceQuotaIncreaseRequestFromTemplate.html
     */
    toGetServiceQuotaIncreaseRequestFromTemplate() {
        return this.to('GetServiceQuotaIncreaseRequestFromTemplate');
    }
    /**
     * Grants permission to list all default service quotas for the specified AWS service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListAWSDefaultServiceQuotas.html
     */
    toListAWSDefaultServiceQuotas() {
        return this.to('ListAWSDefaultServiceQuotas');
    }
    /**
     * Grants permission to request a list of the changes to quotas for a service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListRequestedServiceQuotaChangeHistory.html
     */
    toListRequestedServiceQuotaChangeHistory() {
        return this.to('ListRequestedServiceQuotaChangeHistory');
    }
    /**
     * Grants permission to request a list of the changes to specific service quotas
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListRequestedServiceQuotaChangeHistoryByQuota.html
     */
    toListRequestedServiceQuotaChangeHistoryByQuota() {
        return this.to('ListRequestedServiceQuotaChangeHistoryByQuota');
    }
    /**
     * Grants permission to return a list of the service quota increase requests from the service quota template
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - organizations:DescribeOrganization
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListServiceQuotaIncreaseRequestsInTemplate.html
     */
    toListServiceQuotaIncreaseRequestsInTemplate() {
        return this.to('ListServiceQuotaIncreaseRequestsInTemplate');
    }
    /**
     * Grants permission to list all service quotas for the specified AWS service, in that account, in that Region
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - autoscaling:DescribeAccountLimits
     * - cloudformation:DescribeAccountLimits
     * - dynamodb:DescribeLimits
     * - elasticloadbalancing:DescribeAccountLimits
     * - iam:GetAccountSummary
     * - kinesis:DescribeLimits
     * - rds:DescribeAccountAttributes
     * - route53:GetAccountLimit
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListServiceQuotas.html
     */
    toListServiceQuotas() {
        return this.to('ListServiceQuotas');
    }
    /**
     * Grants permission to list the AWS services available in Service Quotas
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListServices.html
     */
    toListServices() {
        return this.to('ListServices');
    }
    /**
     * Grants permission to view the existing tags on a SQ resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to define and add a quota to the service quota template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifService()
     *
     * Dependent actions:
     * - organizations:DescribeOrganization
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_PutServiceQuotaIncreaseRequestIntoTemplate.html
     */
    toPutServiceQuotaIncreaseRequestIntoTemplate() {
        return this.to('PutServiceQuotaIncreaseRequestIntoTemplate');
    }
    /**
     * Grants permission to submit the request for a service quota increase
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifService()
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_RequestServiceQuotaIncrease.html
     */
    toRequestServiceQuotaIncrease() {
        return this.to('RequestServiceQuotaIncrease');
    }
    /**
     * Grants permission to associate a set of tags with an existing SQ resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove a set of tags from a SQ resource, where tags to be removed match a set of customer-supplied tag keys
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Adds a resource of type quota to the statement
     *
     * https://docs.aws.amazon.com/servicequotas/latest/userguide/identity-access-management.html#resources
     *
     * @param serviceCode - Identifier for the serviceCode.
     * @param quotaCode - Identifier for the quotaCode.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onQuota(serviceCode, quotaCode, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:servicequotas:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:${serviceCode}/${quotaCode}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the specified AWS service
     *
     * https://docs.aws.amazon.com/servicequotas/latest/userguide/identity-access-management.html#condition-keys
     *
     * Applies to actions:
     * - .toPutServiceQuotaIncreaseRequestIntoTemplate()
     * - .toRequestServiceQuotaIncrease()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifService(value, operator) {
        return this.if(`service`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [servicequotas](https://docs.aws.amazon.com/service-authorization/latest/reference/list_servicequotas.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'servicequotas';
        this.accessLevelList = {
            Write: [
                'AssociateServiceQuotaTemplate',
                'DeleteServiceQuotaIncreaseRequestFromTemplate',
                'DisassociateServiceQuotaTemplate',
                'PutServiceQuotaIncreaseRequestIntoTemplate',
                'RequestServiceQuotaIncrease'
            ],
            Read: [
                'GetAWSDefaultServiceQuota',
                'GetAssociationForServiceQuotaTemplate',
                'GetRequestedServiceQuotaChange',
                'GetServiceQuota',
                'GetServiceQuotaIncreaseRequestFromTemplate',
                'ListAWSDefaultServiceQuotas',
                'ListRequestedServiceQuotaChangeHistory',
                'ListRequestedServiceQuotaChangeHistoryByQuota',
                'ListServiceQuotaIncreaseRequestsInTemplate',
                'ListServiceQuotas',
                'ListServices',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Servicequotas = Servicequotas;
//# sourceMappingURL=data:application/json;base64,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