"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SesV2 = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [ses-v2](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonsimpleemailservicev2.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class SesV2 extends shared_1.PolicyStatement {
    /**
     * Grants permission to get metric data on your activity
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_BatchGetMetricData.html
     */
    toBatchGetMetricData() {
        return this.to('BatchGetMetricData');
    }
    /**
     * Grants permission to cancel an export job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifExportSourceType()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CancelExportJob.html
     */
    toCancelExportJob() {
        return this.to('CancelExportJob');
    }
    /**
     * Grants permission to create a new configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateConfigurationSet.html
     */
    toCreateConfigurationSet() {
        return this.to('CreateConfigurationSet');
    }
    /**
     * Grants permission to create a configuration set event destination
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateConfigurationSetEventDestination.html
     */
    toCreateConfigurationSetEventDestination() {
        return this.to('CreateConfigurationSetEventDestination');
    }
    /**
     * Grants permission to create a contact
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateContact.html
     */
    toCreateContact() {
        return this.to('CreateContact');
    }
    /**
     * Grants permission to create a contact list
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateContactList.html
     */
    toCreateContactList() {
        return this.to('CreateContactList');
    }
    /**
     * Grants permission to create a new custom verification email template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateCustomVerificationEmailTemplate.html
     */
    toCreateCustomVerificationEmailTemplate() {
        return this.to('CreateCustomVerificationEmailTemplate');
    }
    /**
     * Grants permission to create a new pool of dedicated IP addresses
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateDedicatedIpPool.html
     */
    toCreateDedicatedIpPool() {
        return this.to('CreateDedicatedIpPool');
    }
    /**
     * Grants permission to create a new predictive inbox placement test
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateDeliverabilityTestReport.html
     */
    toCreateDeliverabilityTestReport() {
        return this.to('CreateDeliverabilityTestReport');
    }
    /**
     * Grants permission to start the process of verifying an email identity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateEmailIdentity.html
     */
    toCreateEmailIdentity() {
        return this.to('CreateEmailIdentity');
    }
    /**
     * Grants permission to create the specified sending authorization policy for the given identity
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateEmailIdentityPolicy.html
     */
    toCreateEmailIdentityPolicy() {
        return this.to('CreateEmailIdentityPolicy');
    }
    /**
     * Grants permission to create an email template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateEmailTemplate.html
     */
    toCreateEmailTemplate() {
        return this.to('CreateEmailTemplate');
    }
    /**
     * Grants permission to create an export job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifExportSourceType()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateExportJob.html
     */
    toCreateExportJob() {
        return this.to('CreateExportJob');
    }
    /**
     * Grants permission to creates an import job for a data destination
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateImportJob.html
     */
    toCreateImportJob() {
        return this.to('CreateImportJob');
    }
    /**
     * Grants permission to create a new multi-region endpoint
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateMultiRegionEndpoint.html
     */
    toCreateMultiRegionEndpoint() {
        return this.to('CreateMultiRegionEndpoint');
    }
    /**
     * Grants permission to delete an existing configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteConfigurationSet.html
     */
    toDeleteConfigurationSet() {
        return this.to('DeleteConfigurationSet');
    }
    /**
     * Grants permission to delete an event destination
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteConfigurationSetEventDestination.html
     */
    toDeleteConfigurationSetEventDestination() {
        return this.to('DeleteConfigurationSetEventDestination');
    }
    /**
     * Grants permission to delete a contact from a contact list
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteContact.html
     */
    toDeleteContact() {
        return this.to('DeleteContact');
    }
    /**
     * Grants permission to delete a contact list with all of its contacts
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteContactList.html
     */
    toDeleteContactList() {
        return this.to('DeleteContactList');
    }
    /**
     * Grants permission to delete an existing custom verification email template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteCustomVerificationEmailTemplate.html
     */
    toDeleteCustomVerificationEmailTemplate() {
        return this.to('DeleteCustomVerificationEmailTemplate');
    }
    /**
     * Grants permission to delete a dedicated IP pool
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteDedicatedIpPool.html
     */
    toDeleteDedicatedIpPool() {
        return this.to('DeleteDedicatedIpPool');
    }
    /**
     * Grants permission to delete an email identity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteEmailIdentity.html
     */
    toDeleteEmailIdentity() {
        return this.to('DeleteEmailIdentity');
    }
    /**
     * Grants permission to delete the specified sending authorization policy for the given identity (an email address or a domain)
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteEmailIdentityPolicy.html
     */
    toDeleteEmailIdentityPolicy() {
        return this.to('DeleteEmailIdentityPolicy');
    }
    /**
     * Grants permission to delete an email template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteEmailTemplate.html
     */
    toDeleteEmailTemplate() {
        return this.to('DeleteEmailTemplate');
    }
    /**
     * Grants permission to delete a multi-region endpoint
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteMultiRegionEndpoint.html
     */
    toDeleteMultiRegionEndpoint() {
        return this.to('DeleteMultiRegionEndpoint');
    }
    /**
     * Grants permission to remove an email address from the suppression list for your account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteSuppressedDestination.html
     */
    toDeleteSuppressedDestination() {
        return this.to('DeleteSuppressedDestination');
    }
    /**
     * Grants permission to get information about the email-sending status and capabilities for your account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetAccount.html
     */
    toGetAccount() {
        return this.to('GetAccount');
    }
    /**
     * Grants permission to retrieve a list of the deny lists on which your dedicated IP addresses or tracked domains appear
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetBlacklistReports.html
     */
    toGetBlacklistReports() {
        return this.to('GetBlacklistReports');
    }
    /**
     * Grants permission to get information about an existing configuration set
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetConfigurationSet.html
     */
    toGetConfigurationSet() {
        return this.to('GetConfigurationSet');
    }
    /**
     * Grants permission to retrieve a list of event destinations that are associated with a configuration set
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetConfigurationSetEventDestinations.html
     */
    toGetConfigurationSetEventDestinations() {
        return this.to('GetConfigurationSetEventDestinations');
    }
    /**
     * Grants permission to return a contact from a contact list
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetContact.html
     */
    toGetContact() {
        return this.to('GetContact');
    }
    /**
     * Grants permission to return contact list metadata
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetContactList.html
     */
    toGetContactList() {
        return this.to('GetContactList');
    }
    /**
     * Grants permission to return the custom email verification template for the template name you specify
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetCustomVerificationEmailTemplate.html
     */
    toGetCustomVerificationEmailTemplate() {
        return this.to('GetCustomVerificationEmailTemplate');
    }
    /**
     * Grants permission to get information about a dedicated IP address
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetDedicatedIp.html
     */
    toGetDedicatedIp() {
        return this.to('GetDedicatedIp');
    }
    /**
     * Grants permission to get information about a dedicated IP pool
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetDedicatedIpPool.html
     */
    toGetDedicatedIpPool() {
        return this.to('GetDedicatedIpPool');
    }
    /**
     * Grants permission to list the dedicated IP addresses a dedicated IP pool
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetDedicatedIps.html
     */
    toGetDedicatedIps() {
        return this.to('GetDedicatedIps');
    }
    /**
     * Grants permission to get the status of the Deliverability dashboard
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetDeliverabilityDashboardOptions.html
     */
    toGetDeliverabilityDashboardOptions() {
        return this.to('GetDeliverabilityDashboardOptions');
    }
    /**
     * Grants permission to retrieve the results of a predictive inbox placement test
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetDeliverabilityTestReport.html
     */
    toGetDeliverabilityTestReport() {
        return this.to('GetDeliverabilityTestReport');
    }
    /**
     * Grants permission to retrieve all the deliverability data for a specific campaign
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetDomainDeliverabilityCampaign.html
     */
    toGetDomainDeliverabilityCampaign() {
        return this.to('GetDomainDeliverabilityCampaign');
    }
    /**
     * Grants permission to retrieve inbox placement and engagement rates for the domains that you use to send email
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetDomainStatisticsReport.html
     */
    toGetDomainStatisticsReport() {
        return this.to('GetDomainStatisticsReport');
    }
    /**
     * Grants permission to get information about a specific identity
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetEmailIdentity.html
     */
    toGetEmailIdentity() {
        return this.to('GetEmailIdentity');
    }
    /**
     * Grants permission to return the requested sending authorization policies for the given identity (an email address or a domain)
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetEmailIdentityPolicies.html
     */
    toGetEmailIdentityPolicies() {
        return this.to('GetEmailIdentityPolicies');
    }
    /**
     * Grants permission to return the template object, which includes the subject line, HTML part, and text part for the template you specify
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetEmailTemplate.html
     */
    toGetEmailTemplate() {
        return this.to('GetEmailTemplate');
    }
    /**
     * Grants permission to get information about an export job
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifExportSourceType()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetExportJob.html
     */
    toGetExportJob() {
        return this.to('GetExportJob');
    }
    /**
     * Grants permission to provide information about an import job
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetImportJob.html
     */
    toGetImportJob() {
        return this.to('GetImportJob');
    }
    /**
     * Grants permission to provide insights about a message
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetMessageInsights.html
     */
    toGetMessageInsights() {
        return this.to('GetMessageInsights');
    }
    /**
     * Grants permission to get information about a multi-region endpoint
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetMultiRegionEndpoint.html
     */
    toGetMultiRegionEndpoint() {
        return this.to('GetMultiRegionEndpoint');
    }
    /**
     * Grants permission to retrieve information about a specific email address that's on the suppression list for your account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetSuppressedDestination.html
     */
    toGetSuppressedDestination() {
        return this.to('GetSuppressedDestination');
    }
    /**
     * Grants permission to list all of the configuration sets for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListConfigurationSets.html
     */
    toListConfigurationSets() {
        return this.to('ListConfigurationSets');
    }
    /**
     * Grants permission to list all of the contact lists available for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListContactLists.html
     */
    toListContactLists() {
        return this.to('ListContactLists');
    }
    /**
     * Grants permission to list the contacts present in a specific contact list
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListContacts.html
     */
    toListContacts() {
        return this.to('ListContacts');
    }
    /**
     * Grants permission to list all of the existing custom verification email templates for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListCustomVerificationEmailTemplates.html
     */
    toListCustomVerificationEmailTemplates() {
        return this.to('ListCustomVerificationEmailTemplates');
    }
    /**
     * Grants permission to list all of the dedicated IP pools for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListDedicatedIpPools.html
     */
    toListDedicatedIpPools() {
        return this.to('ListDedicatedIpPools');
    }
    /**
     * Grants permission to retrieve the list of the predictive inbox placement tests that you've performed, regardless of their statuses, for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListDeliverabilityTestReports.html
     */
    toListDeliverabilityTestReports() {
        return this.to('ListDeliverabilityTestReports');
    }
    /**
     * Grants permission to list deliverability data for campaigns that used a specific domain to send email during a specified time range
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListDomainDeliverabilityCampaigns.html
     */
    toListDomainDeliverabilityCampaigns() {
        return this.to('ListDomainDeliverabilityCampaigns');
    }
    /**
     * Grants permission to list the email identities for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListEmailIdentities.html
     */
    toListEmailIdentities() {
        return this.to('ListEmailIdentities');
    }
    /**
     * Grants permission to list all of the email templates for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListEmailTemplates.html
     */
    toListEmailTemplates() {
        return this.to('ListEmailTemplates');
    }
    /**
     * Grants permission to list all the exports jobs for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifExportSourceType()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListExportJobs.html
     */
    toListExportJobs() {
        return this.to('ListExportJobs');
    }
    /**
     * Grants permission to list all of the import jobs for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListImportJobs.html
     */
    toListImportJobs() {
        return this.to('ListImportJobs');
    }
    /**
     * Grants permission to list all of the multi-region endpoints for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListMultiRegionEndpoints.html
     */
    toListMultiRegionEndpoints() {
        return this.to('ListMultiRegionEndpoints');
    }
    /**
     * Grants permission to list recommendations for your account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListRecommendations.html
     */
    toListRecommendations() {
        return this.to('ListRecommendations');
    }
    /**
     * Grants permission to list email addresses that are on the suppression list for your account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListSuppressedDestinations.html
     */
    toListSuppressedDestinations() {
        return this.to('ListSuppressedDestinations');
    }
    /**
     * Grants permission to retrieve a list of the tags (keys and values) that are associated with a specific resource for your account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to enable or disable the automatic warm-up feature for dedicated IP addresses
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutAccountDedicatedIpWarmupAttributes.html
     */
    toPutAccountDedicatedIpWarmupAttributes() {
        return this.to('PutAccountDedicatedIpWarmupAttributes');
    }
    /**
     * Grants permission to update your account details
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutAccountDetails.html
     */
    toPutAccountDetails() {
        return this.to('PutAccountDetails');
    }
    /**
     * Grants permission to enable or disable the ability to send email for your account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutAccountSendingAttributes.html
     */
    toPutAccountSendingAttributes() {
        return this.to('PutAccountSendingAttributes');
    }
    /**
     * Grants permission to change the settings for the account-level suppression list
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutAccountSuppressionAttributes.html
     */
    toPutAccountSuppressionAttributes() {
        return this.to('PutAccountSuppressionAttributes');
    }
    /**
     * Grants permission to change the settings for VDM for your account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutAccountVdmAttributes.html
     */
    toPutAccountVdmAttributes() {
        return this.to('PutAccountVdmAttributes');
    }
    /**
     * Grants permission to associate a configuration set with a Mail Manager archive
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutConfigurationSetArchivingOptions.html
     */
    toPutConfigurationSetArchivingOptions() {
        return this.to('PutConfigurationSetArchivingOptions');
    }
    /**
     * Grants permission to associate a configuration set with a dedicated IP pool
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutConfigurationSetDeliveryOptions.html
     */
    toPutConfigurationSetDeliveryOptions() {
        return this.to('PutConfigurationSetDeliveryOptions');
    }
    /**
     * Grants permission to enable or disable collection of reputation metrics for emails that you send using a particular configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutConfigurationSetReputationOptions.html
     */
    toPutConfigurationSetReputationOptions() {
        return this.to('PutConfigurationSetReputationOptions');
    }
    /**
     * Grants permission to enable or disable email sending for messages that use a particular configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutConfigurationSetSendingOptions.html
     */
    toPutConfigurationSetSendingOptions() {
        return this.to('PutConfigurationSetSendingOptions');
    }
    /**
     * Grants permission to specify the account suppression list preferences for a particular configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutConfigurationSetSuppressionOptions.html
     */
    toPutConfigurationSetSuppressionOptions() {
        return this.to('PutConfigurationSetSuppressionOptions');
    }
    /**
     * Grants permission to specify a custom domain to use for open and click tracking elements in email that you send for a particular configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutConfigurationSetTrackingOptions.html
     */
    toPutConfigurationSetTrackingOptions() {
        return this.to('PutConfigurationSetTrackingOptions');
    }
    /**
     * Grants permission to override account-level VDM settings for a particular configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutConfigurationSetVdmOptions.html
     */
    toPutConfigurationSetVdmOptions() {
        return this.to('PutConfigurationSetVdmOptions');
    }
    /**
     * Grants permission to move a dedicated IP address to an existing dedicated IP pool
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutDedicatedIpInPool.html
     */
    toPutDedicatedIpInPool() {
        return this.to('PutDedicatedIpInPool');
    }
    /**
     * Grants permission to transition a dedicated IP pool from Standard to Managed
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutDedicatedIpPoolScalingAttributes.html
     */
    toPutDedicatedIpPoolScalingAttributes() {
        return this.to('PutDedicatedIpPoolScalingAttributes');
    }
    /**
     * Grants permission to put Dedicated IP warm up attributes
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutDedicatedIpWarmupAttributes.html
     */
    toPutDedicatedIpWarmupAttributes() {
        return this.to('PutDedicatedIpWarmupAttributes');
    }
    /**
     * Grants permission to enable or disable the Deliverability dashboard
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutDeliverabilityDashboardOption.html
     */
    toPutDeliverabilityDashboardOption() {
        return this.to('PutDeliverabilityDashboardOption');
    }
    /**
     * Grants permission to associate a configuration set with an email identity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutEmailIdentityConfigurationSetAttributes.html
     */
    toPutEmailIdentityConfigurationSetAttributes() {
        return this.to('PutEmailIdentityConfigurationSetAttributes');
    }
    /**
     * Grants permission to enable or disable DKIM authentication for an email identity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutEmailIdentityDkimAttributes.html
     */
    toPutEmailIdentityDkimAttributes() {
        return this.to('PutEmailIdentityDkimAttributes');
    }
    /**
     * Grants permission to configure or change the DKIM authentication settings for an email domain identity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutEmailIdentityDkimSigningAttributes.html
     */
    toPutEmailIdentityDkimSigningAttributes() {
        return this.to('PutEmailIdentityDkimSigningAttributes');
    }
    /**
     * Grants permission to enable or disable feedback forwarding for an email identity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutEmailIdentityFeedbackAttributes.html
     */
    toPutEmailIdentityFeedbackAttributes() {
        return this.to('PutEmailIdentityFeedbackAttributes');
    }
    /**
     * Grants permission to enable or disable the custom MAIL FROM domain configuration for an email identity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutEmailIdentityMailFromAttributes.html
     */
    toPutEmailIdentityMailFromAttributes() {
        return this.to('PutEmailIdentityMailFromAttributes');
    }
    /**
     * Grants permission to add an email address to the suppression list
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutSuppressedDestination.html
     */
    toPutSuppressedDestination() {
        return this.to('PutSuppressedDestination');
    }
    /**
     * Grants permission to replicate email identity DKIM signing key
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifReplicaRegion()
     */
    toReplicateEmailIdentityDkimSigningKey() {
        return this.to('ReplicateEmailIdentityDkimSigningKey');
    }
    /**
     * Grants permission to compose an email message to multiple destinations
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifMultiRegionEndpointId()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_SendBulkEmail.html
     */
    toSendBulkEmail() {
        return this.to('SendBulkEmail');
    }
    /**
     * Grants permission to add an email address to the list of identities and attempts to verify it
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_SendCustomVerificationEmail.html
     */
    toSendCustomVerificationEmail() {
        return this.to('SendCustomVerificationEmail');
    }
    /**
     * Grants permission to send an email message
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifFeedbackAddress()
     * - .ifFromAddress()
     * - .ifFromDisplayName()
     * - .ifRecipients()
     * - .ifMultiRegionEndpointId()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_SendEmail.html
     */
    toSendEmail() {
        return this.to('SendEmail');
    }
    /**
     * Grants permission to add one or more tags (keys and values) to a specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to create a preview of the MIME content of an email when provided with a template and a set of replacement data
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_TestRenderEmailTemplate.html
     */
    toTestRenderEmailTemplate() {
        return this.to('TestRenderEmailTemplate');
    }
    /**
     * Grants permission to remove one or more tags (keys and values) from a specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the configuration of an event destination for a configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_UpdateConfigurationSetEventDestination.html
     */
    toUpdateConfigurationSetEventDestination() {
        return this.to('UpdateConfigurationSetEventDestination');
    }
    /**
     * Grants permission to update a contact's preferences for a list
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_UpdateContact.html
     */
    toUpdateContact() {
        return this.to('UpdateContact');
    }
    /**
     * Grants permission to update contact list metadata
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_UpdateContactList.html
     */
    toUpdateContactList() {
        return this.to('UpdateContactList');
    }
    /**
     * Grants permission to update an existing custom verification email template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_UpdateCustomVerificationEmailTemplate.html
     */
    toUpdateCustomVerificationEmailTemplate() {
        return this.to('UpdateCustomVerificationEmailTemplate');
    }
    /**
     * Grants permission to update the specified sending authorization policy for the given identity (an email address or a domain)
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_UpdateEmailIdentityPolicy.html
     */
    toUpdateEmailIdentityPolicy() {
        return this.to('UpdateEmailIdentityPolicy');
    }
    /**
     * Grants permission to update an email template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_UpdateEmailTemplate.html
     */
    toUpdateEmailTemplate() {
        return this.to('UpdateEmailTemplate');
    }
    /**
     * Adds a resource of type configuration-set to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ConfigurationSet.html
     *
     * @param configurationSetName - Identifier for the configurationSetName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConfigurationSet(configurationSetName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:configuration-set/${configurationSetName}`);
    }
    /**
     * Adds a resource of type contact-list to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ContactList.html
     *
     * @param contactListName - Identifier for the contactListName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onContactList(contactListName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:contact-list/${contactListName}`);
    }
    /**
     * Adds a resource of type custom-verification-email-template to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CustomVerificationEmailTemplateMetadata.html
     *
     * @param templateName - Identifier for the templateName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onCustomVerificationEmailTemplate(templateName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:custom-verification-email-template/${templateName}`);
    }
    /**
     * Adds a resource of type dedicated-ip-pool to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DedicatedIp.html
     *
     * @param dedicatedIPPool - Identifier for the dedicatedIPPool.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDedicatedIpPool(dedicatedIPPool, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:dedicated-ip-pool/${dedicatedIPPool}`);
    }
    /**
     * Adds a resource of type deliverability-test-report to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeliverabilityTestReport.html
     *
     * @param reportId - Identifier for the reportId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDeliverabilityTestReport(reportId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:deliverability-test-report/${reportId}`);
    }
    /**
     * Adds a resource of type export-job to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ExportJobSummary.html
     *
     * @param exportJobId - Identifier for the exportJobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onExportJob(exportJobId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:export-job/${exportJobId}`);
    }
    /**
     * Adds a resource of type identity to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_IdentityInfo.html
     *
     * @param identityName - Identifier for the identityName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIdentity(identityName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:identity/${identityName}`);
    }
    /**
     * Adds a resource of type import-job to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ImportJobSummary.html
     *
     * @param importJobId - Identifier for the importJobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onImportJob(importJobId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:import-job/${importJobId}`);
    }
    /**
     * Adds a resource of type template to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_Template.html
     *
     * @param templateName - Identifier for the templateName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onTemplate(templateName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:template/${templateName}`);
    }
    /**
     * Adds a resource of type multi-region-endpoint to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_MultiRegionEndpoint.html
     *
     * @param endpointName - Identifier for the endpointName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onMultiRegionEndpoint(endpointName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:multi-region-endpoint/${endpointName}`);
    }
    /**
     * Adds a resource of type mailmanager-archive to the statement
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_Archive.html
     *
     * @param archiveId - Identifier for the archiveId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMailmanagerArchive(archiveId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:mailmanager-archive/${archiveId}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateConfigurationSet()
     * - .toCreateContactList()
     * - .toCreateDedicatedIpPool()
     * - .toCreateDeliverabilityTestReport()
     * - .toCreateEmailIdentity()
     * - .toCreateMultiRegionEndpoint()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toBatchGetMetricData()
     * - .toCreateConfigurationSetEventDestination()
     * - .toCreateContact()
     * - .toCreateEmailIdentityPolicy()
     * - .toDeleteConfigurationSet()
     * - .toDeleteConfigurationSetEventDestination()
     * - .toDeleteContact()
     * - .toDeleteContactList()
     * - .toDeleteDedicatedIpPool()
     * - .toDeleteEmailIdentity()
     * - .toDeleteEmailIdentityPolicy()
     * - .toDeleteMultiRegionEndpoint()
     * - .toGetConfigurationSet()
     * - .toGetConfigurationSetEventDestinations()
     * - .toGetContact()
     * - .toGetDedicatedIpPool()
     * - .toGetDedicatedIps()
     * - .toGetDeliverabilityTestReport()
     * - .toGetDomainStatisticsReport()
     * - .toGetEmailIdentity()
     * - .toGetEmailIdentityPolicies()
     * - .toGetMultiRegionEndpoint()
     * - .toListRecommendations()
     * - .toPutConfigurationSetArchivingOptions()
     * - .toPutConfigurationSetDeliveryOptions()
     * - .toPutConfigurationSetReputationOptions()
     * - .toPutConfigurationSetSendingOptions()
     * - .toPutConfigurationSetSuppressionOptions()
     * - .toPutConfigurationSetTrackingOptions()
     * - .toPutConfigurationSetVdmOptions()
     * - .toPutDedicatedIpInPool()
     * - .toPutDedicatedIpPoolScalingAttributes()
     * - .toPutEmailIdentityConfigurationSetAttributes()
     * - .toPutEmailIdentityDkimAttributes()
     * - .toPutEmailIdentityDkimSigningAttributes()
     * - .toPutEmailIdentityFeedbackAttributes()
     * - .toPutEmailIdentityMailFromAttributes()
     * - .toUpdateConfigurationSetEventDestination()
     * - .toUpdateContact()
     * - .toUpdateContactList()
     * - .toUpdateEmailIdentityPolicy()
     *
     * Applies to resource types:
     * - configuration-set
     * - contact-list
     * - dedicated-ip-pool
     * - deliverability-test-report
     * - identity
     * - mailmanager-archive
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateConfigurationSet()
     * - .toCreateContactList()
     * - .toCreateDedicatedIpPool()
     * - .toCreateDeliverabilityTestReport()
     * - .toCreateEmailIdentity()
     * - .toCreateMultiRegionEndpoint()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the SES API version
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toBatchGetMetricData()
     * - .toCancelExportJob()
     * - .toCreateConfigurationSet()
     * - .toCreateConfigurationSetEventDestination()
     * - .toCreateContact()
     * - .toCreateContactList()
     * - .toCreateCustomVerificationEmailTemplate()
     * - .toCreateDedicatedIpPool()
     * - .toCreateDeliverabilityTestReport()
     * - .toCreateEmailIdentity()
     * - .toCreateEmailIdentityPolicy()
     * - .toCreateEmailTemplate()
     * - .toCreateExportJob()
     * - .toCreateImportJob()
     * - .toCreateMultiRegionEndpoint()
     * - .toDeleteConfigurationSet()
     * - .toDeleteConfigurationSetEventDestination()
     * - .toDeleteContact()
     * - .toDeleteContactList()
     * - .toDeleteCustomVerificationEmailTemplate()
     * - .toDeleteDedicatedIpPool()
     * - .toDeleteEmailIdentity()
     * - .toDeleteEmailIdentityPolicy()
     * - .toDeleteEmailTemplate()
     * - .toDeleteMultiRegionEndpoint()
     * - .toDeleteSuppressedDestination()
     * - .toGetAccount()
     * - .toGetBlacklistReports()
     * - .toGetConfigurationSet()
     * - .toGetConfigurationSetEventDestinations()
     * - .toGetContact()
     * - .toGetContactList()
     * - .toGetCustomVerificationEmailTemplate()
     * - .toGetDedicatedIp()
     * - .toGetDedicatedIpPool()
     * - .toGetDedicatedIps()
     * - .toGetDeliverabilityDashboardOptions()
     * - .toGetDeliverabilityTestReport()
     * - .toGetDomainDeliverabilityCampaign()
     * - .toGetDomainStatisticsReport()
     * - .toGetEmailIdentity()
     * - .toGetEmailIdentityPolicies()
     * - .toGetEmailTemplate()
     * - .toGetExportJob()
     * - .toGetImportJob()
     * - .toGetMessageInsights()
     * - .toGetMultiRegionEndpoint()
     * - .toGetSuppressedDestination()
     * - .toListConfigurationSets()
     * - .toListContactLists()
     * - .toListContacts()
     * - .toListCustomVerificationEmailTemplates()
     * - .toListDedicatedIpPools()
     * - .toListDeliverabilityTestReports()
     * - .toListDomainDeliverabilityCampaigns()
     * - .toListEmailIdentities()
     * - .toListEmailTemplates()
     * - .toListExportJobs()
     * - .toListImportJobs()
     * - .toListMultiRegionEndpoints()
     * - .toListRecommendations()
     * - .toListSuppressedDestinations()
     * - .toListTagsForResource()
     * - .toPutAccountDedicatedIpWarmupAttributes()
     * - .toPutAccountDetails()
     * - .toPutAccountSendingAttributes()
     * - .toPutAccountSuppressionAttributes()
     * - .toPutAccountVdmAttributes()
     * - .toPutConfigurationSetArchivingOptions()
     * - .toPutConfigurationSetDeliveryOptions()
     * - .toPutConfigurationSetReputationOptions()
     * - .toPutConfigurationSetSendingOptions()
     * - .toPutConfigurationSetSuppressionOptions()
     * - .toPutConfigurationSetTrackingOptions()
     * - .toPutConfigurationSetVdmOptions()
     * - .toPutDedicatedIpInPool()
     * - .toPutDedicatedIpPoolScalingAttributes()
     * - .toPutDedicatedIpWarmupAttributes()
     * - .toPutDeliverabilityDashboardOption()
     * - .toPutEmailIdentityConfigurationSetAttributes()
     * - .toPutEmailIdentityDkimAttributes()
     * - .toPutEmailIdentityDkimSigningAttributes()
     * - .toPutEmailIdentityFeedbackAttributes()
     * - .toPutEmailIdentityMailFromAttributes()
     * - .toPutSuppressedDestination()
     * - .toSendBulkEmail()
     * - .toSendCustomVerificationEmail()
     * - .toSendEmail()
     * - .toTagResource()
     * - .toTestRenderEmailTemplate()
     * - .toUntagResource()
     * - .toUpdateConfigurationSetEventDestination()
     * - .toUpdateContact()
     * - .toUpdateContactList()
     * - .toUpdateCustomVerificationEmailTemplate()
     * - .toUpdateEmailIdentityPolicy()
     * - .toUpdateEmailTemplate()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifApiVersion(value, operator) {
        return this.if(`ApiVersion`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the export source type
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toCancelExportJob()
     * - .toCreateExportJob()
     * - .toGetExportJob()
     * - .toListExportJobs()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifExportSourceType(value, operator) {
        return this.if(`ExportSourceType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the "Return-Path" address, which specifies where bounces and complaints are sent by email feedback forwarding
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toSendEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFeedbackAddress(value, operator) {
        return this.if(`FeedbackAddress`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the "From" address of a message
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toSendEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFromAddress(value, operator) {
        return this.if(`FromAddress`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the "From" address that is used as the display name of a message
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toSendEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFromDisplayName(value, operator) {
        return this.if(`FromDisplayName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the multi-region endpoint ID that is used to send email
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toSendBulkEmail()
     * - .toSendEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMultiRegionEndpointId(value, operator) {
        return this.if(`MultiRegionEndpointId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the recipient addresses of a message, which include the "To", "CC", and "BCC" addresses
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toSendEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRecipients(value, operator) {
        return this.if(`Recipients`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the replica regions for Replicating domain DKIM signing key
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toReplicateEmailIdentityDkimSigningKey()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReplicaRegion(value, operator) {
        return this.if(`ReplicaRegion`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [ses-v2](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonsimpleemailservicev2.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'ses';
        this.accessLevelList = {
            Read: [
                'BatchGetMetricData',
                'GetAccount',
                'GetBlacklistReports',
                'GetConfigurationSet',
                'GetConfigurationSetEventDestinations',
                'GetContact',
                'GetContactList',
                'GetCustomVerificationEmailTemplate',
                'GetDedicatedIp',
                'GetDedicatedIpPool',
                'GetDedicatedIps',
                'GetDeliverabilityDashboardOptions',
                'GetDeliverabilityTestReport',
                'GetDomainDeliverabilityCampaign',
                'GetDomainStatisticsReport',
                'GetEmailIdentity',
                'GetEmailIdentityPolicies',
                'GetEmailTemplate',
                'GetExportJob',
                'GetImportJob',
                'GetMessageInsights',
                'GetMultiRegionEndpoint',
                'GetSuppressedDestination',
                'ListDomainDeliverabilityCampaigns',
                'ListRecommendations',
                'ListSuppressedDestinations',
                'ListTagsForResource'
            ],
            Write: [
                'CancelExportJob',
                'CreateConfigurationSet',
                'CreateConfigurationSetEventDestination',
                'CreateContact',
                'CreateContactList',
                'CreateCustomVerificationEmailTemplate',
                'CreateDedicatedIpPool',
                'CreateDeliverabilityTestReport',
                'CreateEmailIdentity',
                'CreateEmailTemplate',
                'CreateExportJob',
                'CreateImportJob',
                'CreateMultiRegionEndpoint',
                'DeleteConfigurationSet',
                'DeleteConfigurationSetEventDestination',
                'DeleteContact',
                'DeleteContactList',
                'DeleteCustomVerificationEmailTemplate',
                'DeleteDedicatedIpPool',
                'DeleteEmailIdentity',
                'DeleteEmailTemplate',
                'DeleteMultiRegionEndpoint',
                'DeleteSuppressedDestination',
                'PutAccountDedicatedIpWarmupAttributes',
                'PutAccountDetails',
                'PutAccountSendingAttributes',
                'PutAccountSuppressionAttributes',
                'PutAccountVdmAttributes',
                'PutConfigurationSetArchivingOptions',
                'PutConfigurationSetDeliveryOptions',
                'PutConfigurationSetReputationOptions',
                'PutConfigurationSetSendingOptions',
                'PutConfigurationSetSuppressionOptions',
                'PutConfigurationSetTrackingOptions',
                'PutConfigurationSetVdmOptions',
                'PutDedicatedIpInPool',
                'PutDedicatedIpPoolScalingAttributes',
                'PutDedicatedIpWarmupAttributes',
                'PutDeliverabilityDashboardOption',
                'PutEmailIdentityConfigurationSetAttributes',
                'PutEmailIdentityDkimAttributes',
                'PutEmailIdentityDkimSigningAttributes',
                'PutEmailIdentityFeedbackAttributes',
                'PutEmailIdentityMailFromAttributes',
                'PutSuppressedDestination',
                'SendBulkEmail',
                'SendCustomVerificationEmail',
                'SendEmail',
                'TestRenderEmailTemplate',
                'UpdateConfigurationSetEventDestination',
                'UpdateContact',
                'UpdateContactList',
                'UpdateCustomVerificationEmailTemplate',
                'UpdateEmailTemplate'
            ],
            'Permissions management': [
                'CreateEmailIdentityPolicy',
                'DeleteEmailIdentityPolicy',
                'ReplicateEmailIdentityDkimSigningKey',
                'UpdateEmailIdentityPolicy'
            ],
            List: [
                'ListConfigurationSets',
                'ListContactLists',
                'ListContacts',
                'ListCustomVerificationEmailTemplates',
                'ListDedicatedIpPools',
                'ListDeliverabilityTestReports',
                'ListEmailIdentities',
                'ListEmailTemplates',
                'ListExportJobs',
                'ListImportJobs',
                'ListMultiRegionEndpoints'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.SesV2 = SesV2;
//# sourceMappingURL=data:application/json;base64,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