"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.States = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [states](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsstepfunctions.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class States extends shared_1.PolicyStatement {
    /**
     * Grants permission to create an activity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_CreateActivity.html
     */
    toCreateActivity() {
        return this.to('CreateActivity');
    }
    /**
     * Grants permission to create a state machine
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     * - states:PublishStateMachineVersion
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_CreateStateMachine.html
     */
    toCreateStateMachine() {
        return this.to('CreateStateMachine');
    }
    /**
     * Grants permission to create a state machine alias
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifStateMachineQualifier()
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_CreateStateMachineAlias.html
     */
    toCreateStateMachineAlias() {
        return this.to('CreateStateMachineAlias');
    }
    /**
     * Grants permission to delete an activity
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_DeleteActivity.html
     */
    toDeleteActivity() {
        return this.to('DeleteActivity');
    }
    /**
     * Grants permission to delete a state machine
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_DeleteStateMachine.html
     */
    toDeleteStateMachine() {
        return this.to('DeleteStateMachine');
    }
    /**
     * Grants permission to delete a state machine alias
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifStateMachineQualifier()
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_DeleteStateMachineAlias.html
     */
    toDeleteStateMachineAlias() {
        return this.to('DeleteStateMachineAlias');
    }
    /**
     * Grants permission to delete a state machine version
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifStateMachineQualifier()
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_DeleteStateMachineVersion.html
     */
    toDeleteStateMachineVersion() {
        return this.to('DeleteStateMachineVersion');
    }
    /**
     * Grants permission to describe an activity
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_DescribeActivity.html
     */
    toDescribeActivity() {
        return this.to('DescribeActivity');
    }
    /**
     * Grants permission to describe an execution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_DescribeExecution.html
     */
    toDescribeExecution() {
        return this.to('DescribeExecution');
    }
    /**
     * Grants permission to describe a map run
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_DescribeMapRun.html
     */
    toDescribeMapRun() {
        return this.to('DescribeMapRun');
    }
    /**
     * Grants permission to describe a state machine
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifStateMachineQualifier()
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_DescribeStateMachine.html
     */
    toDescribeStateMachine() {
        return this.to('DescribeStateMachine');
    }
    /**
     * Grants permission to describe a state machine alias
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifStateMachineQualifier()
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_DescribeStateMachineAlias.html
     */
    toDescribeStateMachineAlias() {
        return this.to('DescribeStateMachineAlias');
    }
    /**
     * Grants permission to describe the state machine for an execution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_DescribeStateMachineForExecution.html
     */
    toDescribeStateMachineForExecution() {
        return this.to('DescribeStateMachineForExecution');
    }
    /**
     * Grants permission to be used by workers to retrieve a task (with the specified activity ARN) which has been scheduled for execution by a running state machine
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_GetActivityTask.html
     */
    toGetActivityTask() {
        return this.to('GetActivityTask');
    }
    /**
     * Grants permission to return the history of the specified execution as a list of events
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_GetExecutionHistory.html
     */
    toGetExecutionHistory() {
        return this.to('GetExecutionHistory');
    }
    /**
     * Grants permission to invoke the HTTP Task state
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/step-functions/latest/dg/connect-third-party-apis.html
     */
    toInvokeHTTPEndpoint() {
        return this.to('InvokeHTTPEndpoint');
    }
    /**
     * Grants permission to list the existing activities
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_ListActivities.html
     */
    toListActivities() {
        return this.to('ListActivities');
    }
    /**
     * Grants permission to list the executions of a state machine
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifStateMachineQualifier()
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_ListExecutions.html
     */
    toListExecutions() {
        return this.to('ListExecutions');
    }
    /**
     * Grants permission to list the map runs of an execution
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_ListMapRuns.html
     */
    toListMapRuns() {
        return this.to('ListMapRuns');
    }
    /**
     * Grants permission to list the aliases of a state machine
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifStateMachineQualifier()
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_ListStateMachineAliases.html
     */
    toListStateMachineAliases() {
        return this.to('ListStateMachineAliases');
    }
    /**
     * Grants permission to list the versions of a state machine
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_ListStateMachineVersions.html
     */
    toListStateMachineVersions() {
        return this.to('ListStateMachineVersions');
    }
    /**
     * Grants permission to lists the existing state machines
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_ListStateMachines.html
     */
    toListStateMachines() {
        return this.to('ListStateMachines');
    }
    /**
     * Grants permission to list tags for an AWS Step Functions resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to publish a state machine version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_PublishStateMachineVersion.html
     */
    toPublishStateMachineVersion() {
        return this.to('PublishStateMachineVersion');
    }
    /**
     * Grants permission to redrive an execution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_RedriveExecution.html
     */
    toRedriveExecution() {
        return this.to('RedriveExecution');
    }
    /**
     * Grants permission to reveal sensitive data from an execution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/step-functions/latest/dg/test-state-isolation.html
     */
    toRevealSecrets() {
        return this.to('RevealSecrets');
    }
    /**
     * Grants permission to report that the task identified by the taskToken failed
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_SendTaskFailure.html
     */
    toSendTaskFailure() {
        return this.to('SendTaskFailure');
    }
    /**
     * Grants permission to report to the service that the task represented by the specified taskToken is still making progress
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_SendTaskHeartbeat.html
     */
    toSendTaskHeartbeat() {
        return this.to('SendTaskHeartbeat');
    }
    /**
     * Grants permission to report that the task identified by the taskToken completed successfully
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_SendTaskSuccess.html
     */
    toSendTaskSuccess() {
        return this.to('SendTaskSuccess');
    }
    /**
     * Grants permission to start a state machine execution
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifStateMachineQualifier()
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_StartExecution.html
     */
    toStartExecution() {
        return this.to('StartExecution');
    }
    /**
     * Grants permission to start a Synchronous Express state machine execution
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifStateMachineQualifier()
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_StartSyncExecution.html
     */
    toStartSyncExecution() {
        return this.to('StartSyncExecution');
    }
    /**
     * Grants permission to stop an execution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_StopExecution.html
     */
    toStopExecution() {
        return this.to('StopExecution');
    }
    /**
     * Grants permission to tag an AWS Step Functions resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to test a state machine definition
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - states:RevealSecrets
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_TestState.html
     */
    toTestState() {
        return this.to('TestState');
    }
    /**
     * Grants permission to remove a tag from an AWS Step Functions resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a map run
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_UpdateMapRun.html
     */
    toUpdateMapRun() {
        return this.to('UpdateMapRun');
    }
    /**
     * Grants permission to update a state machine
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     * - states:PublishStateMachineVersion
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_UpdateStateMachine.html
     */
    toUpdateStateMachine() {
        return this.to('UpdateStateMachine');
    }
    /**
     * Grants permission to update a state machine alias
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifStateMachineQualifier()
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_UpdateStateMachineAlias.html
     */
    toUpdateStateMachineAlias() {
        return this.to('UpdateStateMachineAlias');
    }
    /**
     * Grants permission to validate a state machine definition
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/step-functions/latest/apireference/API_ValidateStateMachineDefinition.html
     */
    toValidateStateMachineDefinition() {
        return this.to('ValidateStateMachineDefinition');
    }
    /**
     * Adds a resource of type activity to the statement
     *
     * https://docs.aws.amazon.com/step-functions/latest/dg/concepts-activities.html
     *
     * @param activityName - Identifier for the activityName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onActivity(activityName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:states:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:activity:${activityName}`);
    }
    /**
     * Adds a resource of type execution to the statement
     *
     * https://docs.aws.amazon.com/step-functions/latest/dg/concepts-state-machine-executions.html
     *
     * @param stateMachineName - Identifier for the stateMachineName.
     * @param executionId - Identifier for the executionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onExecution(stateMachineName, executionId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:states:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:execution:${stateMachineName}:${executionId}`);
    }
    /**
     * Adds a resource of type express to the statement
     *
     * https://docs.aws.amazon.com/step-functions/latest/dg/concepts-state-machine-executions.html
     *
     * @param stateMachineName - Identifier for the stateMachineName.
     * @param executionId - Identifier for the executionId.
     * @param expressId - Identifier for the expressId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onExpress(stateMachineName, executionId, expressId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:states:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:express:${stateMachineName}:${executionId}:${expressId}`);
    }
    /**
     * Adds a resource of type statemachine to the statement
     *
     * https://docs.aws.amazon.com/step-functions/latest/dg/concepts-amazon-states-language.html
     *
     * @param stateMachineName - Identifier for the stateMachineName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStatemachine(stateMachineName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:states:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:stateMachine:${stateMachineName}`);
    }
    /**
     * Adds a resource of type statemachineversion to the statement
     *
     * https://docs.aws.amazon.com/step-functions/latest/dg/concepts-cd-aliasing-versioning.html
     *
     * @param stateMachineName - Identifier for the stateMachineName.
     * @param stateMachineVersionId - Identifier for the stateMachineVersionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStatemachineversion(stateMachineName, stateMachineVersionId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:states:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:stateMachine:${stateMachineName}:${stateMachineVersionId}`);
    }
    /**
     * Adds a resource of type statemachinealias to the statement
     *
     * https://docs.aws.amazon.com/step-functions/latest/dg/concepts-cd-aliasing-versioning.html
     *
     * @param stateMachineName - Identifier for the stateMachineName.
     * @param stateMachineAliasName - Identifier for the stateMachineAliasName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStatemachinealias(stateMachineName, stateMachineAliasName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:states:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:stateMachine:${stateMachineName}:${stateMachineAliasName}`);
    }
    /**
     * Adds a resource of type maprun to the statement
     *
     * https://docs.aws.amazon.com/step-functions/latest/dg/concepts-examine-map-run.html
     *
     * @param stateMachineName - Identifier for the stateMachineName.
     * @param mapRunLabel - Identifier for the mapRunLabel.
     * @param mapRunId - Identifier for the mapRunId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onMaprun(stateMachineName, mapRunLabel, mapRunId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:states:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:mapRun:${stateMachineName}/${mapRunLabel}:${mapRunId}`);
    }
    /**
     * Adds a resource of type labelled execution to the statement
     *
     * https://docs.aws.amazon.com/step-functions/latest/dg/concepts-examine-map-run.html
     *
     * @param stateMachineName - Identifier for the stateMachineName.
     * @param mapRunLabel - Identifier for the mapRunLabel.
     * @param executionId - Identifier for the executionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onLabelledExecution(stateMachineName, mapRunLabel, executionId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:states:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:execution:${stateMachineName}/${mapRunLabel}:${executionId}`);
    }
    /**
     * Adds a resource of type labelled express to the statement
     *
     * https://docs.aws.amazon.com/step-functions/latest/dg/concepts-examine-map-run.html
     *
     * @param stateMachineName - Identifier for the stateMachineName.
     * @param mapRunLabel - Identifier for the mapRunLabel.
     * @param executionId - Identifier for the executionId.
     * @param expressId - Identifier for the expressId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onLabelledExpress(stateMachineName, mapRunLabel, executionId, expressId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:states:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:express:${stateMachineName}/${mapRunLabel}:${executionId}:${expressId}`);
    }
    /**
     * Filters access by a tag key and value pair that is allowed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateActivity()
     * - .toCreateStateMachine()
     * - .toTagResource()
     * - .toUpdateStateMachine()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a tag key and value pair of a resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - activity
     * - execution
     * - statemachine
     * - statemachineversion
     * - statemachinealias
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a list of tag keys that are allowed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateActivity()
     * - .toCreateStateMachine()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateStateMachine()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the endpoint that the HTTP Task state allows in the request
     *
     * https://docs.aws.amazon.com/step-functions/latest/dg/connect-third-party-apis.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifHTTPEndpoint(value, operator) {
        return this.if(`HTTPEndpoint`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the method that the HTTP Task state allows in the request
     *
     * https://docs.aws.amazon.com/step-functions/latest/dg/connect-third-party-apis.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifHTTPMethod(value, operator) {
        return this.if(`HTTPMethod`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the qualifier of a state machine ARN
     *
     * https://docs.aws.amazon.com/step-functions/latest/dg/auth-version-alias.html
     *
     * Applies to actions:
     * - .toCreateStateMachineAlias()
     * - .toDeleteStateMachineAlias()
     * - .toDeleteStateMachineVersion()
     * - .toDescribeStateMachine()
     * - .toDescribeStateMachineAlias()
     * - .toListExecutions()
     * - .toListStateMachineAliases()
     * - .toStartExecution()
     * - .toStartSyncExecution()
     * - .toUpdateStateMachineAlias()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifStateMachineQualifier(value, operator) {
        return this.if(`StateMachineQualifier`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [states](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsstepfunctions.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'states';
        this.accessLevelList = {
            Write: [
                'CreateActivity',
                'CreateStateMachine',
                'CreateStateMachineAlias',
                'DeleteActivity',
                'DeleteStateMachine',
                'DeleteStateMachineAlias',
                'DeleteStateMachineVersion',
                'GetActivityTask',
                'InvokeHTTPEndpoint',
                'PublishStateMachineVersion',
                'RedriveExecution',
                'SendTaskFailure',
                'SendTaskHeartbeat',
                'SendTaskSuccess',
                'StartExecution',
                'StartSyncExecution',
                'StopExecution',
                'TestState',
                'UpdateMapRun',
                'UpdateStateMachine',
                'UpdateStateMachineAlias'
            ],
            Read: [
                'DescribeActivity',
                'DescribeExecution',
                'DescribeMapRun',
                'DescribeStateMachine',
                'DescribeStateMachineAlias',
                'DescribeStateMachineForExecution',
                'GetExecutionHistory',
                'RevealSecrets',
                'ValidateStateMachineDefinition'
            ],
            List: [
                'ListActivities',
                'ListExecutions',
                'ListMapRuns',
                'ListStateMachineAliases',
                'ListStateMachineVersions',
                'ListStateMachines',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.States = States;
//# sourceMappingURL=data:application/json;base64,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