"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Textract = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [textract](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazontextract.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Textract extends shared_1.PolicyStatement {
    /**
     * Grants permission to detect instances of real-world document entities within an image provided as input
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/textract/latest/dg/API_AnalyzeDocument.html
     */
    toAnalyzeDocument() {
        return this.to('AnalyzeDocument');
    }
    /**
     * Grants permission to detect instances of real-world document entities within an image provided as input
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/textract/latest/dg/API_AnalyzeExpense.html
     */
    toAnalyzeExpense() {
        return this.to('AnalyzeExpense');
    }
    /**
     * Grants permission to detect relevant information from identity documents provided as input
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/textract/latest/dg/API_AnalyzeID.html
     */
    toAnalyzeID() {
        return this.to('AnalyzeID');
    }
    /**
     * Grants permission to create an Amazon Textract adapter
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/textract/latest/dg/API_CreateAdapter.html
     */
    toCreateAdapter() {
        return this.to('CreateAdapter');
    }
    /**
     * Grants permission to create an Amazon Textract adapter version
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/textract/latest/dg/API_CreateAdapterVersion.html
     */
    toCreateAdapterVersion() {
        return this.to('CreateAdapterVersion');
    }
    /**
     * Grants permission to delete an Amazon Textract adapter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/textract/latest/dg/API_DeleteAdapter.html
     */
    toDeleteAdapter() {
        return this.to('DeleteAdapter');
    }
    /**
     * Grants permission to delete an Amazon Textract adapter version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/textract/latest/dg/API_DeleteAdapterVersion.html
     */
    toDeleteAdapterVersion() {
        return this.to('DeleteAdapterVersion');
    }
    /**
     * Grants permission to detect text in document images
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/textract/latest/dg/API_DetectDocumentText.html
     */
    toDetectDocumentText() {
        return this.to('DetectDocumentText');
    }
    /**
     * Grants permission to get an Amazon Textract adapter
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/textract/latest/dg/API_GetAdapter.html
     */
    toGetAdapter() {
        return this.to('GetAdapter');
    }
    /**
     * Grants permission to get an Amazon Textract adapter version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/textract/latest/dg/API_GetAdapterVersion.html
     */
    toGetAdapterVersion() {
        return this.to('GetAdapterVersion');
    }
    /**
     * Grants permission to return information about a document analysis job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/textract/latest/dg/API_GetDocumentAnalysis.html
     */
    toGetDocumentAnalysis() {
        return this.to('GetDocumentAnalysis');
    }
    /**
     * Grants permission to return information about a document text detection job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/textract/latest/dg/API_GetDocumentTextDetection.html
     */
    toGetDocumentTextDetection() {
        return this.to('GetDocumentTextDetection');
    }
    /**
     * Grants permission to return information about an expense analysis job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/textract/latest/dg/API_GetExpenseAnalysis.html
     */
    toGetExpenseAnalysis() {
        return this.to('GetExpenseAnalysis');
    }
    /**
     * Grants permission to retrieve page-level information regarding a lending analysis job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/textract/latest/dg/API_GetLendingAnalysis.html
     */
    toGetLendingAnalysis() {
        return this.to('GetLendingAnalysis');
    }
    /**
     * Grants permission to retrieve summarized information regarding a lending analysis job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/textract/latest/dg/API_GetLendingAnalysisSummary.html
     */
    toGetLendingAnalysisSummary() {
        return this.to('GetLendingAnalysisSummary');
    }
    /**
     * Grants permission to list Amazon Textract adapter versions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/textract/latest/dg/API_ListAdapterVersions.html
     */
    toListAdapterVersions() {
        return this.to('ListAdapterVersions');
    }
    /**
     * Grants permission to list Amazon Textract adapters
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/textract/latest/dg/API_ListAdapters.html
     */
    toListAdapters() {
        return this.to('ListAdapters');
    }
    /**
     * Grants permission to return a list of tags associated with a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/textract/latest/dg/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to start an asynchronous job to detect instances of real-world document entities within an image or pdf provided as input
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/textract/latest/dg/API_StartDocumentAnalysis.html
     */
    toStartDocumentAnalysis() {
        return this.to('StartDocumentAnalysis');
    }
    /**
     * Grants permission to start an asynchronous job to detect text in document images or pdfs
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/textract/latest/dg/API_StartDocumentTextDetection.html
     */
    toStartDocumentTextDetection() {
        return this.to('StartDocumentTextDetection');
    }
    /**
     * Grants permission to start an asynchronous job to detect instances of invoices or receipts within an image or pdf provided as input
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/textract/latest/dg/API_StartExpenseAnalysis.html
     */
    toStartExpenseAnalysis() {
        return this.to('StartExpenseAnalysis');
    }
    /**
     * Grants permission to start an asynchronous job for detection of entities in a lending document, takes a provided image or PDF as input
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/textract/latest/dg/API_StartLendingAnalysis.html
     */
    toStartLendingAnalysis() {
        return this.to('StartLendingAnalysis');
    }
    /**
     * Grants permission to add one or more tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/textract/latest/dg/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove one or more tags from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/textract/latest/dg/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update Amazon Textract adapter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/textract/latest/dg/API_UpdateAdapter.html
     */
    toUpdateAdapter() {
        return this.to('UpdateAdapter');
    }
    /**
     * Adds a resource of type adapter to the statement
     *
     * https://docs.aws.amazon.com/textract/latest/dg/API_AdapterOverview.html
     *
     * @param adapterId - Identifier for the adapterId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAdapter(adapterId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:textract:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:/adapters/${adapterId}`);
    }
    /**
     * Adds a resource of type adapterversion to the statement
     *
     * https://docs.aws.amazon.com/textract/latest/dg/API_AdapterVersionOverview.html
     *
     * @param adapterId - Identifier for the adapterId.
     * @param adapterVersion - Identifier for the adapterVersion.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAdapterversion(adapterId, adapterVersion, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:textract:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:/adapters/${adapterId}/versions/${adapterVersion}`);
    }
    /**
     * Filters access by tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateAdapter()
     * - .toCreateAdapterVersion()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - adapter
     * - adapterversion
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateAdapter()
     * - .toCreateAdapterVersion()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [textract](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazontextract.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'textract';
        this.accessLevelList = {
            Read: [
                'AnalyzeDocument',
                'AnalyzeExpense',
                'AnalyzeID',
                'DetectDocumentText',
                'GetAdapter',
                'GetAdapterVersion',
                'GetDocumentAnalysis',
                'GetDocumentTextDetection',
                'GetExpenseAnalysis',
                'GetLendingAnalysis',
                'GetLendingAnalysisSummary',
                'ListAdapterVersions',
                'ListAdapters',
                'ListTagsForResource'
            ],
            Write: [
                'CreateAdapter',
                'CreateAdapterVersion',
                'DeleteAdapter',
                'DeleteAdapterVersion',
                'StartDocumentAnalysis',
                'StartDocumentTextDetection',
                'StartExpenseAnalysis',
                'StartLendingAnalysis',
                'UpdateAdapter'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Textract = Textract;
//# sourceMappingURL=data:application/json;base64,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