"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Notifications = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [notifications](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsusernotifications.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Notifications extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate a new Channel with a particular NotificationConfiguration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_AssociateChannel.html
     */
    toAssociateChannel() {
        return this.to('AssociateChannel');
    }
    /**
     * Grants permission to associate an Account contact to a particular Managed Notification Configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_AssociateManagedNotificationAccountContact.html
     */
    toAssociateManagedNotificationAccountContact() {
        return this.to('AssociateManagedNotificationAccountContact');
    }
    /**
     * Grants permission to associate a Channel to a particular Managed Notification Configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_AssociateManagedNotificationAdditionalChannel.html
     */
    toAssociateManagedNotificationAdditionalChannel() {
        return this.to('AssociateManagedNotificationAdditionalChannel');
    }
    /**
     * Grants permission to create a new EventRule, associating it with a NotificationConfiguration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_CreateEventRule.html
     */
    toCreateEventRule() {
        return this.to('CreateEventRule');
    }
    /**
     * Grants permission to create a NotificationConfiguration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_CreateNotificationConfiguration.html
     */
    toCreateNotificationConfiguration() {
        return this.to('CreateNotificationConfiguration');
    }
    /**
     * Grants permission to delete an EventRule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_DeleteEventRule.html
     */
    toDeleteEventRule() {
        return this.to('DeleteEventRule');
    }
    /**
     * Grants permission to delete a NotificationConfiguration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_DeleteNotificationConfiguration.html
     */
    toDeleteNotificationConfiguration() {
        return this.to('DeleteNotificationConfiguration');
    }
    /**
     * Grants permission to deregister a NotificationHub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_DeregisterNotificationHub.html
     */
    toDeregisterNotificationHub() {
        return this.to('DeregisterNotificationHub');
    }
    /**
     * Grants permission to disable Service Trust for AWS User Notifications
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - organizations:DisableAWSServiceAccess
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_DisableNotificationsAccessForOrganization.html
     */
    toDisableNotificationsAccessForOrganization() {
        return this.to('DisableNotificationsAccessForOrganization');
    }
    /**
     * Grants permission to remove a Channel from a NotificationConfiguration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_DisassociateChannel.html
     */
    toDisassociateChannel() {
        return this.to('DisassociateChannel');
    }
    /**
     * Grants permission to remove an Account contact from a Managed Notification Configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_DisassociateManagedNotificationAccountContact.html
     */
    toDisassociateManagedNotificationAccountContact() {
        return this.to('DisassociateManagedNotificationAccountContact');
    }
    /**
     * Grants permission to remove a Channel from a Managed Notification Configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_DisassociateManagedNotificationAdditionalChannel.html
     */
    toDisassociateManagedNotificationAdditionalChannel() {
        return this.to('DisassociateManagedNotificationAdditionalChannel');
    }
    /**
     * Grants permission to enable Service Trust for AWS User Notifications
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - organizations:EnableAWSServiceAccess
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_EnableNotificationsAccessForOrganization.html
     */
    toEnableNotificationsAccessForOrganization() {
        return this.to('EnableNotificationsAccessForOrganization');
    }
    /**
     * Grants permission to get an EventRule
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_GetEventRule.html
     */
    toGetEventRule() {
        return this.to('GetEventRule');
    }
    /**
     * Grants permission to read the opt-in status of an AWS User Notification Service feature
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/notifications/latest/userguide/managing-notification-features.html
     */
    toGetFeatureOptInStatus() {
        return this.to('GetFeatureOptInStatus');
    }
    /**
     * Grants permission to get a Managed Notification Child Event
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_GetManagedNotificationChildEvent.html
     */
    toGetManagedNotificationChildEvent() {
        return this.to('GetManagedNotificationChildEvent');
    }
    /**
     * Grants permission to get a Managed Notification Configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_GetManagedNotificationConfiguration.html
     */
    toGetManagedNotificationConfiguration() {
        return this.to('GetManagedNotificationConfiguration');
    }
    /**
     * Grants permission to get a Managed NotificationEvent
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_GetManagedNotificationEvent.html
     */
    toGetManagedNotificationEvent() {
        return this.to('GetManagedNotificationEvent');
    }
    /**
     * Grants permission to get a NotificationConfiguration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_GetNotificationConfiguration.html
     */
    toGetNotificationConfiguration() {
        return this.to('GetNotificationConfiguration');
    }
    /**
     * Grants permission to get a NotificationEvent
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_GetNotificationEvent.html
     */
    toGetNotificationEvent() {
        return this.to('GetNotificationEvent');
    }
    /**
     * Grants permission to read Service Trust for AWS User Notifications
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_GetNotificationsAccessForOrganization.html
     */
    toGetNotificationsAccessForOrganization() {
        return this.to('GetNotificationsAccessForOrganization');
    }
    /**
     * Grants permission to list Channels by NotificationConfiguration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_ListChannels.html
     */
    toListChannels() {
        return this.to('ListChannels');
    }
    /**
     * Grants permission to list EventRules
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_ListEventRules.html
     */
    toListEventRules() {
        return this.to('ListEventRules');
    }
    /**
     * Grants permission to list Account contacts and Channels associated with a Managed Notification Configuration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_ListManagedNotificationChannelAssociations.html
     */
    toListManagedNotificationChannelAssociations() {
        return this.to('ListManagedNotificationChannelAssociations');
    }
    /**
     * Grants permission to list Managed Notification Child Events
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_ListManagedNotificationChildEvents.html
     */
    toListManagedNotificationChildEvents() {
        return this.to('ListManagedNotificationChildEvents');
    }
    /**
     * Grants permission to list Managed Notification Configurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_ListManagedNotificationConfigurations.html
     */
    toListManagedNotificationConfigurations() {
        return this.to('ListManagedNotificationConfigurations');
    }
    /**
     * Grants permission to list Managed Notification Events
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_ListManagedNotificationEvents.html
     */
    toListManagedNotificationEvents() {
        return this.to('ListManagedNotificationEvents');
    }
    /**
     * Grants permission to list NotificationConfigurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_ListNotificationConfigurations.html
     */
    toListNotificationConfigurations() {
        return this.to('ListNotificationConfigurations');
    }
    /**
     * Grants permission to list NotificationEvents
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_ListNotificationEvents.html
     */
    toListNotificationEvents() {
        return this.to('ListNotificationEvents');
    }
    /**
     * Grants permission to list NotificationHubs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_ListNotificationHubs.html
     */
    toListNotificationHubs() {
        return this.to('ListNotificationHubs');
    }
    /**
     * Grants permission to get tags for a resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to update the opt-in status of an AWS User Notification Service feature
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/notifications/latest/userguide/managing-notification-features.html
     */
    toPutFeatureOptInStatus() {
        return this.to('PutFeatureOptInStatus');
    }
    /**
     * Grants permission to register a NotificationHub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_RegisterNotificationHub.html
     */
    toRegisterNotificationHub() {
        return this.to('RegisterNotificationHub');
    }
    /**
     * Grants permission to tag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an EventRule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_UpdateEventRule.html
     */
    toUpdateEventRule() {
        return this.to('UpdateEventRule');
    }
    /**
     * Grants permission to update a NotificationConfiguration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_UpdateNotificationConfiguration.html
     */
    toUpdateNotificationConfiguration() {
        return this.to('UpdateNotificationConfiguration');
    }
    /**
     * Adds a resource of type EventRule to the statement
     *
     * https://docs.aws.amazon.com/notifications/latest/userguide/resource-level-permissions.html
     *
     * @param notificationConfigurationId - Identifier for the notificationConfigurationId.
     * @param eventRuleId - Identifier for the eventRuleId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onEventRule(notificationConfigurationId, eventRuleId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:notifications::${account ?? this.defaultAccount}:configuration/${notificationConfigurationId}/rule/${eventRuleId}`);
    }
    /**
     * Adds a resource of type NotificationConfiguration to the statement
     *
     * https://docs.aws.amazon.com/notifications/latest/userguide/resource-level-permissions.html
     *
     * @param notificationConfigurationId - Identifier for the notificationConfigurationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onNotificationConfiguration(notificationConfigurationId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:notifications::${account ?? this.defaultAccount}:configuration/${notificationConfigurationId}`);
    }
    /**
     * Adds a resource of type NotificationEvent to the statement
     *
     * https://docs.aws.amazon.com/notifications/latest/userguide/resource-level-permissions.html
     *
     * @param notificationConfigurationId - Identifier for the notificationConfigurationId.
     * @param notificationEventId - Identifier for the notificationEventId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onNotificationEvent(notificationConfigurationId, notificationEventId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:notifications:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:configuration/${notificationConfigurationId}/event/${notificationEventId}`);
    }
    /**
     * Adds a resource of type ManagedNotificationChildEvent to the statement
     *
     * https://docs.aws.amazon.com/notifications/latest/userguide/resource-level-permissions.html
     *
     * @param category - Identifier for the category.
     * @param subcategory - Identifier for the subcategory.
     * @param notificationEventId - Identifier for the notificationEventId.
     * @param notificationChildEventId - Identifier for the notificationChildEventId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onManagedNotificationChildEvent(category, subcategory, notificationEventId, notificationChildEventId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:notifications::${account ?? this.defaultAccount}:managed-notification-configuration/category/${category}/sub-category/${subcategory}/event/${notificationEventId}/child-event/${notificationChildEventId}`);
    }
    /**
     * Adds a resource of type ManagedNotificationConfiguration to the statement
     *
     * https://docs.aws.amazon.com/notifications/latest/userguide/resource-level-permissions.html
     *
     * @param category - Identifier for the category.
     * @param subcategory - Identifier for the subcategory.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onManagedNotificationConfiguration(category, subcategory, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:notifications::${account ?? this.defaultAccount}:managed-notification-configuration/category/${category}/sub-category/${subcategory}`);
    }
    /**
     * Adds a resource of type ManagedNotificationEvent to the statement
     *
     * https://docs.aws.amazon.com/notifications/latest/userguide/resource-level-permissions.html
     *
     * @param category - Identifier for the category.
     * @param subcategory - Identifier for the subcategory.
     * @param notificationEventId - Identifier for the notificationEventId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onManagedNotificationEvent(category, subcategory, notificationEventId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:notifications::${account ?? this.defaultAccount}:managed-notification-configuration/category/${category}/sub-category/${subcategory}/event/${notificationEventId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateNotificationConfiguration()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - NotificationConfiguration
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateNotificationConfiguration()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [notifications](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsusernotifications.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'notifications';
        this.accessLevelList = {
            Write: [
                'AssociateChannel',
                'AssociateManagedNotificationAccountContact',
                'AssociateManagedNotificationAdditionalChannel',
                'CreateEventRule',
                'CreateNotificationConfiguration',
                'DeleteEventRule',
                'DeleteNotificationConfiguration',
                'DeregisterNotificationHub',
                'DisableNotificationsAccessForOrganization',
                'DisassociateChannel',
                'DisassociateManagedNotificationAccountContact',
                'DisassociateManagedNotificationAdditionalChannel',
                'EnableNotificationsAccessForOrganization',
                'PutFeatureOptInStatus',
                'RegisterNotificationHub',
                'UpdateEventRule',
                'UpdateNotificationConfiguration'
            ],
            Read: [
                'GetEventRule',
                'GetFeatureOptInStatus',
                'GetManagedNotificationChildEvent',
                'GetManagedNotificationConfiguration',
                'GetManagedNotificationEvent',
                'GetNotificationConfiguration',
                'GetNotificationEvent',
                'GetNotificationsAccessForOrganization'
            ],
            List: [
                'ListChannels',
                'ListEventRules',
                'ListManagedNotificationChannelAssociations',
                'ListManagedNotificationChildEvents',
                'ListManagedNotificationConfigurations',
                'ListManagedNotificationEvents',
                'ListNotificationConfigurations',
                'ListNotificationEvents',
                'ListNotificationHubs',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Notifications = Notifications;
//# sourceMappingURL=data:application/json;base64,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