"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VpcLattice = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [vpc-lattice](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonvpclattice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class VpcLattice extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate a resource configuration through Amazon EventBridge and AWS Step Functions service networks
     *
     * Access Level: Permissions management
     */
    toAssociateViaAWSServiceEventsAndStates() {
        return this.to('AssociateViaAWSServiceEventsAndStates');
    }
    /**
     * Grants permission to create an access log subscription
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - logs:CreateLogDelivery
     * - logs:GetLogDelivery
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_CreateAccessLogSubscription.html
     */
    toCreateAccessLogSubscription() {
        return this.to('CreateAccessLogSubscription');
    }
    /**
     * Grants permission to create a listener
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifProtocol()
     * - .ifTargetGroupArns()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_CreateListener.html
     */
    toCreateListener() {
        return this.to('CreateListener');
    }
    /**
     * Grants permission to create a resource configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_CreateResourceConfiguration.html
     */
    toCreateResourceConfiguration() {
        return this.to('CreateResourceConfiguration');
    }
    /**
     * Grants permission to create a resource gateway
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifVpcId()
     *
     * Dependent actions:
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_CreateResourceGateway.html
     */
    toCreateResourceGateway() {
        return this.to('CreateResourceGateway');
    }
    /**
     * Grants permission to create a rule
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifTargetGroupArns()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_CreateRule.html
     */
    toCreateRule() {
        return this.to('CreateRule');
    }
    /**
     * Grants permission to create a service
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAuthType()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_CreateService.html
     */
    toCreateService() {
        return this.to('CreateService');
    }
    /**
     * Grants permission to create a service network
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAuthType()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_CreateServiceNetwork.html
     */
    toCreateServiceNetwork() {
        return this.to('CreateServiceNetwork');
    }
    /**
     * Grants permission to create an association between a service network and a resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifResourceConfigurationArn()
     * - .ifServiceNetworkArn()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_CreateServiceNetworkResourceAssociation.html
     */
    toCreateServiceNetworkResourceAssociation() {
        return this.to('CreateServiceNetworkResourceAssociation');
    }
    /**
     * Grants permission to create a service network and service association
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifServiceArn()
     * - .ifServiceNetworkArn()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_CreateServiceNetworkServiceAssociation.html
     */
    toCreateServiceNetworkServiceAssociation() {
        return this.to('CreateServiceNetworkServiceAssociation');
    }
    /**
     * Grants permission to create a service network and VPC association
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifSecurityGroupIds()
     * - .ifServiceNetworkArn()
     * - .ifVpcId()
     *
     * Dependent actions:
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_CreateServiceNetworkVpcAssociation.html
     */
    toCreateServiceNetworkVpcAssociation() {
        return this.to('CreateServiceNetworkVpcAssociation');
    }
    /**
     * Grants permission to create an association between a service network and VPC endpoint
     *
     * Access Level: Permissions management
     */
    toCreateServiceNetworkVpcEndpointAssociation() {
        return this.to('CreateServiceNetworkVpcEndpointAssociation');
    }
    /**
     * Grants permission to create a target group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifVpcId()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_CreateTargetGroup.html
     */
    toCreateTargetGroup() {
        return this.to('CreateTargetGroup');
    }
    /**
     * Grants permission to delete an access log subscription
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - logs:DeleteLogDelivery
     * - logs:GetLogDelivery
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_DeleteAccessLogSubscription.html
     */
    toDeleteAccessLogSubscription() {
        return this.to('DeleteAccessLogSubscription');
    }
    /**
     * Grants permission to delete an auth policy
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_DeleteAuthPolicy.html
     */
    toDeleteAuthPolicy() {
        return this.to('DeleteAuthPolicy');
    }
    /**
     * Grants permission to delete a listener
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_DeleteListener.html
     */
    toDeleteListener() {
        return this.to('DeleteListener');
    }
    /**
     * Grants permission to delete a resource configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_DeleteResourceConfiguration.html
     */
    toDeleteResourceConfiguration() {
        return this.to('DeleteResourceConfiguration');
    }
    /**
     * Grants permission to delete a resource endpoint association
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_DeleteResourceEndpointAssociation.html
     */
    toDeleteResourceEndpointAssociation() {
        return this.to('DeleteResourceEndpointAssociation');
    }
    /**
     * Grants permission to delete a resource gateway
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_DeleteResourceGateway.html
     */
    toDeleteResourceGateway() {
        return this.to('DeleteResourceGateway');
    }
    /**
     * Grants permission to delete a resource policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to delete a rule
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_DeleteRule.html
     */
    toDeleteRule() {
        return this.to('DeleteRule');
    }
    /**
     * Grants permission to delete a service
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_DeleteService.html
     */
    toDeleteService() {
        return this.to('DeleteService');
    }
    /**
     * Grants permission to delete a service network
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_DeleteServiceNetwork.html
     */
    toDeleteServiceNetwork() {
        return this.to('DeleteServiceNetwork');
    }
    /**
     * Grants permission to delete the association between a service network and resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_DeleteServiceNetworkResourceAssociation.html
     */
    toDeleteServiceNetworkResourceAssociation() {
        return this.to('DeleteServiceNetworkResourceAssociation');
    }
    /**
     * Grants permission to delete a service network service association
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifServiceArn()
     * - .ifServiceNetworkArn()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_DeleteServiceNetworkServiceAssociation.html
     */
    toDeleteServiceNetworkServiceAssociation() {
        return this.to('DeleteServiceNetworkServiceAssociation');
    }
    /**
     * Grants permission to delete a service network and VPC association
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifServiceNetworkArn()
     * - .ifVpcId()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_DeleteServiceNetworkVpcAssociation.html
     */
    toDeleteServiceNetworkVpcAssociation() {
        return this.to('DeleteServiceNetworkVpcAssociation');
    }
    /**
     * Grants permission to delete a target group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_DeleteTargetGroup.html
     */
    toDeleteTargetGroup() {
        return this.to('DeleteTargetGroup');
    }
    /**
     * Grants permission to deregister targets from a target group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_DeregisterTargets.html
     */
    toDeregisterTargets() {
        return this.to('DeregisterTargets');
    }
    /**
     * Grants permission to get information about an access log subscription
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - logs:GetLogDelivery
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_GetAccessLogSubscription.html
     */
    toGetAccessLogSubscription() {
        return this.to('GetAccessLogSubscription');
    }
    /**
     * Grants permission to get information about an auth policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_GetAuthPolicy.html
     */
    toGetAuthPolicy() {
        return this.to('GetAuthPolicy');
    }
    /**
     * Grants permission to get information about a listener
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_GetListener.html
     */
    toGetListener() {
        return this.to('GetListener');
    }
    /**
     * Grants permission to get information about a resource configuration
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_GetResourceConfiguration.html
     */
    toGetResourceConfiguration() {
        return this.to('GetResourceConfiguration');
    }
    /**
     * Grants permission to get information about a resource gateway
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_GetResourceGateway.html
     */
    toGetResourceGateway() {
        return this.to('GetResourceGateway');
    }
    /**
     * Grants permission to get information about a resource policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_GetResourcePolicy.html
     */
    toGetResourcePolicy() {
        return this.to('GetResourcePolicy');
    }
    /**
     * Grants permission to get information about a rule
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_GetRule.html
     */
    toGetRule() {
        return this.to('GetRule');
    }
    /**
     * Grants permission to get information about a service
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_GetService.html
     */
    toGetService() {
        return this.to('GetService');
    }
    /**
     * Grants permission to get information about a service network
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_GetServiceNetwork.html
     */
    toGetServiceNetwork() {
        return this.to('GetServiceNetwork');
    }
    /**
     * Grants permission to get information about an association between a service network and resource configuration
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_GetServiceNetworkResourceAssociation.html
     */
    toGetServiceNetworkResourceAssociation() {
        return this.to('GetServiceNetworkResourceAssociation');
    }
    /**
     * Grants permission to get information about a service network and service association
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifServiceArn()
     * - .ifServiceNetworkArn()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_GetServiceNetworkServiceAssociation.html
     */
    toGetServiceNetworkServiceAssociation() {
        return this.to('GetServiceNetworkServiceAssociation');
    }
    /**
     * Grants permission to get information about a service network and VPC association
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifServiceNetworkArn()
     * - .ifVpcId()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_GetServiceNetworkVpcAssociation.html
     */
    toGetServiceNetworkVpcAssociation() {
        return this.to('GetServiceNetworkVpcAssociation');
    }
    /**
     * Grants permission to get information about a target group
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_GetTargetGroup.html
     */
    toGetTargetGroup() {
        return this.to('GetTargetGroup');
    }
    /**
     * Grants permission to list some or all access log subscriptions about a service network or a service
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_ListAccessLogSubscriptions.html
     */
    toListAccessLogSubscriptions() {
        return this.to('ListAccessLogSubscriptions');
    }
    /**
     * Grants permission to list some or all listeners
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_ListListeners.html
     */
    toListListeners() {
        return this.to('ListListeners');
    }
    /**
     * Grants permission to list some or all resource configurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_ListResourceConfigurations.html
     */
    toListResourceConfigurations() {
        return this.to('ListResourceConfigurations');
    }
    /**
     * Grants permission to list some or all associations between a resource configuration and VPC endpoint
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifResourceConfigurationArn()
     * - .ifVpcEndpointId()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_ListResourceEndpointAssociations.html
     */
    toListResourceEndpointAssociations() {
        return this.to('ListResourceEndpointAssociations');
    }
    /**
     * Grants permission to list some or all resource gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_ListResourceGateways.html
     */
    toListResourceGateways() {
        return this.to('ListResourceGateways');
    }
    /**
     * Grants permission to list some or all rules
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_ListRules.html
     */
    toListRules() {
        return this.to('ListRules');
    }
    /**
     * Grants permission to list some or all associations between a service network and resource configuration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_ListServiceNetworkResourceAssociations.html
     */
    toListServiceNetworkResourceAssociations() {
        return this.to('ListServiceNetworkResourceAssociations');
    }
    /**
     * Grants permission to list some or all service network and service associations
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifServiceArn()
     * - .ifServiceNetworkArn()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_ListServiceNetworkServiceAssociations.html
     */
    toListServiceNetworkServiceAssociations() {
        return this.to('ListServiceNetworkServiceAssociations');
    }
    /**
     * Grants permission to list some or all service network and VPC associations
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifServiceNetworkArn()
     * - .ifVpcId()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_ListServiceNetworkVpcAssociations.html
     */
    toListServiceNetworkVpcAssociations() {
        return this.to('ListServiceNetworkVpcAssociations');
    }
    /**
     * Grants permission to list some or all associations between a service network and VPC endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_ListServiceNetworkVpcEndpointAssociations.html
     */
    toListServiceNetworkVpcEndpointAssociations() {
        return this.to('ListServiceNetworkVpcEndpointAssociations');
    }
    /**
     * Grants permission to list the service networks owned by a caller account or shared with the caller account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_ListServiceNetworks.html
     */
    toListServiceNetworks() {
        return this.to('ListServiceNetworks');
    }
    /**
     * Grants permission to list the services owned by a caller account or shared with the caller account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_ListServices.html
     */
    toListServices() {
        return this.to('ListServices');
    }
    /**
     * Grants permission to list tags for a vpc-lattice resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list some or all target groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_ListTargetGroups.html
     */
    toListTargetGroups() {
        return this.to('ListTargetGroups');
    }
    /**
     * Grants permission to list some or all targets in a target group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_ListTargets.html
     */
    toListTargets() {
        return this.to('ListTargets');
    }
    /**
     * Grants permission to create or update the auth policy for a service network or a service
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_PutAuthPolicy.html
     */
    toPutAuthPolicy() {
        return this.to('PutAuthPolicy');
    }
    /**
     * Grants permission to create a resource policy for a resource configuration, service, or service network
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_PutResourcePolicy.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to register targets to a target group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_RegisterTargets.html
     */
    toRegisterTargets() {
        return this.to('RegisterTargets');
    }
    /**
     * Grants permission to tag a vpc-lattice resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a vpc-lattice resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an access log subscription
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - logs:GetLogDelivery
     * - logs:UpdateLogDelivery
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_UpdateAccessLogSubscription.html
     */
    toUpdateAccessLogSubscription() {
        return this.to('UpdateAccessLogSubscription');
    }
    /**
     * Grants permission to update a listener
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifTargetGroupArns()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_UpdateListener.html
     */
    toUpdateListener() {
        return this.to('UpdateListener');
    }
    /**
     * Grants permission to update a resource configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_UpdateResourceConfiguration.html
     */
    toUpdateResourceConfiguration() {
        return this.to('UpdateResourceConfiguration');
    }
    /**
     * Grants permission to update a resource gateway
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifSecurityGroupIds()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_UpdateResourceGateway.html
     */
    toUpdateResourceGateway() {
        return this.to('UpdateResourceGateway');
    }
    /**
     * Grants permission to update a rule
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifTargetGroupArns()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_UpdateRule.html
     */
    toUpdateRule() {
        return this.to('UpdateRule');
    }
    /**
     * Grants permission to update a service
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAuthType()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_UpdateService.html
     */
    toUpdateService() {
        return this.to('UpdateService');
    }
    /**
     * Grants permission to update a service network
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAuthType()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_UpdateServiceNetwork.html
     */
    toUpdateServiceNetwork() {
        return this.to('UpdateServiceNetwork');
    }
    /**
     * Grants permission to update a service network and VPC association
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifSecurityGroupIds()
     * - .ifServiceNetworkArn()
     * - .ifVpcId()
     *
     * Dependent actions:
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_UpdateServiceNetworkVpcAssociation.html
     */
    toUpdateServiceNetworkVpcAssociation() {
        return this.to('UpdateServiceNetworkVpcAssociation');
    }
    /**
     * Grants permission to update a target group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/APIReference/API_UpdateTargetGroup.html
     */
    toUpdateTargetGroup() {
        return this.to('UpdateTargetGroup');
    }
    /**
     * Adds a resource of type AccessLogSubscription to the statement
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/ug/monitoring-access-logs.html
     *
     * @param accessLogSubscriptionId - Identifier for the accessLogSubscriptionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     */
    onAccessLogSubscription(accessLogSubscriptionId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:vpc-lattice:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:accesslogsubscription/${accessLogSubscriptionId}`);
    }
    /**
     * Adds a resource of type Listener to the statement
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/ug/listeners.html
     *
     * @param serviceId - Identifier for the serviceId.
     * @param listenerId - Identifier for the listenerId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifProtocol()
     * - .ifTargetGroupArns()
     */
    onListener(serviceId, listenerId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:vpc-lattice:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:service/${serviceId}/listener/${listenerId}`);
    }
    /**
     * Adds a resource of type ResourceConfiguration to the statement
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/ug/resource-configurations.html
     *
     * @param resourceConfigurationId - Identifier for the resourceConfigurationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     */
    onResourceConfiguration(resourceConfigurationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:vpc-lattice:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:resourceconfiguration/${resourceConfigurationId}`);
    }
    /**
     * Adds a resource of type ResourceEndpointAssociation to the statement
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/ug/resource-endpoint-associations.html
     *
     * @param resourceEndpointAssociationId - Identifier for the resourceEndpointAssociationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifResourceConfigurationArn()
     * - .ifVpcEndpointId()
     */
    onResourceEndpointAssociation(resourceEndpointAssociationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:vpc-lattice:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:resourceendpointassociation/${resourceEndpointAssociationId}`);
    }
    /**
     * Adds a resource of type ResourceGateway to the statement
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/ug/resource-gateways.html
     *
     * @param resourceGatewayId - Identifier for the resourceGatewayId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifVpcId()
     */
    onResourceGateway(resourceGatewayId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:vpc-lattice:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:resourcegateway/${resourceGatewayId}`);
    }
    /**
     * Adds a resource of type Rule to the statement
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/ug/listeners.html#listener-rules
     *
     * @param serviceId - Identifier for the serviceId.
     * @param listenerId - Identifier for the listenerId.
     * @param ruleId - Identifier for the ruleId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifTargetGroupArns()
     */
    onRule(serviceId, listenerId, ruleId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:vpc-lattice:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:service/${serviceId}/listener/${listenerId}/rule/${ruleId}`);
    }
    /**
     * Adds a resource of type Service to the statement
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/ug/services.html
     *
     * @param serviceId - Identifier for the serviceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifAuthType()
     */
    onService(serviceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:vpc-lattice:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:service/${serviceId}`);
    }
    /**
     * Adds a resource of type ServiceNetwork to the statement
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/ug/service-networks.html
     *
     * @param serviceNetworkId - Identifier for the serviceNetworkId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifAuthType()
     */
    onServiceNetwork(serviceNetworkId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:vpc-lattice:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:servicenetwork/${serviceNetworkId}`);
    }
    /**
     * Adds a resource of type ServiceNetworkResourceAssociation to the statement
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/ug/service-network-associations.html#service-network-resource-configuration
     *
     * @param serviceNetworkResourceAssociationId - Identifier for the serviceNetworkResourceAssociationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifResourceConfigurationArn()
     * - .ifServiceNetworkArn()
     */
    onServiceNetworkResourceAssociation(serviceNetworkResourceAssociationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:vpc-lattice:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:servicenetworkresourceassociation/${serviceNetworkResourceAssociationId}`);
    }
    /**
     * Adds a resource of type ServiceNetworkServiceAssociation to the statement
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/ug/service-network-associations.html#service-network-service-associations
     *
     * @param serviceNetworkServiceAssociationId - Identifier for the serviceNetworkServiceAssociationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifServiceArn()
     * - .ifServiceNetworkArn()
     */
    onServiceNetworkServiceAssociation(serviceNetworkServiceAssociationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:vpc-lattice:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:servicenetworkserviceassociation/${serviceNetworkServiceAssociationId}`);
    }
    /**
     * Adds a resource of type ServiceNetworkVpcAssociation to the statement
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/ug/service-network-associations.html#service-network-vpc-associations
     *
     * @param serviceNetworkVpcAssociationId - Identifier for the serviceNetworkVpcAssociationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifSecurityGroupIds()
     * - .ifServiceNetworkArn()
     * - .ifVpcId()
     */
    onServiceNetworkVpcAssociation(serviceNetworkVpcAssociationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:vpc-lattice:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:servicenetworkvpcassociation/${serviceNetworkVpcAssociationId}`);
    }
    /**
     * Adds a resource of type TargetGroup to the statement
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/ug/target-groups.html
     *
     * @param targetGroupId - Identifier for the targetGroupId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifVpcId()
     */
    onTargetGroup(targetGroupId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:vpc-lattice:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:targetgroup/${targetGroupId}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateAccessLogSubscription()
     * - .toCreateListener()
     * - .toCreateResourceConfiguration()
     * - .toCreateResourceGateway()
     * - .toCreateRule()
     * - .toCreateService()
     * - .toCreateServiceNetwork()
     * - .toCreateServiceNetworkResourceAssociation()
     * - .toCreateServiceNetworkServiceAssociation()
     * - .toCreateServiceNetworkVpcAssociation()
     * - .toCreateTargetGroup()
     * - .toTagResource()
     *
     * Applies to resource types:
     * - AccessLogSubscription
     * - Listener
     * - ResourceConfiguration
     * - ResourceEndpointAssociation
     * - ResourceGateway
     * - Rule
     * - Service
     * - ServiceNetwork
     * - ServiceNetworkResourceAssociation
     * - ServiceNetworkServiceAssociation
     * - ServiceNetworkVpcAssociation
     * - TargetGroup
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toDeleteAccessLogSubscription()
     * - .toDeleteListener()
     * - .toDeleteResourceConfiguration()
     * - .toDeleteResourceEndpointAssociation()
     * - .toDeleteResourceGateway()
     * - .toDeleteRule()
     * - .toDeleteService()
     * - .toDeleteServiceNetwork()
     * - .toDeleteServiceNetworkResourceAssociation()
     * - .toDeleteServiceNetworkServiceAssociation()
     * - .toDeleteServiceNetworkVpcAssociation()
     * - .toDeleteTargetGroup()
     * - .toGetAccessLogSubscription()
     * - .toGetListener()
     * - .toGetResourceConfiguration()
     * - .toGetResourceGateway()
     * - .toGetRule()
     * - .toGetService()
     * - .toGetServiceNetwork()
     * - .toGetServiceNetworkResourceAssociation()
     * - .toGetServiceNetworkServiceAssociation()
     * - .toGetServiceNetworkVpcAssociation()
     * - .toGetTargetGroup()
     * - .toTagResource()
     * - .toUpdateAccessLogSubscription()
     * - .toUpdateListener()
     * - .toUpdateResourceConfiguration()
     * - .toUpdateResourceGateway()
     * - .toUpdateRule()
     * - .toUpdateService()
     * - .toUpdateServiceNetwork()
     * - .toUpdateServiceNetworkVpcAssociation()
     * - .toUpdateTargetGroup()
     *
     * Applies to resource types:
     * - AccessLogSubscription
     * - Listener
     * - ResourceConfiguration
     * - ResourceEndpointAssociation
     * - ResourceGateway
     * - Rule
     * - Service
     * - ServiceNetwork
     * - ServiceNetworkResourceAssociation
     * - ServiceNetworkServiceAssociation
     * - ServiceNetworkVpcAssociation
     * - TargetGroup
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateAccessLogSubscription()
     * - .toCreateListener()
     * - .toCreateResourceConfiguration()
     * - .toCreateResourceGateway()
     * - .toCreateRule()
     * - .toCreateService()
     * - .toCreateServiceNetwork()
     * - .toCreateServiceNetworkResourceAssociation()
     * - .toCreateServiceNetworkServiceAssociation()
     * - .toCreateServiceNetworkVpcAssociation()
     * - .toCreateTargetGroup()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateServiceNetworkVpcAssociation()
     *
     * Applies to resource types:
     * - AccessLogSubscription
     * - Listener
     * - ResourceConfiguration
     * - ResourceEndpointAssociation
     * - ResourceGateway
     * - Rule
     * - Service
     * - ServiceNetwork
     * - ServiceNetworkResourceAssociation
     * - ServiceNetworkServiceAssociation
     * - ServiceNetworkVpcAssociation
     * - TargetGroup
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the auth type specified in the request
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/ug/
     *
     * Applies to actions:
     * - .toCreateService()
     * - .toCreateServiceNetwork()
     * - .toUpdateService()
     * - .toUpdateServiceNetwork()
     *
     * Applies to resource types:
     * - Service
     * - ServiceNetwork
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAuthType(value, operator) {
        return this.if(`AuthType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the protocol specified in the request
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/ug/
     *
     * Applies to actions:
     * - .toCreateListener()
     *
     * Applies to resource types:
     * - Listener
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifProtocol(value, operator) {
        return this.if(`Protocol`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ARN of a resource configuration
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/ug/
     *
     * Applies to actions:
     * - .toCreateServiceNetworkResourceAssociation()
     * - .toListResourceEndpointAssociations()
     *
     * Applies to resource types:
     * - ResourceEndpointAssociation
     * - ServiceNetworkResourceAssociation
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifResourceConfigurationArn(value, operator) {
        return this.if(`ResourceConfigurationArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the IDs of security groups
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/ug/
     *
     * Applies to actions:
     * - .toCreateServiceNetworkVpcAssociation()
     * - .toUpdateResourceGateway()
     * - .toUpdateServiceNetworkVpcAssociation()
     *
     * Applies to resource types:
     * - ServiceNetworkVpcAssociation
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecurityGroupIds(value, operator) {
        return this.if(`SecurityGroupIds`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ARN of a service
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/ug/
     *
     * Applies to actions:
     * - .toCreateServiceNetworkServiceAssociation()
     * - .toDeleteServiceNetworkServiceAssociation()
     * - .toGetServiceNetworkServiceAssociation()
     * - .toListServiceNetworkServiceAssociations()
     *
     * Applies to resource types:
     * - ServiceNetworkServiceAssociation
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifServiceArn(value, operator) {
        return this.if(`ServiceArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the ARN of a service network
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/ug/
     *
     * Applies to actions:
     * - .toCreateServiceNetworkResourceAssociation()
     * - .toCreateServiceNetworkServiceAssociation()
     * - .toCreateServiceNetworkVpcAssociation()
     * - .toDeleteServiceNetworkServiceAssociation()
     * - .toDeleteServiceNetworkVpcAssociation()
     * - .toGetServiceNetworkServiceAssociation()
     * - .toGetServiceNetworkVpcAssociation()
     * - .toListServiceNetworkServiceAssociations()
     * - .toListServiceNetworkVpcAssociations()
     * - .toUpdateServiceNetworkVpcAssociation()
     *
     * Applies to resource types:
     * - ServiceNetworkResourceAssociation
     * - ServiceNetworkServiceAssociation
     * - ServiceNetworkVpcAssociation
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifServiceNetworkArn(value, operator) {
        return this.if(`ServiceNetworkArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the ARNs of target groups
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/ug/
     *
     * Applies to actions:
     * - .toCreateListener()
     * - .toCreateRule()
     * - .toUpdateListener()
     * - .toUpdateRule()
     *
     * Applies to resource types:
     * - Listener
     * - Rule
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifTargetGroupArns(value, operator) {
        return this.if(`TargetGroupArns`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the ID of a VPC endpoint
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/ug/
     *
     * Applies to actions:
     * - .toListResourceEndpointAssociations()
     *
     * Applies to resource types:
     * - ResourceEndpointAssociation
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpcEndpointId(value, operator) {
        return this.if(`VpcEndpointId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ID of a virtual private cloud (VPC)
     *
     * https://docs.aws.amazon.com/vpc-lattice/latest/ug/
     *
     * Applies to actions:
     * - .toCreateResourceGateway()
     * - .toCreateServiceNetworkVpcAssociation()
     * - .toCreateTargetGroup()
     * - .toDeleteServiceNetworkVpcAssociation()
     * - .toGetServiceNetworkVpcAssociation()
     * - .toListServiceNetworkVpcAssociations()
     * - .toUpdateServiceNetworkVpcAssociation()
     *
     * Applies to resource types:
     * - ResourceGateway
     * - ServiceNetworkVpcAssociation
     * - TargetGroup
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpcId(value, operator) {
        return this.if(`VpcId`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [vpc-lattice](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonvpclattice.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'vpc-lattice';
        this.accessLevelList = {
            'Permissions management': [
                'AssociateViaAWSServiceEventsAndStates',
                'CreateServiceNetworkVpcEndpointAssociation',
                'DeleteAuthPolicy',
                'PutAuthPolicy'
            ],
            Write: [
                'CreateAccessLogSubscription',
                'CreateListener',
                'CreateResourceConfiguration',
                'CreateResourceGateway',
                'CreateRule',
                'CreateService',
                'CreateServiceNetwork',
                'CreateServiceNetworkResourceAssociation',
                'CreateServiceNetworkServiceAssociation',
                'CreateServiceNetworkVpcAssociation',
                'CreateTargetGroup',
                'DeleteAccessLogSubscription',
                'DeleteListener',
                'DeleteResourceConfiguration',
                'DeleteResourceEndpointAssociation',
                'DeleteResourceGateway',
                'DeleteResourcePolicy',
                'DeleteRule',
                'DeleteService',
                'DeleteServiceNetwork',
                'DeleteServiceNetworkResourceAssociation',
                'DeleteServiceNetworkServiceAssociation',
                'DeleteServiceNetworkVpcAssociation',
                'DeleteTargetGroup',
                'DeregisterTargets',
                'PutResourcePolicy',
                'RegisterTargets',
                'UpdateAccessLogSubscription',
                'UpdateListener',
                'UpdateResourceConfiguration',
                'UpdateResourceGateway',
                'UpdateRule',
                'UpdateService',
                'UpdateServiceNetwork',
                'UpdateServiceNetworkVpcAssociation',
                'UpdateTargetGroup'
            ],
            Read: [
                'GetAccessLogSubscription',
                'GetAuthPolicy',
                'GetListener',
                'GetResourceConfiguration',
                'GetResourceGateway',
                'GetResourcePolicy',
                'GetRule',
                'GetService',
                'GetServiceNetwork',
                'GetServiceNetworkResourceAssociation',
                'GetServiceNetworkServiceAssociation',
                'GetServiceNetworkVpcAssociation',
                'GetTargetGroup',
                'ListTagsForResource'
            ],
            List: [
                'ListAccessLogSubscriptions',
                'ListListeners',
                'ListResourceConfigurations',
                'ListResourceEndpointAssociations',
                'ListResourceGateways',
                'ListRules',
                'ListServiceNetworkResourceAssociations',
                'ListServiceNetworkServiceAssociations',
                'ListServiceNetworkVpcAssociations',
                'ListServiceNetworkVpcEndpointAssociations',
                'ListServiceNetworks',
                'ListServices',
                'ListTargetGroups',
                'ListTargets'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.VpcLattice = VpcLattice;
//# sourceMappingURL=data:application/json;base64,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