"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Workmail = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [workmail](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonworkmail.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Workmail extends shared_1.PolicyStatement {
    /**
     * Grants permission to configure vended log delivery for WorkMail audit logs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/audit-logging.html
     */
    toAllowVendedLogDeliveryForResource() {
        return this.to('AllowVendedLogDeliveryForResource');
    }
    /**
     * Grants permission to add a member (user or group) to the resource's set of delegates
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_AssociateDelegateToResource.html
     */
    toAssociateDelegateToResource() {
        return this.to('AssociateDelegateToResource');
    }
    /**
     * Grants permission to add a member (user or group) to the group's set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_AssociateMemberToGroup.html
     */
    toAssociateMemberToGroup() {
        return this.to('AssociateMemberToGroup');
    }
    /**
     * Grants permission to assume an impersonation role for the given Amazon WorkMail organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_AssumeImpersonationRole.html
     */
    toAssumeImpersonationRole() {
        return this.to('AssumeImpersonationRole');
    }
    /**
     * Grants permission to cancel a currently running mailbox export job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_CancelMailboxExportJob.html
     */
    toCancelMailboxExportJob() {
        return this.to('CancelMailboxExportJob');
    }
    /**
     * Grants permission to add an alias to the set of a given member (user or group) of WorkMail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_CreateAlias.html
     */
    toCreateAlias() {
        return this.to('CreateAlias');
    }
    /**
     * Grants permission to create an AvailabilityConfiguration for the given Amazon WorkMail organization and domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_CreateAvailabilityConfiguration.html
     */
    toCreateAvailabilityConfiguration() {
        return this.to('CreateAvailabilityConfiguration');
    }
    /**
     * Grants permission to create a group that can be used in WorkMail by calling the RegisterToWorkMail operation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_CreateGroup.html
     */
    toCreateGroup() {
        return this.to('CreateGroup');
    }
    /**
     * Grants permission to create an Identity Center application for WorkMail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_CreateIdentityCenterApplication.html
     */
    toCreateIdentityCenterApplication() {
        return this.to('CreateIdentityCenterApplication');
    }
    /**
     * Grants permission to create an impersonation role for the given Amazon WorkMail organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_CreateImpersonationRole.html
     */
    toCreateImpersonationRole() {
        return this.to('CreateImpersonationRole');
    }
    /**
     * Grants permission to create an inbound email flow rule which will apply to all email sent to an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/create-email-rules.html
     */
    toCreateInboundMailFlowRule() {
        return this.to('CreateInboundMailFlowRule');
    }
    /**
     * Grants permission to create a mail domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/add_domain.html
     */
    toCreateMailDomain() {
        return this.to('CreateMailDomain');
    }
    /**
     * Grants permission to create a new mobile device access rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_CreateMobileDeviceAccessRule.html
     */
    toCreateMobileDeviceAccessRule() {
        return this.to('CreateMobileDeviceAccessRule');
    }
    /**
     * Grants permission to create a new Amazon WorkMail organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_CreateOrganization.html
     */
    toCreateOrganization() {
        return this.to('CreateOrganization');
    }
    /**
     * Grants permission to create an outbound email flow rule which will apply to all email sent from an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/create-email-rules.html
     */
    toCreateOutboundMailFlowRule() {
        return this.to('CreateOutboundMailFlowRule');
    }
    /**
     * Grants permission to create a new WorkMail resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_CreateResource.html
     */
    toCreateResource() {
        return this.to('CreateResource');
    }
    /**
     * Grants permission to register an SMTP gateway to a WorkMail organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/smtp-gateway.html
     */
    toCreateSmtpGateway() {
        return this.to('CreateSmtpGateway');
    }
    /**
     * Grants permission to create a user, which can be enabled afterwards by calling the RegisterToWorkMail operation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_CreateUser.html
     */
    toCreateUser() {
        return this.to('CreateUser');
    }
    /**
     * Grants permission to delete an access control rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteAccessControlRule.html
     */
    toDeleteAccessControlRule() {
        return this.to('DeleteAccessControlRule');
    }
    /**
     * Grants permission to remove one or more specified aliases from a set of aliases for a given user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteAlias.html
     */
    toDeleteAlias() {
        return this.to('DeleteAlias');
    }
    /**
     * Grants permission to delete the AvailabilityConfiguration for the given Amazon WorkMail organization and domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteAvailabilityConfiguration.html
     */
    toDeleteAvailabilityConfiguration() {
        return this.to('DeleteAvailabilityConfiguration');
    }
    /**
     * Grants permission to delete the email monitoring configuration for an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteEmailMonitoringConfiguration.html
     */
    toDeleteEmailMonitoringConfiguration() {
        return this.to('DeleteEmailMonitoringConfiguration');
    }
    /**
     * Grants permission to delete a group from WorkMail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteGroup.html
     */
    toDeleteGroup() {
        return this.to('DeleteGroup');
    }
    /**
     * Grants permission to delete an Identity Center application for WorkMail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteIdentityCenterApplication.html
     */
    toDeleteIdentityCenterApplication() {
        return this.to('DeleteIdentityCenterApplication');
    }
    /**
     * Grants permission to delete the identity provider configuration for the organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteIdentityProviderConfiguration.html
     */
    toDeleteIdentityProviderConfiguration() {
        return this.to('DeleteIdentityProviderConfiguration');
    }
    /**
     * Grants permission to delete an impersonation role for the given Amazon WorkMail organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteImpersonationRole.html
     */
    toDeleteImpersonationRole() {
        return this.to('DeleteImpersonationRole');
    }
    /**
     * Grants permission to remove an inbound email flow rule to no longer apply to emails sent to an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/remove-email-flow-rule.html
     */
    toDeleteInboundMailFlowRule() {
        return this.to('DeleteInboundMailFlowRule');
    }
    /**
     * Grants permission to remove an unused mail domain from an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/remove_domain.html
     */
    toDeleteMailDomain() {
        return this.to('DeleteMailDomain');
    }
    /**
     * Grants permission to delete permissions granted to a member (user or group)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteMailboxPermissions.html
     */
    toDeleteMailboxPermissions() {
        return this.to('DeleteMailboxPermissions');
    }
    /**
     * Grants permission to remove a mobile device from a user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/manage-devices.html#remove_mobile_device
     */
    toDeleteMobileDevice() {
        return this.to('DeleteMobileDevice');
    }
    /**
     * Grants permission to delete a mobile device access override
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteMobileDeviceAccessOverride.html
     */
    toDeleteMobileDeviceAccessOverride() {
        return this.to('DeleteMobileDeviceAccessOverride');
    }
    /**
     * Grants permission to delete a mobile device access rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteMobileDeviceAccessRule.html
     */
    toDeleteMobileDeviceAccessRule() {
        return this.to('DeleteMobileDeviceAccessRule');
    }
    /**
     * Grants permission to delete an Amazon WorkMail organization and all underlying AWS resources managed by Amazon WorkMail as part of the organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteOrganization.html
     */
    toDeleteOrganization() {
        return this.to('DeleteOrganization');
    }
    /**
     * Grants permission to remove an outbound email flow rule so that it no longer applies to emails sent from an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/remove-email-flow-rule.html
     */
    toDeleteOutboundMailFlowRule() {
        return this.to('DeleteOutboundMailFlowRule');
    }
    /**
     * Grants permission to delete a personal access token
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeletePersonalAccessToken.html
     */
    toDeletePersonalAccessToken() {
        return this.to('DeletePersonalAccessToken');
    }
    /**
     * Grants permission to delete the specified resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteResource.html
     */
    toDeleteResource() {
        return this.to('DeleteResource');
    }
    /**
     * Grants permission to delete the retention policy based on the supplied organization and policy identifiers
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteRetentionPolicy.html
     */
    toDeleteRetentionPolicy() {
        return this.to('DeleteRetentionPolicy');
    }
    /**
     * Grants permission to remove an SMTP gateway from an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/smtp-gateway.html
     */
    toDeleteSmtpGateway() {
        return this.to('DeleteSmtpGateway');
    }
    /**
     * Grants permission to delete a user from WorkMail and all subsequent systems
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteUser.html
     */
    toDeleteUser() {
        return this.to('DeleteUser');
    }
    /**
     * Grants permission to deliver emails to a WorkMail organization via the SES MailManager DeliverToMailbox action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_DeliverToMailboxAction.html
     */
    toDeliverToMailbox() {
        return this.to('DeliverToMailbox');
    }
    /**
     * Grants permission to mark a user, group, or resource as no longer used in WorkMail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeregisterFromWorkMail.html
     */
    toDeregisterFromWorkMail() {
        return this.to('DeregisterFromWorkMail');
    }
    /**
     * Grants permission to deregister a mail domain from an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeregisterMailDomain.html
     */
    toDeregisterMailDomain() {
        return this.to('DeregisterMailDomain');
    }
    /**
     * Grants permission to retrieve the email monitoring configuration for an organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DescribeEmailMonitoringConfiguration.html
     */
    toDescribeEmailMonitoringConfiguration() {
        return this.to('DescribeEmailMonitoringConfiguration');
    }
    /**
     * Grants permission to read details of an entity
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DescribeEntity.html
     */
    toDescribeEntity() {
        return this.to('DescribeEntity');
    }
    /**
     * Grants permission to read the details for a group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DescribeGroup.html
     */
    toDescribeGroup() {
        return this.to('DescribeGroup');
    }
    /**
     * Grants permission to read the identity provider configuration for the organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DescribeIdentityProviderConfiguration.html
     */
    toDescribeIdentityProviderConfiguration() {
        return this.to('DescribeIdentityProviderConfiguration');
    }
    /**
     * Grants permission to read the settings in a DMARC policy for a specified organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DescribeInboundDmarcSettings.html
     */
    toDescribeInboundDmarcSettings() {
        return this.to('DescribeInboundDmarcSettings');
    }
    /**
     * Grants permission to read the details of an inbound mail flow rule configured for an organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/email-flows.html#email-flows-rule-actions
     */
    toDescribeInboundMailFlowRule() {
        return this.to('DescribeInboundMailFlowRule');
    }
    /**
     * Grants permission to show the details of all mail domains associated with the organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/domains_overview.html
     */
    toDescribeMailDomains() {
        return this.to('DescribeMailDomains');
    }
    /**
     * Grants permission to retrieve details of a mailbox export job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DescribeMailboxExportJob.html
     */
    toDescribeMailboxExportJob() {
        return this.to('DescribeMailboxExportJob');
    }
    /**
     * Grants permission to read details of an organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DescribeOrganization.html
     */
    toDescribeOrganization() {
        return this.to('DescribeOrganization');
    }
    /**
     * Grants permission to read the details of an outbound mail flow rule configured for an organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/email-flows.html#email-flows-rule-outbound
     */
    toDescribeOutboundMailFlowRule() {
        return this.to('DescribeOutboundMailFlowRule');
    }
    /**
     * Grants permission to read the details for a resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DescribeResource.html
     */
    toDescribeResource() {
        return this.to('DescribeResource');
    }
    /**
     * Grants permission to read the details of an SMTP gateway registered to an organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/smtp-gateway.html
     */
    toDescribeSmtpGateway() {
        return this.to('DescribeSmtpGateway');
    }
    /**
     * Grants permission to read details for a user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DescribeUser.html
     */
    toDescribeUser() {
        return this.to('DescribeUser');
    }
    /**
     * Grants permission to remove a member from the resource's set of delegates
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DisassociateDelegateFromResource.html
     */
    toDisassociateDelegateFromResource() {
        return this.to('DisassociateDelegateFromResource');
    }
    /**
     * Grants permission to remove a member from a group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DisassociateMemberFromGroup.html
     */
    toDisassociateMemberFromGroup() {
        return this.to('DisassociateMemberFromGroup');
    }
    /**
     * Grants permission to enable a mail domain in the organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/add_domain.html
     */
    toEnableMailDomain() {
        return this.to('EnableMailDomain');
    }
    /**
     * Grants permission to get the effects of access control rules as they apply to a specified IPv4 address, access protocol action, or user ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_GetAccessControlEffect.html
     */
    toGetAccessControlEffect() {
        return this.to('GetAccessControlEffect');
    }
    /**
     * Grants permission to retrieve the retention policy associated at an organizational level
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_GetDefaultRetentionPolicy.html
     */
    toGetDefaultRetentionPolicy() {
        return this.to('GetDefaultRetentionPolicy');
    }
    /**
     * Grants permission to retrieve an impersonation role for the given Amazon WorkMail organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_GetImpersonationRole.html
     */
    toGetImpersonationRole() {
        return this.to('GetImpersonationRole');
    }
    /**
     * Grants permission to get the effect of the rules associated to an impersonation role for a specific user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_GetImpersonationRoleEffect.html
     */
    toGetImpersonationRoleEffect() {
        return this.to('GetImpersonationRoleEffect');
    }
    /**
     * Grants permission to read the configured journaling and fallback email addresses for email journaling
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/journaling_overview.html
     */
    toGetJournalingRules() {
        return this.to('GetJournalingRules');
    }
    /**
     * Grants permission to retrieve details of a given mail domain in an organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_GetMailDomain.html
     */
    toGetMailDomain() {
        return this.to('GetMailDomain');
    }
    /**
     * Grants permission to get the details of the mail domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/domains_overview.html
     */
    toGetMailDomainDetails() {
        return this.to('GetMailDomainDetails');
    }
    /**
     * Grants permission to read the details of the user's mailbox
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_GetMailboxDetails.html
     */
    toGetMailboxDetails() {
        return this.to('GetMailboxDetails');
    }
    /**
     * Grants permission to simulate the effect of the mobile device access rules for the given attributes of a sample access event
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_GetMobileDeviceAccessEffect.html
     */
    toGetMobileDeviceAccessEffect() {
        return this.to('GetMobileDeviceAccessEffect');
    }
    /**
     * Grants permission to retrieve a mobile device access override
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_GetMobileDeviceAccessOverride.html
     */
    toGetMobileDeviceAccessOverride() {
        return this.to('GetMobileDeviceAccessOverride');
    }
    /**
     * Grants permission to get the details of the mobile device
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/manage-devices.html
     */
    toGetMobileDeviceDetails() {
        return this.to('GetMobileDeviceDetails');
    }
    /**
     * Grants permission to get a list of the mobile devices associated with the user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/manage-devices.html
     */
    toGetMobileDevicesForUser() {
        return this.to('GetMobileDevicesForUser');
    }
    /**
     * Grants permission to get the details of the mobile device policy associated with the organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/edit_organization_mobile_policy.html
     */
    toGetMobilePolicyDetails() {
        return this.to('GetMobilePolicyDetails');
    }
    /**
     * Grants permission to read metadata for a personal access token
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_GetPersonalAccessTokenMetadata.html
     */
    toGetPersonalAccessTokenMetadata() {
        return this.to('GetPersonalAccessTokenMetadata');
    }
    /**
     * Grants permission to list the access control rules
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListAccessControlRules.html
     */
    toListAccessControlRules() {
        return this.to('ListAccessControlRules');
    }
    /**
     * Grants permission to list the aliases associated with a given entity
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListAliases.html
     */
    toListAliases() {
        return this.to('ListAliases');
    }
    /**
     * Grants permission to list all the AvailabilityConfiguration's for the given Amazon WorkMail organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListAvailabilityConfigurations.html
     */
    toListAvailabilityConfigurations() {
        return this.to('ListAvailabilityConfigurations');
    }
    /**
     * Grants permission to read an overview of the members of a group. Users and groups can be members of a group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListGroupMembers.html
     */
    toListGroupMembers() {
        return this.to('ListGroupMembers');
    }
    /**
     * Grants permission to list summaries of the organization's groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListGroups.html
     */
    toListGroups() {
        return this.to('ListGroups');
    }
    /**
     * Grants permission to list the groups to which an entity belongs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListGroupsForEntity.html
     */
    toListGroupsForEntity() {
        return this.to('ListGroupsForEntity');
    }
    /**
     * Grants permission to list the impersonation roles for the given Amazon WorkMail organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListImpersonationRoles.html
     */
    toListImpersonationRoles() {
        return this.to('ListImpersonationRoles');
    }
    /**
     * Grants permission to list inbound mail flow rules configured for an organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/email-flows.html#email-flows-rule-actions
     */
    toListInboundMailFlowRules() {
        return this.to('ListInboundMailFlowRules');
    }
    /**
     * Grants permission to list the mail domains for a given organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListMailDomains.html
     */
    toListMailDomains() {
        return this.to('ListMailDomains');
    }
    /**
     * Grants permission to list mailbox export jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListMailboxExportJobs.html
     */
    toListMailboxExportJobs() {
        return this.to('ListMailboxExportJobs');
    }
    /**
     * Grants permission to list the mailbox permissions associated with a user, group, or resource mailbox
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListMailboxPermissions.html
     */
    toListMailboxPermissions() {
        return this.to('ListMailboxPermissions');
    }
    /**
     * Grants permission to list the mobile device access overrides
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListMobileDeviceAccessOverrides.html
     */
    toListMobileDeviceAccessOverrides() {
        return this.to('ListMobileDeviceAccessOverrides');
    }
    /**
     * Grants permission to list the mobile device access rules
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListMobileDeviceAccessRules.html
     */
    toListMobileDeviceAccessRules() {
        return this.to('ListMobileDeviceAccessRules');
    }
    /**
     * Grants permission to list the non-deleted organizations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListOrganizations.html
     */
    toListOrganizations() {
        return this.to('ListOrganizations');
    }
    /**
     * Grants permission to list outbound mail flow rules configured for an organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/email-flows.html#email-flows-rule-outbound
     */
    toListOutboundMailFlowRules() {
        return this.to('ListOutboundMailFlowRules');
    }
    /**
     * Grants permission to list metadata for personal access tokens
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListPersonalAccessTokens.html
     */
    toListPersonalAccessTokens() {
        return this.to('ListPersonalAccessTokens');
    }
    /**
     * Grants permission to list the delegates associated with a resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListResourceDelegates.html
     */
    toListResourceDelegates() {
        return this.to('ListResourceDelegates');
    }
    /**
     * Grants permission to list the organization's resources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListResources.html
     */
    toListResources() {
        return this.to('ListResources');
    }
    /**
     * Grants permission to list SMTP gateways registered to the organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/smtp-gateway.html
     */
    toListSmtpGateways() {
        return this.to('ListSmtpGateways');
    }
    /**
     * Grants permission to list the tags applied to an Amazon WorkMail organization resource
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list the organization's users
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListUsers.html
     */
    toListUsers() {
        return this.to('ListUsers');
    }
    /**
     * Grants permission to add a new access control rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_PutAccessControlRule.html
     */
    toPutAccessControlRule() {
        return this.to('PutAccessControlRule');
    }
    /**
     * Grants permission to add or update the email monitoring configuration for an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_PutEmailMonitoringConfiguration.html
     */
    toPutEmailMonitoringConfiguration() {
        return this.to('PutEmailMonitoringConfiguration');
    }
    /**
     * Grants permission to add or update the identity provider configuration for the organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_PutIdentityProviderConfiguration.html
     */
    toPutIdentityProviderConfiguration() {
        return this.to('PutIdentityProviderConfiguration');
    }
    /**
     * Grants permission to enable or disable a DMARC policy for a given organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_PutInboundDmarcSettings.html
     */
    toPutInboundDmarcSettings() {
        return this.to('PutInboundDmarcSettings');
    }
    /**
     * Grants permission to set permissions for a user, group, or resource, replacing any existing permissions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_PutMailboxPermissions.html
     */
    toPutMailboxPermissions() {
        return this.to('PutMailboxPermissions');
    }
    /**
     * Grants permission to add or update a mobile device access override
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_PutMobileDeviceAccessOverride.html
     */
    toPutMobileDeviceAccessOverride() {
        return this.to('PutMobileDeviceAccessOverride');
    }
    /**
     * Grants permission to add or update the retention policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_PutRetentionPolicy.html
     */
    toPutRetentionPolicy() {
        return this.to('PutRetentionPolicy');
    }
    /**
     * Grants permission to register a new mail domain in an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_RegisterMailDomain.html
     */
    toRegisterMailDomain() {
        return this.to('RegisterMailDomain');
    }
    /**
     * Grants permission to register an existing and disabled user, group, or resource for use by associating a mailbox and calendaring capabilities
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_RegisterToWorkMail.html
     */
    toRegisterToWorkMail() {
        return this.to('RegisterToWorkMail');
    }
    /**
     * Grants permission to allow the administrator to reset the password for a user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ResetPassword.html
     */
    toResetPassword() {
        return this.to('ResetPassword');
    }
    /**
     * Grants permission to perform a prefix search to find a specific user in a mail group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/groups_overview.html
     */
    toSearchMembers() {
        return this.to('SearchMembers');
    }
    /**
     * Grants permission to set the default mail domain for the organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/default_domain.html
     */
    toSetDefaultMailDomain() {
        return this.to('SetDefaultMailDomain');
    }
    /**
     * Grants permission to set journaling and fallback email addresses for email journaling
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/journaling_overview.html
     */
    toSetJournalingRules() {
        return this.to('SetJournalingRules');
    }
    /**
     * Grants permission to set the details of a mobile policy associated with the organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/edit_organization_mobile_policy.html
     */
    toSetMobilePolicyDetails() {
        return this.to('SetMobilePolicyDetails');
    }
    /**
     * Grants permission to start a new mailbox export job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_StartMailboxExportJob.html
     */
    toStartMailboxExportJob() {
        return this.to('StartMailboxExportJob');
    }
    /**
     * Grants permission to tag the specified Amazon WorkMail organization resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to performs a test on an availability provider to ensure that access is allowed
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_TestAvailabilityConfiguration.html
     */
    toTestAvailabilityConfiguration() {
        return this.to('TestAvailabilityConfiguration');
    }
    /**
     * Grants permission to test what inbound rules will apply to an email with a given sender and recipient
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/test-email-flow-rule.html
     */
    toTestInboundMailFlowRules() {
        return this.to('TestInboundMailFlowRules');
    }
    /**
     * Grants permission to test what outbound rules will apply to an email with a given sender and recipient
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/test-email-flow-rule.html
     */
    toTestOutboundMailFlowRules() {
        return this.to('TestOutboundMailFlowRules');
    }
    /**
     * Grants permission to untag the specified Amazon WorkMail organization resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an existing AvailabilityConfiguration for the given Amazon WorkMail organization and domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_UpdateAvailabilityConfiguration.html
     */
    toUpdateAvailabilityConfiguration() {
        return this.to('UpdateAvailabilityConfiguration');
    }
    /**
     * Grants permission to update which domain is the default domain for an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_UpdateDefaultMailDomain.html
     */
    toUpdateDefaultMailDomain() {
        return this.to('UpdateDefaultMailDomain');
    }
    /**
     * Grants permission to update details of a group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_UpdateGroup.html
     */
    toUpdateGroup() {
        return this.to('UpdateGroup');
    }
    /**
     * Grants permission to update an existing impersonation role for the given Amazon WorkMail organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_UpdateImpersonationRole.html
     */
    toUpdateImpersonationRole() {
        return this.to('UpdateImpersonationRole');
    }
    /**
     * Grants permission to update the details of an inbound email flow rule which will apply to all email sent to an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/modify-email-flow-rule.html
     */
    toUpdateInboundMailFlowRule() {
        return this.to('UpdateInboundMailFlowRule');
    }
    /**
     * Grants permission to update the maximum size (in MB) of the user's mailbox
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_UpdateMailboxQuota.html
     */
    toUpdateMailboxQuota() {
        return this.to('UpdateMailboxQuota');
    }
    /**
     * Grants permission to update a mobile device access rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_UpdateMobileDeviceAccessRule.html
     */
    toUpdateMobileDeviceAccessRule() {
        return this.to('UpdateMobileDeviceAccessRule');
    }
    /**
     * Grants permission to update the details of an outbound email flow rule which will apply to all email sent from an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/modify-email-flow-rule.html
     */
    toUpdateOutboundMailFlowRule() {
        return this.to('UpdateOutboundMailFlowRule');
    }
    /**
     * Grants permission to update the primary email for a user, group, or resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_UpdatePrimaryEmailAddress.html
     */
    toUpdatePrimaryEmailAddress() {
        return this.to('UpdatePrimaryEmailAddress');
    }
    /**
     * Grants permission to update details for the resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_UpdateResource.html
     */
    toUpdateResource() {
        return this.to('UpdateResource');
    }
    /**
     * Grants permission to update the details of an existing SMTP gateway registered to an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/smtp-gateway.html
     */
    toUpdateSmtpGateway() {
        return this.to('UpdateSmtpGateway');
    }
    /**
     * Grants permission to update details of a user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_UpdateUser.html
     */
    toUpdateUser() {
        return this.to('UpdateUser');
    }
    /**
     * Grants permission to remotely wipe the mobile device associated with a user's account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/manage-devices.html#remote_wipe_device
     */
    toWipeMobileDevice() {
        return this.to('WipeMobileDevice');
    }
    /**
     * Adds a resource of type organization to the statement
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/organizations_overview.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onOrganization(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:workmail:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:organization/${resourceId}`);
    }
    /**
     * Filters access by the tag key-value pairs that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toListTagsForResource()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - organization
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toListTagsForResource()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [workmail](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonworkmail.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'workmail';
        this.accessLevelList = {
            Write: [
                'AllowVendedLogDeliveryForResource',
                'AssociateDelegateToResource',
                'AssociateMemberToGroup',
                'AssumeImpersonationRole',
                'CancelMailboxExportJob',
                'CreateAlias',
                'CreateAvailabilityConfiguration',
                'CreateGroup',
                'CreateIdentityCenterApplication',
                'CreateImpersonationRole',
                'CreateInboundMailFlowRule',
                'CreateMailDomain',
                'CreateMobileDeviceAccessRule',
                'CreateOrganization',
                'CreateOutboundMailFlowRule',
                'CreateResource',
                'CreateSmtpGateway',
                'CreateUser',
                'DeleteAccessControlRule',
                'DeleteAlias',
                'DeleteAvailabilityConfiguration',
                'DeleteEmailMonitoringConfiguration',
                'DeleteGroup',
                'DeleteIdentityCenterApplication',
                'DeleteIdentityProviderConfiguration',
                'DeleteImpersonationRole',
                'DeleteInboundMailFlowRule',
                'DeleteMailDomain',
                'DeleteMailboxPermissions',
                'DeleteMobileDevice',
                'DeleteMobileDeviceAccessOverride',
                'DeleteMobileDeviceAccessRule',
                'DeleteOrganization',
                'DeleteOutboundMailFlowRule',
                'DeletePersonalAccessToken',
                'DeleteResource',
                'DeleteRetentionPolicy',
                'DeleteSmtpGateway',
                'DeleteUser',
                'DeliverToMailbox',
                'DeregisterFromWorkMail',
                'DeregisterMailDomain',
                'DisassociateDelegateFromResource',
                'DisassociateMemberFromGroup',
                'EnableMailDomain',
                'PutAccessControlRule',
                'PutEmailMonitoringConfiguration',
                'PutIdentityProviderConfiguration',
                'PutInboundDmarcSettings',
                'PutMailboxPermissions',
                'PutMobileDeviceAccessOverride',
                'PutRetentionPolicy',
                'RegisterMailDomain',
                'RegisterToWorkMail',
                'ResetPassword',
                'SetDefaultMailDomain',
                'SetJournalingRules',
                'SetMobilePolicyDetails',
                'StartMailboxExportJob',
                'TestInboundMailFlowRules',
                'TestOutboundMailFlowRules',
                'UpdateAvailabilityConfiguration',
                'UpdateDefaultMailDomain',
                'UpdateGroup',
                'UpdateImpersonationRole',
                'UpdateInboundMailFlowRule',
                'UpdateMailboxQuota',
                'UpdateMobileDeviceAccessRule',
                'UpdateOutboundMailFlowRule',
                'UpdatePrimaryEmailAddress',
                'UpdateResource',
                'UpdateSmtpGateway',
                'UpdateUser',
                'WipeMobileDevice'
            ],
            Read: [
                'DescribeEmailMonitoringConfiguration',
                'DescribeEntity',
                'DescribeIdentityProviderConfiguration',
                'DescribeInboundDmarcSettings',
                'DescribeInboundMailFlowRule',
                'DescribeMailboxExportJob',
                'DescribeOutboundMailFlowRule',
                'DescribeSmtpGateway',
                'GetAccessControlEffect',
                'GetDefaultRetentionPolicy',
                'GetImpersonationRole',
                'GetImpersonationRoleEffect',
                'GetJournalingRules',
                'GetMailDomain',
                'GetMailDomainDetails',
                'GetMailboxDetails',
                'GetMobileDeviceAccessEffect',
                'GetMobileDeviceAccessOverride',
                'GetMobileDeviceDetails',
                'GetMobileDevicesForUser',
                'GetMobilePolicyDetails',
                'GetPersonalAccessTokenMetadata',
                'ListAccessControlRules',
                'ListAvailabilityConfigurations',
                'ListMobileDeviceAccessOverrides',
                'ListMobileDeviceAccessRules',
                'SearchMembers',
                'TestAvailabilityConfiguration'
            ],
            List: [
                'DescribeGroup',
                'DescribeMailDomains',
                'DescribeOrganization',
                'DescribeResource',
                'DescribeUser',
                'ListAliases',
                'ListGroupMembers',
                'ListGroups',
                'ListGroupsForEntity',
                'ListImpersonationRoles',
                'ListInboundMailFlowRules',
                'ListMailDomains',
                'ListMailboxExportJobs',
                'ListMailboxPermissions',
                'ListOrganizations',
                'ListOutboundMailFlowRules',
                'ListPersonalAccessTokens',
                'ListResourceDelegates',
                'ListResources',
                'ListSmtpGateways',
                'ListTagsForResource',
                'ListUsers'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Workmail = Workmail;
//# sourceMappingURL=data:application/json;base64,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