"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Workspaces = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [workspaces](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonworkspaces.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Workspaces extends shared_1.PolicyStatement {
    /**
     * Grants permission to accept invitations from other AWS accounts to share the same configuration for WorkSpaces BYOL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_AcceptAccountLinkInvitation.html
     */
    toAcceptAccountLinkInvitation() {
        return this.to('AcceptAccountLinkInvitation');
    }
    /**
     * Grants permission to associate connection aliases with directories
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_AssociateConnectionAlias.html
     */
    toAssociateConnectionAlias() {
        return this.to('AssociateConnectionAlias');
    }
    /**
     * Grants permission to associate IP access control groups with directories
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_AssociateIpGroups.html
     */
    toAssociateIpGroups() {
        return this.to('AssociateIpGroups');
    }
    /**
     * Grants permission to associate a workspace application with a WorkSpace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_AssociateWorkspaceApplication.html
     */
    toAssociateWorkspaceApplication() {
        return this.to('AssociateWorkspaceApplication');
    }
    /**
     * Grants permission to add rules to IP access control groups
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - workspaces:UpdateRulesOfIpGroup
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_AuthorizeIpRules.html
     */
    toAuthorizeIpRules() {
        return this.to('AuthorizeIpRules');
    }
    /**
     * Grants permission to copy a WorkSpace image
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - workspaces:DescribeWorkspaceImages
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CopyWorkspaceImage.html
     */
    toCopyWorkspaceImage() {
        return this.to('CopyWorkspaceImage');
    }
    /**
     * Grants permission to invite other AWS accounts to share the same configuration for WorkSpaces BYOL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateAccountLinkInvitation.html
     */
    toCreateAccountLinkInvitation() {
        return this.to('CreateAccountLinkInvitation');
    }
    /**
     * Grants permission to create an Amazon Connect client add-in within a directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateConnectClientAddIn.html
     */
    toCreateConnectClientAddIn() {
        return this.to('CreateConnectClientAddIn');
    }
    /**
     * Grants permission to create connection aliases for use with cross-Region redirection
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateConnectionAlias.html
     */
    toCreateConnectionAlias() {
        return this.to('CreateConnectionAlias');
    }
    /**
     * Grants permission to create IP access control groups
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateIpGroup.html
     */
    toCreateIpGroup() {
        return this.to('CreateIpGroup');
    }
    /**
     * Grants permission to create one or more Standby WorkSpaces
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateStandbyWorkspaces.html
     */
    toCreateStandbyWorkspaces() {
        return this.to('CreateStandbyWorkspaces');
    }
    /**
     * Grants permission to create tags for WorkSpaces resources
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateTags.html
     */
    toCreateTags() {
        return this.to('CreateTags');
    }
    /**
     * Grants permission to create an updated WorkSpace image
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateUpdatedWorkspaceImage.html
     */
    toCreateUpdatedWorkspaceImage() {
        return this.to('CreateUpdatedWorkspaceImage');
    }
    /**
     * Grants permission to create a WorkSpace bundle
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - workspaces:CreateTags
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateWorkspaceBundle.html
     */
    toCreateWorkspaceBundle() {
        return this.to('CreateWorkspaceBundle');
    }
    /**
     * Grants permission to create a new WorkSpace image
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateWorkspaceImage.html
     */
    toCreateWorkspaceImage() {
        return this.to('CreateWorkspaceImage');
    }
    /**
     * Grants permission to create one or more WorkSpaces
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateWorkspaces.html
     */
    toCreateWorkspaces() {
        return this.to('CreateWorkspaces');
    }
    /**
     * Grants permission to create a WorkSpaces Pool
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateWorkspacesPool.html
     */
    toCreateWorkspacesPool() {
        return this.to('CreateWorkspacesPool');
    }
    /**
     * Grants permission to delete invitations to other AWS accounts to share the same configuration for WorkSpaces BYOL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteAccountLinkInvitation.html
     */
    toDeleteAccountLinkInvitation() {
        return this.to('DeleteAccountLinkInvitation');
    }
    /**
     * Grants permission to delete AWS WorkSpaces Client branding data within a directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteClientBranding.html
     */
    toDeleteClientBranding() {
        return this.to('DeleteClientBranding');
    }
    /**
     * Grants permission to delete an Amazon Connect client add-in that is configured within a directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteConnectClientAddIn.html
     */
    toDeleteConnectClientAddIn() {
        return this.to('DeleteConnectClientAddIn');
    }
    /**
     * Grants permission to delete connection aliases
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteConnectionAlias.html
     */
    toDeleteConnectionAlias() {
        return this.to('DeleteConnectionAlias');
    }
    /**
     * Grants permission to delete IP access control groups
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteIpGroup.html
     */
    toDeleteIpGroup() {
        return this.to('DeleteIpGroup');
    }
    /**
     * Grants permission to delete tags from WorkSpaces resources
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteTags.html
     */
    toDeleteTags() {
        return this.to('DeleteTags');
    }
    /**
     * Grants permission to delete WorkSpace bundles
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteWorkspaceBundle.html
     */
    toDeleteWorkspaceBundle() {
        return this.to('DeleteWorkspaceBundle');
    }
    /**
     * Grants permission to delete WorkSpace images
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteWorkspaceImage.html
     */
    toDeleteWorkspaceImage() {
        return this.to('DeleteWorkspaceImage');
    }
    /**
     * Grants permission to deploy all pending workspace applications on a WorkSpace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DeployWorkspaceApplications.html
     */
    toDeployWorkspaceApplications() {
        return this.to('DeployWorkspaceApplications');
    }
    /**
     * Grants permission to deregister directories from use with Amazon WorkSpaces
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DeregisterWorkspaceDirectory.html
     */
    toDeregisterWorkspaceDirectory() {
        return this.to('DeregisterWorkspaceDirectory');
    }
    /**
     * Grants permission to retrieve the configuration of Bring Your Own License (BYOL) for WorkSpaces accounts
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeAccount.html
     */
    toDescribeAccount() {
        return this.to('DescribeAccount');
    }
    /**
     * Grants permission to retrieve modifications to the configuration of Bring Your Own License (BYOL) for WorkSpaces accounts
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeAccountModifications.html
     */
    toDescribeAccountModifications() {
        return this.to('DescribeAccountModifications');
    }
    /**
     * Grants permission to retrieve information about resources associated with a WorkSpace application
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeApplicationAssociations.html
     */
    toDescribeApplicationAssociations() {
        return this.to('DescribeApplicationAssociations');
    }
    /**
     * Grants permission to obtain information about WorkSpace applications
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeApplications.html
     */
    toDescribeApplications() {
        return this.to('DescribeApplications');
    }
    /**
     * Grants permission to retrieve information about resources associated with a WorkSpace bundle
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeBundleAssociations.html
     */
    toDescribeBundleAssociations() {
        return this.to('DescribeBundleAssociations');
    }
    /**
     * Grants permission to retrieve AWS WorkSpaces Client branding data within a directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeClientBranding.html
     */
    toDescribeClientBranding() {
        return this.to('DescribeClientBranding');
    }
    /**
     * Grants permission to retrieve information about WorkSpaces clients
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeClientProperties.html
     */
    toDescribeClientProperties() {
        return this.to('DescribeClientProperties');
    }
    /**
     * Grants permission to retrieve a list of Amazon Connect client add-ins that have been created
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeConnectClientAddIns.html
     */
    toDescribeConnectClientAddIns() {
        return this.to('DescribeConnectClientAddIns');
    }
    /**
     * Grants permission to retrieve the permissions that the owners of connection aliases have granted to other AWS accounts for connection aliases
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeConnectionAliasPermissions.html
     */
    toDescribeConnectionAliasPermissions() {
        return this.to('DescribeConnectionAliasPermissions');
    }
    /**
     * Grants permission to retrieve a list that describes the connection aliases used for cross-Region redirection
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeConnectionAliases.html
     */
    toDescribeConnectionAliases() {
        return this.to('DescribeConnectionAliases');
    }
    /**
     * Grants permission to retrieve information about resources associated with a WorkSpace image
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeImageAssociations.html
     */
    toDescribeImageAssociations() {
        return this.to('DescribeImageAssociations');
    }
    /**
     * Grants permission to retrieve information about IP access control groups
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeIpGroups.html
     */
    toDescribeIpGroups() {
        return this.to('DescribeIpGroups');
    }
    /**
     * Grants permission to describe the tags for WorkSpaces resources
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeTags.html
     */
    toDescribeTags() {
        return this.to('DescribeTags');
    }
    /**
     * Grants permission to retrieve information about resources associated with a WorkSpace
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaceAssociations.html
     */
    toDescribeWorkspaceAssociations() {
        return this.to('DescribeWorkspaceAssociations');
    }
    /**
     * Grants permission to obtain information about WorkSpace bundles
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaceBundles.html
     */
    toDescribeWorkspaceBundles() {
        return this.to('DescribeWorkspaceBundles');
    }
    /**
     * Grants permission to retrieve information about directories that are registered with WorkSpaces
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaceDirectories.html
     */
    toDescribeWorkspaceDirectories() {
        return this.to('DescribeWorkspaceDirectories');
    }
    /**
     * Grants permission to retrieve information about WorkSpace image permissions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaceImagePermissions.html
     */
    toDescribeWorkspaceImagePermissions() {
        return this.to('DescribeWorkspaceImagePermissions');
    }
    /**
     * Grants permission to retrieve information about WorkSpace images
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaceImages.html
     */
    toDescribeWorkspaceImages() {
        return this.to('DescribeWorkspaceImages');
    }
    /**
     * Grants permission to retrieve information about WorkSpace snapshots
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaceSnapshots.html
     */
    toDescribeWorkspaceSnapshots() {
        return this.to('DescribeWorkspaceSnapshots');
    }
    /**
     * Grants permission to obtain information about WorkSpaces
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaces.html
     */
    toDescribeWorkspaces() {
        return this.to('DescribeWorkspaces');
    }
    /**
     * Grants permission to obtain the connection status of WorkSpaces
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspacesConnectionStatus.html
     */
    toDescribeWorkspacesConnectionStatus() {
        return this.to('DescribeWorkspacesConnectionStatus');
    }
    /**
     * Grants permission to retrieve information about the sessions of a WorkSpaces Pool
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspacesPoolSessions.html
     */
    toDescribeWorkspacesPoolSessions() {
        return this.to('DescribeWorkspacesPoolSessions');
    }
    /**
     * Grants permission to retrieve information about WorkSpaces Pools
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspacesPools.html
     */
    toDescribeWorkspacesPools() {
        return this.to('DescribeWorkspacesPools');
    }
    /**
     * Grants permission to disassociate connection aliases from directories
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DisassociateConnectionAlias.html
     */
    toDisassociateConnectionAlias() {
        return this.to('DisassociateConnectionAlias');
    }
    /**
     * Grants permission to disassociate IP access control groups from directories
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DisassociateIpGroups.html
     */
    toDisassociateIpGroups() {
        return this.to('DisassociateIpGroups');
    }
    /**
     * Grants permission to disassociate a workspace application from a WorkSpace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DisassociateWorkspaceApplication.html
     */
    toDisassociateWorkspaceApplication() {
        return this.to('DisassociateWorkspaceApplication');
    }
    /**
     * Grants permission to retrieve a link with another AWS Account for sharing configuration for WorkSpaces BYOL
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_GetAccountLink.html
     */
    toGetAccountLink() {
        return this.to('GetAccountLink');
    }
    /**
     * Grants permission to import AWS WorkSpaces Client branding data within a directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ImportClientBranding.html
     */
    toImportClientBranding() {
        return this.to('ImportClientBranding');
    }
    /**
     * Grants permission to import Bring Your Own License (BYOL) images into Amazon WorkSpaces
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DescribeImages
     * - ec2:ModifyImageAttribute
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ImportWorkspaceImage.html
     */
    toImportWorkspaceImage() {
        return this.to('ImportWorkspaceImage');
    }
    /**
     * Grants permission to retrieve links with the AWS Account(s) that share your configuration for WorkSpaces BYOL
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ListAccountLinks.html
     */
    toListAccountLinks() {
        return this.to('ListAccountLinks');
    }
    /**
     * Grants permission to list the available CIDR ranges for enabling Bring Your Own License (BYOL) for WorkSpaces accounts
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ListAvailableManagementCidrRanges.html
     */
    toListAvailableManagementCidrRanges() {
        return this.to('ListAvailableManagementCidrRanges');
    }
    /**
     * Grants permission to migrate WorkSpaces
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_MigrateWorkspace.html
     */
    toMigrateWorkspace() {
        return this.to('MigrateWorkspace');
    }
    /**
     * Grants permission to modify the configuration of Bring Your Own License (BYOL) for WorkSpaces accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyAccount.html
     */
    toModifyAccount() {
        return this.to('ModifyAccount');
    }
    /**
     * Grants permission to modify the certificate-based authorization properties of a directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyCertificateBasedAuthProperties.html
     */
    toModifyCertificateBasedAuthProperties() {
        return this.to('ModifyCertificateBasedAuthProperties');
    }
    /**
     * Grants permission to modify the properties of WorkSpaces clients
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyClientProperties.html
     */
    toModifyClientProperties() {
        return this.to('ModifyClientProperties');
    }
    /**
     * Grants permission to configure the specified directory between Standard TLS and FIPS 140-2 validated mode
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyEndpointEncryptionMode.html
     */
    toModifyEndpointEncryptionMode() {
        return this.to('ModifyEndpointEncryptionMode');
    }
    /**
     * Grants permission to modify the SAML properties of a directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifySamlProperties.html
     */
    toModifySamlProperties() {
        return this.to('ModifySamlProperties');
    }
    /**
     * Grants permission to modify the self-service WorkSpace management capabilities for your users
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifySelfservicePermissions.html
     */
    toModifySelfservicePermissions() {
        return this.to('ModifySelfservicePermissions');
    }
    /**
     * Grants permission to modify the streaming properties
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyStreamingProperties.html
     */
    toModifyStreamingProperties() {
        return this.to('ModifyStreamingProperties');
    }
    /**
     * Grants permission to specify which devices and operating systems users can use to access their WorkSpaces
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyWorkspaceAccessProperties.html
     */
    toModifyWorkspaceAccessProperties() {
        return this.to('ModifyWorkspaceAccessProperties');
    }
    /**
     * Grants permission to modify the default properties used to create WorkSpaces
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyWorkspaceCreationProperties.html
     */
    toModifyWorkspaceCreationProperties() {
        return this.to('ModifyWorkspaceCreationProperties');
    }
    /**
     * Grants permission to modify WorkSpace properties, including the running mode and the AutoStop period
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyWorkspaceProperties.html
     */
    toModifyWorkspaceProperties() {
        return this.to('ModifyWorkspaceProperties');
    }
    /**
     * Grants permission to modify the state of WorkSpaces
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyWorkspaceState.html
     */
    toModifyWorkspaceState() {
        return this.to('ModifyWorkspaceState');
    }
    /**
     * Grants permission to reboot WorkSpaces
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_RebootWorkspaces.html
     */
    toRebootWorkspaces() {
        return this.to('RebootWorkspaces');
    }
    /**
     * Grants permission to rebuild WorkSpaces
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_RebuildWorkspaces.html
     */
    toRebuildWorkspaces() {
        return this.to('RebuildWorkspaces');
    }
    /**
     * Grants permission to register directories for use with Amazon WorkSpaces
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_RegisterWorkspaceDirectory.html
     */
    toRegisterWorkspaceDirectory() {
        return this.to('RegisterWorkspaceDirectory');
    }
    /**
     * Grants permission to reject invitations from other AWS accounts to share the same configuration for WorkSpaces BYOL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_RejectAccountLinkInvitation.html
     */
    toRejectAccountLinkInvitation() {
        return this.to('RejectAccountLinkInvitation');
    }
    /**
     * Grants permission to restore WorkSpaces
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_RestoreWorkspace.html
     */
    toRestoreWorkspace() {
        return this.to('RestoreWorkspace');
    }
    /**
     * Grants permission to remove rules from IP access control groups
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - workspaces:UpdateRulesOfIpGroup
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_RevokeIpRules.html
     */
    toRevokeIpRules() {
        return this.to('RevokeIpRules');
    }
    /**
     * Grants permission to start AutoStop WorkSpaces
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_StartWorkspaces.html
     */
    toStartWorkspaces() {
        return this.to('StartWorkspaces');
    }
    /**
     * Grants permission to start a WorkSpaces Pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_StartWorkspacesPool.html
     */
    toStartWorkspacesPool() {
        return this.to('StartWorkspacesPool');
    }
    /**
     * Grants permission to stop AutoStop WorkSpaces
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_StopWorkspaces.html
     */
    toStopWorkspaces() {
        return this.to('StopWorkspaces');
    }
    /**
     * Grants permission to stop a WorkSpaces Pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_StopWorkspacesPool.html
     */
    toStopWorkspacesPool() {
        return this.to('StopWorkspacesPool');
    }
    /**
     * Grants permission to federated users to sign in by using their existing credentials and stream their workspace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserId()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_Stream.html
     */
    toStream() {
        return this.to('Stream');
    }
    /**
     * Grants permission to terminate WorkSpaces
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_TerminateWorkspaces.html
     */
    toTerminateWorkspaces() {
        return this.to('TerminateWorkspaces');
    }
    /**
     * Grants permission to terminate a WorkSpaces Pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_TerminateWorkspacesPool.html
     */
    toTerminateWorkspacesPool() {
        return this.to('TerminateWorkspacesPool');
    }
    /**
     * Grants permission to terminate a WorkSpaces Pool session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_TerminateWorkspacesPoolSession.html
     */
    toTerminateWorkspacesPoolSession() {
        return this.to('TerminateWorkspacesPoolSession');
    }
    /**
     * Grants permission to update an Amazon Connect client add-in. Use this action to update the name and endpoint URL of an Amazon Connect client add-in
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_UpdateConnectClientAddIn.html
     */
    toUpdateConnectClientAddIn() {
        return this.to('UpdateConnectClientAddIn');
    }
    /**
     * Grants permission to share or unshare connection aliases with other accounts
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_UpdateConnectionAliasPermission.html
     */
    toUpdateConnectionAliasPermission() {
        return this.to('UpdateConnectionAliasPermission');
    }
    /**
     * Grants permission to replace rules for IP access control groups
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - workspaces:AuthorizeIpRules
     * - workspaces:RevokeIpRules
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_UpdateRulesOfIpGroup.html
     */
    toUpdateRulesOfIpGroup() {
        return this.to('UpdateRulesOfIpGroup');
    }
    /**
     * Grants permission to update the WorkSpace images used in WorkSpace bundles
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_UpdateWorkspaceBundle.html
     */
    toUpdateWorkspaceBundle() {
        return this.to('UpdateWorkspaceBundle');
    }
    /**
     * Grants permission to share or unshare WorkSpace images with other accounts by specifying whether other accounts have permission to copy the image
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_UpdateWorkspaceImagePermission.html
     */
    toUpdateWorkspaceImagePermission() {
        return this.to('UpdateWorkspaceImagePermission');
    }
    /**
     * Grants permission to update the WorkSpaces pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_UpdateWorkspacesPool.html
     */
    toUpdateWorkspacesPool() {
        return this.to('UpdateWorkspacesPool');
    }
    /**
     * Adds a resource of type directoryid to the statement
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/manage-workspaces-directory.html
     *
     * @param directoryId - Identifier for the directoryId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDirectoryid(directoryId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:workspaces:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:directory/${directoryId}`);
    }
    /**
     * Adds a resource of type workspacebundle to the statement
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/bundles.html
     *
     * @param bundleId - Identifier for the bundleId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorkspacebundle(bundleId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:workspaces:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workspacebundle/${bundleId}`);
    }
    /**
     * Adds a resource of type workspaceid to the statement
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/wsp_workspace_management.html
     *
     * @param workspaceId - Identifier for the workspaceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorkspaceid(workspaceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:workspaces:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workspace/${workspaceId}`);
    }
    /**
     * Adds a resource of type workspaceimage to the statement
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/bundles.html
     *
     * @param imageId - Identifier for the imageId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorkspaceimage(imageId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:workspaces:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workspaceimage/${imageId}`);
    }
    /**
     * Adds a resource of type workspaceipgroup to the statement
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/amazon-workspaces-ip-access-control-groups.html
     *
     * @param groupId - Identifier for the groupId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorkspaceipgroup(groupId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:workspaces:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workspaceipgroup/${groupId}`);
    }
    /**
     * Adds a resource of type workspacespoolid to the statement
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/amazon-workspaces-pool.html
     *
     * @param poolId - Identifier for the poolId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorkspacespoolid(poolId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:workspaces:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workspacespool/${poolId}`);
    }
    /**
     * Adds a resource of type connectionalias to the statement
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/cross-region-redirection.html
     *
     * @param connectionAliasId - Identifier for the connectionAliasId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConnectionalias(connectionAliasId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:workspaces:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:connectionalias/${connectionAliasId}`);
    }
    /**
     * Adds a resource of type workspaceapplication to the statement
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/application-bundle-management.html
     *
     * @param workSpaceApplicationId - Identifier for the workSpaceApplicationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorkspaceapplication(workSpaceApplicationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:workspaces:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workspaceapplication/${workSpaceApplicationId}`);
    }
    /**
     * Filters access based on the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCopyWorkspaceImage()
     * - .toCreateConnectionAlias()
     * - .toCreateIpGroup()
     * - .toCreateStandbyWorkspaces()
     * - .toCreateTags()
     * - .toCreateUpdatedWorkspaceImage()
     * - .toCreateWorkspaceBundle()
     * - .toCreateWorkspaceImage()
     * - .toCreateWorkspaces()
     * - .toCreateWorkspacesPool()
     * - .toDeleteTags()
     * - .toRegisterWorkspaceDirectory()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toAssociateWorkspaceApplication()
     * - .toDeployWorkspaceApplications()
     * - .toDescribeApplicationAssociations()
     * - .toDescribeBundleAssociations()
     * - .toDescribeImageAssociations()
     * - .toDescribeWorkspaceAssociations()
     * - .toDisassociateWorkspaceApplication()
     *
     * Applies to resource types:
     * - directoryid
     * - workspacebundle
     * - workspaceid
     * - workspaceimage
     * - workspaceipgroup
     * - workspacespoolid
     * - connectionalias
     * - workspaceapplication
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCopyWorkspaceImage()
     * - .toCreateConnectionAlias()
     * - .toCreateIpGroup()
     * - .toCreateStandbyWorkspaces()
     * - .toCreateTags()
     * - .toCreateUpdatedWorkspaceImage()
     * - .toCreateWorkspaceBundle()
     * - .toCreateWorkspaceImage()
     * - .toCreateWorkspaces()
     * - .toCreateWorkspacesPool()
     * - .toDeleteTags()
     * - .toRegisterWorkspaceDirectory()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ID of the Workspaces user
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/external-identity-providers-setting-up-saml.html#external-identity-providers-embed-inline-policy-for-IAM-role
     *
     * Applies to actions:
     * - .toStream()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUserId(value, operator) {
        return this.if(`userId`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [workspaces](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonworkspaces.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'workspaces';
        this.accessLevelList = {
            Write: [
                'AcceptAccountLinkInvitation',
                'AssociateConnectionAlias',
                'AssociateIpGroups',
                'AssociateWorkspaceApplication',
                'AuthorizeIpRules',
                'CopyWorkspaceImage',
                'CreateAccountLinkInvitation',
                'CreateConnectClientAddIn',
                'CreateConnectionAlias',
                'CreateIpGroup',
                'CreateStandbyWorkspaces',
                'CreateUpdatedWorkspaceImage',
                'CreateWorkspaceBundle',
                'CreateWorkspaceImage',
                'CreateWorkspaces',
                'CreateWorkspacesPool',
                'DeleteAccountLinkInvitation',
                'DeleteClientBranding',
                'DeleteConnectClientAddIn',
                'DeleteConnectionAlias',
                'DeleteIpGroup',
                'DeleteWorkspaceBundle',
                'DeleteWorkspaceImage',
                'DeployWorkspaceApplications',
                'DeregisterWorkspaceDirectory',
                'DisassociateConnectionAlias',
                'DisassociateIpGroups',
                'DisassociateWorkspaceApplication',
                'ImportClientBranding',
                'ImportWorkspaceImage',
                'MigrateWorkspace',
                'ModifyAccount',
                'ModifyCertificateBasedAuthProperties',
                'ModifyClientProperties',
                'ModifyEndpointEncryptionMode',
                'ModifySamlProperties',
                'ModifyStreamingProperties',
                'ModifyWorkspaceAccessProperties',
                'ModifyWorkspaceCreationProperties',
                'ModifyWorkspaceProperties',
                'ModifyWorkspaceState',
                'RebootWorkspaces',
                'RebuildWorkspaces',
                'RegisterWorkspaceDirectory',
                'RejectAccountLinkInvitation',
                'RestoreWorkspace',
                'RevokeIpRules',
                'StartWorkspaces',
                'StartWorkspacesPool',
                'StopWorkspaces',
                'StopWorkspacesPool',
                'Stream',
                'TerminateWorkspaces',
                'TerminateWorkspacesPool',
                'TerminateWorkspacesPoolSession',
                'UpdateConnectClientAddIn',
                'UpdateRulesOfIpGroup',
                'UpdateWorkspaceBundle',
                'UpdateWorkspacesPool'
            ],
            Tagging: [
                'CreateTags',
                'DeleteTags'
            ],
            Read: [
                'DescribeAccount',
                'DescribeAccountModifications',
                'DescribeClientBranding',
                'DescribeConnectionAliasPermissions',
                'DescribeConnectionAliases',
                'DescribeIpGroups',
                'DescribeTags',
                'DescribeWorkspaceDirectories',
                'DescribeWorkspaceImagePermissions',
                'DescribeWorkspacesConnectionStatus',
                'GetAccountLink'
            ],
            List: [
                'DescribeApplicationAssociations',
                'DescribeApplications',
                'DescribeBundleAssociations',
                'DescribeClientProperties',
                'DescribeConnectClientAddIns',
                'DescribeImageAssociations',
                'DescribeWorkspaceAssociations',
                'DescribeWorkspaceBundles',
                'DescribeWorkspaceImages',
                'DescribeWorkspaceSnapshots',
                'DescribeWorkspaces',
                'DescribeWorkspacesPoolSessions',
                'DescribeWorkspacesPools',
                'ListAccountLinks',
                'ListAvailableManagementCidrRanges'
            ],
            'Permissions management': [
                'ModifySelfservicePermissions',
                'UpdateConnectionAliasPermission',
                'UpdateWorkspaceImagePermission'
            ]
        };
    }
}
exports.Workspaces = Workspaces;
//# sourceMappingURL=data:application/json;base64,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