import { AccessLevelList } from '../../shared/access-level';
import { PolicyStatement, Operator } from '../../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [workspaces-instances](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsworkspacesmanagedinstances.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class WorkspacesInstances extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to associate a workspace managed volume to a workspace managed instance in your account
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:AttachVolume
     * - ec2:DescribeVolumes
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_AssociateVolume.html
     */
    toAssociateVolume(): this;
    /**
     * Grants permission to create a workspace managed volume in your account
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateVolume
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateVolume.html
     */
    toCreateVolume(): this;
    /**
     * Grants permission to create a workspace managed instance in your account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:DescribeInstances
     * - ec2:RunInstances
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateWorkspaceInstance.html
     */
    toCreateWorkspaceInstance(): this;
    /**
     * Grants permission to delete a workspace managed volume in your account
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DeleteVolume
     * - ec2:DescribeVolumes
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteVolume.html
     */
    toDeleteVolume(): this;
    /**
     * Grants permission to delete a workspace managed instance in your account
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:TerminateInstances
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteWorkspaceInstance.html
     */
    toDeleteWorkspaceInstance(): this;
    /**
     * Grants permission to disassociate a workspace managed volume from a workspace managed instance in your account
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DescribeVolumes
     * - ec2:DetachVolume
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DisassociateVolume.html
     */
    toDisassociateVolume(): this;
    /**
     * Grants permission to get details for a specific workspace managed instance in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_GetWorkspaceInstance.html
     */
    toGetWorkspaceInstance(): this;
    /**
     * Grants permission to list all supported instance types
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ListInstanceTypes.html
     */
    toListInstanceTypes(): this;
    /**
     * Grants permission to list all supported AWS regions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ListRegions.html
     */
    toListRegions(): this;
    /**
     * Grants permission to list user tags for resources in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ListTagsForResource.html
     */
    toListTagsForResource(): this;
    /**
     * Grants permission to list workspace managed instances in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ListWorkspaceInstances.html
     */
    toListWorkspaceInstances(): this;
    /**
     * Grants permission to add user tags to resources in your account
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_TagResource.html
     */
    toTagResource(): this;
    /**
     * Grants permission to remove user tags from resources in your account
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_UntagResource.html
     */
    toUntagResource(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type WorkspaceInstanceId to the statement
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/managed-workspaces-instances.html
     *
     * @param workspaceInstanceId - Identifier for the workspaceInstanceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorkspaceInstanceId(workspaceInstanceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type VolumeId to the statement
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/managed-workspaces-volumes.html
     *
     * @param volumeId - Identifier for the volumeId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVolumeId(volumeId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access based on the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateWorkspaceInstance()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access based on the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - WorkspaceInstanceId
     * - VolumeId
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access based on the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateWorkspaceInstance()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [workspaces-instances](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsworkspacesmanagedinstances.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
