"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Operator = void 0;
/**
 * Use condition operators in the `Condition` element to match the condition key and value in the policy against values in the request context. For more information about the `Condition` element, see [IAM JSON Policy Elements: Condition](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
 *
 * The condition operator that you can use in a policy depends on the condition key you choose. You can choose a global condition key or a service-specific condition key. To learn which condition operator you can use for a global condition key, see [AWS Global Condition Context Keys](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html). To learn which condition operator you can use for a service-specific condition key, see [Actions, Resources, and Condition Keys for AWS Services](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_actions-resources-contextkeys.html) and choose the service that you want to view.
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html
 */
class Operator {
    constructor() {
        this.base = '';
        this.hasIfExists = false;
        this.hasForAllValues = false;
        this.hasForAnyValue = false;
    }
    setBase(base) {
        if (this.base.length) {
            throw new Error(`Operator already set: ${this.base}`);
        }
        this.base = base;
        return this;
    }
    toString() {
        let value = this.base;
        if (this.base.length == 0) {
            throw new Error('No operator set');
        }
        if (this.hasIfExists) {
            value += 'IfExists';
        }
        if (this.hasForAllValues) {
            value = `ForAllValues:${value}`;
        }
        if (this.hasForAnyValue) {
            value = `ForAnyValue:${value}`;
        }
        if (this.base == 'Null' && value != this.base) {
            throw new Error(`You cannot add modifiers to the "Null" operator: ${value}`);
        }
        return value;
    }
    /**
     * You can add `IfExists` to the end of any condition operator name except the `Null` condition. For example, `StringLikeIfExists`. You do this to say "If the policy key is present in the context of the request, process the key as specified in the policy. If the key is not present, evaluate the condition element as true." Other condition elements in the statement can still result in a nonmatch, but not a missing key when checked with `...IfExists`.
     */
    ifExists() {
        this.hasIfExists = true;
        return this;
    }
    /**
     * Tests whether the value of every member of the request set is a subset of the condition key set. The condition returns true if every key value in the request matches at least one value in the policy. It also returns true if there are no keys in the request, or if the key values resolve to a null data set, such as an empty string.
     */
    forAllValues() {
        this.hasForAllValues = true;
        return this;
    }
    /**
     * Tests whether at least one member of the set of request values matches at least one member of the set of condition key values. The condition returns true if any one of the key values in the request matches any one of the condition values in the policy. For no matching key or a null dataset, the condition returns false.
     */
    forAnyValue() {
        this.hasForAnyValue = true;
        return this;
    }
    /**
     * Exact match, case sensitive.
     */
    stringEquals() {
        return this.setBase(Operator.stringEquals);
    }
    /**
     * Negated exact match, case sensitive.
     */
    stringNotEquals() {
        return this.setBase(Operator.stringNotEquals);
    }
    /**
     * Exact match, ignore case.
     */
    stringEqualsIgnoreCase() {
        return this.setBase(Operator.stringEqualsIgnoreCase);
    }
    /**
     * Negated exact match, ignore case.
     */
    stringNotEqualsIgnoreCase() {
        return this.setBase(Operator.stringNotEqualsIgnoreCase);
    }
    /**
     * Case-sensitive match.
     *
     * The values can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`) anywhere in the string.
     */
    stringLike() {
        return this.setBase(Operator.stringLike);
    }
    /**
     * Negated case-sensitive matching.
     *
     * The values can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`) anywhere in the string.
     */
    stringNotLike() {
        return this.setBase(Operator.stringNotLike);
    }
    /**
     * Exact match.
     */
    numericEquals() {
        return this.setBase(Operator.numericEquals);
    }
    /**
     * Negated exact match.
     */
    numericNotEquals() {
        return this.setBase(Operator.numericNotEquals);
    }
    /**
     * Match numbers lower than value: `<`
     */
    numericLessThan() {
        return this.setBase(Operator.numericLessThan);
    }
    /**
     * Match numbers lower or equal to value: `<=`
     */
    numericLessThanEquals() {
        return this.setBase(Operator.numericLessThanEquals);
    }
    /**
     * Match numbers higher than value: `>`
     */
    numericGreaterThan() {
        return this.setBase(Operator.numericGreaterThan);
    }
    /**
     * Match numbers higher or equal value: `>=`
     */
    numericGreaterThanEquals() {
        return this.setBase(Operator.numericGreaterThanEquals);
    }
    /**
     * Match a specific date: `=`
     */
    dateEquals() {
        return this.setBase(Operator.dateEquals);
    }
    /**
     * Negated match for a specific date: `!=`
     */
    dateNotEquals() {
        return this.setBase(Operator.dateNotEquals);
    }
    /**
     * Match before a specific date and time: `<`
     */
    dateLessThan() {
        return this.setBase(Operator.dateLessThan);
    }
    /**
     * Match at or before a specific date and time: `<=`
     */
    dateLessThanEquals() {
        return this.setBase(Operator.dateLessThanEquals);
    }
    /**
     * Match after a specific a date and time: `>`
     */
    dateGreaterThan() {
        return this.setBase(Operator.dateGreaterThan);
    }
    /**
     * Match at or after a specific date and time: `>=`
     */
    dateGreaterThanEquals() {
        return this.setBase(Operator.dateGreaterThanEquals);
    }
    /**
     * Boolean match
     */
    bool() {
        return this.setBase(Operator.bool);
    }
    /**
     * The BinaryEquals condition operator lets you construct Condition elements that test key values that are in binary format. It compares the value of the specified key byte for byte against a [base-64](https://en.wikipedia.org/wiki/Base64) encoded representation of the binary value in the policy.
     */
    binaryEquals() {
        return this.setBase(Operator.binaryEquals);
    }
    /**
     * The BinaryEquals condition operator lets you construct Condition elements that test key values that are in binary format. It compares the value of the specified key byte for byte against a [base-64](https://en.wikipedia.org/wiki/Base64) encoded representation of the binary value in the policy.
     */
    binaryNotEquals() {
        return this.setBase(Operator.binaryNotEquals);
    }
    /**
     * Match an IP address or range: `=`
     */
    ipAddress() {
        return this.setBase(Operator.ipAddress);
    }
    /**
     * All IP addresses except the specified IP address or range `!=`
     */
    notIpAddress() {
        return this.setBase(Operator.notIpAddress);
    }
    /**
     * Match of the ARN, case sensitive.
     *
     * Each of the six colon-delimited components of the ARN is checked separately and each can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`).
     *
     * `ArnEquals` and `ArnLike` behave identical.
     */
    arnEquals() {
        return this.setBase(Operator.arnEquals);
    }
    /**
     * Negated match of the ARN, case sensitive.
     *
     * Each of the six colon-delimited components of the ARN is checked separately and each can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`).
     *
     * `ArnNotEquals` and `ArnNotLike` behave identical.
     */
    arnNotEquals() {
        return this.setBase(Operator.arnNotEquals);
    }
    /**
     * Match of the ARN, case sensitive.
     *
     * Each of the six colon-delimited components of the ARN is checked separately and each can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`).
     */
    arnLike() {
        return this.setBase(Operator.arnLike);
    }
    /**
     * Negated match of the ARN, case sensitive.
     *
     * Each of the six colon-delimited components of the ARN is checked separately and each can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`).
     *
     * `ArnNotEquals` and `ArnNotLike` behave identical.
     */
    arnNotLike() {
        return this.setBase(Operator.arnNotLike);
    }
}
exports.Operator = Operator;
/**
 * Exact match, case sensitive.
 */
Operator.stringEquals = 'StringEquals';
/**
 * Negated exact match, case sensitive.
 */
Operator.stringNotEquals = 'StringNotEquals';
/**
 * Exact match, ignore case.
 */
Operator.stringEqualsIgnoreCase = 'StringEqualsIgnoreCase';
/**
 * Negated exact match, ignore case.
 */
Operator.stringNotEqualsIgnoreCase = 'StringNotEqualsIgnoreCase';
/**
 * Case-sensitive match.
 *
 * The values can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`) anywhere in the string.
 */
Operator.stringLike = 'StringLike';
/**
 * Negated case-sensitive matching.
 *
 * The values can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`) anywhere in the string.
 */
Operator.stringNotLike = 'StringNotLike';
/**
 * Exact match.
 */
Operator.numericEquals = 'NumericEquals';
/**
 * Negated exact match.
 */
Operator.numericNotEquals = 'NumericNotEquals';
/**
 * Match numbers lower than value: `<`
 */
Operator.numericLessThan = 'NumericLessThan';
/**
 * Match numbers lower or equal to value: `<=`
 */
Operator.numericLessThanEquals = 'NumericLessThanEquals';
/**
 * Match numbers higher than value: `>`
 */
Operator.numericGreaterThan = 'NumericGreaterThan';
/**
 * Match numbers higher or equal value: `>=`
 */
Operator.numericGreaterThanEquals = 'NumericGreaterThanEquals';
/**
 * Match a specific date: `=`
 */
Operator.dateEquals = 'DateEquals';
/**
 * Negated match for a specific date: `!=`
 */
Operator.dateNotEquals = 'DateNotEquals';
/**
 * Match before a specific date and time: `<`
 */
Operator.dateLessThan = 'DateLessThan';
/**
 * Match at or before a specific date and time: `<=`
 */
Operator.dateLessThanEquals = 'DateLessThanEquals';
/**
 * Match after a specific a date and time: `>`
 */
Operator.dateGreaterThan = 'DateGreaterThan';
/**
 * Match at or after a specific date and time: `>=`
 */
Operator.dateGreaterThanEquals = 'DateGreaterThanEquals';
/**
 * Boolean match
 */
Operator.bool = 'Bool';
/**
 * The BinaryEquals condition operator lets you construct Condition elements that test key values that are in binary format. It compares the value of the specified key byte for byte against a [base-64](https://en.wikipedia.org/wiki/Base64) encoded representation of the binary value in the policy.
 */
Operator.binaryEquals = 'BinaryEquals';
/**
 * The BinaryEquals condition operator lets you construct Condition elements that test key values that are in binary format. It compares the value of the specified key byte for byte against a [base-64](https://en.wikipedia.org/wiki/Base64) encoded representation of the binary value in the policy.
 */
Operator.binaryNotEquals = 'BinaryNotEquals';
/**
 * Match an IP address or range: `=`
 */
Operator.ipAddress = 'IpAddress';
/**
 * All IP addresses except the specified IP address or range `!=`
 */
Operator.notIpAddress = 'NotIpAddress';
/**
 * Match of the ARN, case sensitive.
 *
 * Each of the six colon-delimited components of the ARN is checked separately and each can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`).
 *
 * `ArnEquals` and `ArnLike` behave identical.
 */
Operator.arnEquals = 'ArnEquals';
/**
 * Negated match of the ARN, case sensitive.
 *
 * Each of the six colon-delimited components of the ARN is checked separately and each can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`).
 *
 * `ArnNotEquals` and `ArnNotLike` behave identical.
 */
Operator.arnNotEquals = 'ArnNotEquals';
/**
 * Match of the ARN, case sensitive.
 *
 * Each of the six colon-delimited components of the ARN is checked separately and each can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`).
 */
Operator.arnLike = 'ArnLike';
/**
 * Negated match of the ARN, case sensitive.
 *
 * Each of the six colon-delimited components of the ARN is checked separately and each can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`).
 *
 * `ArnNotEquals` and `ArnNotLike` behave identical.
 */
Operator.arnNotLike = 'ArnNotLike';
/**
 * Check if a key is present at the time of authorization. In the policy statement, use either true (the key doesn't exist — it is null) or false (the key exists and its value is not null).
 */
Operator.null = 'Null';
//# sourceMappingURL=data:application/json;base64,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