"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStatementWithResources = void 0;
const _3_actions_1 = require("./3-actions");
/**
 * Adds "resource" functionality to the Policy Statement
 */
class PolicyStatementWithResources extends _3_actions_1.PolicyStatementWithActions {
    constructor() {
        super(...arguments);
        this.useNotResource = false;
        this.floydResources = [];
        this.skipAutoResource = false;
        this.cdkResourcesApplied = false;
    }
    /**
     * Injects resources into the statement.
     *
     * Only relevant for the main package. In CDK mode this only calls super.
     */
    toJSON() {
        // @ts-ignore only available after swapping 1-base
        if (typeof this.addResources == 'function') {
            this.cdkApplyResources();
            return super.toJSON();
        }
        const mode = this.useNotResource ? 'NotResource' : 'Resource';
        const statement = super.toJSON();
        const self = this;
        this.ensureResource();
        if (this.floydResources.length) {
            const resources = this.floydResources.filter((elem, pos) => {
                return self.floydResources.indexOf(elem) == pos;
            });
            statement[mode] = resources.length > 1 ? resources : resources[0];
        }
        return statement;
    }
    toStatementJson() {
        this.ensureResource();
        this.cdkApplyResources();
        // @ts-ignore only available after swapping 1-base
        return super.toStatementJson();
    }
    freeze() {
        // @ts-ignore only available after swapping 1-base
        if (!this.frozen) {
            this.ensureResource();
            this.cdkApplyResources();
        }
        return super.freeze();
    }
    cdkApplyResources() {
        if (!this.cdkResourcesApplied) {
            const mode = this.useNotResource ? 'addNotResources' : 'addResources';
            const self = this;
            const uniqueResources = this.floydResources.filter((elem, pos) => {
                return self.floydResources.indexOf(elem) == pos;
            });
            // @ts-ignore only available after swapping 1-base
            this[mode](...uniqueResources);
            this.cdkResourcesApplied = true;
        }
    }
    /**
     * Switches the statement to use [`NotResource`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notresource.html).
     */
    notResource() {
        this.useNotResource = true;
        return this;
    }
    /**
     * Checks weather any resource was applied to the policy.
     */
    hasResources() {
        return this.floydResources.length > 0;
    }
    /**
     * Limit statement to specified resources.
     *
     * To allow all resources, pass `*`
     */
    on(...arns) {
        this.floydResources.push(...arns);
        return this;
    }
    /**
     * Add all resources (`*`) to the statement
     *
     * This is the default behavior, unless the statement has principals.
     */
    onAllResources() {
        this.floydResources.push('*');
        return this;
    }
    ensureResource() {
        if (this.hasResources())
            return;
        // @ts-ignore only available after swapping 1-base
        if (this.hasResource)
            return;
        // @ts-ignore only available after swapping 1-base
        if (this.hasPrincipal)
            return; //assume policies may not have resources
        if (this.skipAutoResource)
            return;
        // a statement requires resources. if none was added, we assume the user wants all resources
        this.onAllResources();
    }
}
exports.PolicyStatementWithResources = PolicyStatementWithResources;
//# sourceMappingURL=data:application/json;base64,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