"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Discovery = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [discovery](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsapplicationdiscoveryservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Discovery extends shared_1.PolicyStatement {
    /**
     * Grants permission to AssociateConfigurationItemsToApplication API. AssociateConfigurationItemsToApplication associates one or more configuration items with an application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_AssociateConfigurationItemsToApplication.html
     */
    toAssociateConfigurationItemsToApplication() {
        return this.to('AssociateConfigurationItemsToApplication');
    }
    /**
     * Grants permission to BatchDeleteAgents API. BatchDeleteAgents deletes one or more agents/data collectors associated with your account, each identified by its agent ID. Deleting a data collector does not delete the previous data collected
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_BatchDeleteAgents.html
     */
    toBatchDeleteAgents() {
        return this.to('BatchDeleteAgents');
    }
    /**
     * Grants permission to BatchDeleteImportData API. BatchDeleteImportData deletes one or more Migration Hub import tasks, each identified by their import ID. Each import task has a number of records, which can identify servers or applications
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_BatchDeleteImportData.html
     */
    toBatchDeleteImportData() {
        return this.to('BatchDeleteImportData');
    }
    /**
     * Grants permission to CreateApplication API. CreateApplication creates an application with the given name and description
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_CreateApplication.html
     */
    toCreateApplication() {
        return this.to('CreateApplication');
    }
    /**
     * Grants permission to CreateTags API. CreateTags creates one or more tags for configuration items. Tags are metadata that help you categorize IT assets. This API accepts a list of multiple configuration items
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_CreateTags.html
     */
    toCreateTags() {
        return this.to('CreateTags');
    }
    /**
     * Grants permission to DeleteApplications API. DeleteApplications deletes a list of applications and their associations with configuration items
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_DeleteApplications.html
     */
    toDeleteApplications() {
        return this.to('DeleteApplications');
    }
    /**
     * Grants permission to DeleteTags API. DeleteTags deletes the association between configuration items and one or more tags. This API accepts a list of multiple configuration items
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_DeleteTags.html
     */
    toDeleteTags() {
        return this.to('DeleteTags');
    }
    /**
     * Grants permission to DescribeAgents API. DescribeAgents lists agents or the Connector by ID or lists all agents/Connectors associated with your user if you did not specify an ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_DescribeAgents.html
     */
    toDescribeAgents() {
        return this.to('DescribeAgents');
    }
    /**
     * Grants permission to DescribeBatchDeleteConfigurationTask API. DescribeBatchDeleteConfigurationTask returns attributes about a batched deletion task to delete a set of configuration items. The supplied task ID should be the task ID receieved from the output of StartBatchDeleteConfigurationTask
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_DescribeBatchDeleteConfigurationTask.html
     */
    toDescribeBatchDeleteConfigurationTask() {
        return this.to('DescribeBatchDeleteConfigurationTask');
    }
    /**
     * Grants permission to DescribeConfigurations API. DescribeConfigurations retrieves attributes for a list of configuration item IDs. All of the supplied IDs must be for the same asset type (server, application, process, or connection). Output fields are specific to the asset type selected. For example, the output for a server configuration item includes a list of attributes about the server, such as host name, operating system, and number of network cards
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_DescribeConfigurations.html
     */
    toDescribeConfigurations() {
        return this.to('DescribeConfigurations');
    }
    /**
     * Grants permission to DescribeContinuousExports API. DescribeContinuousExports lists exports as specified by ID. All continuous exports associated with your user can be listed if you call DescribeContinuousExports as is without passing any parameters
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_DescribeContinuousExports.html
     */
    toDescribeContinuousExports() {
        return this.to('DescribeContinuousExports');
    }
    /**
     * Grants permission to DescribeExportConfigurations API. DescribeExportConfigurations retrieves the status of a given export process. You can retrieve status from a maximum of 100 processes
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_DescribeExportConfigurations.html
     */
    toDescribeExportConfigurations() {
        return this.to('DescribeExportConfigurations');
    }
    /**
     * Grants permission to DescribeExportTasks API. DescribeExportTasks retrieve status of one or more export tasks. You can retrieve the status of up to 100 export tasks
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_DescribeExportTasks.html
     */
    toDescribeExportTasks() {
        return this.to('DescribeExportTasks');
    }
    /**
     * Grants permission to DescribeImportTasks API. DescribeImportTasks returns an array of import tasks for your user, including status information, times, IDs, the Amazon S3 Object URL for the import file, and more
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_DescribeImportTasks.html
     */
    toDescribeImportTasks() {
        return this.to('DescribeImportTasks');
    }
    /**
     * Grants permission to DescribeTags API. DescribeTags retrieves a list of configuration items that are tagged with a specific tag. Or retrieves a list of all tags assigned to a specific configuration item
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_DescribeTags.html
     */
    toDescribeTags() {
        return this.to('DescribeTags');
    }
    /**
     * Grants permission to DisassociateConfigurationItemsFromApplication API. DisassociateConfigurationItemsFromApplication disassociates one or more configuration items from an application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_DisassociateConfigurationItemsFromApplication.html
     */
    toDisassociateConfigurationItemsFromApplication() {
        return this.to('DisassociateConfigurationItemsFromApplication');
    }
    /**
     * Grants permission to ExportConfigurations API. ExportConfigurations exports all discovered configuration data to an Amazon S3 bucket or an application that enables you to view and evaluate the data. Data includes tags and tag associations, processes, connections, servers, and system performance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_ExportConfigurations.html
     */
    toExportConfigurations() {
        return this.to('ExportConfigurations');
    }
    /**
     * Grants permission to GetDiscoverySummary API. GetDiscoverySummary retrieves a short summary of discovered assets
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_GetDiscoverySummary.html
     */
    toGetDiscoverySummary() {
        return this.to('GetDiscoverySummary');
    }
    /**
     * Grants permission to GetNetworkConnectionGraph API. GetNetworkConnectionGraph accepts input list of one of - Ip Addresses, server ids or node ids. Returns a list of nodes and edges which help customer visualize network connection graph. This API is used for visualize network graph functionality in MigrationHub console
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_GetNetworkConnectionGraph.html
     */
    toGetNetworkConnectionGraph() {
        return this.to('GetNetworkConnectionGraph');
    }
    /**
     * Grants permission to ListConfigurations API. ListConfigurations retrieves a list of configuration items according to criteria you specify in a filter. The filter criteria identify relationship requirements
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_ListConfigurations.html
     */
    toListConfigurations() {
        return this.to('ListConfigurations');
    }
    /**
     * Grants permission to ListServerNeighbors API. ListServerNeighbors retrieves a list of servers which are one network hop away from a specified server
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_ListServerNeighbors.html
     */
    toListServerNeighbors() {
        return this.to('ListServerNeighbors');
    }
    /**
     * Grants permission to StartBatchDeleteConfigurationTask API. StartBatchDeleteConfigurationTask starts an asynchronous batch deletion of your configuration items. All of the supplied IDs must be for the same asset type (server, application, process, or connection). Output is a unique task ID you can use to check back on the deletions progress
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_StartBatchDeleteConfigurationTask.html
     */
    toStartBatchDeleteConfigurationTask() {
        return this.to('StartBatchDeleteConfigurationTask');
    }
    /**
     * Grants permission to StartContinuousExport API. StartContinuousExport start the continuous flow of agent's discovered data into Amazon Athena
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:AttachRolePolicy
     * - iam:CreatePolicy
     * - iam:CreateRole
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_StartContinuousExport.html
     */
    toStartContinuousExport() {
        return this.to('StartContinuousExport');
    }
    /**
     * Grants permission to StartDataCollectionByAgentIds API. StartDataCollectionByAgentIds instructs the specified agents or Connectors to start collecting data
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_StartDataCollectionByAgentIds.html
     */
    toStartDataCollectionByAgentIds() {
        return this.to('StartDataCollectionByAgentIds');
    }
    /**
     * Grants permission to StartExportTask API. StartExportTask export the configuration data about discovered configuration items and relationships to an S3 bucket in a specified format
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_StartExportTask.html
     */
    toStartExportTask() {
        return this.to('StartExportTask');
    }
    /**
     * Grants permission to StartImportTask API. StartImportTask starts an import task. The Migration Hub import feature allows you to import details of your on-premises environment directly into AWS without having to use the Application Discovery Service (ADS) tools such as the Discovery Connector or Discovery Agent. This gives you the option to perform migration assessment and planning directly from your imported data including the ability to group your devices as applications and track their migration status
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - discovery:AssociateConfigurationItemsToApplication
     * - discovery:CreateApplication
     * - discovery:CreateTags
     * - discovery:GetDiscoverySummary
     * - discovery:ListConfigurations
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_StartImportTask.html
     */
    toStartImportTask() {
        return this.to('StartImportTask');
    }
    /**
     * Grants permission to StopContinuousExport API. StopContinuousExport stops the continuous flow of agent's discovered data into Amazon Athena
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_StopContinuousExport.html
     */
    toStopContinuousExport() {
        return this.to('StopContinuousExport');
    }
    /**
     * Grants permission to StopDataCollectionByAgentIds API. StopDataCollectionByAgentIds instructs the specified agents or Connectors to stop collecting data
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_StopDataCollectionByAgentIds.html
     */
    toStopDataCollectionByAgentIds() {
        return this.to('StopDataCollectionByAgentIds');
    }
    /**
     * Grants permission to UpdateApplication API. UpdateApplication updates metadata about an application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/application-discovery/latest/APIReference/API_UpdateApplication.html
     */
    toUpdateApplication() {
        return this.to('UpdateApplication');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toDeleteTags()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [discovery](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsapplicationdiscoveryservice.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'discovery';
        this.accessLevelList = {
            Write: [
                'AssociateConfigurationItemsToApplication',
                'BatchDeleteAgents',
                'BatchDeleteImportData',
                'CreateApplication',
                'DeleteApplications',
                'DisassociateConfigurationItemsFromApplication',
                'ExportConfigurations',
                'StartBatchDeleteConfigurationTask',
                'StartContinuousExport',
                'StartDataCollectionByAgentIds',
                'StartExportTask',
                'StartImportTask',
                'StopContinuousExport',
                'StopDataCollectionByAgentIds',
                'UpdateApplication'
            ],
            Tagging: [
                'CreateTags',
                'DeleteTags'
            ],
            Read: [
                'DescribeAgents',
                'DescribeBatchDeleteConfigurationTask',
                'DescribeConfigurations',
                'DescribeContinuousExports',
                'DescribeExportConfigurations',
                'DescribeExportTasks',
                'DescribeTags',
                'GetDiscoverySummary',
                'GetNetworkConnectionGraph'
            ],
            List: [
                'DescribeImportTasks',
                'ListConfigurations',
                'ListServerNeighbors'
            ]
        };
    }
}
exports.Discovery = Discovery;
//# sourceMappingURL=data:application/json;base64,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