"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Artifact = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [artifact](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsartifact.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Artifact extends shared_1.PolicyStatement {
    /**
     * Grants permission to accept an AWS agreement that has not yet been accepted by the customer account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/artifact/latest/APIReference/API_AcceptAgreement.html
     */
    toAcceptAgreement() {
        return this.to('AcceptAgreement');
    }
    /**
     * Grants permission to accept the terms of an NDA Document for a given agreement resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/artifact/latest/APIReference/API_AcceptNdaForAgreement.html
     */
    toAcceptNdaForAgreement() {
        return this.to('AcceptNdaForAgreement');
    }
    /**
     * Grants permission to download an AWS agreement that has not yet been accepted or a customer agreement that has been accepted by the customer account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/artifact/latest/ug/managing-agreements.html
     */
    toDownloadAgreement() {
        return this.to('DownloadAgreement');
    }
    /**
     * Grants permission to download an AWS compliance report package
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/artifact/latest/ug/getting-started.html
     */
    toGet() {
        return this.to('Get');
    }
    /**
     * Grants permission to get the account settings for Artifact
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/artifact/latest/APIReference/API_GetAccountSettings.html
     */
    toGetAccountSettings() {
        return this.to('GetAccountSettings');
    }
    /**
     * Grants permission to get an AWS agreement that has not yet been accepted by the customer account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/artifact/latest/APIReference/API_GetAgreement.html
     */
    toGetAgreement() {
        return this.to('GetAgreement');
    }
    /**
     * Grants permission to get an AWS agreement that has been accepted by the customer account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/artifact/latest/APIReference/API_GetCustomerAgreement.html
     */
    toGetCustomerAgreement() {
        return this.to('GetCustomerAgreement');
    }
    /**
     * Grants permission to retrieve the NDA Document for a given agreement resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/artifact/latest/APIReference/API_GetNdaForAgreement.html
     */
    toGetNdaForAgreement() {
        return this.to('GetNdaForAgreement');
    }
    /**
     * Grants permission to download a report
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/artifact/latest/APIReference/API_GetReport.html
     */
    toGetReport() {
        return this.to('GetReport');
    }
    /**
     * Grants permission to download metadata associated with a report
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/artifact/latest/APIReference/API_GetReportMetadata.html
     */
    toGetReportMetadata() {
        return this.to('GetReportMetadata');
    }
    /**
     * Grants permission to download a term associated with a report
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/artifact/latest/APIReference/API_GetTermForReport.html
     */
    toGetTermForReport() {
        return this.to('GetTermForReport');
    }
    /**
     * Grants permission to list AWS agreements
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/artifact/latest/APIReference/API_ListAgreements.html
     */
    toListAgreements() {
        return this.to('ListAgreements');
    }
    /**
     * Grants permission to list customer-agreement resources that have been accepted by the customer account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/artifact/latest/APIReference/API_ListCustomerAgreements.html
     */
    toListCustomerAgreements() {
        return this.to('ListCustomerAgreements');
    }
    /**
     * Grants permission to list reports in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/artifact/latest/APIReference/API_ListReports.html
     */
    toListReports() {
        return this.to('ListReports');
    }
    /**
     * Grants permission to put account settings for Artifact
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/artifact/latest/APIReference/API_PutAccountSettings.html
     */
    toPutAccountSettings() {
        return this.to('PutAccountSettings');
    }
    /**
     * Grants permission to terminate a customer agreement that was previously accepted by the customer account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/artifact/latest/APIReference/API_TerminateAgreement.html
     */
    toTerminateAgreement() {
        return this.to('TerminateAgreement');
    }
    /**
     * Adds a resource of type report-package to the statement
     *
     * https://docs.aws.amazon.com/artifact/latest/ug/what-is-aws-artifact.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onReportPackage(resourceName, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:artifact:::report-package/${resourceName}`);
    }
    /**
     * Adds a resource of type customer-agreement to the statement
     *
     * https://docs.aws.amazon.com/artifact/latest/ug/managing-agreements.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onCustomerAgreement(resourceName, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:artifact::${account ?? this.defaultAccount}:customer-agreement/${resourceName}`);
    }
    /**
     * Adds a resource of type agreement to the statement
     *
     * https://docs.aws.amazon.com/artifact/latest/ug/managing-agreements.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAgreement(resourceName, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:artifact:::agreement/${resourceName}`);
    }
    /**
     * Adds a resource of type report to the statement
     *
     * https://docs.aws.amazon.com/artifact/latest/ug/what-is-aws-artifact.html
     *
     * @param reportId - Identifier for the reportId.
     * @param version - Identifier for the version.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onReport(reportId, version, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:artifact:${region ?? this.defaultRegion}::report/${reportId}:${version}`);
    }
    /**
     * Filters access by which category reports are associated with
     *
     * https://docs.aws.amazon.com/artifact/latest/ug/using-condition-keys.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReportCategory(value, operator) {
        return this.if(`ReportCategory`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by which series reports are associated with
     *
     * https://docs.aws.amazon.com/artifact/latest/ug/using-condition-keys.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReportSeries(value, operator) {
        return this.if(`ReportSeries`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [artifact](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsartifact.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'artifact';
        this.accessLevelList = {
            Write: [
                'AcceptAgreement',
                'AcceptNdaForAgreement',
                'PutAccountSettings',
                'TerminateAgreement'
            ],
            Read: [
                'DownloadAgreement',
                'Get',
                'GetAccountSettings',
                'GetAgreement',
                'GetCustomerAgreement',
                'GetNdaForAgreement',
                'GetReport',
                'GetReportMetadata',
                'GetTermForReport'
            ],
            List: [
                'ListAgreements',
                'ListCustomerAgreements',
                'ListReports'
            ]
        };
    }
}
exports.Artifact = Artifact;
//# sourceMappingURL=data:application/json;base64,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