"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Auditmanager = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [auditmanager](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsauditmanager.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Auditmanager extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate an evidence folder with an assessment report in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_AssociateAssessmentReportEvidenceFolder.html
     */
    toAssociateAssessmentReportEvidenceFolder() {
        return this.to('AssociateAssessmentReportEvidenceFolder');
    }
    /**
     * Grants permission to associate a list of evidence to an assessment report in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_BatchAssociateAssessmentReportEvidence.html
     */
    toBatchAssociateAssessmentReportEvidence() {
        return this.to('BatchAssociateAssessmentReportEvidence');
    }
    /**
     * Grants permission to create delegations for an assessment in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_BatchCreateDelegationByAssessment.html
     */
    toBatchCreateDelegationByAssessment() {
        return this.to('BatchCreateDelegationByAssessment');
    }
    /**
     * Grants permission to delete delegations for an assessment in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_BatchDeleteDelegationByAssessment.html
     */
    toBatchDeleteDelegationByAssessment() {
        return this.to('BatchDeleteDelegationByAssessment');
    }
    /**
     * Grants permission to disassociate a list of evidence from an assessment report in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_BatchDisassociateAssessmentReportEvidence.html
     */
    toBatchDisassociateAssessmentReportEvidence() {
        return this.to('BatchDisassociateAssessmentReportEvidence');
    }
    /**
     * Grants permission to import a list of evidence to an assessment control in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_BatchImportEvidenceToAssessmentControl.html
     */
    toBatchImportEvidenceToAssessmentControl() {
        return this.to('BatchImportEvidenceToAssessmentControl');
    }
    /**
     * Grants permission to create an assessment to be used with AWS Audit Manager
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_CreateAssessment.html
     */
    toCreateAssessment() {
        return this.to('CreateAssessment');
    }
    /**
     * Grants permission to create a framework for use in AWS Audit Manager
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_CreateAssessmentFramework.html
     */
    toCreateAssessmentFramework() {
        return this.to('CreateAssessmentFramework');
    }
    /**
     * Grants permission to create an assessment report in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_CreateAssessmentReport.html
     */
    toCreateAssessmentReport() {
        return this.to('CreateAssessmentReport');
    }
    /**
     * Grants permission to create a control to be used in AWS Audit Manager
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_CreateControl.html
     */
    toCreateControl() {
        return this.to('CreateControl');
    }
    /**
     * Grants permission to delete an assessment in AWS Audit Manager
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_DeleteAssessment.html
     */
    toDeleteAssessment() {
        return this.to('DeleteAssessment');
    }
    /**
     * Grants permission to delete an assessment framework in AWS Audit Manager
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_DeleteAssessmentFramework.html
     */
    toDeleteAssessmentFramework() {
        return this.to('DeleteAssessmentFramework');
    }
    /**
     * Grants permission to delete a share request for a custom framework in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_DeleteAssessmentFrameworkShare.html
     */
    toDeleteAssessmentFrameworkShare() {
        return this.to('DeleteAssessmentFrameworkShare');
    }
    /**
     * Grants permission to delete an assessment report in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_DeleteAssessmentReport.html
     */
    toDeleteAssessmentReport() {
        return this.to('DeleteAssessmentReport');
    }
    /**
     * Grants permission to delete a control in AWS Audit Manager
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_DeleteControl.html
     */
    toDeleteControl() {
        return this.to('DeleteControl');
    }
    /**
     * Grants permission to deregister an account in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_ DeregisterAccount.html
     */
    toDeregisterAccount() {
        return this.to('DeregisterAccount');
    }
    /**
     * Grants permission to deregister the delegated administrator account for AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_DeregisterOrganizationAdminAccount.html
     */
    toDeregisterOrganizationAdminAccount() {
        return this.to('DeregisterOrganizationAdminAccount');
    }
    /**
     * Grants permission to disassociate an evidence folder from an assessment report in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_DisassociateAssessmentReportEvidenceFolder.html
     */
    toDisassociateAssessmentReportEvidenceFolder() {
        return this.to('DisassociateAssessmentReportEvidenceFolder');
    }
    /**
     * Grants permission to get the status of an account in AWS Audit Manager
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetAccountStatus.html
     */
    toGetAccountStatus() {
        return this.to('GetAccountStatus');
    }
    /**
     * Grants permission to get an assessment created in AWS Audit Manager
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetAssessment.html
     */
    toGetAssessment() {
        return this.to('GetAssessment');
    }
    /**
     * Grants permission to get an assessment framework in AWS Audit Manager
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetAssessmentFramework.html
     */
    toGetAssessmentFramework() {
        return this.to('GetAssessmentFramework');
    }
    /**
     * Grants permission to get the URL for an assessment report in AWS Audit Manager
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetAssessmentReportUrl.html
     */
    toGetAssessmentReportUrl() {
        return this.to('GetAssessmentReportUrl');
    }
    /**
     * Grants permission to get changelogs for an assessment in AWS Audit Manager
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetChangeLogs.html
     */
    toGetChangeLogs() {
        return this.to('GetChangeLogs');
    }
    /**
     * Grants permission to get a control in AWS Audit Manager
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetControl.html
     */
    toGetControl() {
        return this.to('GetControl');
    }
    /**
     * Grants permission to get all delegations in AWS Audit Manager
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetDelegations.html
     */
    toGetDelegations() {
        return this.to('GetDelegations');
    }
    /**
     * Grants permission to get evidence from AWS Audit Manager
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetEvidence.html
     */
    toGetEvidence() {
        return this.to('GetEvidence');
    }
    /**
     * Grants permission to get all the evidence from an evidence folder in AWS Audit Manager
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetEvidenceByEvidenceFolder.html
     */
    toGetEvidenceByEvidenceFolder() {
        return this.to('GetEvidenceByEvidenceFolder');
    }
    /**
     * Grants permission to get a presigned Amazon S3 URL that can be used to upload a file as manual evidence
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetEvidenceFileUploadUrl.html
     */
    toGetEvidenceFileUploadUrl() {
        return this.to('GetEvidenceFileUploadUrl');
    }
    /**
     * Grants permission to get the evidence folder from AWS Audit Manager
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetEvidenceFolder.html
     */
    toGetEvidenceFolder() {
        return this.to('GetEvidenceFolder');
    }
    /**
     * Grants permission to get the evidence folders from an assessment in AWS Audit Manager
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetEvidenceFoldersByAssessment.html
     */
    toGetEvidenceFoldersByAssessment() {
        return this.to('GetEvidenceFoldersByAssessment');
    }
    /**
     * Grants permission to get the evidence folders from an assessment control in AWS Audit Manager
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetEvidenceFoldersByAssessmentControl.html
     */
    toGetEvidenceFoldersByAssessmentControl() {
        return this.to('GetEvidenceFoldersByAssessmentControl');
    }
    /**
     * Grants permission to get analytics data for all active assessments
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetInsights.html
     */
    toGetInsights() {
        return this.to('GetInsights');
    }
    /**
     * Grants permission to get analytics data for a specific active assessment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetInsightsByAssessment.html
     */
    toGetInsightsByAssessment() {
        return this.to('GetInsightsByAssessment');
    }
    /**
     * Grants permission to get the delegated administrator account in AWS Audit Manager
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetOrganizationAdminAccount.html
     */
    toGetOrganizationAdminAccount() {
        return this.to('GetOrganizationAdminAccount');
    }
    /**
     * Grants permission to get the services in scope for an assessment in AWS Audit Manager
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetServicesInScope.html
     */
    toGetServicesInScope() {
        return this.to('GetServicesInScope');
    }
    /**
     * Grants permission to get all settings configured in AWS Audit Manager
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_GetSettings.html
     */
    toGetSettings() {
        return this.to('GetSettings');
    }
    /**
     * Grants permission to list analytics data for controls in a specific control domain and active assessment
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_ListAssessmentControlInsightsByControlDomain.html
     */
    toListAssessmentControlInsightsByControlDomain() {
        return this.to('ListAssessmentControlInsightsByControlDomain');
    }
    /**
     * Grants permission to list all sent or received share requests for custom frameworks in AWS Audit Manager
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_ListAssessmentFrameworkShareRequests.html
     */
    toListAssessmentFrameworkShareRequests() {
        return this.to('ListAssessmentFrameworkShareRequests');
    }
    /**
     * Grants permission to list all assessment frameworks in AWS Audit Manager
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_ListAssessmentFrameworks.html
     */
    toListAssessmentFrameworks() {
        return this.to('ListAssessmentFrameworks');
    }
    /**
     * Grants permission to list all assessment reports in AWS Audit Manager
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_ListAssessmentReports.html
     */
    toListAssessmentReports() {
        return this.to('ListAssessmentReports');
    }
    /**
     * Grants permission to list all assessments in AWS Audit Manager
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_ListAssessments.html
     */
    toListAssessments() {
        return this.to('ListAssessments');
    }
    /**
     * Grants permission to list analytics data for control domains across all active assessments
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_ListControlDomainInsights.html
     */
    toListControlDomainInsights() {
        return this.to('ListControlDomainInsights');
    }
    /**
     * Grants permission to list analytics data for control domains in a specific active assessment
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_ListControlDomainInsightsByAssessment.html
     */
    toListControlDomainInsightsByAssessment() {
        return this.to('ListControlDomainInsightsByAssessment');
    }
    /**
     * Grants permission to list analytics data for controls in a specific control domain across all active assessments
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_ListControlInsightsByControlDomain.html
     */
    toListControlInsightsByControlDomain() {
        return this.to('ListControlInsightsByControlDomain');
    }
    /**
     * Grants permission to list all controls in AWS Audit Manager
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_ListControls.html
     */
    toListControls() {
        return this.to('ListControls');
    }
    /**
     * Grants permission to list all the data source keywords in AWS Audit Manager
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_ListKeywordsForDataSource.html
     */
    toListKeywordsForDataSource() {
        return this.to('ListKeywordsForDataSource');
    }
    /**
     * Grants permission to list all notifications in AWS Audit Manager
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_ListNotifications.html
     */
    toListNotifications() {
        return this.to('ListNotifications');
    }
    /**
     * Grants permission to list tags for an AWS Audit Manager resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to register an account in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_RegisterAccount.html
     */
    toRegisterAccount() {
        return this.to('RegisterAccount');
    }
    /**
     * Grants permission to register an account within the organization as the delegated administrator for AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_RegisterOrganizationAdminAccount.html
     */
    toRegisterOrganizationAdminAccount() {
        return this.to('RegisterOrganizationAdminAccount');
    }
    /**
     * Grants permission to create a share request for a custom framework in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_StartAssessmentFrameworkShare.html
     */
    toStartAssessmentFrameworkShare() {
        return this.to('StartAssessmentFrameworkShare');
    }
    /**
     * Grants permission to tag an AWS Audit Manager resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag an AWS Audit Manager resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an assessment in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_UpdateAssessment.html
     */
    toUpdateAssessment() {
        return this.to('UpdateAssessment');
    }
    /**
     * Grants permission to update an assessment control in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_UpdateAssessmentControl.html
     */
    toUpdateAssessmentControl() {
        return this.to('UpdateAssessmentControl');
    }
    /**
     * Grants permission to update the status of an assessment control set in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_UpdateAssessmentControlSetStatus.html
     */
    toUpdateAssessmentControlSetStatus() {
        return this.to('UpdateAssessmentControlSetStatus');
    }
    /**
     * Grants permission to update an assessment framework in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_UpdateAssessmentFramework.html
     */
    toUpdateAssessmentFramework() {
        return this.to('UpdateAssessmentFramework');
    }
    /**
     * Grants permission to update a share request for a custom framework in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_UpdateAssessmentFrameworkShare.html
     */
    toUpdateAssessmentFrameworkShare() {
        return this.to('UpdateAssessmentFrameworkShare');
    }
    /**
     * Grants permission to update the status of an assessment in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_UpdateAssessmentStatus.html
     */
    toUpdateAssessmentStatus() {
        return this.to('UpdateAssessmentStatus');
    }
    /**
     * Grants permission to update a control in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_UpdateControl.html
     */
    toUpdateControl() {
        return this.to('UpdateControl');
    }
    /**
     * Grants permission to update settings in AWS Audit Manager
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_UpdateSettings.html
     */
    toUpdateSettings() {
        return this.to('UpdateSettings');
    }
    /**
     * Grants permission to validate the integrity of an assessment report in AWS Audit Manager
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/audit-manager/latest/APIReference/API_ValidateAssessmentReportIntegrity.html
     */
    toValidateAssessmentReportIntegrity() {
        return this.to('ValidateAssessmentReportIntegrity');
    }
    /**
     * Adds a resource of type assessment to the statement
     *
     * https://docs.aws.amazon.com/audit-manager/latest/userguide/API_Assessment.html
     *
     * @param assessmentId - Identifier for the assessmentId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAssessment(assessmentId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:auditmanager:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:assessment/${assessmentId}`);
    }
    /**
     * Adds a resource of type assessmentFramework to the statement
     *
     * https://docs.aws.amazon.com/audit-manager/latest/userguide/API_AssessmentFramework.html
     *
     * @param assessmentFrameworkId - Identifier for the assessmentFrameworkId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAssessmentFramework(assessmentFrameworkId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:auditmanager:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:assessmentFramework/${assessmentFrameworkId}`);
    }
    /**
     * Adds a resource of type assessmentControlSet to the statement
     *
     * https://docs.aws.amazon.com/audit-manager/latest/userguide/API_AssessmentControlSet.html
     *
     * @param assessmentId - Identifier for the assessmentId.
     * @param controlSetId - Identifier for the controlSetId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAssessmentControlSet(assessmentId, controlSetId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:auditmanager:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:assessment/${assessmentId}/controlSet/${controlSetId}`);
    }
    /**
     * Adds a resource of type control to the statement
     *
     * https://docs.aws.amazon.com/audit-manager/latest/userguide/API_Control.html
     *
     * @param controlId - Identifier for the controlId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onControl(controlId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:auditmanager:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:control/${controlId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateAssessment()
     * - .toCreateAssessmentFramework()
     * - .toCreateControl()
     * - .toDeleteAssessment()
     * - .toDeleteAssessmentFramework()
     * - .toDeleteControl()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - control
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateAssessment()
     * - .toCreateAssessmentFramework()
     * - .toCreateControl()
     * - .toDeleteAssessment()
     * - .toDeleteAssessmentFramework()
     * - .toDeleteControl()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [auditmanager](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsauditmanager.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'auditmanager';
        this.accessLevelList = {
            Write: [
                'AssociateAssessmentReportEvidenceFolder',
                'BatchAssociateAssessmentReportEvidence',
                'BatchCreateDelegationByAssessment',
                'BatchDeleteDelegationByAssessment',
                'BatchDisassociateAssessmentReportEvidence',
                'BatchImportEvidenceToAssessmentControl',
                'CreateAssessment',
                'CreateAssessmentFramework',
                'CreateAssessmentReport',
                'CreateControl',
                'DeleteAssessment',
                'DeleteAssessmentFramework',
                'DeleteAssessmentFrameworkShare',
                'DeleteAssessmentReport',
                'DeleteControl',
                'DeregisterAccount',
                'DeregisterOrganizationAdminAccount',
                'DisassociateAssessmentReportEvidenceFolder',
                'RegisterAccount',
                'RegisterOrganizationAdminAccount',
                'StartAssessmentFrameworkShare',
                'UpdateAssessment',
                'UpdateAssessmentControl',
                'UpdateAssessmentControlSetStatus',
                'UpdateAssessmentFramework',
                'UpdateAssessmentFrameworkShare',
                'UpdateAssessmentStatus',
                'UpdateControl',
                'UpdateSettings'
            ],
            Read: [
                'GetAccountStatus',
                'GetAssessment',
                'GetAssessmentFramework',
                'GetAssessmentReportUrl',
                'GetChangeLogs',
                'GetControl',
                'GetEvidence',
                'GetEvidenceByEvidenceFolder',
                'GetEvidenceFileUploadUrl',
                'GetEvidenceFolder',
                'GetEvidenceFoldersByAssessment',
                'GetEvidenceFoldersByAssessmentControl',
                'GetInsights',
                'GetInsightsByAssessment',
                'GetOrganizationAdminAccount',
                'GetServicesInScope',
                'GetSettings',
                'ListTagsForResource',
                'ValidateAssessmentReportIntegrity'
            ],
            List: [
                'GetDelegations',
                'ListAssessmentControlInsightsByControlDomain',
                'ListAssessmentFrameworkShareRequests',
                'ListAssessmentFrameworks',
                'ListAssessmentReports',
                'ListAssessments',
                'ListControlDomainInsights',
                'ListControlDomainInsightsByAssessment',
                'ListControlInsightsByControlDomain',
                'ListControls',
                'ListKeywordsForDataSource',
                'ListNotifications'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Auditmanager = Auditmanager;
//# sourceMappingURL=data:application/json;base64,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