"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Billing = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [billing](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsbilling.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Billing extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a billing view
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_billing_CreateBillingView.html
     */
    toCreateBillingView() {
        return this.to('CreateBillingView');
    }
    /**
     * Grants permission to delete a billing view
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_billing_DeleteBillingView.html
     */
    toDeleteBillingView() {
        return this.to('DeleteBillingView');
    }
    /**
     * Grants permission to delete a billing view resource policy
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_billing_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to perform queries on billing information
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html#user-permissions
     */
    toGetBillingData() {
        return this.to('GetBillingData');
    }
    /**
     * Grants permission to view detailed line item billing information
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html#user-permissions
     */
    toGetBillingDetails() {
        return this.to('GetBillingDetails');
    }
    /**
     * Grants permission to view notifications sent by AWS related to your accounts billing information
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html#user-permissions
     */
    toGetBillingNotifications() {
        return this.to('GetBillingNotifications');
    }
    /**
     * Grants permission to view billing preferences such as reserved instance, savings plans and credits sharing
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html#user-permissions
     */
    toGetBillingPreferences() {
        return this.to('GetBillingPreferences');
    }
    /**
     * Grants permission to get the metadata for a specified billing view
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_billing_GetBillingView.html
     */
    toGetBillingView() {
        return this.to('GetBillingView');
    }
    /**
     * Grants permission to view the account's contract information including the contract number, end-user organization names, PO numbers and if the account is used to service public-sector customers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html#user-permissions
     */
    toGetContractInformation() {
        return this.to('GetContractInformation');
    }
    /**
     * Grants permission to view credits that have been redeemed
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html#user-permissions
     */
    toGetCredits() {
        return this.to('GetCredits');
    }
    /**
     * Grants permission to retrieve the state of the Allow IAM Access billing preference
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html#user-permissions
     */
    toGetIAMAccessPreference() {
        return this.to('GetIAMAccessPreference');
    }
    /**
     * Grants permission to get the resource policy specified billing view
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_billing_GetResourcePolicy.html
     */
    toGetResourcePolicy() {
        return this.to('GetResourcePolicy');
    }
    /**
     * Grants permission to retrieve the account's default Seller of Record
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html#user-permissions
     */
    toGetSellerOfRecord() {
        return this.to('GetSellerOfRecord');
    }
    /**
     * Grants permission to get a list of all your available billing views
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_billing_ListBillingViews.html
     */
    toListBillingViews() {
        return this.to('ListBillingViews');
    }
    /**
     * Grants permission to get the list of source views for a specified billing view
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_billing_ListSourceViewsForBillingView.html
     */
    toListSourceViewsForBillingView() {
        return this.to('ListSourceViewsForBillingView');
    }
    /**
     * Grants permission to get the list of tags for a specified billing view
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_billing_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to set the account's contract information end-user organization names and if the account is used to service public-sector customers
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html#user-permissions
     */
    toPutContractInformation() {
        return this.to('PutContractInformation');
    }
    /**
     * Grants permission to put a billing view resource policy
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_billing_PutResourcePolicy.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to redeem an AWS credit
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html#user-permissions
     */
    toRedeemCredits() {
        return this.to('RedeemCredits');
    }
    /**
     * Grants permission to add tags to a specified billing view
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_billing_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove a tag from a specified billing view
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_billing_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update billing preferences such as reserved instance, savings plans and credits sharing
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html#user-permissions
     */
    toUpdateBillingPreferences() {
        return this.to('UpdateBillingPreferences');
    }
    /**
     * Grants permission to update a billing view
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_billing_UpdateBillingView.html
     */
    toUpdateBillingView() {
        return this.to('UpdateBillingView');
    }
    /**
     * Grants permission to update the Allow IAM Access billing preference
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html#user-permissions
     */
    toUpdateIAMAccessPreference() {
        return this.to('UpdateIAMAccessPreference');
    }
    /**
     * Adds a resource of type billingview to the statement
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBillingview(resourceId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:billing::${account ?? this.defaultAccount}:billingview/${resourceId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateBillingView()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toDeleteBillingView()
     * - .toDeleteResourcePolicy()
     * - .toGetBillingView()
     * - .toGetResourcePolicy()
     * - .toListSourceViewsForBillingView()
     * - .toListTagsForResource()
     * - .toPutResourcePolicy()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateBillingView()
     *
     * Applies to resource types:
     * - billingview
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateBillingView()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [billing](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsbilling.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'billing';
        this.accessLevelList = {
            Write: [
                'CreateBillingView',
                'DeleteBillingView',
                'PutContractInformation',
                'RedeemCredits',
                'UpdateBillingPreferences',
                'UpdateBillingView',
                'UpdateIAMAccessPreference'
            ],
            'Permissions management': [
                'DeleteResourcePolicy',
                'GetResourcePolicy',
                'PutResourcePolicy'
            ],
            Read: [
                'GetBillingData',
                'GetBillingDetails',
                'GetBillingNotifications',
                'GetBillingPreferences',
                'GetBillingView',
                'GetContractInformation',
                'GetCredits',
                'GetIAMAccessPreference',
                'GetSellerOfRecord',
                'ListBillingViews',
                'ListTagsForResource'
            ],
            List: [
                'ListSourceViewsForBillingView'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Billing = Billing;
//# sourceMappingURL=data:application/json;base64,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